-----------------------------------------------------------------------
-- Title      : CPRI Demo Package
-- Project    : CPRI
-----------------------------------------------------------------------
-- File       : cpri_demo_pkg.vhd
-- Author     : Xilinx
-----------------------------------------------------------------------
-- Description: Contains useful definitions for the CPRI core.
-----------------------------------------------------------------------
-- (c) Copyright 2004 - 2018 Xilinx, Inc. All rights reserved.
--
-- This file contains confidential and proprietary information
-- of Xilinx, Inc. and is protected under U.S. and
-- international copyright and other intellectual property
-- laws.
--
-- DISCLAIMER
-- This disclaimer is not a license and does not grant any
-- rights to the materials distributed herewith. Except as
-- otherwise provided in a valid license issued to you by
-- Xilinx, and to the maximum extent permitted by applicable
-- law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
-- WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
-- AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
-- BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
-- INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
-- (2) Xilinx shall not be liable (whether in contract or tort,
-- including negligence, or under any other theory of
-- liability) for any loss or damage of any kind or nature
-- related to, arising under or in connection with these
-- materials, including for any direct, or any indirect,
-- special, incidental, or consequential loss or damage
-- (including loss of data, profits, goodwill, or any type of
-- loss or damage suffered as a result of any action brought
-- by a third party) even if such damage or loss was
-- reasonably foreseeable or Xilinx had been advised of the
-- possibility of the same.
--
-- CRITICAL APPLICATIONS
-- Xilinx products are not designed or intended to be fail-
-- safe, or for use in any application requiring fail-safe
-- performance, such as life-support or safety devices or
-- systems, Class III medical devices, nuclear facilities,
-- applications related to the deployment of airbags, or any
-- other applications that could lead to death, personal
-- injury, or severe property or environmental damage
-- (individually and collectively, "Critical
-- Applications"). Customer assumes the sole risk and
-- liability of any use of Xilinx products in Critical
-- Applications, subject only to applicable laws and
-- regulations governing limitations on product liability.
--
-- THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
-- PART OF THIS FILE AT ALL TIMES.
-----------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

library work;
use work.iic_controller_pkg.all;

package cpri_demo_pkg is

  generic (
    N_CPRI : integer := 1;
    DEMO   : demo_t  := UNKNOWN
  );

--=======================================================================================
-- Constants
--=======================================================================================

  -- Functions to determine bit size depending on demo design
  function speed_nbit( demo : demo_t ) return natural;
  function txalign_nbit( demo : demo_t ) return natural;
  function datapath_nbit( demo : demo_t ) return natural;
  function gtdmonitor_nbit( demo : demo_t ) return natural;

  -- Record signals size variation depending on demo design
  constant NB_SPEED   : natural;
  constant NB_DATAP   : natural;
  constant NB_GTDMON  : natural;
  constant NB_TXALIGN : natural;

-- R21 measurement
  constant FIXED_HARD_FEC_LATENCY     : unsigned(31 downto 0) := x"000009CC"; -- 2508 UI (38 clks x66)
  constant FIXED_GTYE4_LATENCY_64B66B : unsigned(31 downto 0) := x"00000235"; -- 565 UI
  constant FIXED_GTYE4_LATENCY_8B10B  : unsigned(31 downto 0) := x"0000019B"; -- 411.5 UI
  constant PIPELINE_DELAY_FEC         : unsigned(31 downto 0) := x"000004A4"; -- 1188 UI (18 clks x66)
  constant PIPELINE_DELAY_64B66B      : unsigned(31 downto 0) := x"0000035A"; -- 858 UI (13 clks x66)
  constant PIPELINE_DELAY_8B10B       : unsigned(31 downto 0) := x"00000320"; -- 800 UI (10 clks x80)
  constant LINE_RATE_LATENCY_24G      : unsigned(31 downto 0) := x"00000016"; -- 22 UI
  constant LINE_RATE_LATENCY_12G      : unsigned(31 downto 0) := x"0000000A"; -- 10 UI
  constant LINE_RATE_LATENCY_10G      : unsigned(31 downto 0) := x"00000008"; -- 8 UI
  constant LINE_RATE_LATENCY_8G       : unsigned(31 downto 0) := x"00000006"; -- 6 UI

--=======================================================================================
-- Types
--=======================================================================================

  -- Definition of std_logic(_vector) array types
  type mstd_t is array(N_CPRI-1 downto 0) of std_logic;
  
  type sync_slv_00b_t is array(2 downto 0) of std_logic_vector( 0 downto 0);
  type sync_slv_04b_t is array(2 downto 0) of std_logic_vector( 3 downto 0);
  type sync_slv_08b_t is array(2 downto 0) of std_logic_vector( 7 downto 0);
  type sync_slv_10b_t is array(2 downto 0) of std_logic_vector( 9 downto 0);
  type sync_slv_15b_t is array(2 downto 0) of std_logic_vector(14 downto 0);

  -- AXI Bus management Finite-State Machine
  type mgmnt_fsm_t is (IDLE, TO_IDLE, MGMNT_WR0, MGMNT_RD0, MGMNT_RD1, MGMNT_WRA);

  type fsm_axi_rec is
    record
      nstate : mgmnt_fsm_t;
      addr   : std_logic_vector(11 downto 0);
      wdata  : std_logic_vector(31 downto 0);
    end record;

  type fsm_axi_t is array (natural range<>) of fsm_axi_rec;

  -- Initialize available management AXI commands ROM
  function mgmnt_rom_init(part : demo_t) return fsm_axi_t;

  constant MGMNT_CMD_PMA0      : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"020"
                                                , wdata  => x"00000008" );
  constant MGMNT_CMD_24FG      : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00004000" );
  constant MGMNT_CMD_12FG      : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00002000" );
  constant MGMNT_CMD_10FG      : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00001000" );
  constant MGMNT_CMD_8FG       : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00000800" );
  constant MGMNT_CMD_24G       : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00000400" );
  constant MGMNT_CMD_12G       : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00000200" );
  constant MGMNT_CMD_8G        : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00000100" );
  constant MGMNT_CMD_10G       : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00000080" );
  constant MGMNT_CMD_9G        : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00000040" );
  constant MGMNT_CMD_6G        : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00000020" );
  constant MGMNT_CMD_4G        : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00000010" );
  constant MGMNT_CMD_3G        : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00000008" );
  constant MGMNT_CMD_2G        : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00000004" );
  constant MGMNT_CMD_1G        : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"034"
                                                , wdata  => x"00000002" );
  constant MGMNT_CMD_FEC_ERR_0 : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"07C"
                                                , wdata  => x"00000000" );
  constant MGMNT_CMD_FEC_ERR_1 : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"07C"
                                                , wdata  => x"00000001" );
  constant MGMNT_CMD_FEC_ERR_2 : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"07C"
                                                , wdata  => x"00000002" );
  constant MGMNT_CMD_FEC_ERR_3 : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"07C"
                                                , wdata  => x"00000004" );
  constant MGMNT_CMD_FEC_ERR_4 : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"07C"
                                                , wdata  => x"00000008" );
  constant MGMNT_CMD_FEC_ERR_5 : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"07C"
                                                , wdata  => x"0000000F" );
  constant MGMNT_CMD_SOFT_RST  : fsm_axi_rec := ( nstate => TO_IDLE
                                                , addr   => x"038"
                                                , wdata  => x"95CD0000" );

  subtype mgmnt_cmd_index_t is integer range 0 to 10;

  constant MGMNT_INDX_LINE_RATE_0      : mgmnt_cmd_index_t := 0;
  constant MGMNT_INDX_LINE_RATE_1      : mgmnt_cmd_index_t := 1;
  constant MGMNT_INDX_LINE_RATE_2      : mgmnt_cmd_index_t := 2;
  constant MGMNT_INDX_LINE_RATE_3      : mgmnt_cmd_index_t := 3;
  constant MGMNT_INDX_FEC_ERROR_0PPM   : mgmnt_cmd_index_t := 4;
  constant MGMNT_INDX_FEC_ERROR_1e0PPM : mgmnt_cmd_index_t := 5;
  constant MGMNT_INDX_FEC_ERROR_1e1PPM : mgmnt_cmd_index_t := 6;
  constant MGMNT_INDX_FEC_ERROR_1e2PPM : mgmnt_cmd_index_t := 7;
  constant MGMNT_INDX_FEC_ERROR_1e3PPM : mgmnt_cmd_index_t := 8;
  constant MGMNT_INDX_FEC_ERROR_1e4PPM : mgmnt_cmd_index_t := 9;
  constant MGMNT_INDX_SOFT_RESET       : mgmnt_cmd_index_t := 10;

  -- Core Master / Slave control
  type cntrl_rec is
    record
      core_is_master   : std_logic_vector(1 downto 0);
      core_in_loopback : std_logic_vector(1 downto 0);
      pma_loopback     : std_logic_vector(0 downto 0);
      test_start_clk   : std_logic_vector(0 downto 0);
      test_start_eth   : std_logic_vector(0 downto 0);
    end record;

  type cntrl_t is array (N_CPRI-1 downto 0) of cntrl_rec;

  -- R21 Measurement
  type delay_rec is
    record
      b         : unsigned(31 downto 0);
      c         : unsigned(31 downto 0);
      c_tx      : unsigned(31 downto 0);
      d         : unsigned(31 downto 0);
      r21_timer : unsigned(35 downto 0);
    end record;

  type delay_t is array (N_CPRI-1 downto 0) of delay_rec;

  -- GT debug signals
  type gt_rec is
    record
      cplllock        : std_logic_vector( 0 downto 0);
      qplllock        : std_logic_vector( 0 downto 0);
      txresetdone     : std_logic_vector( 0 downto 0);
      rxresetdone     : std_logic_vector( 0 downto 0);
      rxpmaresetdone  : std_logic_vector( 0 downto 0);
      rxdatavalid     : std_logic_vector( 0 downto 0);
      rxheadervalid   : std_logic_vector( 0 downto 0);
      dmonitorout     : std_logic_vector(NB_GTDMON-1 downto 0);
      txdata          : std_logic_vector(NB_DATAP-1  downto 0);
      txheader        : std_logic_vector( 1 downto 0);
      rxdata          : std_logic_vector(NB_DATAP-1  downto 0);
      rxheader        : std_logic_vector( 1 downto 0);
      txphaligndone   : std_logic_vector( 0 downto 0);
      txphinitdone    : std_logic_vector( 0 downto 0);
      txdlysresetdone : std_logic_vector( 0 downto 0);
      rxphaligndone   : std_logic_vector( 0 downto 0);
      rxdlysresetdone : std_logic_vector( 0 downto 0);
      rxsyncdone      : std_logic_vector( 0 downto 0);
      rxgearboxslip   : std_logic_vector( 0 downto 0);
    end record;

  type gt_t is array (N_CPRI-1 downto 0) of gt_rec;

  -- Ethernet data generation and checking
  type eth_rec is
    record
      txd                 : std_logic_vector( 7 downto 0);
      tx_er               : std_logic_vector( 0 downto 0);
      tx_en               : std_logic_vector( 0 downto 0);
      col                 : std_logic_vector( 0 downto 0);
      crs                 : std_logic_vector( 0 downto 0);
      rxd                 : std_logic_vector( 7 downto 0);
      rx_er               : std_logic_vector( 0 downto 0);
      rx_dv               : std_logic_vector( 0 downto 0);
      rx_avail            : std_logic;
      rx_ready            : std_logic;
      rx_fifo_full        : std_logic;
      rx_fifo_almost_full : std_logic;
      tx_count            : std_logic_vector(31 downto 0);
      rx_frame_count      : std_logic_vector(31 downto 0);
      error_count         : std_logic_vector(31 downto 0);
      error               : std_logic_vector( 0 downto 0);
    end record;

  type eth_t is array (N_CPRI-1 downto 0) of eth_rec;

  -- HDLC Data generation and checking
  type hdlc_rec is
    record
      rx_data       : std_logic_vector( 0 downto 0);
      tx_data       : std_logic_vector( 0 downto 0);
      rx_data_valid : std_logic_vector( 0 downto 0);
      tx_enable     : std_logic_vector( 0 downto 0);
      error         : std_logic_vector( 0 downto 0);
      bit_count     : std_logic_vector(31 downto 0);
      error_count   : std_logic_vector(31 downto 0);
    end record;

  type hdlc_t is array (N_CPRI-1 downto 0) of hdlc_rec;

  -- IQ Data generation and checking
  type iq_rec is
    record
      tx                       : std_logic_vector(NB_DATAP-1 downto 0);
      rx                       : std_logic_vector(NB_DATAP-1 downto 0);
      tx_enable                : std_logic_vector( 0 downto 0);
      rx_chk_enable            : std_logic_vector( 0 downto 0);
      basic_frame_first_word   : std_logic_vector( 0 downto 0);
      basic_frame_first_word_r : std_logic;
      error                    : std_logic_vector( 0 downto 0);
      error_count              : std_logic_vector(31 downto 0);
      frame_count              : std_logic_vector(31 downto 0);
    end record;

  type iq_t is array (N_CPRI-1 downto 0) of iq_rec;

  -- Management Interface
  type mgmnt_rec is
    record
      read_req  : std_logic_vector( 0 downto 0);
      write_req : std_logic_vector( 0 downto 0);
      cmd       : std_logic_vector(10 downto 0);
      addr      : std_logic_vector(11 downto 0);
      rd_data   : std_logic_vector(31 downto 0);
      wr_data   : std_logic_vector(31 downto 0);
      state     : mgmnt_fsm_t;
      axi_cmd   : fsm_axi_rec;
    end record;

  type mgmnt_t is array (N_CPRI-1 downto 0) of mgmnt_rec;

  -- AXI Lite Management Interface
  type s_axi_rec is
    record
      aresetn                 : std_logic;
      awaddr                  : std_logic_vector(11 downto 0);
      awvalid                 : std_logic_vector( 0 downto 0);
      awready                 : std_logic_vector( 0 downto 0);
      wdata                   : std_logic_vector(31 downto 0);
      wvalid                  : std_logic_vector( 0 downto 0);
      wready                  : std_logic_vector( 0 downto 0);
      bresp                   : std_logic_vector( 1 downto 0);
      bvalid                  : std_logic_vector( 0 downto 0);
      bready                  : std_logic_vector( 0 downto 0);
      araddr                  : std_logic_vector(11 downto 0);
      arvalid                 : std_logic_vector( 0 downto 0);
      arready                 : std_logic_vector( 0 downto 0);
      rdata                   : std_logic_vector(31 downto 0);
      rresp                   : std_logic_vector( 1 downto 0);
      rvalid                  : std_logic_vector( 0 downto 0);
      rready                  : std_logic_vector( 0 downto 0);
      reset_request           : std_logic;
      reset_request_in        : std_logic;
      reset_acknowledge_out   : std_logic_vector( 0 downto 0);
      sdi_request_in          : std_logic_vector( 0 downto 0);
      sdi_request_out         : std_logic_vector( 0 downto 0);
      local_los               : std_logic_vector( 0 downto 0);
      local_lof               : std_logic_vector( 0 downto 0);
      local_rai               : std_logic_vector( 0 downto 0);
      remote_los              : std_logic_vector( 0 downto 0);
      remote_lof              : std_logic_vector( 0 downto 0);
      remote_rai              : std_logic_vector( 0 downto 0);
      fifo_transit_time       : std_logic_vector(13 downto 0);
      coarse_timer_value      : std_logic_vector(17 downto 0);
      barrel_shift_value      : std_logic_vector( 6 downto 0);
      tx_gb_latency_value     : std_logic_vector(15 downto 0);
      rx_gb_latency_value     : std_logic_vector(15 downto 0);
	  hfec_fifo_latency_value : std_logic_vector(15 downto 0);
    end record;

  type s_axi_t is array (N_CPRI-1 downto 0) of s_axi_rec;

  -- CPRI status
  type stat_rec is
    record
      alarm          : std_logic_vector( 0 downto 0);
      speed4         : std_logic_vector( 3 downto 0);
      speed5         : std_logic_vector( 4 downto 0);
      speed          : std_logic_vector(NB_SPEED-1 downto 0);
      code           : std_logic_vector( 3 downto 0);
      fec_line_rate  : std_logic_vector( 0 downto 0);
      rx_delay_value : std_logic_vector( 6 downto 0);
    end record;

  type stat_t is array (N_CPRI-1 downto 0) of stat_rec;

  -- FSM State flags
  type state_rec is
    record
      reset   : std_logic_vector(0 downto 0);
      l1sync  : std_logic_vector(0 downto 0);
      proto   : std_logic_vector(0 downto 0);
      cm      : std_logic_vector(0 downto 0);
      passive : std_logic_vector(0 downto 0);
      vs      : std_logic_vector(0 downto 0);
      operate : std_logic_vector(0 downto 0);
    end record;

  type state_t is array (N_CPRI-1 downto 0) of state_rec;

  -- Synchronization
  type sync_rec is
    record
      nodebfn_tx_strobe   : std_logic_vector( 0 downto 0);
      nodebfn_tx_nr       : std_logic_vector(11 downto 0);
      nodebfn_rx_strobe   : std_logic_vector( 0 downto 0);
      nodebfn_rx_nr       : std_logic_vector(11 downto 0);
      nodebfn_rx_nr_store : std_logic_vector(11 downto 0); 
      bfn_tx_strobe       : std_logic;
      bfn_tx_nr           : std_logic_vector(11 downto 0);
      bfn_tx_strobe_slv   : std_logic;
      bfn_tx_nr_slv       : std_logic_vector(11 downto 0);
    end record;

  type sync_t is array (N_CPRI-1 downto 0) of sync_rec;

  -- TX Alignment
  type tx_align_rec is
    record
      txphalign_in    : std_logic_vector(NB_TXALIGN-1 downto 0);
      txphalign_out   : std_logic_vector(NB_TXALIGN-1 downto 0);
      txdlysreset_in  : std_logic_vector(NB_TXALIGN-1 downto 0);
      txdlysreset_out : std_logic_vector(NB_TXALIGN-1 downto 0);
      txphinit_in     : std_logic_vector(NB_TXALIGN-1 downto 0);
      txphinit_out    : std_logic_vector(NB_TXALIGN-1 downto 0);
      txdlyen         : std_logic_vector(NB_TXALIGN-1 downto 0);
    end record;

  type tx_align_t is array (N_CPRI-1 downto 0) of tx_align_rec;

  -- Vendor specific data generation and checking
  type vendor_rec is
    record
      tx_xs          : std_logic_vector(  1 downto 0);
      tx_ns          : std_logic_vector(  5 downto 0);
      tx_data        : std_logic_vector(127 downto 0);
      rx_xs          : std_logic_vector(  1 downto 0);
      rx_ns          : std_logic_vector(  5 downto 0);
      rx_data        : std_logic_vector(127 downto 0);
      vs_error       : std_logic_vector(  0 downto 0);
      vs_word_count  : std_logic_vector( 31 downto 0);
      vs_error_count : std_logic_vector( 31 downto 0);
    end record;

  type vendor_t is array (N_CPRI-1 downto 0) of vendor_rec;

  -- VIO Reset control
  type vio_rec is
    record
      reset     : std_logic_vector(0 downto 0);
      reset_req : std_logic_vector(0 downto 0);
      reset_p   : std_logic;
      reset_sr  : std_logic_vector(4 downto 0);
    end record;

  type vio_t is array (N_CPRI-1 downto 0) of vio_rec;

  -- Hard/Soft FEC Interface
  type fec_rec is
    record
      rx_rsfec_enable         : std_logic;
      rx_serdes_data          : std_logic_vector(63 downto 0);
      rx_serdes_head          : std_logic_vector( 1 downto 0);
      rx_gbx_slip             : std_logic;
      cdc_reset               : std_logic;
      fifo_fill_level         : std_logic_vector( 6 downto 0);
      average                 : std_logic_vector(15 downto 0);
      average_rdy             : std_logic;
      average_ack             : std_logic;
      fifo_error              : std_logic;
      cdc_rxdata              : std_logic_vector(63 downto 0);
      cdc_rxheader            : std_logic_vector( 1 downto 0);
      pcs_rxdata              : std_logic_vector(63 downto 0);
      pcs_rxheader            : std_logic_vector( 1 downto 0);
      pcs_rxheadervalid       : std_logic;
      stat_rx_delay_value     : std_logic_vector( 6 downto 0);
      stat_rx_align_status    : std_logic;
      stat_cw_inc             : std_logic;
      stat_corrected_cw_inc   : std_logic;
      stat_uncorrected_cw_inc : std_logic;
	  hfec_fifo_latency       : std_logic_vector(15 downto 0);
	  hfec_fifo_latency_rdy   : std_logic;
	  hfec_fifo_latency_ack   : std_logic;
    end record;

  type fec_t is array (N_CPRI-1 downto 0) of fec_rec;

  -- Hard-FEC wrapper demo design types
  type cpri_sync_slv_00b_t is array (N_CPRI-1 downto 0) of sync_slv_00b_t;
  type cpri_sync_slv_04b_t is array (N_CPRI-1 downto 0) of sync_slv_04b_t;
  type cpri_sync_slv_10b_t is array (N_CPRI-1 downto 0) of sync_slv_10b_t;
  type cpri_sync_slv_15b_t is array (N_CPRI-1 downto 0) of sync_slv_15b_t;

--=======================================================================================
-- Function prototypes
--=======================================================================================

  -- One-hot to 4bits encoder
  function onehot_to_4b(signal speed : std_logic_vector) return std_logic_vector;

  -- One-hot to 5bits encoder
  function onehot_to_5b(signal speed : std_logic_vector) return std_logic_vector;

  -- Delay step B calculation, as indicated at CPRI product guide
  function b_step_delay(signal speed                   : std_logic_vector;
                        signal hfec_fifo_latency_value : std_logic_vector;
                        signal barrel_shift_value      : std_logic_vector) return unsigned;

  -- Delay step C calculation, as indicated at CPRI product guide
  function c_step_delay(signal speed               : std_logic_vector;
                        signal fifo_transit_time   : std_logic_vector;
                        signal rx_gb_latency_value : std_logic_vector) return unsigned;

  -- Delay step CTX calculation, as indicated at CPRI product guide
  function ctx_step_delay(signal speed               : std_logic_vector;
                          signal tx_gb_latency_value : std_logic_vector) return unsigned;

  -- Delay step D calculation, as indicated at CPRI product guide
  function d_step_delay(signal speed : std_logic_vector;
                        signal b     : unsigned;
                        signal c     : unsigned;
                        signal c_tx  : unsigned) return unsigned;

  -- Delay step R21 calculation, as indicated at CPRI product guide
  function r21_step_delay(signal speed              : std_logic_vector;
                          signal d                  : unsigned;
                          signal core_is_master     : std_logic;
                          signal coarse_timer_value : std_logic_vector) return unsigned;

--=======================================================================================
-- Procedure prototypes
--=======================================================================================

  -- Reset Management I/F
  procedure MGMNT_FSM_RST(constant nstate      : mgmnt_fsm_t;
                          signal s_axi_awaddr  : out std_logic_vector(11 downto 0);
                          signal s_axi_awvalid : out std_logic_vector( 0 downto 0);
                          signal s_axi_wdata   : out std_logic_vector(31 downto 0);
                          signal s_axi_bready  : out std_logic_vector( 0 downto 0);
                          signal s_axi_wvalid  : out std_logic_vector( 0 downto 0);
                          signal s_axi_araddr  : out std_logic_vector(11 downto 0);
                          signal s_axi_arvalid : out std_logic_vector( 0 downto 0);
                          signal s_axi_rready  : out std_logic_vector( 0 downto 0);
                          signal mgmnt_state   : out mgmnt_fsm_t);

  -- Management I/F FSM - INIT2 state
  procedure MGMNT_FSM_TO_IDLE(signal s_axi_bvalid : in  std_logic_vector( 0 downto 0);
                              signal s_axi_bready : out std_logic_vector( 0 downto 0);
                              signal mgmnt_state  : out mgmnt_fsm_t);

  -- Management I/F FSM - MGMNT_PMA0 state
  procedure MGMNT_FSM_WREG(signal axi_data      : in  fsm_axi_rec;
                           signal s_axi_wready  : in  std_logic_vector( 0 downto 0);
                           signal s_axi_awaddr  : out std_logic_vector(11 downto 0);
                           signal s_axi_awvalid : out std_logic_vector( 0 downto 0);
                           signal s_axi_wdata   : out std_logic_vector(31 downto 0);
                           signal s_axi_wvalid  : out std_logic_vector( 0 downto 0);
                           signal mgmnt_state   : out mgmnt_fsm_t);

  -- Management I/F FSM - IDLE state
  procedure MGMNT_FSM_IDLE(signal mgmnt_cmd_lst : in  fsm_axi_t;
                           signal write_req     : in  std_logic_vector;
                           signal read_req      : in  std_logic_vector;
                           signal mgmnt_cmd     : in  std_logic_vector;
                           signal pma_loopback  : in  std_logic_vector;
                           signal s_axi_wready  : in  std_logic_vector( 0 downto 0);
                           signal s_axi_awaddr  : out std_logic_vector(11 downto 0);
                           signal s_axi_awvalid : out std_logic_vector( 0 downto 0);
                           signal s_axi_wdata   : out std_logic_vector(31 downto 0);
                           signal s_axi_wvalid  : out std_logic_vector( 0 downto 0);
                           signal s_axi_bready  : out std_logic_vector( 0 downto 0);
                           signal s_axi_araddr  : out std_logic_vector(11 downto 0);
                           signal s_axi_arvalid : out std_logic_vector( 0 downto 0);
                           signal s_axi_rready  : out std_logic_vector( 0 downto 0);
                           signal mgmnt_state   : out mgmnt_fsm_t;
                           signal axi_cmd       : out fsm_axi_rec);

  -- Management I/F FSM - MGMNT_WR0 state
  procedure MGMNT_FSM_WR0(signal mgmnt_addr    : in  std_logic_vector;
                          signal mgmnt_wr_data : in  std_logic_vector;
                          signal s_axi_wready  : in  std_logic_vector( 0 downto 0);
                          signal s_axi_awaddr  : out std_logic_vector(11 downto 0);
                          signal s_axi_awvalid : out std_logic_vector( 0 downto 0);
                          signal s_axi_wdata   : out std_logic_vector(31 downto 0);
                          signal s_axi_wvalid  : out std_logic_vector( 0 downto 0);
                          signal mgmnt_state   : out mgmnt_fsm_t);

  -- Management I/F FSM - MGMNT_RD0 state
  procedure MGMNT_FSM_RD0(signal mgmnt_addr    : in  std_logic_vector;
                          signal s_axi_arready : in  std_logic_vector( 0 downto 0);
                          signal s_axi_araddr  : out std_logic_vector(11 downto 0);
                          signal s_axi_arvalid : out std_logic_vector( 0 downto 0);
                          signal s_axi_rready  : out std_logic_vector( 0 downto 0);
                          signal mgmnt_state   : out mgmnt_fsm_t);

  -- Management I/F FSM - MGMNT_RD1 state
  procedure MGMNT_FSM_RD1(signal s_axi_rdata   : in  std_logic_vector(31 downto 0);
                          signal s_axi_rvalid  : in  std_logic_vector( 0 downto 0);
                          signal s_axi_arvalid : out std_logic_vector( 0 downto 0);
                          signal s_axi_rready  : out std_logic_vector( 0 downto 0);
                          signal mgmnt_rd_data : out std_logic_vector;
                          signal mgmnt_state   : out mgmnt_fsm_t);

  -- Recovered clock generator
  procedure RECCLK_CALC(signal speed  : in    std_logic_vector;
                        signal recclk : inout std_logic;
                        signal count  : inout unsigned);

end cpri_demo_pkg;

package body cpri_demo_pkg is

--=======================================================================================
-- Constants
--=======================================================================================

-- Record signals size variation depending on demo design
  constant NB_SPEED   : natural := speed_nbit( DEMO );
  constant NB_TXALIGN : natural := txalign_nbit( DEMO );
  constant NB_DATAP   : natural := datapath_nbit( DEMO );
  constant NB_GTDMON  : natural := gtdmonitor_nbit( DEMO );

--=======================================================================================
-- Function definitions
--=======================================================================================

  -- Functions to determine bit size depending on demo design
  function speed_nbit( demo : demo_t ) return natural is
  begin
    case demo is
      when kcu105    =>
        return 8;
      when zcu102    =>
        return 10;
      when vcu108_ms =>
        return 10;
      when others    =>
        return 15;
    end case;
  end speed_nbit;

  function txalign_nbit( demo : demo_t ) return natural is
  begin
    case demo is
      when vcu118_hfw =>
        return 4;
      when others     =>
        return 3;
    end case;
  end txalign_nbit;

  function datapath_nbit( demo : demo_t ) return natural is
  begin
    case demo is
      when kcu105    =>
        return 32;
      when zcu102    =>
        return 32;
      when vcu108_ms =>
        return 32;
      when others    =>
        return 64;
      end case;
  end datapath_nbit;

  function gtdmonitor_nbit( demo : demo_t ) return natural is
  begin
    case demo is
      when kcu105    =>
        return 17;
      when vcu108    =>
        return 17;
      when vcu108_ms =>
        return 17;
      when others    =>
        return 16;
    end case;
  end gtdmonitor_nbit;

  -- Initialize available management AXI commands ROM
  function mgmnt_rom_init(part : demo_t)
  return fsm_axi_t is
    variable mgmnt_rom : fsm_axi_t(3 downto 0);
  begin
    case part is
      when kcu105    =>
        mgmnt_rom := ( 0 => MGMNT_CMD_10G,
                       1 => MGMNT_CMD_9G,
                       2 => MGMNT_CMD_6G,
                       3 => MGMNT_CMD_4G );
      when vcu108 =>
        mgmnt_rom := ( 0 => MGMNT_CMD_24FG,
                       1 => MGMNT_CMD_24G,
                       2 => MGMNT_CMD_12FG,
                       3 => MGMNT_CMD_12G );
      when vcu108_ms =>
        mgmnt_rom := ( 0 => MGMNT_CMD_12G,
                       1 => MGMNT_CMD_10G,
                       2 => MGMNT_CMD_9G,
                       3 => MGMNT_CMD_8G );  
      when vcu118    =>
        mgmnt_rom := ( 0 => MGMNT_CMD_24FG,
                       1 => MGMNT_CMD_24G,
                       2 => MGMNT_CMD_24G,
                       3 => MGMNT_CMD_24G );
      when vcu118_hfw =>
        mgmnt_rom := ( 0 => MGMNT_CMD_24FG,
                       1 => MGMNT_CMD_24G,
                       2 => MGMNT_CMD_24G,
                       3 => MGMNT_CMD_24G );
      when zcu102     =>
        mgmnt_rom := ( 0 => MGMNT_CMD_12G,
                       1 => MGMNT_CMD_10G,
                       2 => MGMNT_CMD_9G,
                       3 => MGMNT_CMD_8G );
      when others =>
        mgmnt_rom := ( 0 => MGMNT_CMD_24G,
                       1 => MGMNT_CMD_12G,
                       2 => MGMNT_CMD_8G,
                       3 => MGMNT_CMD_10G );
    end case;

    return mgmnt_rom;
  end mgmnt_rom_init;

  -- One-hot to 4bits encoder
  function onehot_to_4b(signal speed : std_logic_vector)
  return std_logic_vector is
    variable speed_tmp : unsigned(14 downto 0) := (others => '0');
    variable res : std_logic_vector(3 downto 0) := (others => '0');
  begin
    -- Make array expression to be locally static, line rate speed size adjustment
    speed_tmp := resize(unsigned(speed), speed_tmp'length);
    -- 4-bit encode
    case speed_tmp is
      when "000000000000001" => res := "0001"; -- 1  (614 M)
      when "000000000000010" => res := "0010"; -- 2  (1228 M)
      when "000000000000100" => res := "0011"; -- 3  (2457 M)
      when "000000000001000" => res := "0100"; -- 4  (3072 M)
      when "000000000010000" => res := "0101"; -- 5  (4915 M)
      when "000000000100000" => res := "0110"; -- 6  (6144 M)
      when "000000001000000" => res := "0111"; -- 7  (9830 M)
      when "000000010000000" => res := "1000"; -- 8  (10137 M)
      when "000000100000000" => res := "1001"; -- 9  (8110  M)
      when "000001000000000" => res := "1010"; -- 10 (12165 M)
      when "000010000000000" => res := "1011"; -- 11 (24330 M)
      when "000100000000000" => res := "1100"; -- 12 (8110  M FEC)
      when "001000000000000" => res := "1101"; -- 13 (10137 M FEC)
      when "010000000000000" => res := "1110"; -- 14 (12165 M FEC)
      when "100000000000000" => res := "1111"; -- 15 (24330 M FEC)
      when others            => res := "0000";
    end case;

    return res;
  end onehot_to_4b;

  -- One-hot to 5bits encoder
  function onehot_to_5b(signal speed : std_logic_vector)
  return std_logic_vector is
    variable speed_tmp : unsigned(14 downto 0) := (others => '0');
    variable res : std_logic_vector(4 downto 0) := (others => '0');
  begin
    -- Make array expression to be locally static, line rate speed size adjustment
    speed_tmp := resize(unsigned(speed), speed_tmp'length);
    -- 5-bit encode
    case speed_tmp is
      when "000000000000001" => res := "00001"; -- 1  (614 M)
      when "000000000000010" => res := "00010"; -- 2  (1228 M)
      when "000000000000100" => res := "00011"; -- 3  (2457 M)
      when "000000000001000" => res := "00100"; -- 4  (3072 M)
      when "000000000010000" => res := "00101"; -- 5  (4915 M)
      when "000000000100000" => res := "00110"; -- 6  (6144 M)
      when "000000001000000" => res := "00111"; -- 7  (9830 M)
      when "000000010000000" => res := "01000"; -- 8  (10137 M)
      when "000000100000000" => res := "01001"; -- 9  (8110  M)
      when "000001000000000" => res := "01010"; -- 10 (12165 M)
      when "000010000000000" => res := "01011"; -- 11 (24330 M)
      when "000100000000000" => res := "01100"; -- 12 (8110  M FEC)
      when "001000000000000" => res := "01101"; -- 13 (10137 M FEC)
      when "010000000000000" => res := "01110"; -- 14 (12165 M FEC)
      when "100000000000000" => res := "01111"; -- 15 (24330 M FEC)
      when others            => res := "00000";
    end case;

    return res;
  end onehot_to_5b;

  -- Delay step B calculation, as indicated at CPRI product guide
  function b_step_delay(signal speed                   : std_logic_vector;
                        signal hfec_fifo_latency_value : std_logic_vector;
                        signal barrel_shift_value      : std_logic_vector)
  return unsigned is
    variable b : unsigned(31 downto 0) := (others => '0');
    variable speed_tmp : unsigned(14 downto 0) := (others => '0');
  begin
    -- Make array expression to be locally static, line rate speed size adjustment
    speed_tmp := resize(unsigned(speed), speed_tmp'length);
    -- R21 delay calculation based on actual line rate
    case speed_tmp is
      when "100000000000000" =>
        b := FIXED_GTYE4_LATENCY_64B66B + LINE_RATE_LATENCY_24G + FIXED_HARD_FEC_LATENCY + (unsigned(hfec_fifo_latency_value)*66);
      when "010000000000000" =>
        b := FIXED_GTYE4_LATENCY_64B66B + LINE_RATE_LATENCY_12G + FIXED_HARD_FEC_LATENCY + (unsigned(hfec_fifo_latency_value)*66);
      when "001000000000000" =>
        b := FIXED_GTYE4_LATENCY_64B66B + LINE_RATE_LATENCY_10G + FIXED_HARD_FEC_LATENCY + (unsigned(hfec_fifo_latency_value)*66);
      when "000100000000000" =>
        b := FIXED_GTYE4_LATENCY_64B66B + LINE_RATE_LATENCY_8G  + FIXED_HARD_FEC_LATENCY + (unsigned(hfec_fifo_latency_value)*66);
      when "000010000000000" =>
        b := FIXED_GTYE4_LATENCY_64B66B + LINE_RATE_LATENCY_24G;
      when "000001000000000" =>
        b := FIXED_GTYE4_LATENCY_64B66B + LINE_RATE_LATENCY_12G;
      when "000000010000000" =>
        b := FIXED_GTYE4_LATENCY_64B66B + LINE_RATE_LATENCY_10G;
      when "000000100000000" =>
        b := FIXED_GTYE4_LATENCY_64B66B + LINE_RATE_LATENCY_8G;
      when others =>
        b := FIXED_GTYE4_LATENCY_8B10B + unsigned(barrel_shift_value);
    end case;
  
    return b;
  end b_step_delay;
  
  -- Delay step C calculation, as indicated at CPRI product guide
  function c_step_delay(signal speed               : std_logic_vector;
                        signal fifo_transit_time   : std_logic_vector;
                        signal rx_gb_latency_value : std_logic_vector)
  return unsigned is
    variable c : unsigned(31 downto 0) := (others => '0');
    variable speed_tmp : unsigned(14 downto 0) := (others => '0');
  begin
    -- Make array expression to be locally static, line rate speed size adjustment
    speed_tmp := resize(unsigned(speed), speed_tmp'length);
    -- R21 delay calculation based on actual line rate
    if speed_tmp(14 downto 7) /= "00000000" then
      c := "0000" & (((unsigned(fifo_transit_time)*66)/128)
                  + (unsigned(rx_gb_latency_value(15 downto 3))));
    else
      c := "0000" & ((unsigned(fifo_transit_time)*80)/128);
    end if;
  
    return c;
  end c_step_delay;
  
  -- Delay step CTX calculation, as indicated at CPRI product guide
  function ctx_step_delay(signal speed               : std_logic_vector;
                          signal tx_gb_latency_value : std_logic_vector)
  return unsigned is
    variable c_tx : unsigned(31 downto 0) := (others => '0');
    variable speed_tmp : unsigned(14 downto 0) := (others => '0');
  begin
    -- Make array expression to be locally static, line rate speed size adjustment
    speed_tmp := resize(unsigned(speed), speed_tmp'length);
    -- R21 delay calculation based on actual line rate
    if speed_tmp(14 downto 7) /= "00000000" then
      c_tx := x"0000" & "000" & (unsigned(tx_gb_latency_value(15 downto 3)));
    else
      c_tx := (others => '0');
    end if;
  
    return c_tx;
  end ctx_step_delay;
  
  -- Delay step D calculation, as indicated at CPRI product guide
  function d_step_delay(signal speed : std_logic_vector;
                        signal b     : unsigned;
                        signal c     : unsigned;
                        signal c_tx  : unsigned)
  return unsigned is
    variable d : unsigned(31 downto 0) := (others => '0');
    variable speed_tmp : unsigned(14 downto 0) := (others => '0');
  begin
    -- Make array expression to be locally static, line rate speed size adjustment
    speed_tmp := resize(unsigned(speed), speed_tmp'length);
    -- R21 delay calculation based on actual line rate
    if speed_tmp(14 downto 11) /= "0000" then
      d := b + c + c_tx + PIPELINE_DELAY_FEC;
    elsif speed_tmp(10 downto 7) /= "0000" then
      d := b + c + c_tx + PIPELINE_DELAY_64B66B;
    else
      d := b + c + c_tx + PIPELINE_DELAY_8B10B;
    end if;
  
    return d;
  end d_step_delay;
  
  -- Delay step R21 calculation, as indicated at CPRI product guide
  function r21_step_delay(signal speed              : std_logic_vector;
                          signal d                  : unsigned;
                          signal core_is_master     : std_logic;
                          signal coarse_timer_value : std_logic_vector)
  return unsigned is
    variable r21_timer : unsigned(35 downto 0) := (others => '0');
    variable speed_tmp : unsigned(14 downto 0) := (others => '0');
  begin
    -- Make array expression to be locally static, line rate speed size adjustment
    speed_tmp := resize(unsigned(speed), speed_tmp'length);
    -- R21 delay calculation based on actual line rate
    if speed_tmp(14 downto 7) /= "00000000" then
      if core_is_master = '0' then
        r21_timer := (unsigned(coarse_timer_value)*66) + d;
      else
        r21_timer := (unsigned(coarse_timer_value)*66) - d;
      end if;
    else
      if core_is_master = '0' then
        r21_timer := (unsigned(coarse_timer_value)*80) + d;
      else
        r21_timer := (unsigned(coarse_timer_value)*80) - d;
      end if;
    end if;
  
    return r21_timer;
  end r21_step_delay;

--=======================================================================================
-- Procedure definitions
--=======================================================================================

  -- Reset Management I/F
  procedure MGMNT_FSM_RST(constant nstate      : mgmnt_fsm_t;
                          signal s_axi_awaddr  : out std_logic_vector(11 downto 0);
                          signal s_axi_awvalid : out std_logic_vector( 0 downto 0);
                          signal s_axi_wdata   : out std_logic_vector(31 downto 0);
                          signal s_axi_bready  : out std_logic_vector( 0 downto 0);
                          signal s_axi_wvalid  : out std_logic_vector( 0 downto 0);
                          signal s_axi_araddr  : out std_logic_vector(11 downto 0);
                          signal s_axi_arvalid : out std_logic_vector( 0 downto 0);
                          signal s_axi_rready  : out std_logic_vector( 0 downto 0);
                          signal mgmnt_state   : out mgmnt_fsm_t) is
  begin
    s_axi_awaddr  <= (others => '0');
    s_axi_awvalid <= (others => '0');
    s_axi_wdata   <= (others => '0');
    s_axi_wvalid  <= (others => '0');
    s_axi_bready  <= (others => '0');
    s_axi_araddr  <= (others => '0');
    s_axi_arvalid <= (others => '0');
    s_axi_rready  <= (others => '0');
    mgmnt_state   <= nstate;
  end;

  -- Management I/F FSM - INIT2 state
  procedure MGMNT_FSM_TO_IDLE(signal s_axi_bvalid : in  std_logic_vector( 0 downto 0);
                              signal s_axi_bready : out std_logic_vector( 0 downto 0);
                              signal mgmnt_state  : out mgmnt_fsm_t) is
  begin
    -- Response phase
    if(s_axi_bvalid(0) = '1') then
      mgmnt_state  <= IDLE;
      s_axi_bready <= (others => '1');
    end if;
  end;

  -- Management I/F FSM - IDLE state
  procedure MGMNT_FSM_IDLE(signal mgmnt_cmd_lst : in  fsm_axi_t;
                           signal write_req     : in  std_logic_vector;
                           signal read_req      : in  std_logic_vector;
                           signal mgmnt_cmd     : in  std_logic_vector;
                           signal pma_loopback  : in  std_logic_vector;
                           signal s_axi_wready  : in  std_logic_vector( 0 downto 0);
                           signal s_axi_awaddr  : out std_logic_vector(11 downto 0);
                           signal s_axi_awvalid : out std_logic_vector( 0 downto 0);
                           signal s_axi_wdata   : out std_logic_vector(31 downto 0);
                           signal s_axi_wvalid  : out std_logic_vector( 0 downto 0);
                           signal s_axi_bready  : out std_logic_vector( 0 downto 0);
                           signal s_axi_araddr  : out std_logic_vector(11 downto 0);
                           signal s_axi_arvalid : out std_logic_vector( 0 downto 0);
                           signal s_axi_rready  : out std_logic_vector( 0 downto 0);
                           signal mgmnt_state   : out mgmnt_fsm_t;
                           signal axi_cmd       : out fsm_axi_rec) is
  begin
    s_axi_awaddr  <= (others => '0');
    s_axi_awvalid <= (others => '0');
    s_axi_wdata   <= (others => '0');
    s_axi_wvalid  <= (others => '0');
    s_axi_bready  <= (others => '0');
    s_axi_araddr  <= (others => '0');
    s_axi_arvalid <= (others => '0');
    s_axi_rready  <= (others => '0');
    if (write_req(0) = '1') then
      mgmnt_state <= MGMNT_WR0;
    end if;
    if (read_req(0) = '1') then
      mgmnt_state <= MGMNT_RD0;
    end if;
    if (pma_loopback(0) = '1') then
      mgmnt_state <= MGMNT_WRA;
      axi_cmd     <= MGMNT_CMD_PMA0;
    end if;
    if (mgmnt_cmd(MGMNT_INDX_LINE_RATE_0) = '1') then
      mgmnt_state <= MGMNT_WRA;
      axi_cmd     <= mgmnt_cmd_lst(0);
    end if;
    if (mgmnt_cmd(MGMNT_INDX_LINE_RATE_1) = '1') then
      mgmnt_state <= MGMNT_WRA;
      axi_cmd     <= mgmnt_cmd_lst(1);
    end if;
    if (mgmnt_cmd(MGMNT_INDX_LINE_RATE_2) = '1') then
      mgmnt_state <= MGMNT_WRA;
      axi_cmd     <= mgmnt_cmd_lst(2);
    end if;
    if (mgmnt_cmd(MGMNT_INDX_LINE_RATE_3) = '1') then
      mgmnt_state <= MGMNT_WRA;
      axi_cmd     <= mgmnt_cmd_lst(3);
    end if;
    if (mgmnt_cmd(MGMNT_INDX_FEC_ERROR_0PPM) = '1') then
      mgmnt_state <= MGMNT_WRA;
      axi_cmd     <= MGMNT_CMD_FEC_ERR_0;
    end if;
    if (mgmnt_cmd(MGMNT_INDX_FEC_ERROR_1e0PPM) = '1') then
      mgmnt_state <= MGMNT_WRA;
      axi_cmd     <= MGMNT_CMD_FEC_ERR_1;
    end if;
    if (mgmnt_cmd(MGMNT_INDX_FEC_ERROR_1e1PPM) = '1') then
      mgmnt_state <= MGMNT_WRA;
      axi_cmd     <= MGMNT_CMD_FEC_ERR_2;
    end if;
    if (mgmnt_cmd(MGMNT_INDX_FEC_ERROR_1e2PPM) = '1') then
      mgmnt_state <= MGMNT_WRA;
      axi_cmd     <= MGMNT_CMD_FEC_ERR_3;
    end if;
    if (mgmnt_cmd(MGMNT_INDX_FEC_ERROR_1e3PPM) = '1') then
      mgmnt_state <= MGMNT_WRA;
      axi_cmd     <= MGMNT_CMD_FEC_ERR_4;
    end if;
    if (mgmnt_cmd(MGMNT_INDX_FEC_ERROR_1e4PPM) = '1') then
      mgmnt_state <= MGMNT_WRA;
      axi_cmd     <= MGMNT_CMD_FEC_ERR_5;
    end if;
    if (mgmnt_cmd(MGMNT_INDX_SOFT_RESET) = '1') then
      mgmnt_state <= MGMNT_WRA;
      axi_cmd     <= MGMNT_CMD_SOFT_RST;
    end if;
  end;

  -- Management I/F FSM - MGMNT_PMA0 state
  procedure MGMNT_FSM_WREG(signal axi_data      : in  fsm_axi_rec;
                           signal s_axi_wready  : in  std_logic_vector( 0 downto 0);
                           signal s_axi_awaddr  : out std_logic_vector(11 downto 0);
                           signal s_axi_awvalid : out std_logic_vector( 0 downto 0);
                           signal s_axi_wdata   : out std_logic_vector(31 downto 0);
                           signal s_axi_wvalid  : out std_logic_vector( 0 downto 0);
                           signal mgmnt_state   : out mgmnt_fsm_t) is
  begin
    -- Write address phase
    s_axi_awaddr  <= axi_data.addr;
    s_axi_awvalid <= (others => '1');
    s_axi_wdata   <= axi_data.wdata;
    s_axi_wvalid  <= (others => '1');

    if (s_axi_wready(0) = '1') then
      mgmnt_state   <= axi_data.nstate;
      s_axi_awvalid <= (others => '0');
      s_axi_wvalid  <= (others => '0');
    end if;
  end;

  -- Management I/F FSM - MGMNT_WR0 state
  procedure MGMNT_FSM_WR0(signal mgmnt_addr    : in  std_logic_vector;
                          signal mgmnt_wr_data : in  std_logic_vector;
                          signal s_axi_wready  : in  std_logic_vector( 0 downto 0);
                          signal s_axi_awaddr  : out std_logic_vector(11 downto 0);
                          signal s_axi_awvalid : out std_logic_vector( 0 downto 0);
                          signal s_axi_wdata   : out std_logic_vector(31 downto 0);
                          signal s_axi_wvalid  : out std_logic_vector( 0 downto 0);
                          signal mgmnt_state   : out mgmnt_fsm_t) is
  begin
    -- Write address phase
    s_axi_awaddr  <= mgmnt_addr;
    s_axi_awvalid <= (others => '1');
    s_axi_wdata   <= mgmnt_wr_data;
    s_axi_wvalid  <= (others => '1');
    
    if (s_axi_wready(0) = '1') then
      mgmnt_state   <= TO_IDLE;
      s_axi_awvalid <= (others => '0');
      s_axi_wvalid  <= (others => '0');
    end if;
  end;

  -- Management I/F FSM - MGMNT_RD0 state
  procedure MGMNT_FSM_RD0(signal mgmnt_addr    : in  std_logic_vector;
                          signal s_axi_arready : in  std_logic_vector( 0 downto 0);
                          signal s_axi_araddr  : out std_logic_vector(11 downto 0);
                          signal s_axi_arvalid : out std_logic_vector( 0 downto 0);
                          signal s_axi_rready  : out std_logic_vector( 0 downto 0);
                          signal mgmnt_state   : out mgmnt_fsm_t) is
  begin
    -- Read address phase
    s_axi_araddr  <= mgmnt_addr;
    s_axi_arvalid <= (others => '1');
    s_axi_rready  <= (others => '1');
    
    if s_axi_arready(0) = '1' then
      mgmnt_state <= MGMNT_RD1;
    end if;
  end;

  -- Management I/F FSM - MGMNT_RD1 state
  procedure MGMNT_FSM_RD1(signal s_axi_rdata   : in  std_logic_vector(31 downto 0);
                          signal s_axi_rvalid  : in  std_logic_vector( 0 downto 0);
                          signal s_axi_arvalid : out std_logic_vector( 0 downto 0);
                          signal s_axi_rready  : out std_logic_vector( 0 downto 0);
                          signal mgmnt_rd_data : out std_logic_vector;
                          signal mgmnt_state   : out mgmnt_fsm_t) is
  begin
    -- Read response phase
    s_axi_arvalid  <= (others => '0');
    
    if s_axi_rvalid(0) = '1' then
      mgmnt_state   <= IDLE;
      s_axi_rready  <= (others => '0');
      mgmnt_rd_data <= s_axi_rdata;
    end if;
  end;

  -- Recovered clock generator
  procedure RECCLK_CALC(signal speed  : in    std_logic_vector;
                        signal recclk : inout std_logic;
                        signal count  : inout unsigned) is
    variable speed_tmp : unsigned(14 downto 0) := (others => '0');
  begin
    -- Make array expression to be locally static, line rate speed size adjustment
    speed_tmp := resize(unsigned(speed), speed_tmp'length);
    -- Generate recovered clock
    if speed_tmp = "000010000000000" or speed_tmp = "100000000000000" then    -- 24.3G
      if count = 11 then 
        recclk <= not recclk;
        count <= "00000";
      else
        count <= count + 1;
      end if;
    elsif speed_tmp = "000001000000000" or speed_tmp = "010000000000000" then -- 12.1G
      if count = 5 then 
        recclk <= not recclk;
        count <= "00000";
      else
        count <= count + 1;
      end if;
    elsif speed_tmp = "000000100000000" or speed_tmp = "000100000000000" or speed_tmp = "000000000010000" then -- 8G/4.9G
      if count = 3 then 
        recclk <= not recclk;
        count <= "00000";
      else
        count <= count + 1;
      end if;
    elsif speed_tmp = "000000010000000" or speed_tmp = "001000000000000" or speed_tmp = "000000000100000"  then -- 10.1G/6.1G
      if count = 4 then 
        recclk <= not recclk;
        count <= "00000";
      else
        count <= count + 1;
      end if;
    elsif speed_tmp = "000000001000000" then -- 9.8G
      if count = 7 then 
        recclk <= not recclk;
        count <= "00000";
      else
        count <= count + 1;
      end if;
    elsif speed_tmp = "000000000001000" then -- 3G
      if count = 2 then
        recclk <= '1';
        count <= count + 1;
      elsif count = 4 then
        recclk <= '0';
        count <= "00000";
      else
        count <= count + 1;
      end if;
    elsif speed_tmp = "000000000000100" then -- 2.4G
      if count = 1 then
        recclk <= not recclk;
        count <= "00000";
      else
        count <= count + 1;
      end if;
    else                     -- 1.2G
      recclk <= not recclk;  -- 0.6G
    end if;
  end;

end cpri_demo_pkg;
