-- Title      : Example top level for CPRI design on Virtex Ultrascale
-- Project    : CPRI
-----------------------------------------------------------------------
-- File       : cpri_vcu108_32bit_ms.vhd
-- Author     : Xilinx
-----------------------------------------------------------------------
-- Description:
--   This example design for the 64-bit CPRI core is designed to be
--   hosted on the VCU108 Evalutation Board using Chipscope to provide
--   Control and Monitoring of the CPRI core.
--
-----------------------------------------------------------------------
-- (c) Copyright 2004 - 2016 Xilinx, Inc. All rights reserved.
--
-- This file contains confidential and proprietary information
-- of Xilinx, Inc. and is protected under U.S. and
-- international copyright and other intellectual property
-- laws.
--
-- DISCLAIMER
-- This disclaimer is not a license and does not grant any
-- rights to the materials distributed herewith. Except as
-- otherwise provided in a valid license issued to you by
-- Xilinx, and to the maximum extent permitted by applicable
-- law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
-- WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
-- AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
-- BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
-- INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
-- (2) Xilinx shall not be liable (whether in contract or tort,
-- including negligence, or under any other theory of
-- liability) for any loss or damage of any kind or nature
-- related to, arising under or in connection with these
-- materials, including for any direct, or any indirect,
-- special, incidental, or consequential loss or damage
-- (including loss of data, profits, goodwill, or any type of
-- loss or damage suffered as a result of any action brought
-- by a third party) even if such damage or loss was
-- reasonably foreseeable or Xilinx had been advised of the
-- possibility of the same.
--
-- CRITICAL APPLICATIONS
-- Xilinx products are not designed or intended to be fail-
-- safe, or for use in any application requiring fail-safe
-- performance, such as life-support or safety devices or
-- systems, Class III medical devices, nuclear facilities,
-- applications related to the deployment of airbags, or any
-- other applications that could lead to death, personal
-- injury, or severe property or environmental damage
-- (individually and collectively, "Critical
-- Applications"). Customer assumes the sole risk and
-- liability of any use of Xilinx products in Critical
-- Applications, subject only to applicable laws and
-- regulations governing limitations on product liability.
--
-- THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
-- PART OF THIS FILE AT ALL TIMES.
-----------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

library unisim;
use unisim.vcomponents.all;

library work;
use work.iic_controller_pkg.all;

entity cpri_vcu108_ms is
port (
  -- Clocks
  clk300_p       : in std_logic;
  clk300_n       : in std_logic;

  -- clock out to Si5328 jitter filter
  clk_si5328_p   : out std_logic;
  clk_si5328_n   : out std_logic;

  -- clock in from Si570 as Master GTY ref clock
  refclk_m_p     : in  std_logic;
  refclk_m_n     : in  std_logic;

  -- clock in from Si5328 as Slave GTY ref clock
  refclk_s_p     : in  std_logic;
  refclk_s_n     : in  std_logic;

  -- Clock Control
  iic_sda        : inout std_logic;
  iic_scl        : inout std_logic;

  si5328_rst_n   : out std_logic;
  iic_mux_rst_n  : out std_logic;

  -- Reset (push button)
  pb_reset       : in  std_logic;

  -- Clock Monitor ports
  clk_si570_mon  : out std_logic;
  clk_si5328_mon : out std_logic;

  -- CPRI link (Master)
  txp_m          : out std_logic;
  txn_m          : out std_logic;
  rxp_m          : in  std_logic;
  rxn_m          : in  std_logic;

  -- CPRI link (Slave)
  txp_s          : out std_logic;
  txn_s          : out std_logic;
  rxp_s          : in  std_logic;
  rxn_s          : in  std_logic;

  -- FPGA fan
  fan_on         : out std_logic;

  -- Status Leds
  leds           : out std_logic_vector(7 downto 0)
);

package p is new work.cpri_demo_pkg
  generic map (N_CPRI => 2,
               DEMO   => vcu108_ms);

use p.all;

end cpri_vcu108_ms;

architecture rtl of cpri_vcu108_ms is

  --=========================================
  -- VIO - Control & Management

  component vio_control
  port (
    clk         : in  std_logic;
    probe_in0   : in  std_logic_vector(31 downto 0);
    probe_out0  : out std_logic_vector(0 downto 0);
    probe_out1  : out std_logic_vector(0 downto 0);
    probe_out2  : out std_logic_vector(11 downto 0);
    probe_out3  : out std_logic_vector(31 downto 0);
    probe_out4  : out std_logic_vector(0 downto 0);
    probe_out5  : out std_logic_vector(0 downto 0);
    probe_out6  : out std_logic_vector(0 downto 0);
    probe_out7  : out std_logic_vector(0 downto 0);
    probe_out8  : out std_logic_vector(0 downto 0);
    probe_out9  : out std_logic_vector(0 downto 0);
    probe_out10 : out std_logic_vector(0 downto 0);
    probe_out11 : out std_logic_vector(0 downto 0);
    probe_out12 : out std_logic_vector(0 downto 0);
    probe_out13 : out std_logic_vector(0 downto 0)
   );
  end component;

  component vio_status
  port (
    clk        : in std_logic;
    probe_in0  : in std_logic_vector(15 downto 0);
    probe_in1  : in std_logic_vector(15 downto 0);
    probe_in2  : in std_logic_vector(15 downto 0);
    probe_in3  : in std_logic_vector(15 downto 0);
    probe_in4  : in std_logic_vector(15 downto 0);
    probe_in5  : in std_logic_vector(15 downto 0);
    probe_in6  : in std_logic_vector(15 downto 0);
    probe_in7  : in std_logic_vector(15 downto 0);
    probe_in8  : in std_logic_vector(13 downto 0);
    probe_in9  : in std_logic_vector(17 downto 0);
    probe_in10 : in std_logic_vector(6 downto 0);
    probe_in11 : in std_logic_vector(0 downto 0);
    probe_in12 : in std_logic_vector(0 downto 0);
    probe_in13 : in std_logic_vector(0 downto 0);
    probe_in14 : in std_logic_vector(0 downto 0);
    probe_in15 : in std_logic_vector(0 downto 0);
    probe_in16 : in std_logic_vector(0 downto 0);
    probe_in17 : in std_logic_vector(0 downto 0);
    probe_in18 : in std_logic_vector(0 downto 0);
    probe_in19 : in std_logic_vector(0 downto 0);
    probe_in20 : in std_logic_vector(4 downto 0);
    probe_in21 : in std_logic_vector(0 downto 0);
    probe_in22 : in std_logic_vector(0 downto 0);
    probe_in23 : in std_logic_vector(0 downto 0);
    probe_in24 : in std_logic_vector(0 downto 0);
    probe_in25 : in std_logic_vector(0 downto 0);
    probe_in26 : in std_logic_vector(0 downto 0);
    probe_in27 : in std_logic_vector(0 downto 0);
    probe_in28 : in std_logic_vector(35 downto 0);
    probe_in29 : in std_logic_vector(0 downto 0);
    probe_in30 : in std_logic_vector(0 downto 0);
    probe_in31 : in std_logic_vector(0 downto 0);
    probe_in32 : in std_logic_vector(0 downto 0);
    probe_in33 : in std_logic_vector(15 downto 0);
    probe_in34 : in std_logic_vector(15 downto 0)
    );
  end component;

  component ila_0
  port (
    clk     : in std_logic;
    probe0  : in std_logic_vector(0 downto 0);
    probe1  : in std_logic_vector(11 downto 0);
    probe2  : in std_logic_vector(0 downto 0);
    probe3  : in std_logic_vector(31 downto 0);
    probe4  : in std_logic_vector(0 downto 0);
    probe5  : in std_logic_vector(0 downto 0);
    probe6  : in std_logic_vector(1 downto 0);
    probe7  : in std_logic_vector(5 downto 0);
    probe8  : in std_logic_vector(15 downto 0);
    probe9  : in std_logic_vector(7 downto 0);
    probe10 : in std_logic_vector(0 downto 0);
    probe11 : in std_logic_vector(0 downto 0);
    probe12 : in std_logic_vector(0 downto 0);
    probe13 : in std_logic_vector(0 downto 0);
    probe14 : in std_logic_vector(0 downto 0);
    probe15 : in std_logic_vector(0 downto 0);
    probe16 : in std_logic_vector(0 downto 0);
    probe17 : in std_logic_vector(0 downto 0);
    probe18 : in std_logic_vector(0 downto 0);
    probe19 : in std_logic_vector(0 downto 0);
    probe20 : in std_logic_vector(0 downto 0);
    probe21 : in std_logic_vector(0 downto 0);
    probe22 : in std_logic_vector(0 downto 0);
    probe23 : in std_logic_vector(0 downto 0)
  );
  end component;
  
  component ila_1
  port (
    clk     : in std_logic;
    probe0  : in std_logic_vector(0 downto 0);
    probe1  : in std_logic_vector(11 downto 0);
    probe2  : in std_logic_vector(0 downto 0);
    probe3  : in std_logic_vector(31 downto 0);
    probe4  : in std_logic_vector(0 downto 0);
    probe5  : in std_logic_vector(3 downto 0);
    probe6  : in std_logic_vector(9 downto 0);
    probe7  : in std_logic_vector(0 downto 0);
    probe8  : in std_logic_vector(0 downto 0);
    probe9  : in std_logic_vector(0 downto 0);
    probe10 : in std_logic_vector(1 downto 0);
    probe11 : in std_logic_vector(5 downto 0);
    probe12 : in std_logic_vector(15 downto 0);
    probe13 : in std_logic_vector(0 downto 0);
    probe14 : in std_logic_vector(7 downto 0);
    probe15 : in std_logic_vector(0 downto 0);
    probe16 : in std_logic_vector(0 downto 0);
    probe17 : in std_logic_vector(0 downto 0)
  );
  end component;

  -------------------------------------------------------------------------------
  -- HW demo signals
  -------------------------------------------------------------------------------

  attribute keep          : string;
  attribute mark_debug    : string;
  attribute ASYNC_REG     : string;
  attribute SHREG_EXTRACT : string;

  -- Clocks
  signal clk              : std_logic_vector(1 downto 0);
  signal clk_ok           : std_logic_vector(1 downto 0);

  signal clk300b          : std_logic;
  signal clk300           : std_logic;
  
  signal aux_clkb         : std_logic;
  signal aux_clk          : std_logic;

  signal clk10b           : std_logic;
  signal clk10            : std_logic;

  signal eth_clkb         : std_logic;
  signal eth_clk          : std_logic;

  signal hires_clkb       : std_logic;
  signal hires_clk        : std_logic;

  signal clk_si5328       : std_logic;

  signal refclk           : std_logic_vector(1 downto 0);
  signal refclki          : std_logic_vector(1 downto 0);
  signal refclki_bufg     : std_logic_vector(1 downto 0);

  signal alive_refclk     : unsigned(24 downto 0);

  signal hires_clk_ok     : std_logic;
  signal refclk_ok        : std_logic_vector(0 downto 0);
  signal refclk_ok_count  : unsigned(24 downto 0);
  attribute mark_debug of refclk_ok : signal is "true";

  signal fb               : std_logic;
  signal mmcm_locked      : std_logic_vector(0 downto 0);
  attribute mark_debug of mmcm_locked : signal is "true";

  signal iic_done         : std_logic_vector(0 downto 0);
  attribute mark_debug of iic_done : signal is "true"; 

  -- Slave recovered clock
  signal recclk           : std_logic;
  signal recclk_ok        : std_logic;
  signal oddr_sr          : std_logic;

  -- Resets
  signal mmcm_reset       : std_logic;
  signal iic_reset        : std_logic;
  signal reset            : std_logic;
  signal reset_s          : std_logic_vector(1 downto 0);

  -- Fan
  signal fan              : std_logic;
  signal fan_count        : unsigned(8 downto 0);

  -- Shared GT COMMON block signals
  signal common_qpll0clk    : std_logic;
  signal common_qpll0refclk : std_logic;
  signal common_qpll0lock   : std_logic;
  signal common_qpll1clk    : std_logic;
  signal common_qpll1refclk : std_logic;
  signal common_qpll1lock   : std_logic;
  signal common_qpll0_pd    : std_logic;
  signal common_qpll1_pd    : std_logic;

  -- Decoded speed & status 
  signal count       : unsigned(4 downto 0) := (others => '0');
  signal oddr_d1     : std_logic;
  signal oddr_d2     : std_logic;
  signal recclkout_i : std_logic;

  -- Core Master / Slave control
  signal cntrl : cntrl_t;
  attribute mark_debug of cntrl : signal is "true";

  -- R21 Measurement
  signal delay : delay_t;
  attribute mark_debug of delay : signal is "true";

  -- Ethernet data generation and checking
  signal eth : eth_t;
  attribute mark_debug of eth : signal is "true";

  -- GT debug signals
  signal gt : gt_t;
  attribute mark_debug of gt : signal is "true";

  -- HDLC Data generation and checking
  signal hdlc : hdlc_t;
  attribute mark_debug of hdlc : signal is "true";

  -- IQ Data generation and checking
  signal iq : iq_t;
  attribute mark_debug of iq : signal is "true";

  signal iq_rx_chk_enable_clk : cpri_sync_slv_00b_t;
  signal iq_rx_chk_enable_eth : cpri_sync_slv_00b_t;
  attribute ASYNC_REG     of iq_rx_chk_enable_clk : signal is "true";
  attribute SHREG_EXTRACT of iq_rx_chk_enable_clk : signal is "no";
  attribute ASYNC_REG     of iq_rx_chk_enable_eth : signal is "true";
  attribute SHREG_EXTRACT of iq_rx_chk_enable_eth : signal is "no";

  -- Management Interface
  signal mgmnt : mgmnt_t := (others => ( (others => '0')
                                       , (others => '0')
                                       , (others => '0')
                                       , (others => '0')
                                       , (others => '0')
                                       , (others => '0')
                                       , IDLE
                                       , MGMNT_CMD_PMA0 ));

  signal MGMNT_CMD_ROM : fsm_axi_t(3 downto 0) := mgmnt_rom_init( vcu108_ms );

  attribute mark_debug of mgmnt : signal is "true";

  -- AXI Lite Management Interface
  signal s_axi : s_axi_t;
  attribute mark_debug of s_axi : signal is "true";

  -- CPRI core status
  signal stat : stat_t;
  attribute mark_debug of stat : signal is "true";

  signal stat_code_clk     : cpri_sync_slv_04b_t;
  signal stat_speed_clk    : cpri_sync_slv_10b_t;
  signal stat_speed_recclk : cpri_sync_slv_10b_t;
  attribute ASYNC_REG of stat_speed_recclk     : signal is "true";
  attribute SHREG_EXTRACT of stat_speed_recclk : signal is "no";
  attribute ASYNC_REG of stat_speed_clk        : signal is "true";
  attribute SHREG_EXTRACT of stat_speed_clk    : signal is "no";
  attribute ASYNC_REG of stat_code_clk         : signal is "true";
  attribute SHREG_EXTRACT of stat_code_clk     : signal is "no";

  -- FSM State flags
  signal state : state_t;
  attribute mark_debug of state : signal is "true";

  -- Synchronization
  signal sync : sync_t;
  attribute mark_debug of sync : signal is "true";

  -- TX Alignment
  signal tx_align : tx_align_t;
  attribute mark_debug of tx_align : signal is "true";

  -- Vendor specific data generation and checking
  signal vendor : vendor_t;
  attribute mark_debug of vendor : signal is "true";

  -- VIO Reset control
  signal vio : vio_t;
  attribute mark_debug of vio : signal is "true";
  
  signal vio_reset_d : cpri_sync_slv_00b_t;

  -- Decorator signals just for VIO/ILA probes naming purposes
  type probe_t is array(N_CPRI-1 downto 0) of std_logic_vector(0 downto 0);

  signal line_rate_12G : probe_t;
  signal line_rate_10G : probe_t;
  signal line_rate_9G  : probe_t;
  signal line_rate_8G  : probe_t;

  attribute mark_debug of line_rate_12G : signal is "true";
  attribute mark_debug of line_rate_10G : signal is "true";
  attribute mark_debug of line_rate_9G  : signal is "true";
  attribute mark_debug of line_rate_8G  : signal is "true";

begin

  cpri_m_block_i: entity work.cpri_0_support
  port map (
    -- Additional Debug Ports
    -- DRP Access
    gt_drpdaddr                => (others => '0'),
    gt_drpdi                   => (others => '0'),
    gt_drpen                   => '0',
    gt_drpwe                   => '0',
    gt_drpdo                   => open,
    gt_drprdy                  => open,
    -- TX Reset and Initialization
    gt_txpmareset              => '0',
    gt_txpcsreset              => '0',
    gt_txresetdone             => gt(1).txresetdone(0),
    -- RX Reset and Initialization
    gt_rxpmareset              => '0',
    gt_rxpcsreset              => '0',
    gt_rxpmaresetdone          => gt(1).rxpmaresetdone(0),
    gt_rxresetdone             => gt(1).rxresetdone(0),
    -- Clocking
    gt_txphaligndone           => open,
    gt_txphinitdone            => open,
    gt_txdlysresetdone         => open,
    gt_rxphaligndone           => open,
    gt_rxdlysresetdone         => open,
    gt_rxsyncdone              => open,
    gt_cplllock                => gt(1).cplllock(0),
    -- Signal Integrity and Functionality
    gt_rxrate                  => (others => '0'),
    gt_eyescantrigger          => '0',
    gt_eyescanreset            => '0',
    gt_eyescandataerror        => open,
    gt_rxpolarity              => '0',
    gt_txpolarity              => '0',
    gt_rxdfelpmreset           => '0',
    gt_rxlpmen                 => '0',
    gt_txprecursor             => (others => '0'),
    gt_txpostcursor            => (others => '0'),
    gt_txdiffctrl              => "01100",
    gt_txprbsforceerr          => '0',
    gt_txprbssel               => (others => '0'),
    gt_rxprbssel               => (others => '0'),
    gt_rxprbserr               => open,
    gt_rxprbscntreset          => '0',
    gt_rxcdrhold               => '0',
    gt_dmonitorout             => gt(1).dmonitorout,
    gt_rxheader                => open,
    gt_rxheadervalid           => open,
    gt_rxdisperr               => open,
    gt_rxnotintable            => open,
    gt_rxcommadet              => open,
    gt_pcsrsvdin               => x"0000",
    -- Transceiver monitor interface
    txdata                     => gt(1).txdata,
    txcharisk                  => open,
    txheader                   => gt(1).txheader,
    txsequence                 => open,
    rxdata                     => gt(1).rxdata,
    rxchariscomma              => open,
    rxcharisk                  => open,
    rxdisperr                  => open,
    rxnotintable               => open,
    rxdatavalid                => gt(1).rxdatavalid(0),
    rxheader                   => gt(1).rxheader,
    rxheadervalid              => gt(1).rxheadervalid(0),
    rxgearboxslip              => open,
    ----------------- END OF DEBUG PORTS -----------------
    reset                      => reset,
    iq_tx_enable               => iq(1).tx_enable(0),
    basic_frame_first_word     => iq(1).basic_frame_first_word(0),
    iq_tx                      => iq(1).tx,
    iq_rx                      => iq(1).rx,
    qpll0clk_in                => common_qpll0clk,
    qpll0refclk_in             => common_qpll0refclk,
    qpll1clk_in                => common_qpll1clk,
    qpll1refclk_in             => common_qpll1refclk,
    qpll0lock_in               => common_qpll0lock,
    qpll1lock_in               => common_qpll1lock,
    qpll0_pd                   => open,
    qpll1_pd                   => open,
    vendor_tx_data             => vendor(1).tx_data,
    vendor_tx_xs               => vendor(1).tx_xs,
    vendor_tx_ns               => vendor(1).tx_ns,
    vendor_rx_data             => vendor(1).rx_data,
    vendor_rx_xs               => vendor(1).rx_xs,
    vendor_rx_ns               => vendor(1).rx_ns,
    nodebfn_tx_strobe          => sync(1).nodebfn_tx_strobe(0),
    nodebfn_tx_nr              => sync(1).nodebfn_tx_nr,
    nodebfn_rx_strobe          => sync(1).nodebfn_rx_strobe(0),
    nodebfn_rx_nr              => sync(1).nodebfn_rx_nr,
    eth_txd                    => eth(1).txd,
    eth_tx_er                  => eth(1).tx_er(0),
    eth_tx_en                  => eth(1).tx_en(0),
    eth_col                    => eth(1).col(0),
    eth_crs                    => eth(1).crs(0),
    eth_rxd                    => eth(1).rxd,
    eth_rx_er                  => eth(1).rx_er(0),
    eth_rx_dv                  => eth(1).rx_dv(0),
    eth_rx_avail               => eth(1).rx_avail,
    eth_rx_ready               => eth(1).rx_ready,
    rx_fifo_almost_full        => eth(1).rx_fifo_almost_full,
    rx_fifo_full               => eth(1).rx_fifo_full,
    eth_tx_clk                 => eth_clk,
    eth_rx_clk                 => eth_clk,
    hdlc_rx_data               => hdlc(1).rx_data(0),
    hdlc_tx_data               => hdlc(1).tx_data(0),
    hdlc_rx_data_valid         => hdlc(1).rx_data_valid(0),
    hdlc_tx_enable             => hdlc(1).tx_enable(0),
    stat_alarm                 => stat(1).alarm(0),
    stat_code                  => stat(1).code,
    stat_speed                 => stat(1).speed,
    s_axi_aclk                 => aux_clk,
    s_axi_aresetn              => s_axi(1).aresetn,
    s_axi_awaddr               => s_axi(1).awaddr,
    s_axi_awvalid              => s_axi(1).awvalid(0),
    s_axi_awready              => s_axi(1).awready(0),
    s_axi_wdata                => s_axi(1).wdata,
    s_axi_wvalid               => s_axi(1).wvalid(0),
    s_axi_wready               => s_axi(1).wready(0),
    s_axi_bresp                => s_axi(1).bresp,
    s_axi_bvalid               => s_axi(1).bvalid(0),
    s_axi_bready               => s_axi(1).bready(0),
    s_axi_araddr               => s_axi(1).araddr,
    s_axi_arvalid              => s_axi(1).arvalid(0),
    s_axi_arready              => s_axi(1).arready(0),
    s_axi_rdata                => s_axi(1).rdata,
    s_axi_rresp                => s_axi(1).rresp,
    s_axi_rvalid               => s_axi(1).rvalid(0),
    s_axi_rready               => s_axi(1).rready(0),
    l1_timer_expired           => '0',
    vs_negotiation_complete    => '1',
    reset_request_in           => s_axi(1).reset_request_in,
    sdi_request_in             => s_axi(1).sdi_request_in(0),
    reset_acknowledge_out      => s_axi(1).reset_acknowledge_out(0),
    sdi_request_out            => s_axi(1).sdi_request_out(0),
    local_los                  => s_axi(1).local_los(0),
    local_lof                  => s_axi(1).local_lof(0),
    local_rai                  => s_axi(1).local_rai(0),
    remote_los                 => s_axi(1).remote_los(0),
    remote_lof                 => s_axi(1).remote_lof(0),
    remote_rai                 => s_axi(1).remote_rai(0),
    fifo_transit_time          => s_axi(1).fifo_transit_time,
    coarse_timer_value         => s_axi(1).coarse_timer_value,
    barrel_shift_value         => s_axi(1).barrel_shift_value,
    tx_gb_latency_value        => s_axi(1).tx_gb_latency_value,
    rx_gb_latency_value        => s_axi(1).rx_gb_latency_value,
    txp                        => txp_m,
    txn                        => txn_m,
    rxp                        => rxp_m,
    rxn                        => rxn_m,
    lossoflight                => '0',
    txinhibit                  => open,
    core_is_master             => cntrl(1).core_is_master(1),

    refclk                     => refclk(1),
    gtwiz_reset_clk_freerun_in => clk10,
    hires_clk                  => hires_clk,
    hires_clk_ok               => hires_clk_ok,
    qpll_select                => '0',
    recclk_ok                  => open,
    clk_ok_out                 => clk_ok(1),
    recclk                     => open,
    clk_out                    => clk(1),

    txphaligndone_in           => tx_align(1).txphalign_in,
    txdlysresetdone_in         => tx_align(1).txdlysreset_in,
    txphinitdone_in            => tx_align(1).txphinit_in,
    txphinit_out               => tx_align(1).txphinit_in,
    phase_alignment_done_out   => open,
    txdlysreset_out            => tx_align(1).txdlysreset_in,
    txphalign_out              => tx_align(1).txphalign_in,
    txdlyen_out                => open
  );

  cpri_s_block_i: entity work.cpri_1_support
  port map (
    -- Additional Debug Ports
    -- DRP Access
    gt_drpdaddr                => (others => '0'),
    gt_drpdi                   => (others => '0'),
    gt_drpen                   => '0',
    gt_drpwe                   => '0',
    gt_drpdo                   => open,
    gt_drprdy                  => open,
    -- TX Reset and Initialization
    gt_txpmareset              => '0',
    gt_txpcsreset              => '0',
    gt_txresetdone             => gt(0).txresetdone(0),
    -- RX Reset and Initialization
    gt_rxpmareset              => '0',
    gt_rxpcsreset              => '0',
    gt_rxpmaresetdone          => gt(0).rxpmaresetdone(0),
    gt_rxresetdone             => gt(0).rxresetdone(0),
    -- Clocking
    gt_txphaligndone           => open,
    gt_txphinitdone            => open,
    gt_txdlysresetdone         => open,
    gt_rxphaligndone           => open,
    gt_rxdlysresetdone         => open,
    gt_rxsyncdone              => open,
    gt_cplllock                => gt(0).cplllock(0),
    -- Signal Integrity and Functionality
    gt_rxrate                  => (others => '0'),
    gt_eyescantrigger          => '0',
    gt_eyescanreset            => '0',
    gt_eyescandataerror        => open,
    gt_rxpolarity              => '0',
    gt_txpolarity              => '0',
    gt_rxdfelpmreset           => '0',
    gt_rxlpmen                 => '0',
    gt_txprecursor             => (others => '0'),
    gt_txpostcursor            => (others => '0'),
    gt_txdiffctrl              => "01100",
    gt_txprbsforceerr          => '0',
    gt_txprbssel               => (others => '0'),
    gt_rxprbssel               => (others => '0'),
    gt_rxprbserr               => open,
    gt_rxprbscntreset          => '0',
    gt_rxcdrhold               => '0',
    gt_dmonitorout             => gt(0).dmonitorout,
    gt_rxheader                => open,
    gt_rxheadervalid           => open,
    gt_rxdisperr               => open,
    gt_rxnotintable            => open,
    gt_rxcommadet              => open,
    gt_pcsrsvdin               => x"0000",
    -- Transceiver monitor interface
    txdata                     => gt(0).txdata,
    txcharisk                  => open,
    txheader                   => gt(0).txheader,
    txsequence                 => open,
    rxdata                     => gt(0).rxdata,
    rxchariscomma              => open,
    rxcharisk                  => open,
    rxdisperr                  => open,
    rxnotintable               => open,
    rxdatavalid                => gt(0).rxdatavalid(0),
    rxheader                   => gt(0).rxheader,
    rxheadervalid              => gt(0).rxheadervalid(0),
    rxgearboxslip              => open,
    ----------------- END OF DEBUG PORTS -----------------
    reset                      => reset,
    iq_tx_enable               => iq(0).tx_enable(0),
    basic_frame_first_word     => iq(0).basic_frame_first_word(0),
    iq_tx                      => iq(0).tx,
    iq_rx                      => iq(0).rx,
    qpll0clk_in                => common_qpll0clk,
    qpll0refclk_in             => common_qpll0refclk,
    qpll1clk_in                => common_qpll1clk,
    qpll1refclk_in             => common_qpll1refclk,
    qpll0lock_in               => common_qpll0lock,
    qpll1lock_in               => common_qpll1lock,
    qpll0_pd                   => open,
    qpll1_pd                   => open,
    vendor_tx_data             => vendor(0).tx_data,
    vendor_tx_xs               => vendor(0).tx_xs,
    vendor_tx_ns               => vendor(0).tx_ns,
    vendor_rx_data             => vendor(0).rx_data,
    vendor_rx_xs               => vendor(0).rx_xs,
    vendor_rx_ns               => vendor(0).rx_ns,
    nodebfn_tx_strobe          => sync(0).nodebfn_tx_strobe(0),
    nodebfn_tx_nr              => sync(0).nodebfn_tx_nr,
    nodebfn_rx_strobe          => sync(0).nodebfn_rx_strobe(0),
    nodebfn_rx_nr              => sync(0).nodebfn_rx_nr,
    eth_txd                    => eth(0).txd,
    eth_tx_er                  => eth(0).tx_er(0),
    eth_tx_en                  => eth(0).tx_en(0),
    eth_col                    => eth(0).col(0),
    eth_crs                    => eth(0).crs(0),
    eth_rxd                    => eth(0).rxd,
    eth_rx_er                  => eth(0).rx_er(0),
    eth_rx_dv                  => eth(0).rx_dv(0),
    eth_rx_avail               => eth(0).rx_avail,
    eth_rx_ready               => eth(0).rx_ready,
    rx_fifo_almost_full        => eth(0).rx_fifo_almost_full,
    rx_fifo_full               => eth(0).rx_fifo_full,
    eth_tx_clk                 => eth_clk,
    eth_rx_clk                 => eth_clk,
    hdlc_rx_data               => hdlc(0).rx_data(0),
    hdlc_tx_data               => hdlc(0).tx_data(0),
    hdlc_rx_data_valid         => hdlc(0).rx_data_valid(0),
    hdlc_tx_enable             => hdlc(0).tx_enable(0),
    stat_alarm                 => stat(0).alarm(0),
    stat_code                  => stat(0).code,
    stat_speed                 => stat(0).speed,
    s_axi_aclk                 => aux_clk,
    s_axi_aresetn              => s_axi(0).aresetn,
    s_axi_awaddr               => s_axi(0).awaddr,
    s_axi_awvalid              => s_axi(0).awvalid(0),
    s_axi_awready              => s_axi(0).awready(0),
    s_axi_wdata                => s_axi(0).wdata,
    s_axi_wvalid               => s_axi(0).wvalid(0),
    s_axi_wready               => s_axi(0).wready(0),
    s_axi_bresp                => s_axi(0).bresp,
    s_axi_bvalid               => s_axi(0).bvalid(0),
    s_axi_bready               => s_axi(0).bready(0),
    s_axi_araddr               => s_axi(0).araddr,
    s_axi_arvalid              => s_axi(0).arvalid(0),
    s_axi_arready              => s_axi(0).arready(0),
    s_axi_rdata                => s_axi(0).rdata,
    s_axi_rresp                => s_axi(0).rresp,
    s_axi_rvalid               => s_axi(0).rvalid(0),
    s_axi_rready               => s_axi(0).rready(0),
    l1_timer_expired           => '0',
    vs_negotiation_complete    => '1',
    reset_request_in           => s_axi(0).reset_request_in,
    sdi_request_in             => s_axi(0).sdi_request_in(0),
    reset_acknowledge_out      => s_axi(0).reset_acknowledge_out(0),
    sdi_request_out            => s_axi(0).sdi_request_out(0),
    local_los                  => s_axi(0).local_los(0),
    local_lof                  => s_axi(0).local_lof(0),
    local_rai                  => s_axi(0).local_rai(0),
    remote_los                 => s_axi(0).remote_los(0),
    remote_lof                 => s_axi(0).remote_lof(0),
    remote_rai                 => s_axi(0).remote_rai(0),
    fifo_transit_time          => s_axi(0).fifo_transit_time,
    coarse_timer_value         => s_axi(0).coarse_timer_value,
    barrel_shift_value         => s_axi(0).barrel_shift_value,
    tx_gb_latency_value        => s_axi(0).tx_gb_latency_value,
    rx_gb_latency_value        => s_axi(0).rx_gb_latency_value,
    txp                        => txp_s,
    txn                        => txn_s,
    rxp                        => rxp_s,
    rxn                        => rxn_s,
    lossoflight                => '0',
    txinhibit                  => open,
    core_is_master             => cntrl(0).core_is_master(1),

    refclk                     => refclk(0),
    gtwiz_reset_clk_freerun_in => clk10,
    hires_clk                  => hires_clk,
    hires_clk_ok               => hires_clk_ok,
    qpll_select                => '0',
    recclk_ok                  => recclk_ok,
    clk_ok_out                 => clk_ok(0),
    recclk                     => recclk,
    clk_out                    => clk(0),

    txphaligndone_in           => tx_align(0).txphalign_in,
    txdlysresetdone_in         => tx_align(0).txdlysreset_in,
    txphinitdone_in            => tx_align(0).txphinit_in,
    txphinit_out               => tx_align(0).txphinit_in,
    phase_alignment_done_out   => open,
    txdlysreset_out            => tx_align(0).txdlysreset_in,
    txphalign_out              => tx_align(0).txphalign_in,
    txdlyen_out                => open
  );

  -- Shared GT_COMMON block
  cpri_common_i : entity work.cpri_0_gt_rst_support
  port map(
    refclk          => refclk(1),
    aux_clk         => aux_clk,
    reset           => reset,
    qpll0_pd        => common_qpll0_pd,
    qpll1_pd        => common_qpll1_pd,
    qpll_select     => '0',
    speed_select    => stat(1).speed,
    qpll0clk_out    => common_qpll0clk,
    qpll0refclk_out => common_qpll0refclk,
    qpll1clk_out    => common_qpll1clk,
    qpll1refclk_out => common_qpll1refclk,
    qpll0lock_out   => common_qpll0lock,
    qpll1lock_out   => common_qpll1lock,
    gt_qplllock     => gt(1).qplllock(0)
  );

  --===============================
  -- HW Demo
  --===============================

  -------------------------------------------------------------------------------
  -- Clock Generation
  -------------------------------------------------------------------------------

  -- 300 MHz clock in from VCU108
  clk300_ibuf : IBUFDS
  port map (
    I  => clk300_p,
    IB => clk300_n,
    O  => clk300b
  );

  clk300_bufg : BUFG
  port map (
    I => clk300b,
    O => clk300
  );

  -- MMCM generates several clocks from the 300MHz input
  --
  --  CLKOUT 0 : hires clk  (400 MHz)
  --  CLKOUT 1 : aux_clk    (100 MHz)
  --  CLKOUT 2 : clk10      (10 MHz)
  --  CLKOUT 3 : eth_clk    (125 MHz for GMII)
  --
  mmcm_i : MMCM_BASE
  generic map (
    CLKFBOUT_MULT_F    => 3.000,
    CLKOUT0_DIVIDE_F   => 2.250,
    CLKOUT1_DIVIDE     => 9,
    CLKOUT2_DIVIDE     => 90,
    CLKOUT3_DIVIDE     => 9,
    CLKFBOUT_PHASE     => 0.000,
    CLKIN1_PERIOD      => 3.333,
    CLKOUT0_DUTY_CYCLE => 0.5,
    CLKOUT0_PHASE      => 0.000,
    CLOCK_HOLD         => false,
    DIVCLK_DIVIDE      => 1,
    REF_JITTER1        => 0.010,
    STARTUP_WAIT       => false
  )
  port map (
    CLKIN1             => clk300,
    CLKFBOUTB          => open,
    CLKOUT0            => hires_clkb,
    CLKOUT1            => aux_clkb,
    CLKOUT2            => clk10b,
    CLKOUT3            => eth_clkb,
    CLKFBIN            => fb,
    CLKFBOUT           => fb,
    LOCKED             => mmcm_locked(0),
    PWRDWN             => '0',
    RST                => mmcm_reset
  );

  aux_bufg : BUFG
  port map (
    I => aux_clkb,
    O => aux_clk
  );

  hires_bufg : BUFG
  port map (
    I => hires_clkb,
    O => hires_clk
  );

  clk10_bufg : BUFG
  port map (
    I => clk10b,
    O => clk10
  );

  eth_bufg : BUFG
  port map (
    I => eth_clkb,
    O => eth_clk
  );

  -------------------------------------
  -- GT Reference clock Generation
  -------------------------------------

  -- Si570 is set up to provide reference clock to Master CPRI core

  -- Reference clock IN from Si570
  refclk_m_ibufds : IBUFDS_GTE3
  port map (
    I     => refclk_m_p,
    IB    => refclk_m_n,
    O     => refclk(1),
    CEB   => '0',
    ODIV2 => refclki(1)
  );

  -- Reference clock IN from Si5328
  refclk_s_ibufds : IBUFDS_GTE3
  port map (
    I     => refclk_s_p,
    IB    => refclk_s_n,
    O     => refclk(0),
    CEB   => '0',
    ODIV2 => refclki(0)
  );

  -------------------------------------------------------------------------------
  -- This counter is used to insert a ~100 us delay after the IIC controller has
  -- set up the reference clock to allow the CPRI core to come out of reset with a
  -- stable GTX reference clock.
  -- 11-bit counter @ 10 MHz => 1024 cycles => 102 us

  process (clk10, iic_reset)
  begin
    if (iic_reset = '1') then
      refclk_ok_count   <= (others => '0');
      refclk_ok(0)      <= '0';
    elsif rising_edge(clk10) then
      if (iic_done(0) = '1') then
        if (refclk_ok_count(refclk_ok_count'high) = '0') then
          refclk_ok_count <= refclk_ok_count + 1;
        end if;
      else
        refclk_ok_count <= (others => '0');
      end if;
      refclk_ok(0) <= std_logic(refclk_ok_count(refclk_ok_count'high));
    end if;
  end process;

  -- reset IIC slaves in sympathy with IIC controller
  si5328_rst_n  <= not iic_reset;
  iic_mux_rst_n <= not iic_reset;

  -------------------------------------------------------------------------------
  -- IIC controller
  --
  -- Note that the IIC controller must be clocked at 100 MHz as there is a fixed
  -- ratio between this and it's operating frequency on the IIC bus.

  i_iic_controller : entity work.iic_controller
  generic map( vcu108_ms )
  port map (
    reset   => iic_reset,
    clk100  => aux_clk,
    iic_sda => iic_sda,
    iic_scl => iic_scl,
    status  => open,
    done    => iic_done(0)
  );

  -------------------------------------------------------------------------------
  -- Recovered clock
  --
  -- Recovered clock out to the Si5328
  obufds0 : OBUFDS
  port map (
    I  => clk_si5328,
    O  => clk_si5328_p,
    OB => clk_si5328_n
  );

  -- Line rate  Recovered clock   Div    Recclk_out
  -- 12.16512       368.64        24       15.36
  -- 10.1376        307.20        20       15.36
  --  9.8304        245.76        16       15.36
  --  8.1108        245.76        16       15.36
  --  6.144         153.60        10       15.36
  --  4.9152        122.88         8       15.36
  --  3.072         76.80          5       15.36
  --  2.4576        61.44          4       15.36
  --  1.2288        30.72          2       15.36
  --  0.6144        15.36          1       15.36
  process (recclk)
  begin
    if rising_edge (recclk) then
      RECCLK_CALC( speed  => stat_speed_recclk(0)(2)
                 , recclk => recclkout_i
                 , count  => count);
    end if;
  end process;

  -- For 614Mb/s use ODDRE1 to multiply by 2
  process (recclk)
  begin
    if rising_edge(recclk) then
      if stat_speed_recclk(0)(2) = "0000000001" then
        oddr_d1 <= '1';
      else
        oddr_d1 <= recclkout_i;
      end if;
    end if;
  end process;

  process (recclk)
  begin
    if falling_edge(recclk) then
      if stat_speed_recclk(0)(2) = "0000000001" then
        oddr_d2 <= '0';
      else
        oddr_d2 <= recclkout_i;
      end if;
    end if;
  end process;

  -- Output 15.36MHz
  recclkout_gen : ODDRE1
  port map (D1  => oddr_d1,
            D2  => oddr_d2,
            C   => recclk,
            SR  => oddr_sr,
            Q   => clk_si5328);

  oddr_sr <= '1' when ( cntrl(0).core_is_master(1) = '1'
                     or cntrl(0).core_in_loopback(1) = '1'
                     or recclk_ok = '0') else '0';

  -------------------------------------------------------------------------------
  -- LEDs and Monitor ports
  -------------------------------------------------------------------------------

  -- LEDs

  -- Divide refclk for flashing LED
  process(refclki_bufg(0))
  begin
    if rising_edge(refclki_bufg(0)) then
      alive_refclk <= alive_refclk + 1;
    end if;
  end process;

  leds(7)          <= alive_refclk(alive_refclk'high);             -- Refclk
  leds(6)          <= mmcm_locked(0);                              -- clocks locked
  leds(5)          <= stat(1).alarm(0) or stat(0).alarm(0);        -- Alarm
  leds(4 downto 1) <= stat(1).speed4;                              -- Speed indication
  leds(0)          <= state(1).operate(0) and state(0).operate(0); -- State = OPERATIONAL

  -- Fan control
  process (clk10)
  begin
    if rising_edge(clk10) then
      fan_count <= fan_count + 1;
      if fan_count = 506 then
        fan <= '1';
      elsif fan_count = 0 then
        fan <= '0';
      end if;
    end if;
  end process;

  fan_on <= fan;

  -- Clock Monitor outputs on GPIO SMAs

  -- Si570  (refclk(1))
  oddr1 : ODDRE1
  port map(
    D1 => '1',
    D2 => '0',
    C  => refclki_bufg(1),
    Q  => clk_si570_mon,
    SR => '0'
  );

  -- Si5328 (refclk(0))
  oddr2 : ODDRE1
  port map(
    D1 => '1',
    D2 => '0',
    C  => refclki_bufg(0),
    Q  => clk_si5328_mon,
    SR => '0'
  );

  -- Complete reset can be invoked from push button or vio
  mmcm_reset   <= pb_reset;

  -- hold I2C reset until MMCM locked
  iic_reset    <= not mmcm_locked(0);

  hires_clk_ok <= mmcm_locked(0);

  -- Bulk of design held in reset until reference clock is setup and stabilised
  reset <= (not refclk_ok(0)) or vio(1).reset(0) or vio(0).reset(0);

cpri_gen_i : for i in N_CPRI-1 downto 0 generate

  -- IQ Data generator (Master)
  iq_tx_gen_i : entity work.iq_tx_gen
  port map (
    clk               => clk(i),
    iq_tx_enable      => iq(i).tx_enable(0),
    iq_tx             => iq(i).tx,
    nodebfn_tx_strobe => sync(i).bfn_tx_strobe,
    nodebfn_tx_nr     => sync(i).bfn_tx_nr
  );

  -- IQ Data monitor (Master)
  iq_rx_chk_i : entity work.iq_rx_chk
  port map (
    clk                    => clk(i),
    enable                 => cntrl(i).test_start_clk(0),
    speed                  => stat_speed_clk(i)(2),
    basic_frame_first_word => iq(i).basic_frame_first_word(0),
    iq_rx                  => iq(i).rx,
    nodebfn_rx_strobe      => sync(i).nodebfn_rx_strobe(0),
    nodebfn_rx_nr          => sync(i).nodebfn_rx_nr,
    frame_count            => iq(i).frame_count,
    error_count            => iq(i).error_count,
    nodebfn_rx_nr_store    => sync(i).nodebfn_rx_nr_store,
    word_err               => iq(i).error(0)
  );

  -- Logic to generate the slave synchronization strobe.
  s_p2: process (clk(i))
  begin
    if rising_edge(clk(i)) then
      -- Slave-block TX capture
      sync(i).bfn_tx_strobe_slv <= sync(i).nodebfn_rx_strobe(0);
      -- Registered first word flag
      iq(i).basic_frame_first_word_r <= iq(i).basic_frame_first_word(0);
      -- Change tx nr on falling edge of tx strobe
      if( sync(i).bfn_tx_strobe_slv = '1' and iq(i).basic_frame_first_word_r = '1' ) then
        sync(i).bfn_tx_nr_slv <= std_logic_vector(unsigned(sync(i).nodebfn_rx_nr) + 1);
      end if;
    end if;
  end process;

  sync(i).nodebfn_tx_strobe(0) <= sync(i).bfn_tx_strobe_slv when
                                 (cntrl(i).core_is_master(1) = '0' and cntrl(i).core_in_loopback(1) = '0') else
                                  sync(i).bfn_tx_strobe;

  sync(i).nodebfn_tx_nr        <= sync(i).bfn_tx_nr_slv when
                                 (cntrl(i).core_is_master(1) = '0' and cntrl(i).core_in_loopback(1) = '0') else
                                  sync(i).bfn_tx_nr;

  -- Synchronise stat_speed to the tx user clock (master)
  process(clk(i))
  begin
    if rising_edge(clk(i)) then
      stat_speed_clk(i)(0) <= stat(i).speed;
      stat_speed_clk(i)(1) <= stat_speed_clk(i)(0);
      stat_speed_clk(i)(2) <= stat_speed_clk(i)(1);
    end if;
  end process;

  -- Synchronise stat_speed to the recovered clock
  process(recclk)
  begin
    if rising_edge(recclk) then
      stat_speed_recclk(i)(0) <= stat(i).speed;
      stat_speed_recclk(i)(1) <= stat_speed_recclk(i)(0);
      stat_speed_recclk(i)(2) <= stat_speed_recclk(i)(1);
    end if;
  end process;

  -- Synchronise stat_code to the aux clk
  process(clk(i))
  begin
    if rising_edge(clk(i)) then
      stat_code_clk(i)(0) <= stat(i).code;
      stat_code_clk(i)(1) <= stat_code_clk(i)(0);
      stat_code_clk(i)(2) <= stat_code_clk(i)(1);
    end if;
  end process;


  iq(i).rx_chk_enable(0) <= '1' when stat(i).code = "1111" and 
                                     s_axi(i).reset_acknowledge_out(0) = '0' else '0';

  -- Synchronise to the core clock
  process(clk(i))
  begin
    if rising_edge(clk(i)) then
      iq_rx_chk_enable_clk(i)(0) <= iq(i).rx_chk_enable;
      iq_rx_chk_enable_clk(i)(1) <= iq_rx_chk_enable_clk(i)(0);
      iq_rx_chk_enable_clk(i)(2) <= iq_rx_chk_enable_clk(i)(1);
    end if;
  end process;

  cntrl(i).test_start_clk <= iq_rx_chk_enable_clk(i)(2) and iq_rx_chk_enable_clk(i)(1);

  -- Synchronise to the ethernet clock
  process(eth_clk)
  begin
    if rising_edge(eth_clk) then
      iq_rx_chk_enable_eth(i)(0) <= iq(i).rx_chk_enable;
      iq_rx_chk_enable_eth(i)(1) <= iq_rx_chk_enable_eth(i)(0);
      iq_rx_chk_enable_eth(i)(2) <= iq_rx_chk_enable_eth(i)(1);
    end if;
  end process;

  cntrl(i).test_start_eth <= iq_rx_chk_enable_eth(i)(2) and iq_rx_chk_enable_eth(i)(1);

  -- HDLC data generator and monitor
  hdlc_tx_rx_i : entity work.hdlc_stim
  port map(
    clk                    => clk(i),
    clk_ok                 => clk_ok(i),
    reset                  => reset_s(i),
    hdlc_rx_data           => hdlc(i).rx_data(0),
    hdlc_tx_data           => hdlc(i).tx_data(0),
    hdlc_rx_data_valid     => hdlc(i).rx_data_valid(0),
    hdlc_tx_enable         => hdlc(i).tx_enable(0),
    start_hdlc             => cntrl(i).test_start_clk(0),
    bit_count              => hdlc(i).bit_count,
    error_count            => hdlc(i).error_count,
    hdlc_error             => hdlc(i).error(0)
  );

  -- Ethernet data generator and monitor
  gmii_tx_rx_i : entity work.gmii_stim
  port map(
    eth_clk                => eth_clk,
    tx_eth_enable          => eth(i).tx_en(0),
    tx_eth_data            => eth(i).txd,
    tx_eth_err             => eth(i).tx_er(0),
    tx_eth_overflow        => eth(i).col(0),
    tx_eth_half            => eth(i).crs(0),
    rx_eth_data_valid      => eth(i).rx_dv(0),
    rx_eth_data            => eth(i).rxd,
    rx_eth_err             => eth(i).rx_er(0),
    tx_start               => cntrl(i).test_start_eth(0),
    rx_start               => cntrl(i).test_start_eth(0),
    tx_count               => eth(i).tx_count,
    rx_count               => eth(i).rx_frame_count,
    err_count              => eth(i).error_count,
    eth_rx_error           => eth(i).error(0),
    eth_rx_avail           => eth(i).rx_avail,
    eth_rx_ready           => eth(i).rx_ready,
    rx_fifo_almost_full    => eth(i).rx_fifo_almost_full,
    rx_fifo_full           => eth(i).rx_fifo_full
  );

  -- Vendor specific data generator and monitor
  vendor_tx_rx_i : entity work.vendor_stim
  port map(
    clk                    => clk(i),
    reset                  => reset_s(i),
    iq_tx_enable           => iq(i).tx_enable(0),
    bffw                   => iq(i).basic_frame_first_word(0),
    vendor_tx_data         => vendor(i).tx_data,
    vendor_tx_xs           => vendor(i).tx_xs,
    vendor_tx_ns           => vendor(i).tx_ns,
    vendor_rx_data         => vendor(i).rx_data,
    vendor_rx_xs           => vendor(i).rx_xs,
    vendor_rx_ns           => vendor(i).rx_ns,
    start_vendor           => cntrl(i).test_start_clk(0),
    word_count             => vendor(i).vs_word_count,
    error_count            => vendor(i).vs_error_count,
    vendor_error           => vendor(i).vs_error(0),
    speed                  => stat_speed_clk(i)(2)
  );
 
  -- Clock Monitor outputs on GPIO SMAs  
  -- Buffer refclk for Logic use and Output monitor
  refclk_bufg : BUFG_GT
  port map (
    I       => refclki(i),
    CE      => '1',
    CEMASK  => '1',
    CLR     => '0',
    CLRMASK => '1',
    DIV     => "000",
    O       => refclki_bufg(i)
  );

  -------------------------------------------------------------------------------
  -- Resets
  -------------------------------------------------------------------------------
  
  -- Vivado VIO is synchronous only, so a VIO reset needs to be
  -- handled appropriately using a clock which is not reset by the
  -- reset itself
  process (clk300)
  begin
    if rising_edge(clk300) then
      -- handle as async to clk300
      vio_reset_d(i)(0) <= vio(i).reset;
      vio_reset_d(i)(1) <= vio_reset_d(i)(0);
      vio_reset_d(i)(2) <= vio_reset_d(i)(1);
    
      -- vio_reset rising edge creates a reset pulse to the mmcm
      if (vio_reset_d(i)(1)(0) = '1' and vio_reset_d(i)(2)(0) = '0') then
        vio(i).reset_sr <= "11111";
      else
        vio(i).reset_sr <= vio(i).reset_sr(3 downto 0) & '0';
      end if;
      
      vio(i).reset_p <= vio(i).reset_sr(4);
    end if;
  end process; 

  -- hold the AXI managment interface (& FSM) reset until refclk is up and CPRI core
  -- reset is released
  s_axi(i).aresetn <= refclk_ok(0);

  -- synchronise reset for blocks that use this reset in a synchronous fashion
  process (clk(i), reset)
  begin
    if (reset = '1') then
      reset_s(i) <= '1';
    elsif rising_edge(clk(i)) then
      reset_s(i) <= '0';
    end if;
  end process;

  -------------------------------------------------------------------------------
  -- VIO - Control and Management
  -------------------------------------------------------------------------------
  mgmnt(i).cmd(MGMNT_INDX_LINE_RATE_0) <= line_rate_12G(i)(0);
  mgmnt(i).cmd(MGMNT_INDX_LINE_RATE_1) <= line_rate_10G(i)(0);
  mgmnt(i).cmd(MGMNT_INDX_LINE_RATE_2) <= line_rate_9G(i)(0);
  mgmnt(i).cmd(MGMNT_INDX_LINE_RATE_3) <= line_rate_8G(i)(0);

  i_vio_control : vio_control 
  port map(
    clk           => aux_clk,
    probe_in0     => mgmnt(i).rd_data,             -- input [31 : 0] probe_in0
    probe_out0    => mgmnt(i).read_req,            -- output [0 : 0] probe_out0
    probe_out1    => mgmnt(i).write_req,           -- output [0 : 0] probe_out1
    probe_out2    => mgmnt(i).addr,                -- output [11 : 0] probe_out2
    probe_out3    => mgmnt(i).wr_data,             -- output [31 : 0] probe_out3
    probe_out4    => vio(i).reset,                 -- output [0 : 0] probe_out4
    probe_out5    => vio(i).reset_req,             -- output [0 : 0] probe_out5
    probe_out6    => s_axi(i).sdi_request_in,      -- output [0 : 0] probe_out6
    probe_out7(0) => cntrl(i).core_is_master(0),   -- output [0 : 0] probe_out7
    probe_out8(0) => cntrl(i).core_in_loopback(0), -- output [0 : 0] probe_out7
    probe_out9    => line_rate_12G(i),             -- output [0 : 0] probe_out9
    probe_out10   => line_rate_10G(i),             -- output [0 : 0] probe_out10
    probe_out11   => line_rate_9G(i),              -- output [0 : 0] probe_out11
    probe_out12   => line_rate_8G(i),              -- output [0 : 0] probe_out12
    probe_out13   => cntrl(i).pma_loopback         -- output [0 : 0] probe_out13
  );

  -- AXI interface
  -- State machine for controlling writes and reads to and from the
  -- AXI management registers via VIO
  p_mgmnt_states : process(aux_clk, reset)
  begin
    if (reset = '1') then
      MGMNT_FSM_RST( nstate        => IDLE
                   , s_axi_awaddr  => s_axi(i).awaddr
                   , s_axi_awvalid => s_axi(i).awvalid
                   , s_axi_wdata   => s_axi(i).wdata
                   , s_axi_bready  => s_axi(i).bready
                   , s_axi_wvalid  => s_axi(i).wvalid
                   , s_axi_araddr  => s_axi(i).araddr
                   , s_axi_arvalid => s_axi(i).arvalid
                   , s_axi_rready  => s_axi(i).rready
                   , mgmnt_state   => mgmnt(i).state );
    elsif rising_edge(aux_clk) then
      case mgmnt(i).state is
        when IDLE =>
          MGMNT_FSM_IDLE(  mgmnt_cmd_lst => MGMNT_CMD_ROM
                        ,  write_req     => mgmnt(i).write_req
                        ,  read_req      => mgmnt(i).read_req
                        ,  mgmnt_cmd     => mgmnt(i).cmd
                        ,  pma_loopback  => cntrl(i).pma_loopback
                        ,  s_axi_wready  => s_axi(i).wready
                        ,  s_axi_awaddr  => s_axi(i).awaddr
                        ,  s_axi_awvalid => s_axi(i).awvalid
                        ,  s_axi_wdata   => s_axi(i).wdata
                        ,  s_axi_wvalid  => s_axi(i).wvalid
                        ,  s_axi_bready  => s_axi(i).bready
                        ,  s_axi_araddr  => s_axi(i).araddr
                        ,  s_axi_arvalid => s_axi(i).arvalid
                        ,  s_axi_rready  => s_axi(i).rready
                        ,  mgmnt_state   => mgmnt(i).state
                        ,  axi_cmd       => mgmnt(i).axi_cmd );
        when TO_IDLE =>
          MGMNT_FSM_TO_IDLE( s_axi_bvalid  => s_axi(i).bvalid
                           , s_axi_bready  => s_axi(i).bready
                           , mgmnt_state   => mgmnt(i).state );
        when MGMNT_WR0 =>
          MGMNT_FSM_WR0(   mgmnt_addr    => mgmnt(i).addr
                       ,   mgmnt_wr_data => mgmnt(i).wr_data
                       ,   s_axi_wready  => s_axi(i).wready
                       ,   s_axi_awaddr  => s_axi(i).awaddr
                       ,   s_axi_awvalid => s_axi(i).awvalid
                       ,   s_axi_wdata   => s_axi(i).wdata
                       ,   s_axi_wvalid  => s_axi(i).wvalid
                       ,   mgmnt_state   => mgmnt(i).state );
        when MGMNT_RD0 =>
          MGMNT_FSM_RD0(   mgmnt_addr    => mgmnt(i).addr
                       ,   s_axi_arready => s_axi(i).arready
                       ,   s_axi_araddr  => s_axi(i).araddr
                       ,   s_axi_arvalid => s_axi(i).arvalid
                       ,   s_axi_rready  => s_axi(i).rready
                       ,   mgmnt_state   => mgmnt(i).state );
        when MGMNT_RD1 =>
          MGMNT_FSM_RD1(   s_axi_rdata   => s_axi(i).rdata
                       ,   s_axi_rvalid  => s_axi(i).rvalid
                       ,   s_axi_arvalid => s_axi(i).arvalid
                       ,   s_axi_rready  => s_axi(i).rready
                       ,   mgmnt_rd_data => mgmnt(i).rd_data
                       ,   mgmnt_state   => mgmnt(i).state );
        when MGMNT_WRA =>
          MGMNT_FSM_WREG(  axi_data      => mgmnt(i).axi_cmd
                        ,  s_axi_wready  => s_axi(i).wready
                        ,  s_axi_awaddr  => s_axi(i).awaddr
                        ,  s_axi_awvalid => s_axi(i).awvalid
                        ,  s_axi_wdata   => s_axi(i).wdata
                        ,  s_axi_wvalid  => s_axi(i).wvalid
                        ,  mgmnt_state   => mgmnt(i).state );
      end case;
    end if;
  end process;

  -- Discrete Control outputs
  s_axi(i).reset_request_in <= vio(i).reset_req(0) or s_axi(i).reset_request;

  s_axi(i).reset_request    <= '1' when stat(i).code /= "1111" else '0';

  -- Set the core into slave mode via the VIO
  cntrl(i).core_is_master(1)   <= cntrl(i).core_is_master(0);

  -- Set the core to work in loopback mode via the VIO
  -- This is only neccessary in cores running in slave mode.
  cntrl(i).core_in_loopback(1) <= cntrl(i).core_in_loopback(0);

  --------------------------------
  -- VIO - Status Monitor
  --------------------------------
  i_vio_status : vio_status
  PORT MAP (
    clk           => aux_clk,
    probe_in0     => iq(i).frame_count(31 downto 16),
    probe_in1     => iq(i).error_count(15 downto 0),
    probe_in2     => eth(i).rx_frame_count(31 downto 16),
    probe_in3     => eth(i).error_count(15 downto 0),
    probe_in4     => hdlc(i).bit_count(31 downto 16),
    probe_in5     => hdlc(i).error_count(15 downto 0),
    probe_in6     => vendor(i).vs_word_count(31 downto 16),
    probe_in7     => vendor(i).vs_error_count(15 downto 0),
    probe_in8     => s_axi(i).fifo_transit_time,
    probe_in9     => s_axi(i).coarse_timer_value,
    probe_in10    => s_axi(i).barrel_shift_value,
    probe_in11    => s_axi(i).reset_acknowledge_out,
    probe_in12    => s_axi(i).sdi_request_out,
    probe_in13    => s_axi(i).local_los,
    probe_in14    => s_axi(i).local_lof,
    probe_in15    => s_axi(i).local_rai,
    probe_in16    => s_axi(i).remote_los,
    probe_in17    => s_axi(i).remote_lof,
    probe_in18    => s_axi(i).remote_rai,
    probe_in19    => stat(i).alarm,
    probe_in20    => stat(i).speed5,
    probe_in21    => state(i).reset,
    probe_in22    => state(i).l1sync,
    probe_in23    => state(i).proto,
    probe_in24    => state(i).cm,
    probe_in25    => state(i).passive,
    probe_in26    => state(i).vs,
    probe_in27    => state(i).operate,
    probe_in28    => std_logic_vector(delay(i).r21_timer),
    probe_in29    => mmcm_locked,
    probe_in30(0) => clk_ok(i),
    probe_in31    => iic_done,
    probe_in32(0) => refclk_ok(0),
    probe_in33    => s_axi(i).tx_gb_latency_value,
    probe_in34    => s_axi(i).rx_gb_latency_value
  );

  -- Encode speed to :
  --  - 5-bit for VIO display (decodes to line rate when displayed as unsigned)
  --  - 4-bit for LEDs (decodes to line rate when displayed as unsigned)
  speed_2bit_proc_m : process( stat(i).speed )
  begin
    stat(i).speed4 <= onehot_to_4b( stat(i).speed );
    stat(i).speed5 <= onehot_to_5b( stat(i).speed );
  end process;

  -- Decode Rx State to individual bits
  state(i).reset(0)   <= '1' when stat(i).code = "0000" else '0';
  state(i).l1sync(0)  <= '1' when stat(i).code = "0001" else '0';
  state(i).proto(0)   <= '1' when stat(i).code = "0010" else '0';
  state(i).cm(0)      <= '1' when stat(i).code = "0011" else '0';
  state(i).passive(0) <= '1' when stat(i).code = "1011" else '0';
  state(i).vs(0)      <= '1' when stat(i).code = "1110" else '0';
  state(i).operate(0) <= '1' when stat(i).code = "1111" else '0';

  i_ila_tx : ila_0
  port map (
    clk     =>  clk(i),
    probe0  =>  sync(i).nodebfn_tx_strobe,
    probe1  =>  sync(i).nodebfn_tx_nr,
    probe2  =>  iq(i).tx_enable,
    probe3  =>  iq(i).tx,
    probe4  =>  hdlc(i).tx_data,
    probe5  =>  hdlc(i).tx_enable,
    probe6  =>  vendor(i).tx_xs,
    probe7  =>  vendor(i).tx_ns,
    probe8  =>  vendor(i).tx_data(15 downto 0),
    probe9  =>  eth(i).txd,
    probe10 =>  eth(i).tx_er,
    probe11 =>  eth(i).tx_en,
    probe12 =>  eth(i).col,
    probe13 =>  eth(i).crs,
    probe14 =>  s_axi(i).awvalid,
    probe15 =>  s_axi(i).wvalid,
    probe16 =>  s_axi(i).wready,
    probe17 =>  s_axi(i).awready,
    probe18 =>  s_axi(i).bready,
    probe19 =>  s_axi(i).bvalid,
    probe20 =>  s_axi(i).arvalid,
    probe21 =>  s_axi(i).rvalid,
    probe22 =>  s_axi(i).arready,
    probe23 =>  s_axi(i).rready
  );

  i_ila_rx : ila_1
  PORT MAP (
    clk     => clk(i),
    probe0  => sync(i).nodebfn_rx_strobe,
    probe1  => sync(i).nodebfn_rx_nr,
    probe2  => iq(i).basic_frame_first_word,
    probe3  => iq(i).rx,
    probe4  => iq(i).error,
    probe5  => stat_code_clk(i)(0),
    probe6  => stat_speed_clk(i)(2),
    probe7  => hdlc(i).rx_data,
    probe8  => hdlc(i).rx_data_valid,
    probe9  => hdlc(i).error,
    probe10 => vendor(i).rx_xs,
    probe11 => vendor(i).rx_ns,
    probe12 => vendor(i).rx_data(15 downto 0),
    probe13 => vendor(i).vs_error,
    probe14 => eth(i).rxd,
    probe15 => eth(i).rx_er,
    probe16 => eth(i).rx_dv,
    probe17 => eth(i).error
  );
  
  -- Calculate R-21 measurement, T14 in master and Toffset in slave.
  -- b,c and d refer to the steps in the calculation descibed in 
  -- the "Performing the Cable Delay Calculation" sub-section in
  -- the CPRI product guide.
  p_calc_b : process(aux_clk)
  begin
    if rising_edge(aux_clk) then
      delay(i).b <= b_step_delay( speed              => stat(i).speed
                                , rx_delay_value     => stat(i).rx_delay_value
                                , barrel_shift_value => s_axi(i).barrel_shift_value );
    end if;
  end process;

  p_calc_c : process(aux_clk)
  begin
    if rising_edge(aux_clk) then
      delay(i).c <= c_step_delay( speed               => stat(i).speed
                                , fifo_transit_time   => s_axi(i).fifo_transit_time
                                , rx_gb_latency_value => s_axi(i).rx_gb_latency_value );
    end if;
  end process;

  p_calc_c_tx : process(aux_clk)
  begin
    if rising_edge(aux_clk) then
      delay(i).c_tx <= ctx_step_delay( speed               => stat(i).speed
                                     , tx_gb_latency_value => s_axi(i).tx_gb_latency_value );
    end if;
  end process;

  p_calc_d : process(aux_clk)
  begin
    if rising_edge(aux_clk) then
      delay(i).d <= d_step_delay( speed => stat(i).speed
                                , b     => delay(i).b
                                , c     => delay(i).c
                                , c_tx  => delay(i).c_tx );
    end if;
  end process;

  p_calc_r21 : process(aux_clk)
  begin
    if rising_edge(aux_clk) then
      delay(i).r21_timer <= r21_step_delay( speed              => stat(i).speed
                                          , d                  => delay(i).d
                                          , core_is_master     => cntrl(i).core_is_master(1)
                                          , coarse_timer_value => s_axi(i).coarse_timer_value );
    end if;
  end process;
end generate cpri_gen_i;

end rtl;