-- Title      : Example top level for CPRI design on Virtex Ultrascale+
-- Project    : CPRI
-----------------------------------------------------------------------
-- File       : cpri_vcu118_64bit.vhd
-- Author     : Xilinx
-----------------------------------------------------------------------
-- Description:
--   This example design for the 64-bit CPRI core is designed to be
--   hosted on the VCU118 Evalutation Board using Chipscope to provide
--   Control and Monitoring of the CPRI core.
--
-----------------------------------------------------------------------
-- (c) Copyright 2004 - 2018 Xilinx, Inc. All rights reserved.
--
-- This file contains confidential and proprietary information
-- of Xilinx, Inc. and is protected under U.S. and
-- international copyright and other intellectual property
-- laws.
--
-- DISCLAIMER
-- This disclaimer is not a license and does not grant any
-- rights to the materials distributed herewith. Except as
-- otherwise provided in a valid license issued to you by
-- Xilinx, and to the maximum extent permitted by applicable
-- law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
-- WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
-- AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
-- BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
-- INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
-- (2) Xilinx shall not be liable (whether in contract or tort,
-- including negligence, or under any other theory of
-- liability) for any loss or damage of any kind or nature
-- related to, arising under or in connection with these
-- materials, including for any direct, or any indirect,
-- special, incidental, or consequential loss or damage
-- (including loss of data, profits, goodwill, or any type of
-- loss or damage suffered as a result of any action brought
-- by a third party) even if such damage or loss was
-- reasonably foreseeable or Xilinx had been advised of the
-- possibility of the same.
--
-- CRITICAL APPLICATIONS
-- Xilinx products are not designed or intended to be fail-
-- safe, or for use in any application requiring fail-safe
-- performance, such as life-support or safety devices or
-- systems, Class III medical devices, nuclear facilities,
-- applications related to the deployment of airbags, or any
-- other applications that could lead to death, personal
-- injury, or severe property or environmental damage
-- (individually and collectively, "Critical
-- Applications"). Customer assumes the sole risk and
-- liability of any use of Xilinx products in Critical
-- Applications, subject only to applicable laws and
-- regulations governing limitations on product liability.
--
-- THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
-- PART OF THIS FILE AT ALL TIMES.
-----------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

library unisim;
use unisim.vcomponents.all;

library work;
use work.iic_controller_pkg.all;

entity cpri_vcu118 is
port (
  -- Clocks
  clk300_p       : in std_logic;
  clk300_n       : in std_logic;

  -- clock in from Si570 as GTY ref clock
  refclk_p       : in  std_logic;
  refclk_n       : in  std_logic;

  -- Clock Control
  iic_sda        : inout std_logic;
  iic_scl        : inout std_logic;

  iic_mux_rst_n  : out std_logic;

  -- Reset (push button)
  pb_reset       : in  std_logic;

  -- Status Leds
  leds           : out std_logic_vector(7 downto 0);

  -- Clock Monitor ports
  core_clk_mon   : out std_logic;
  clk_si570_mon  : out std_logic;

  -- CPRI link
  txp            : out std_logic;
  txn            : out std_logic;
  rxp            : in  std_logic;
  rxn            : in  std_logic
);

package p is new work.cpri_demo_pkg
  generic map (N_CPRI => 1,
               DEMO   => vcu118);

use p.all;

end cpri_vcu118;

architecture rtl of cpri_vcu118 is

  --=========================================
  -- CPRI core (block level)

  component cpri_0
  port (
    -- Additional Debug Ports
    -- DRP Access
    gt_drpdaddr                : in  std_logic_vector(9 downto 0);
    gt_drpdi                   : in  std_logic_vector(15 downto 0);
    gt_drpen                   : in  std_logic;
    gt_drpwe                   : in  std_logic;
    gt_drpdo                   : out std_logic_vector(15 downto 0);
    gt_drprdy                  : out std_logic;
    -- TX Reset and Initialization
    gt_txpmareset              : in  std_logic;
    gt_txpcsreset              : in  std_logic;
    gt_txresetdone             : out std_logic;
    -- RX Reset and Initialization
    gt_rxpmareset              : in  std_logic;
    gt_rxpcsreset              : in  std_logic;
    gt_rxpmaresetdone          : out std_logic;
    gt_rxresetdone             : out std_logic;
    -- Clocking
    gt_txphaligndone           : out std_logic;
    gt_txphinitdone            : out std_logic;
    gt_txdlysresetdone         : out std_logic;
    gt_rxphaligndone           : out std_logic;
    gt_rxdlysresetdone         : out std_logic;
    gt_rxsyncdone              : out std_logic;
    gt_cplllock                : out std_logic;
    gt_qplllock                : out std_logic;
    -- Signal Integrity and Functionality
    gt_rxrate                  : in  std_logic_vector(2 downto 0);
    gt_eyescantrigger          : in  std_logic;
    gt_eyescanreset            : in  std_logic;
    gt_eyescandataerror        : out std_logic;
    gt_rxpolarity              : in  std_logic;
    gt_txpolarity              : in  std_logic;
    gt_rxdfelpmreset           : in  std_logic;
    gt_rxlpmen                 : in  std_logic;
    gt_txprecursor             : in  std_logic_vector(4 downto 0);
    gt_txpostcursor            : in  std_logic_vector(4 downto 0);
    gt_txdiffctrl              : in  std_logic_vector(4 downto 0);
    gt_txprbsforceerr          : in  std_logic;
    gt_txprbssel               : in  std_logic_vector(3 downto 0);
    gt_rxprbssel               : in  std_logic_vector(3 downto 0);
    gt_rxprbserr               : out std_logic;
    gt_rxprbscntreset          : in  std_logic;
    gt_rxcdrhold               : in  std_logic;
    gt_dmonitorout             : out std_logic_vector(15 downto 0);
    gt_rxheader                : out std_logic_vector(1 downto 0);
    gt_rxheadervalid           : out std_logic;
    gt_rxdisperr               : out std_logic_vector(7 downto 0);
    gt_rxnotintable            : out std_logic_vector(7 downto 0);
    gt_rxcommadet              : out std_logic;
    gt_pcsrsvdin               : in  std_logic_vector(15 downto 0);
    -- Transceiver monitor interface
    txdata                     : out std_logic_vector(63 downto 0);
    txcharisk                  : out std_logic_vector(7 downto 0);
    txheader                   : out std_logic_vector(1 downto 0);
    txsequence                 : out std_logic_vector(6 downto 0);
    rxdata                     : out std_logic_vector(63 downto 0);
    rxchariscomma              : out std_logic_vector(7 downto 0);
    rxcharisk                  : out std_logic_vector(7 downto 0);
    rxdisperr                  : out std_logic_vector(7 downto 0);
    rxnotintable               : out std_logic_vector(7 downto 0);
    rxdatavalid                : out std_logic;
    rxheader                   : out std_logic_vector(1 downto 0);
    rxheadervalid              : out std_logic;
    rxgearboxslip              : out std_logic;
    ----------------- END OF DEBUG PORTS -----------------

    reset                      : in  std_logic;
    -- I/Q interface common
    iq_tx_enable               : out std_logic;
    basic_frame_first_word     : out std_logic;
    iq_tx                      : in  std_logic_vector (63 downto 0);
    iq_rx                      : out std_logic_vector (63 downto 0);
    -- GT Common Ports
    qpll0clk_out               : out std_logic;
    qpll0refclk_out            : out std_logic;
    qpll1clk_out               : out std_logic;
    qpll1refclk_out            : out std_logic;
    qpll0lock_out              : out std_logic;
    qpll1lock_out              : out std_logic;
    -- Vendor Specific Data
    vendor_tx_data             : in  std_logic_vector(127 downto 0);
    vendor_tx_xs               : out std_logic_vector(1 downto 0);
    vendor_tx_ns               : out std_logic_vector(5 downto 0);
    vendor_rx_data             : out std_logic_vector(127 downto 0);
    vendor_rx_xs               : out std_logic_vector(1 downto 0);
    vendor_rx_ns               : out std_logic_vector(5 downto 0);
    vs_negotiation_complete    : in  std_logic;
    -- Synchronization
    nodebfn_tx_strobe          : in  std_logic;
    nodebfn_tx_nr              : in  std_logic_vector(11 downto 0);
    nodebfn_rx_strobe          : out std_logic;
    nodebfn_rx_nr              : out std_logic_vector(11 downto 0);
    -- Ethernet interface
    eth_txd                    : in  std_logic_vector(7 downto 0);
    eth_tx_er                  : in  std_logic;
    eth_tx_en                  : in  std_logic;
    eth_col                    : out std_logic;
    eth_crs                    : out std_logic;
    eth_rxd                    : out std_logic_vector(7 downto 0);
    eth_rx_er                  : out std_logic;
    eth_rx_dv                  : out std_logic;
    eth_rx_avail               : out std_logic;
    eth_rx_ready               : in  std_logic;
    rx_fifo_almost_full        : out std_logic;
    rx_fifo_full               : out std_logic;
    eth_tx_clk                 : in  std_logic;
    eth_rx_clk                 : in  std_logic;
    -- HDLC interface
    hdlc_rx_data               : out std_logic;
    hdlc_tx_data               : in  std_logic;
    hdlc_rx_data_valid         : out std_logic;
    hdlc_tx_enable             : out std_logic;
    -- Status interface
    stat_alarm                 : out std_logic;
    stat_code                  : out std_logic_vector(3 downto 0);
    stat_speed                 : out std_logic_vector(14 downto 0);
    reset_request_in           : in  std_logic;
    sdi_request_in             : in  std_logic;
    reset_acknowledge_out      : out std_logic;
    sdi_request_out            : out std_logic;
    local_los                  : out std_logic;
    local_lof                  : out std_logic;
    local_rai                  : out std_logic;
    remote_los                 : out std_logic;
    remote_lof                 : out std_logic;
    remote_rai                 : out std_logic;
    fifo_transit_time          : out std_logic_vector(13 downto 0);
    coarse_timer_value         : out std_logic_vector(17 downto 0);
    barrel_shift_value         : out std_logic_vector(6 downto 0);
    tx_gb_latency_value        : out std_logic_vector(15 downto 0);
    rx_gb_latency_value        : out std_logic_vector(15 downto 0);
	hfec_fifo_latency_value    : out std_logic_vector(15 downto 0);
    hyperframe_number          : out std_logic_vector(7 downto 0);
    -- AXI-Lite Interface
    s_axi_aclk                 : in  std_logic;
    s_axi_aresetn              : in  std_logic;
    s_axi_awaddr               : in  std_logic_vector(11 downto 0);
    s_axi_awvalid              : in  std_logic;
    s_axi_awready              : out std_logic;
    s_axi_wdata                : in  std_logic_vector(31 downto 0);
    s_axi_wvalid               : in  std_logic;
    s_axi_wready               : out std_logic;
    s_axi_bresp                : out std_logic_vector(1 downto 0);
    s_axi_bvalid               : out std_logic;
    s_axi_bready               : in  std_logic;
    s_axi_araddr               : in  std_logic_vector(11 downto 0);
    s_axi_arvalid              : in  std_logic;
    s_axi_arready              : out std_logic;
    s_axi_rdata                : out std_logic_vector(31 downto 0);
    s_axi_rresp                : out std_logic_vector(1 downto 0);
    s_axi_rvalid               : out std_logic;
    s_axi_rready               : in  std_logic;
    l1_timer_expired           : in  std_logic;
    -- Transceiver signals
    txp                        : out std_logic;
    txn                        : out std_logic;
    rxp                        : in  std_logic;
    rxn                        : in  std_logic;
    lossoflight                : in  std_logic;
    txinhibit                  : out std_logic;
    -- Clocks
    refclk                     : in  std_logic;
    gtwiz_reset_clk_freerun_in : in std_logic;
    core_is_master             : in  std_logic;
    hires_clk                  : in  std_logic;
    hires_clk_ok               : in  std_logic;
    qpll_select                : in  std_logic;
    recclk_ok                  : out std_logic;
    clk_ok_out                 : out std_logic;
    recclk                     : out std_logic;
    clk_out                    : out std_logic;
    rxrecclkout                : out std_logic;
    txusrclk_out               : out std_logic;
    -- Tx alignment
    txphinit_out               : out std_logic_vector(2 downto 0);
    phase_alignment_done_out   : out std_logic;
    txdlysreset_out            : out std_logic_vector(2 downto 0);
    txphalign_out              : out std_logic_vector(2 downto 0);
    txdlyen_out                : out std_logic_vector(2 downto 0);
    txphaligndone_in           : in  std_logic_vector(2 downto 0);
    txdlysresetdone_in         : in  std_logic_vector(2 downto 0);
    txphinitdone_in            : in  std_logic_vector(2 downto 0);
    -- Interface for external CPRI cores Hard FEC support
    -- Hard FEC control - external CPRI cores
    rx_rsfec_enable_ch1         : in  std_logic;
    rx_rsfec_enable_ch2         : in  std_logic;
    rx_rsfec_enable_ch3         : in  std_logic;
    -- Ingress data - external CPRI cores
    rx_rec_clk_ch1              : in  std_logic;
    rx_serdes_data_ch1          : in  std_logic_vector(63 downto 0);
    rx_serdes_head_ch1          : in  std_logic_vector( 1 downto 0);
    rx_gbx_slip_ch1             : out std_logic;
    rx_rec_clk_ch2              : in  std_logic;
    rx_serdes_data_ch2          : in  std_logic_vector(63 downto 0);
    rx_serdes_head_ch2          : in  std_logic_vector( 1 downto 0);
    rx_gbx_slip_ch2             : out std_logic;
    rx_rec_clk_ch3              : in  std_logic;
    rx_serdes_data_ch3          : in  std_logic_vector(63 downto 0);
    rx_serdes_head_ch3          : in  std_logic_vector( 1 downto 0);
    rx_gbx_slip_ch3             : out std_logic;
    -- CDC FIFO control & status - external CPRI cores
    cdc_reset_ch1               : in  std_logic;
    fifo_fill_level_ch1         : in  std_logic_vector(6 downto 0);
    average_ch1                 : out std_logic_vector(15 downto 0);
    average_rdy_ch1             : out std_logic;
    average_ack_ch1             : in  std_logic;
    fifo_error_ch1              : out std_logic;
    cdc_reset_ch2               : in  std_logic;
    fifo_fill_level_ch2         : in  std_logic_vector(6 downto 0);
    average_ch2                 : out std_logic_vector(15 downto 0);
    average_rdy_ch2             : out std_logic;
    average_ack_ch2             : in  std_logic;
    fifo_error_ch2              : out std_logic;
    cdc_reset_ch3               : in  std_logic;
    fifo_fill_level_ch3         : in  std_logic_vector(6 downto 0);
    average_ch3                 : out std_logic_vector(15 downto 0);
    average_rdy_ch3             : out std_logic;
    average_ack_ch3             : in  std_logic;
    fifo_error_ch3              : out std_logic;
    -- CDC FIFO output (no FEC decode)
    cdc_rxdata_ch1              : out std_logic_vector(63 downto 0);
    cdc_rxheader_ch1            : out std_logic_vector( 1 downto 0);
    cdc_rxdata_ch2              : out std_logic_vector(63 downto 0);
    cdc_rxheader_ch2            : out std_logic_vector( 1 downto 0);
    cdc_rxdata_ch3              : out std_logic_vector(63 downto 0);
    cdc_rxheader_ch3            : out std_logic_vector( 1 downto 0);
    -- Egress data & status - external CPRI cores
    pcs_txclk_ch1               : out std_logic;
    pcs_txclk_ok_ch1            : out std_logic;
    pcs_rxdata_ch1              : out std_logic_vector(63 downto 0);
    pcs_rxheader_ch1            : out std_logic_vector( 1 downto 0);
    pcs_rxheadervalid_ch1       : out std_logic;
    stat_rx_align_status_ch1    : out std_logic;
    stat_cw_inc_ch1             : out std_logic;
    stat_corrected_cw_inc_ch1   : out std_logic;
    stat_uncorrected_cw_inc_ch1 : out std_logic;
	hfec_fifo_latency_ch1       : out std_logic_vector(15 downto 0);
	hfec_fifo_latency_rdy_ch1   : out std_logic;
	hfec_fifo_latency_ack_ch1   : in  std_logic;
    pcs_txclk_ch2               : out std_logic;
    pcs_txclk_ok_ch2            : out std_logic;
    pcs_rxdata_ch2              : out std_logic_vector(63 downto 0);
    pcs_rxheader_ch2            : out std_logic_vector( 1 downto 0);
    pcs_rxheadervalid_ch2       : out std_logic;
    stat_rx_align_status_ch2    : out std_logic;
    stat_cw_inc_ch2             : out std_logic;
    stat_corrected_cw_inc_ch2   : out std_logic;
    stat_uncorrected_cw_inc_ch2 : out std_logic;
	hfec_fifo_latency_ch2       : out std_logic_vector(15 downto 0);
	hfec_fifo_latency_rdy_ch2   : out std_logic;
	hfec_fifo_latency_ack_ch2   : in  std_logic;
    pcs_txclk_ch3               : out std_logic;
    pcs_txclk_ok_ch3            : out std_logic;
    pcs_rxdata_ch3              : out std_logic_vector(63 downto 0);
    pcs_rxheader_ch3            : out std_logic_vector( 1 downto 0);
    pcs_rxheadervalid_ch3       : out std_logic;
    stat_rx_align_status_ch3    : out std_logic;
    stat_cw_inc_ch3             : out std_logic;
    stat_corrected_cw_inc_ch3   : out std_logic;
    stat_uncorrected_cw_inc_ch3 : out std_logic;
	hfec_fifo_latency_ch3       : out std_logic_vector(15 downto 0);
	hfec_fifo_latency_rdy_ch3   : out std_logic;
	hfec_fifo_latency_ack_ch3   : in  std_logic
  );
  end component;

  -- VIO - Control & Management
  component vio_control
  port (
    clk         : in  std_logic;
    probe_in0   : in  std_logic_vector(31 downto 0);
    probe_out0  : out std_logic_vector(0 downto 0);
    probe_out1  : out std_logic_vector(0 downto 0);
    probe_out2  : out std_logic_vector(11 downto 0);
    probe_out3  : out std_logic_vector(31 downto 0);
    probe_out4  : out std_logic_vector(0 downto 0);
    probe_out5  : out std_logic_vector(0 downto 0);
    probe_out6  : out std_logic_vector(0 downto 0);
    probe_out7  : out std_logic_vector(0 downto 0);
    probe_out8  : out std_logic_vector(0 downto 0);
    probe_out9  : out std_logic_vector(0 downto 0);
    probe_out10 : out std_logic_vector(0 downto 0);
    probe_out11 : out std_logic_vector(0 downto 0);
    probe_out12 : out std_logic_vector(0 downto 0);
    probe_out13 : out std_logic_vector(0 downto 0);
    probe_out14 : out std_logic_vector(0 downto 0);
    probe_out15 : out std_logic_vector(0 downto 0);
    probe_out16 : out std_logic_vector(0 downto 0)
  );
  end component;

  component vio_status
  port (
    clk        : in std_logic;
    probe_in0  : in std_logic_vector(15 downto 0);
    probe_in1  : in std_logic_vector(15 downto 0);
    probe_in2  : in std_logic_vector(15 downto 0);
    probe_in3  : in std_logic_vector(15 downto 0);
    probe_in4  : in std_logic_vector(15 downto 0);
    probe_in5  : in std_logic_vector(15 downto 0);
    probe_in6  : in std_logic_vector(15 downto 0);
    probe_in7  : in std_logic_vector(15 downto 0);
    probe_in8  : in std_logic_vector(13 downto 0);
    probe_in9  : in std_logic_vector(17 downto 0);
    probe_in10 : in std_logic_vector(6 downto 0);
    probe_in11 : in std_logic_vector(0 downto 0);
    probe_in12 : in std_logic_vector(0 downto 0);
    probe_in13 : in std_logic_vector(0 downto 0);
    probe_in14 : in std_logic_vector(0 downto 0);
    probe_in15 : in std_logic_vector(0 downto 0);
    probe_in16 : in std_logic_vector(0 downto 0);
    probe_in17 : in std_logic_vector(0 downto 0);
    probe_in18 : in std_logic_vector(0 downto 0);
    probe_in19 : in std_logic_vector(0 downto 0);
    probe_in20 : in std_logic_vector(4 downto 0);
    probe_in21 : in std_logic_vector(0 downto 0);
    probe_in22 : in std_logic_vector(0 downto 0);
    probe_in23 : in std_logic_vector(0 downto 0);
    probe_in24 : in std_logic_vector(0 downto 0);
    probe_in25 : in std_logic_vector(0 downto 0);
    probe_in26 : in std_logic_vector(0 downto 0);
    probe_in27 : in std_logic_vector(0 downto 0);
    probe_in28 : in std_logic_vector(35 downto 0);
    probe_in29 : in std_logic_vector(0 downto 0);
    probe_in30 : in std_logic_vector(0 downto 0);
    probe_in31 : in std_logic_vector(0 downto 0);
    probe_in32 : in std_logic_vector(0 downto 0);
    probe_in33 : in std_logic_vector(15 downto 0);
    probe_in34 : in std_logic_vector(15 downto 0);
	probe_in35 : in std_logic_vector(15 downto 0)
  );
  end component;

  component ila_0
  port (
    clk     : in std_logic;
    probe0  : in std_logic_vector(0 downto 0);
    probe1  : in std_logic_vector(11 downto 0);
    probe2  : in std_logic_vector(0 downto 0);
    probe3  : in std_logic_vector(63 downto 0);
    probe4  : in std_logic_vector(0 downto 0);
    probe5  : in std_logic_vector(0 downto 0);
    probe6  : in std_logic_vector(1 downto 0);
    probe7  : in std_logic_vector(5 downto 0);
    probe8  : in std_logic_vector(15 downto 0);
    probe9  : in std_logic_vector(7 downto 0);
    probe10 : in std_logic_vector(0 downto 0);
    probe11 : in std_logic_vector(0 downto 0);
    probe12 : in std_logic_vector(0 downto 0);
    probe13 : in std_logic_vector(0 downto 0);
    probe14 : in std_logic_vector(0 downto 0);
    probe15 : in std_logic_vector(0 downto 0);
    probe16 : in std_logic_vector(0 downto 0);
    probe17 : in std_logic_vector(0 downto 0);
    probe18 : in std_logic_vector(0 downto 0);
    probe19 : in std_logic_vector(0 downto 0);
    probe20 : in std_logic_vector(0 downto 0);
    probe21 : in std_logic_vector(0 downto 0);
    probe22 : in std_logic_vector(0 downto 0);
    probe23 : in std_logic_vector(0 downto 0)
  );
  end component;
  
  component ila_1
  port (
    clk     : in std_logic;
    probe0  : in std_logic_vector(0 downto 0);
    probe1  : in std_logic_vector(11 downto 0);
    probe2  : in std_logic_vector(0 downto 0);
    probe3  : in std_logic_vector(63 downto 0);
    probe4  : in std_logic_vector(0 downto 0);
    probe5  : in std_logic_vector(3 downto 0);
    probe6  : in std_logic_vector(14 downto 0);
    probe7  : in std_logic_vector(0 downto 0);
    probe8  : in std_logic_vector(0 downto 0);
    probe9  : in std_logic_vector(0 downto 0);
    probe10 : in std_logic_vector(1 downto 0);
    probe11 : in std_logic_vector(5 downto 0);
    probe12 : in std_logic_vector(15 downto 0);
    probe13 : in std_logic_vector(0 downto 0);
    probe14 : in std_logic_vector(7 downto 0);
    probe15 : in std_logic_vector(0 downto 0);
    probe16 : in std_logic_vector(0 downto 0);
    probe17 : in std_logic_vector(0 downto 0)
  );
  end component;

  -------------------------------------------------------------------------------
  -- HW demo signals
  -------------------------------------------------------------------------------

  attribute keep          : string;
  attribute mark_debug    : string;
  attribute ASYNC_REG     : string;
  attribute SHREG_EXTRACT : string;

  -- Clocks
  signal clk              : std_logic;
  signal clk_ok           : std_logic_vector(0 downto 0);
  signal recclk           : std_logic;
  signal recclk_ok        : std_logic_vector(0 downto 0);
  attribute mark_debug of clk_ok : signal is "true";
  attribute mark_debug of recclk_ok : signal is "true";

  signal clk300b          : std_logic;
  signal clk300           : std_logic;
  
  signal aux_clkb         : std_logic;
  signal aux_clk          : std_logic;

  signal clk10b           : std_logic;
  signal clk10            : std_logic;

  signal eth_clkb         : std_logic;
  signal eth_clk          : std_logic;

  signal hires_clkb       : std_logic;
  signal hires_clk        : std_logic;
  signal hires_clk_ok     : std_logic;

  signal refclk           : std_logic;
  signal refclki          : std_logic;
  signal refclki_bufg     : std_logic;

  signal fb               : std_logic;
  signal mmcm_locked      : std_logic_vector(0 downto 0);
  attribute mark_debug of mmcm_locked : signal is "true";

  signal iic_done         : std_logic_vector(0 downto 0);
  attribute mark_debug of iic_done : signal is "true";

  signal alive_refclk     : unsigned(24 downto 0);

  signal refclk_ok_count  : unsigned(25 downto 0);
  signal refclk_ok        : std_logic_vector(0 downto 0);
  attribute mark_debug of refclk_ok : signal is "true";

  -- resets
  signal mmcm_reset       : std_logic;
  signal iic_reset        : std_logic;
  signal reset            : std_logic;
  signal reset_s          : std_logic;

  -- Core Master / Slave control
  signal cntrl : cntrl_rec;
  attribute mark_debug of cntrl : signal is "true";

  -- R21 Measurement
  signal delay : delay_rec;
  attribute mark_debug of delay : signal is "true";

  -- Ethernet data generation and checking
  signal eth : eth_rec;
  attribute mark_debug of eth : signal is "true";

  -- GT debug signals
  signal gt : gt_rec;
  attribute mark_debug of gt : signal is "true";

  -- HDLC Data generation and checking
  signal hdlc : hdlc_rec;
  attribute mark_debug of hdlc : signal is "true";

  -- IQ Data generation and checking
  signal iq : iq_rec;
  attribute mark_debug of iq : signal is "true";

  signal iq_rx_chk_enable_clk : sync_slv_00b_t;
  signal iq_rx_chk_enable_eth : sync_slv_00b_t;

  attribute ASYNC_REG     of iq_rx_chk_enable_clk : signal is "true";
  attribute SHREG_EXTRACT of iq_rx_chk_enable_clk : signal is "no";
  attribute ASYNC_REG     of iq_rx_chk_enable_eth : signal is "true";
  attribute SHREG_EXTRACT of iq_rx_chk_enable_eth : signal is "no";

  -- Management Interface
  signal mgmnt : mgmnt_rec := ( read_req  => (others => '0')
                              , write_req => (others => '1')
                              , cmd       => (others => '0')
                              , addr      => (others => '0')
                              , rd_data   => (others => '0')
                              , wr_data   => (others => '0')
                              , state     => IDLE
                              , axi_cmd   => MGMNT_CMD_PMA0 );

  attribute mark_debug of mgmnt : signal is "true";

  signal MGMNT_CMD_ROM : fsm_axi_t(3 downto 0) := mgmnt_rom_init( vcu118 );

  -- AXI Lite Management Interface
  signal s_axi : s_axi_rec;
  attribute mark_debug of s_axi : signal is "true";

  -- CPRI core status
  signal stat : stat_rec;
  attribute mark_debug of stat : signal is "true";

  signal stat_code_clk     : sync_slv_04b_t;
  signal stat_speed_clk    : sync_slv_15b_t;
  signal stat_speed_recclk : sync_slv_15b_t;

  attribute ASYNC_REG of stat_speed_recclk     : signal is "true";
  attribute SHREG_EXTRACT of stat_speed_recclk : signal is "no";
  attribute ASYNC_REG of stat_speed_clk        : signal is "true";
  attribute SHREG_EXTRACT of stat_speed_clk    : signal is "no";
  attribute ASYNC_REG of stat_code_clk         : signal is "true";
  attribute SHREG_EXTRACT of stat_code_clk     : signal is "no";

  -- FSM State flags
  signal state : state_rec;
  attribute mark_debug of state : signal is "true";

  -- Synchronization
  signal sync : sync_rec;
  attribute mark_debug of sync : signal is "true";

  -- TX Alignment
  signal tx_align : tx_align_rec;
  attribute mark_debug of tx_align : signal is "true";

  -- Vendor specific data generation and checking
  signal vendor : vendor_rec;
  attribute mark_debug of vendor : signal is "true";

  -- VIO Reset control
  signal vio : vio_rec;
  attribute mark_debug of vio : signal is "true";

  signal vio_reset_d : sync_slv_00b_t;

  -- Decorator signals just for VIO/ILA probes naming purposes
  signal line_rate_24G_FEC  : std_logic_vector(0 downto 0);
  signal line_rate_24G      : std_logic_vector(0 downto 0);
  signal fec_err_inj_0ppm   : std_logic_vector(0 downto 0);
  signal fec_err_inj_1e0ppm : std_logic_vector(0 downto 0);
  signal fec_err_inj_1e1ppm : std_logic_vector(0 downto 0);
  signal fec_err_inj_1e2ppm : std_logic_vector(0 downto 0);
  signal fec_err_inj_1e3ppm : std_logic_vector(0 downto 0);
  signal fec_err_inj_1e4ppm : std_logic_vector(0 downto 0);
  signal soft_reset         : std_logic_vector(0 downto 0);

  attribute mark_debug of line_rate_24G_FEC  : signal is "true";
  attribute mark_debug of line_rate_24G      : signal is "true";
  attribute mark_debug of fec_err_inj_0ppm   : signal is "true";
  attribute mark_debug of fec_err_inj_1e0ppm : signal is "true";
  attribute mark_debug of fec_err_inj_1e1ppm : signal is "true";
  attribute mark_debug of fec_err_inj_1e2ppm : signal is "true";
  attribute mark_debug of fec_err_inj_1e3ppm : signal is "true";
  attribute mark_debug of fec_err_inj_1e4ppm : signal is "true";
  attribute mark_debug of soft_reset         : signal is "true";

  -- R21 measurement
  constant FIXED_FEC_LATENCY          : unsigned(31 downto 0) := x"000009CC"; -- 2508 UI (38 clks x66)
  constant FIXED_GTYE4_LATENCY_64B66B : unsigned(31 downto 0) := x"00000235"; -- 565 UI
  constant FIXED_GTYE4_LATENCY_8B10B  : unsigned(31 downto 0) := x"0000019B"; -- 411.5 UI
  constant PIPELINE_DELAY_FEC         : unsigned(31 downto 0) := x"000004A4"; -- 1188 UI (18 clks x66)
  constant PIPELINE_DELAY_64B66B      : unsigned(31 downto 0) := x"0000035A"; -- 858 UI (13 clks x66)
  constant PIPELINE_DELAY_8B10B       : unsigned(31 downto 0) := x"00000320"; -- 800 UI (10 clks x80)
  constant LINE_RATE_LATENCY_24_3G    : unsigned(31 downto 0) := x"00000016"; -- 22 UI
  constant LINE_RATE_LATENCY_12_1G    : unsigned(31 downto 0) := x"0000000A"; -- 10 UI
  constant LINE_RATE_LATENCY_10_1G    : unsigned(31 downto 0) := x"00000008"; -- 8 UI
  constant LINE_RATE_LATENCY_8_1G     : unsigned(31 downto 0) := x"00000006"; -- 6 UI
  constant LINE_RATE_LATENCY_9_8G     : unsigned(31 downto 0) := x"00000008"; -- 8 UI
  constant LINE_RATE_LATENCY_6_1G     : unsigned(31 downto 0) := x"00000004"; -- 4 UI
  constant LINE_RATE_LATENCY_4_9G     : unsigned(31 downto 0) := x"00000003"; -- 3 UI
  constant LINE_RATE_LATENCY_3_0G     : unsigned(31 downto 0) := x"00000001"; -- 1 UI

  signal a         : unsigned(31 downto 0);
  signal b         : unsigned(31 downto 0);
  signal c         : unsigned(31 downto 0);
  signal d         : unsigned(31 downto 0);
  signal r21_timer : unsigned(35 downto 0);
  attribute mark_debug of r21_timer : signal is "true";

begin

  cpri_block_i: cpri_0
  port map (
    -- Additional Debug Ports
    -- DRP Access
    gt_drpdaddr                 => (others => '0'),
    gt_drpdi                    => (others => '0'),
    gt_drpen                    => '0',
    gt_drpwe                    => '0',
    gt_drpdo                    => open,    
    gt_drprdy                   => open,
    -- TX Reset and Initialization
    gt_txpmareset               => '0',
    gt_txpcsreset               => '0',
    gt_txresetdone              => gt.txresetdone(0),
    -- RX Reset and Initialization
    gt_rxpmareset               => '0',
    gt_rxpcsreset               => '0',
    gt_rxpmaresetdone           => gt.rxpmaresetdone(0),
    gt_rxresetdone              => gt.rxresetdone(0),
    -- Clocking
    gt_txphaligndone            => gt.txphaligndone(0),
    gt_txphinitdone             => gt.txphinitdone(0),
    gt_txdlysresetdone          => gt.txdlysresetdone(0),
    gt_rxphaligndone            => gt.rxphaligndone(0),
    gt_rxdlysresetdone          => gt.rxdlysresetdone(0),
    gt_rxsyncdone               => gt.rxsyncdone(0),
    gt_cplllock                 => gt.cplllock(0),
    -- Signal Integrity and Functionality
    gt_rxrate                   => (others => '0'),
    gt_eyescantrigger           => '0',
    gt_eyescanreset             => '0',
    gt_eyescandataerror         => open,
    gt_rxpolarity               => '0',
    gt_txpolarity               => '0',
    gt_rxdfelpmreset            => '0',
    gt_rxlpmen                  => '0',
    gt_txprecursor              => (others => '0'),
    gt_txpostcursor             => (others => '0'),
    gt_txdiffctrl               => "01100",
    gt_txprbsforceerr           => '0',
    gt_txprbssel                => (others => '0'),
    gt_rxprbssel                => (others => '0'),
    gt_rxprbserr                => open,
    gt_rxprbscntreset           => '0',
    gt_rxcdrhold                => '0',
    gt_dmonitorout              => gt.dmonitorout,
    gt_rxdisperr                => open,
    gt_rxnotintable             => open,
    gt_rxheader                 => open,
    gt_rxheadervalid            => open,
    gt_rxcommadet               => open,
    gt_pcsrsvdin                => (others => '0'),
    txdata                      => gt.txdata,
    txcharisk                   => open,
    txheader                    => gt.txheader,
    txsequence                  => open,
    rxdata                      => gt.rxdata,
    rxchariscomma               => open,
    rxcharisk                   => open,
    rxdisperr                   => open,
    rxnotintable                => open,
    rxdatavalid                 => gt.rxdatavalid(0),
    rxheader                    => gt.rxheader,
    rxheadervalid               => gt.rxheadervalid(0),
    rxgearboxslip               => gt.rxgearboxslip(0),
    ----------------- END OF DEBUG PORTS -----------------
    reset                       => reset,
    -- I/Q interface common
    iq_tx_enable                => iq.tx_enable(0),
    basic_frame_first_word      => iq.basic_frame_first_word(0),
    iq_tx                       => iq.tx,
    iq_rx                       => iq.rx,
    -- GT Common Ports
    qpll0clk_out                => open,
    qpll0refclk_out             => open,
    qpll1clk_out                => open,
    qpll1refclk_out             => open,
    qpll0lock_out               => open,
    qpll1lock_out               => open,
    -- Vendor Specific Data
    vendor_tx_data              => vendor.tx_data,
    vendor_tx_xs                => vendor.tx_xs,
    vendor_tx_ns                => vendor.tx_ns,
    vendor_rx_data              => vendor.rx_data,
    vendor_rx_xs                => vendor.rx_xs,
    vendor_rx_ns                => vendor.rx_ns,
    vs_negotiation_complete     => '1',
    -- Synchronization
    nodebfn_tx_strobe           => sync.nodebfn_tx_strobe(0),
    nodebfn_tx_nr               => sync.nodebfn_tx_nr,
    nodebfn_rx_strobe           => sync.nodebfn_rx_strobe(0),
    nodebfn_rx_nr               => sync.nodebfn_rx_nr,
    -- Ethernet interface
    eth_txd                     => eth.txd,
    eth_tx_er                   => eth.tx_er(0),
    eth_tx_en                   => eth.tx_en(0),
    eth_col                     => eth.col(0),
    eth_crs                     => eth.crs(0),
    eth_rxd                     => eth.rxd,
    eth_rx_er                   => eth.rx_er(0),
    eth_rx_dv                   => eth.rx_dv(0),
    eth_rx_avail                => eth.rx_avail,
    eth_rx_ready                => eth.rx_ready,
    rx_fifo_almost_full         => eth.rx_fifo_almost_full,
    rx_fifo_full                => eth.rx_fifo_full,
    eth_tx_clk                  => eth_clk,
    eth_rx_clk                  => eth_clk,
    -- HDLC interface
    hdlc_rx_data                => hdlc.rx_data(0),
    hdlc_tx_data                => hdlc.tx_data(0),
    hdlc_rx_data_valid          => hdlc.rx_data_valid(0),
    hdlc_tx_enable              => hdlc.tx_enable(0),
    -- Status interface
    stat_alarm                  => stat.alarm(0),
    stat_code                   => stat.code,
    stat_speed                  => stat.speed,
    reset_request_in            => s_axi.reset_request_in,
    sdi_request_in              => s_axi.sdi_request_in(0),
    reset_acknowledge_out       => s_axi.reset_acknowledge_out(0),
    sdi_request_out             => s_axi.sdi_request_out(0),
    local_los                   => s_axi.local_los(0),
    local_lof                   => s_axi.local_lof(0),
    local_rai                   => s_axi.local_rai(0),
    remote_los                  => s_axi.remote_los(0),
    remote_lof                  => s_axi.remote_lof(0),
    remote_rai                  => s_axi.remote_rai(0),
    fifo_transit_time           => s_axi.fifo_transit_time,
    coarse_timer_value          => s_axi.coarse_timer_value,
    barrel_shift_value          => s_axi.barrel_shift_value,
    tx_gb_latency_value         => s_axi.tx_gb_latency_value,
    rx_gb_latency_value         => s_axi.rx_gb_latency_value,
	hfec_fifo_latency_value     => s_axi.hfec_fifo_latency_value,
    hyperframe_number           => open,
    -- AXI-Lite Management interface
    s_axi_aclk                  => aux_clk,
    s_axi_aresetn               => s_axi.aresetn,
    s_axi_awaddr                => s_axi.awaddr,
    s_axi_awvalid               => s_axi.awvalid(0),
    s_axi_awready               => s_axi.awready(0),
    s_axi_wdata                 => s_axi.wdata,
    s_axi_wvalid                => s_axi.wvalid(0),
    s_axi_wready                => s_axi.wready(0),
    s_axi_bresp                 => s_axi.bresp,
    s_axi_bvalid                => s_axi.bvalid(0),
    s_axi_bready                => s_axi.bready(0),
    s_axi_araddr                => s_axi.araddr,
    s_axi_arvalid               => s_axi.arvalid(0),
    s_axi_arready               => s_axi.arready(0),
    s_axi_rdata                 => s_axi.rdata,
    s_axi_rresp                 => s_axi.rresp,
    s_axi_rvalid                => s_axi.rvalid(0),
    s_axi_rready                => s_axi.rready(0),
    l1_timer_expired            => '0',
    -- Transceiver signals
    txp                         => txp,
    txn                         => txn,
    rxp                         => rxp,
    rxn                         => rxn,
    lossoflight                 => '0',
    txinhibit                   => open,
    -- Clocking
    core_is_master              => cntrl.core_is_master(1),
    refclk                      => refclk,
    gtwiz_reset_clk_freerun_in  => clk10,
    hires_clk                   => hires_clk,
    hires_clk_ok                => hires_clk_ok,
    qpll_select                 => '0',
    recclk_ok                   => recclk_ok(0),
    clk_ok_out                  => clk_ok(0),
    recclk                      => recclk,
    clk_out                     => clk,
    rxrecclkout                 => open,
    txusrclk_out                => open,
    -- Tx alignment
    txphaligndone_in            => tx_align.txphalign_in,
    txdlysresetdone_in          => tx_align.txdlysreset_in,
    txphinitdone_in             => tx_align.txphinit_in,
    txphinit_out                => tx_align.txphinit_in,
    phase_alignment_done_out    => open,
    txdlysreset_out             => tx_align.txdlysreset_in,
    txphalign_out               => tx_align.txphalign_in,
    txdlyen_out                 => open,
    -- Interface for up to 3 external CPRI cores for HFEC support
    -- Hard FEC control - external CPRI cores
    rx_rsfec_enable_ch1         => '0',
    rx_rsfec_enable_ch2         => '0',
    rx_rsfec_enable_ch3         => '0',
    -- Ingress data - external CPRI cores
    rx_rec_clk_ch1              => '0',
    rx_serdes_data_ch1          => (others => '0'),
    rx_serdes_head_ch1          => (others => '0'),
    rx_gbx_slip_ch1             => open,
    rx_rec_clk_ch2              => '0',
    rx_serdes_data_ch2          => (others => '0'),
    rx_serdes_head_ch2          => (others => '0'),
    rx_gbx_slip_ch2             => open,
    rx_rec_clk_ch3              => '0',
    rx_serdes_data_ch3          => (others => '0'),
    rx_serdes_head_ch3          => (others => '0'),
    rx_gbx_slip_ch3             => open,
    -- CDC FIFO control & status - external CPRI cores
    cdc_reset_ch1               => '1',
    fifo_fill_level_ch1         => (others => '0'),
    average_ch1                 => open,
    average_rdy_ch1             => open,
    average_ack_ch1             => '0',
    fifo_error_ch1              => open,
    cdc_reset_ch2               => '1',
    fifo_fill_level_ch2         => (others => '0'),
    average_ch2                 => open,
    average_rdy_ch2             => open,
    average_ack_ch2             => '0',
    fifo_error_ch2              => open,
    cdc_reset_ch3               => '1',
    fifo_fill_level_ch3         => (others => '0'),
    average_ch3                 => open,
    average_rdy_ch3             => open,
    average_ack_ch3             => '0',
    fifo_error_ch3              => open,
    -- CDC FIFO output (no FEC decode)
    cdc_rxdata_ch1              => open,
    cdc_rxheader_ch1            => open,
    cdc_rxdata_ch2              => open,
    cdc_rxheader_ch2            => open,
    cdc_rxdata_ch3              => open,
    cdc_rxheader_ch3            => open,
    -- Egress data & status - external CPRI cores
    pcs_txclk_ch1               => open,
    pcs_txclk_ok_ch1            => open,
    pcs_rxdata_ch1              => open,
    pcs_rxheader_ch1            => open,
    pcs_rxheadervalid_ch1       => open,
    stat_rx_align_status_ch1    => open,
    stat_cw_inc_ch1             => open,
    stat_corrected_cw_inc_ch1   => open,
    stat_uncorrected_cw_inc_ch1 => open,
	hfec_fifo_latency_ch1       => open,
	hfec_fifo_latency_rdy_ch1   => open,
	hfec_fifo_latency_ack_ch1   => '1',
    pcs_txclk_ch2               => open,
    pcs_txclk_ok_ch2            => open,
    pcs_rxdata_ch2              => open,
    pcs_rxheader_ch2            => open,
    pcs_rxheadervalid_ch2       => open,
    stat_rx_align_status_ch2    => open,
    stat_cw_inc_ch2             => open,
    stat_corrected_cw_inc_ch2   => open,
    stat_uncorrected_cw_inc_ch2 => open,
	hfec_fifo_latency_ch2       => open,
	hfec_fifo_latency_rdy_ch2   => open,
	hfec_fifo_latency_ack_ch2   => '1',
    pcs_txclk_ch3               => open,
    pcs_txclk_ok_ch3            => open,
    pcs_rxdata_ch3              => open,
    pcs_rxheader_ch3            => open,
    pcs_rxheadervalid_ch3       => open,
    stat_rx_align_status_ch3    => open,
    stat_cw_inc_ch3             => open,
    stat_corrected_cw_inc_ch3   => open,
    stat_uncorrected_cw_inc_ch3 => open,
	hfec_fifo_latency_ch3       => open,
	hfec_fifo_latency_rdy_ch3   => open,
	hfec_fifo_latency_ack_ch3   => '1'
  );

  -- IQ Data generator
  iq_tx_gen_i : entity work.iq_tx_gen
  port map (
    clk               => clk,
    iq_tx_enable      => iq.tx_enable(0),
    iq_tx             => iq.tx,
    nodebfn_tx_strobe => sync.nodebfn_tx_strobe(0),
    nodebfn_tx_nr     => sync.nodebfn_tx_nr
  );

  -- Synchronise stat_speed to the tx user clock
  process(clk)
  begin
    if rising_edge(clk) then
      stat_speed_clk(0) <= stat.speed;
      stat_speed_clk(1) <= stat_speed_clk(0);
      stat_speed_clk(2) <= stat_speed_clk(1);
    end if;
  end process;

  -- Synchronise stat_speed to the recovered clock
  process(recclk)
  begin
    if rising_edge(recclk) then
      stat_speed_recclk(0) <= stat.speed;
      stat_speed_recclk(1) <= stat_speed_recclk(0);
      stat_speed_recclk(2) <= stat_speed_recclk(1);
    end if;
  end process;

  -- Synchronise stat_code to the aux clk
  process(clk)
  begin
    if rising_edge(clk) then
      stat_code_clk(0) <= stat.code;
      stat_code_clk(1) <= stat_code_clk(0);
      stat_code_clk(2) <= stat_code_clk(1);
    end if;
  end process;

  -- IQ Data monitor
  iq_rx_chk_i : entity work.iq_rx_chk
  port map (
    clk                    => clk,
    enable                 => cntrl.test_start_clk(0),
    speed                  => stat_speed_clk(2),
    basic_frame_first_word => iq.basic_frame_first_word(0),
    iq_rx                  => iq.rx,
    nodebfn_rx_strobe      => sync.nodebfn_rx_strobe(0),
    nodebfn_rx_nr          => sync.nodebfn_rx_nr,
    frame_count            => iq.frame_count,
    error_count            => iq.error_count,
    nodebfn_rx_nr_store    => sync.nodebfn_rx_nr_store,
    word_err               => iq.error(0)
  );

  iq.rx_chk_enable(0) <= '1' when stat.code = "1111" and 
                                  s_axi.reset_acknowledge_out(0) = '0' else '0';

  -- Synchronise to the core clock
  process(clk)
  begin
    if rising_edge(clk) then
      iq_rx_chk_enable_clk(0) <= iq.rx_chk_enable;
      iq_rx_chk_enable_clk(1) <= iq_rx_chk_enable_clk(0);
      iq_rx_chk_enable_clk(2) <= iq_rx_chk_enable_clk(1);
    end if;
  end process;

  cntrl.test_start_clk <= iq_rx_chk_enable_clk(2) and iq_rx_chk_enable_clk(1);

  -- Synchronise to the ethernet clock
  process(eth_clk)
  begin
    if rising_edge(eth_clk) then
      iq_rx_chk_enable_eth(0) <= iq.rx_chk_enable;
      iq_rx_chk_enable_eth(1) <= iq_rx_chk_enable_eth(0);
      iq_rx_chk_enable_eth(2) <= iq_rx_chk_enable_eth(1);
    end if;
  end process;

  cntrl.test_start_eth <= iq_rx_chk_enable_eth(2) and iq_rx_chk_enable_eth(1);

  -- HDLC data generator and monitor
  hdlc_tx_rx_i : entity work.hdlc_stim
  port map(
    clk                    => clk,
    clk_ok                 => clk_ok(0),
    reset                  => reset_s,
    hdlc_rx_data           => hdlc.rx_data(0),
    hdlc_tx_data           => hdlc.tx_data(0),
    hdlc_rx_data_valid     => hdlc.rx_data_valid(0),
    hdlc_tx_enable         => hdlc.tx_enable(0),
    start_hdlc             => cntrl.test_start_clk(0),
    bit_count              => hdlc.bit_count,
    error_count            => hdlc.error_count,
    hdlc_error             => hdlc.error(0)
  );

  -- Ethernet data generator and monitor
  gmii_tx_rx_i : entity work.gmii_stim
  port map(
    eth_clk                => eth_clk,
    tx_eth_enable          => eth.tx_en(0),
    tx_eth_data            => eth.txd,
    tx_eth_err             => eth.tx_er(0),
    tx_eth_overflow        => eth.col(0),
    tx_eth_half            => eth.crs(0),
    rx_eth_data_valid      => eth.rx_dv(0),
    rx_eth_data            => eth.rxd,
    rx_eth_err             => eth.rx_er(0),
    tx_start               => cntrl.test_start_eth(0),
    rx_start               => cntrl.test_start_eth(0),
    tx_count               => eth.tx_count,
    rx_count               => eth.rx_frame_count,
    err_count              => eth.error_count,
    eth_rx_error           => eth.error(0),
    eth_rx_avail           => eth.rx_avail,
    eth_rx_ready           => eth.rx_ready,
    rx_fifo_almost_full    => eth.rx_fifo_almost_full,
    rx_fifo_full           => eth.rx_fifo_full
  );

  -- Vendor specific data generator and monitor
  vendor_tx_rx_i : entity work.vendor_stim
  port map(
    clk                    => clk,
    reset                  => reset_s,
    iq_tx_enable           => iq.tx_enable(0),
    bffw                   => iq.basic_frame_first_word(0),
    vendor_tx_data         => vendor.tx_data,
    vendor_tx_xs           => vendor.tx_xs,
    vendor_tx_ns           => vendor.tx_ns,
    vendor_rx_data         => vendor.rx_data,
    vendor_rx_xs           => vendor.rx_xs,
    vendor_rx_ns           => vendor.rx_ns,
    start_vendor           => cntrl.test_start_clk(0),
    word_count             => vendor.vs_word_count,
    error_count            => vendor.vs_error_count,
    vendor_error           => vendor.vs_error(0),
    speed                  => stat_speed_clk(2)
  );

  --===============================
  -- HW Demo
  --===============================

  -------------------------------------------------------------------------------
  -- Clock Generation
  -------------------------------------------------------------------------------

  -- 300 MHz clock in from VCU118
  clk300_ibuf : IBUFDS
  port map (
    I  => clk300_p,
    IB => clk300_n,
    O  => clk300b
  );

  clk300_bufg : BUFG
  port map (
    I => clk300b,
    O => clk300
  );

  -- MMCM generates several clocks from the 300MHz input
  --
  --  CLKOUT 0 : hires clk  (400 MHz)
  --  CLKOUT 1 : aux_clk    (100 MHz)
  --  CLKOUT 2 : clk10      (10 MHz)
  --  CLKOUT 3 : eth_clk    (125 MHz for GMII)
  --
  mmcm_i : MMCM_BASE
  generic map (
    CLKFBOUT_MULT_F    => 3.000,
    CLKOUT0_DIVIDE_F   => 2.250,
    CLKOUT1_DIVIDE     => 9,
    CLKOUT2_DIVIDE     => 90,
    CLKOUT3_DIVIDE     => 9,
    CLKFBOUT_PHASE     => 0.000,
    CLKIN1_PERIOD      => 3.333,
    CLKOUT0_DUTY_CYCLE => 0.5,
    CLKOUT0_PHASE      => 0.000,
    CLOCK_HOLD         => false,
    DIVCLK_DIVIDE      => 1,
    REF_JITTER1        => 0.010,
    STARTUP_WAIT       => false )
  port map (
    CLKIN1             => clk300,
    CLKFBOUTB          => open,
    CLKOUT0            => hires_clkb,
    CLKOUT1            => aux_clkb,
    CLKOUT2            => clk10b,
    CLKOUT3            => eth_clkb,
    CLKFBIN            => fb,
    CLKFBOUT           => fb,
    LOCKED             => mmcm_locked(0),
    PWRDWN             => '0',
    RST                => mmcm_reset
  );

  aux_bufg : BUFG
  port map (
    I => aux_clkb,
    O => aux_clk
  );

  hires_bufg : BUFG
  port map (
    I => hires_clkb,
    O => hires_clk
  );

  clk10_bufg : BUFG
  port map (
    I => clk10b,
    O => clk10
  );

  eth_bufg : BUFG
  port map (
    I => eth_clkb,
    O => eth_clk
  );

  -------------------------------------
  -- GT Reference clock Generation
  -------------------------------------

  -- Si570 is set up to provide reference clock from the 
  -- 114.285MHz reference on the demonstration board.

  -- Reference clock IN from Si570
  refclk_ibufds : IBUFDS_GTE4
  port map (
    I     => refclk_p,
    IB    => refclk_n,
    O     => refclk,
    CEB   => '0',
    ODIV2 => refclki
  );

  ---- Buffer refclk for Logic use and Output monitor
  refclk_bufg : BUFG_GT
  port map (
    I       => refclki,
    CE      => '1',
    CEMASK  => '1',
    CLR     => '0',
    CLRMASK => '1',
    DIV     => "000",
    O       => refclki_bufg
  );

  -------------------------------------------------------------------------------
  -- IIC controller
  --
  -- Note that the IIC controller must be clocked at 100 MHz as there is a fixed
  -- ratio between this and it's operating frequency on the IIC bus.

  i_iic_controller : entity work.iic_controller
  generic map ( vcu118 )
  port map (
    reset   => iic_reset,
    clk100  => aux_clk,
    iic_sda => iic_sda,
    iic_scl => iic_scl,
    status  => open,
    done    => iic_done(0)
  );

  -------------------------------------------------------------------------------
  -- This counter is used to insert a ~100 us delay after the IIC controller has
  -- set up the reference clock to allow the CPRI core to come out of reset with a
  -- stable GTX reference clock.
  -- 26-bit counter @ 10 MHz => 67108864 cycles => 6.71s

  process (clk10, iic_reset)
  begin
    if (iic_reset = '1') then
      refclk_ok_count   <= (others => '0');
      refclk_ok(0)      <= '0';
    elsif rising_edge(clk10) then
      if (iic_done(0) = '1') then
        if (refclk_ok_count(refclk_ok_count'high) = '0') then
          refclk_ok_count <= refclk_ok_count + 1;
        end if;
      else
        refclk_ok_count <= (others => '0');
      end if;
      refclk_ok(0) <= std_logic(refclk_ok_count(refclk_ok_count'high));
    end if;
  end process;

  -- reset IIC slaves in sympathy with IIC controller
  iic_mux_rst_n <= not iic_reset;

  -------------------------------------------------------------------------------
  -- Resets
  -------------------------------------------------------------------------------

  -- Vivado VIO is synchronous only, so a VIO reset needs to be handled
  -- appropriately using a clock which is not reset by the reset itself
  process (clk300)
  begin
    if rising_edge(clk300) then
      -- handle as async to clk300
      vio_reset_d(0) <= vio.reset;
      vio_reset_d(1) <= vio_reset_d(0);
      vio_reset_d(2) <= vio_reset_d(1);
    
      -- vio_reset rising edge creates a reset pulse to the mmcm
      if (vio_reset_d(1)(0) = '1' and vio_reset_d(2)(0) = '0') then
        vio.reset_sr <= "11111";
      else
        vio.reset_sr <= vio.reset_sr(3 downto 0) & '0';
      end if;
      
      vio.reset_p <= vio.reset_sr(4);
    end if;
  end process;

  -- Complete reset can be invoked from push button
  mmcm_reset   <= pb_reset;

  -- hold I2C reset until MMCM locked
  iic_reset    <= not mmcm_locked(0);

  hires_clk_ok <= mmcm_locked(0);

  -- Bulk of design held in reset until reference clock is setup and stabilised
  -- reset <= not refclk_ok(0);
  reset <= (not refclk_ok(0)) or vio.reset(0);

  -- hold the AXI managment interface (& FSM) reset until refclk is up and CPRI core
  -- reset is released
  -- s_axi_aresetn <= not reset;
  s_axi.aresetn <= refclk_ok(0);

  -- synchronise reset for blocks that use this reset in a synchronous fashion
  process (clk, reset)
  begin
    if (reset = '1') then
      reset_s <= '1';
    elsif rising_edge(clk) then
      reset_s <= '0';
    end if;
  end process;

  -------------------------------------------------------------------------------
  -- LEDs and Monitor ports
  -------------------------------------------------------------------------------

  -- LEDs

  -- Divide refclk for flashing LED
  process(refclki_bufg)
  begin
    if rising_edge(refclki_bufg) then
      alive_refclk <= alive_refclk + 1;
    end if;
  end process;

  leds(7)          <= alive_refclk(alive_refclk'high); -- Refclk
  leds(6)          <= mmcm_locked(0);                  -- clocks locked
  leds(5)          <= stat.alarm(0);                   -- Alarm
  leds(4 downto 1) <= stat.speed4;                     -- Speed indication
  leds(0)          <= state.operate(0);                -- State = OPERATIONAL

  -- Clock Monitor outputs on GPIO SMAs

  -- Core clock
  oddr1 : ODDRE1
  port map(
    D1 => '1',
    D2 => '0',
    C  => clk,
    Q  => core_clk_mon,
    SR => '0'
  );

  -- Si5328 (refclk)
  oddr2 : ODDRE1
  port map(
    D1 => '1',
    D2 => '0',
    C  => refclki_bufg,
    Q  => clk_si570_mon,
    SR => '0'
  );

  -------------------------------------------------------------------------------
  -- VIO - Control and Management
  -------------------------------------------------------------------------------
  mgmnt.cmd(MGMNT_INDX_LINE_RATE_0)      <= line_rate_24G_FEC(0);
  mgmnt.cmd(MGMNT_INDX_LINE_RATE_1)      <= line_rate_24G(0);
  mgmnt.cmd(MGMNT_INDX_FEC_ERROR_0PPM)   <= fec_err_inj_0ppm(0);
  mgmnt.cmd(MGMNT_INDX_FEC_ERROR_1e0PPM) <= fec_err_inj_1e0ppm(0);
  mgmnt.cmd(MGMNT_INDX_FEC_ERROR_1e1PPM) <= fec_err_inj_1e1ppm(0);
  mgmnt.cmd(MGMNT_INDX_FEC_ERROR_1e2PPM) <= fec_err_inj_1e2ppm(0);
  mgmnt.cmd(MGMNT_INDX_FEC_ERROR_1e3PPM) <= fec_err_inj_1e3ppm(0);
  mgmnt.cmd(MGMNT_INDX_FEC_ERROR_1e4PPM) <= fec_err_inj_1e4ppm(0);
  mgmnt.cmd(MGMNT_INDX_SOFT_RESET)       <= soft_reset(0);

  i_vio_control : vio_control 
  port map(
    clk         => aux_clk,
    probe_in0   => mgmnt.rd_data,        -- input  [31 : 0] probe_in0
    probe_out0  => mgmnt.read_req,       -- output [ 0 : 0] probe_out0
    probe_out1  => mgmnt.write_req,      -- output [ 0 : 0] probe_out1
    probe_out2  => mgmnt.addr,           -- output [11 : 0] probe_out2
    probe_out3  => mgmnt.wr_data,        -- output [31 : 0] probe_out3
    probe_out4  => vio.reset,            -- output [ 0 : 0] probe_out4
    probe_out5  => vio.reset_req,        -- output [ 0 : 0] probe_out5
    probe_out6  => s_axi.sdi_request_in, -- output [ 0 : 0] probe_out6
    probe_out7  => line_rate_24G_FEC,    -- output [ 0 : 0] probe_out7
    probe_out8  => line_rate_24G,        -- output [ 0 : 0] probe_out8
    probe_out9  => fec_err_inj_0ppm,     -- output [ 0 : 0] probe_out9
    probe_out10 => fec_err_inj_1e0ppm,   -- output [ 0 : 0] probe_out10
    probe_out11 => fec_err_inj_1e1ppm,   -- output [ 0 : 0] probe_out11
    probe_out12 => fec_err_inj_1e2ppm,   -- output [ 0 : 0] probe_out12
    probe_out13 => fec_err_inj_1e3ppm,   -- output [ 0 : 0] probe_out13
    probe_out14 => fec_err_inj_1e4ppm,   -- output [ 0 : 0] probe_out14
    probe_out15 => cntrl.pma_loopback,   -- output [ 0 : 0] probe_out15
    probe_out16 => soft_reset            -- output [ 0 : 0] probe_out16
  );

  -- AXI interface
  -- State machine for controlling writes and reads to and from the
  -- AXI management registers via VIO
  p_mgmnt_states : process(aux_clk, reset)
  begin
    if (reset = '1') then
      MGMNT_FSM_RST( nstate        => IDLE
                   , s_axi_awaddr  => s_axi.awaddr
                   , s_axi_awvalid => s_axi.awvalid
                   , s_axi_wdata   => s_axi.wdata
                   , s_axi_bready  => s_axi.bready
                   , s_axi_wvalid  => s_axi.wvalid
                   , s_axi_araddr  => s_axi.araddr
                   , s_axi_arvalid => s_axi.arvalid
                   , s_axi_rready  => s_axi.rready
                   , mgmnt_state   => mgmnt.state );
    elsif rising_edge(aux_clk) then
      case mgmnt.state is
        when IDLE =>
          MGMNT_FSM_IDLE(  mgmnt_cmd_lst   => MGMNT_CMD_ROM
                        ,  write_req       => mgmnt.write_req
                        ,  read_req        => mgmnt.read_req
                        ,  mgmnt_cmd       => mgmnt.cmd
                        ,  pma_loopback    => cntrl.pma_loopback
                        ,  s_axi_wready    => s_axi.wready
                        ,  s_axi_awaddr    => s_axi.awaddr
                        ,  s_axi_awvalid   => s_axi.awvalid
                        ,  s_axi_wdata     => s_axi.wdata
                        ,  s_axi_wvalid    => s_axi.wvalid
                        ,  s_axi_bready    => s_axi.bready
                        ,  s_axi_araddr    => s_axi.araddr
                        ,  s_axi_arvalid   => s_axi.arvalid
                        ,  s_axi_rready    => s_axi.rready
                        ,  mgmnt_state     => mgmnt.state
                        ,  axi_cmd         => mgmnt.axi_cmd );
        when TO_IDLE =>
          MGMNT_FSM_TO_IDLE( s_axi_bvalid  => s_axi.bvalid
                           , s_axi_bready  => s_axi.bready
                           , mgmnt_state   => mgmnt.state );
        when MGMNT_WR0 =>
          MGMNT_FSM_WR0(   mgmnt_addr      => mgmnt.addr
                       ,   mgmnt_wr_data   => mgmnt.wr_data
                       ,   s_axi_wready    => s_axi.wready
                       ,   s_axi_awaddr    => s_axi.awaddr
                       ,   s_axi_awvalid   => s_axi.awvalid
                       ,   s_axi_wdata     => s_axi.wdata
                       ,   s_axi_wvalid    => s_axi.wvalid
                       ,   mgmnt_state     => mgmnt.state );
        when MGMNT_RD0 =>
          MGMNT_FSM_RD0(   mgmnt_addr      => mgmnt.addr
                       ,   s_axi_arready   => s_axi.arready
                       ,   s_axi_araddr    => s_axi.araddr
                       ,   s_axi_arvalid   => s_axi.arvalid
                       ,   s_axi_rready    => s_axi.rready
                       ,   mgmnt_state     => mgmnt.state );
        when MGMNT_RD1 =>
          MGMNT_FSM_RD1(   s_axi_rdata     => s_axi.rdata
                       ,   s_axi_rvalid    => s_axi.rvalid
                       ,   s_axi_arvalid   => s_axi.arvalid
                       ,   s_axi_rready    => s_axi.rready
                       ,   mgmnt_rd_data   => mgmnt.rd_data
                       ,   mgmnt_state     => mgmnt.state );
        when MGMNT_WRA =>
          MGMNT_FSM_WREG(  axi_data        => mgmnt.axi_cmd
                        ,  s_axi_wready    => s_axi.wready
                        ,  s_axi_awaddr    => s_axi.awaddr
                        ,  s_axi_awvalid   => s_axi.awvalid
                        ,  s_axi_wdata     => s_axi.wdata
                        ,  s_axi_wvalid    => s_axi.wvalid
                        ,  mgmnt_state     => mgmnt.state );
      end case;
    end if;
  end process;

  -- Discrete Control outputs
  s_axi.reset_request_in <= vio.reset_req(0) or s_axi.reset_request;

  s_axi.reset_request    <= '1' when stat.code /= "1111" else '0';

  -- Core MUST be master in case of selecting to use a Hard FEC
  cntrl.core_is_master   <= (others => '1');

  -- Core in loopback is set low, slave configuration not supported in this design
  cntrl.core_in_loopback <= (others => '0');

  -------------------------
  -- VIO - Status Monitor
  -------------------------
  i_vio_status : vio_status
  PORT MAP (
    clk        => aux_clk,
    probe_in0  => iq.frame_count(31 downto 16),
    probe_in1  => iq.error_count(15 downto 0),
    probe_in2  => eth.rx_frame_count(31 downto 16),
    probe_in3  => eth.error_count(15 downto 0),
    probe_in4  => hdlc.bit_count(31 downto 16),
    probe_in5  => hdlc.error_count(15 downto 0),
    probe_in6  => vendor.vs_word_count(31 downto 16),
    probe_in7  => vendor.vs_error_count(15 downto 0),
    probe_in8  => s_axi.fifo_transit_time,
    probe_in9  => s_axi.coarse_timer_value,
    probe_in10 => s_axi.barrel_shift_value,
    probe_in11 => s_axi.reset_acknowledge_out,
    probe_in12 => s_axi.sdi_request_out,
    probe_in13 => s_axi.local_los,
    probe_in14 => s_axi.local_lof,
    probe_in15 => s_axi.local_rai,
    probe_in16 => s_axi.remote_los,
    probe_in17 => s_axi.remote_lof,
    probe_in18 => s_axi.remote_rai,
    probe_in19 => stat.alarm,
    probe_in20 => stat.speed5,
    probe_in21 => state.reset,
    probe_in22 => state.l1sync,
    probe_in23 => state.proto,
    probe_in24 => state.cm,
    probe_in25 => state.passive,
    probe_in26 => state.vs,
    probe_in27 => state.operate,
    probe_in28 => std_logic_vector(r21_timer),
    probe_in29 => mmcm_locked,
    probe_in30 => clk_ok,
    probe_in31 => iic_done,
    probe_in32 => refclk_ok,
    probe_in33 => s_axi.tx_gb_latency_value,
    probe_in34 => s_axi.rx_gb_latency_value,
	probe_in35 => s_axi.hfec_fifo_latency_value
  );

  -- Encode speed to :
  --  - 5-bit for VIO display (decodes to line rate when displayed as unsigned)
  --  - 4-bit for LEDs (decodes to line rate when displayed as unsigned)
  speed_2bit_proc : process(stat.speed)
  begin
    stat.speed4 <= onehot_to_4b( stat.speed );
    stat.speed5 <= onehot_to_5b( stat.speed );
  end process;

  -- Decode Rx State to individual bits
  state.reset(0)   <= '1' when stat.code = "0000" else '0';
  state.l1sync(0)  <= '1' when stat.code = "0001" else '0';
  state.proto(0)   <= '1' when stat.code = "0010" else '0';
  state.cm(0)      <= '1' when stat.code = "0011" else '0';
  state.passive(0) <= '1' when stat.code = "1011" else '0';
  state.vs(0)      <= '1' when stat.code = "1110" else '0';
  state.operate(0) <= '1' when stat.code = "1111" else '0';

  i_ila_tx : ila_0
  port map (
    clk     => clk,
    probe0  => sync.nodebfn_tx_strobe,
    probe1  => sync.nodebfn_tx_nr,
    probe2  => iq.tx_enable,
    probe3  => iq.tx,
    probe4  => hdlc.tx_data,
    probe5  => hdlc.tx_enable,
    probe6  => vendor.tx_xs,
    probe7  => vendor.tx_ns,
    probe8  => vendor.tx_data(15 downto 0),
    probe9  => eth.txd,
    probe10 => eth.tx_er,
    probe11 => eth.tx_en,
    probe12 => eth.col,
    probe13 => eth.crs,
    probe14 => s_axi.awvalid,
    probe15 => s_axi.wvalid,
    probe16 => s_axi.wready,
    probe17 => s_axi.awready,
    probe18 => s_axi.bready,
    probe19 => s_axi.bvalid,
    probe20 => s_axi.arvalid,
    probe21 => s_axi.rvalid,
    probe22 => s_axi.arready,
    probe23 => s_axi.rready
  );

  i_ila_rx : ila_1
  port map (
    clk     => clk,
    probe0  => sync.nodebfn_rx_strobe,
    probe1  => sync.nodebfn_rx_nr,
    probe2  => iq.basic_frame_first_word,
    probe3  => iq.rx,
    probe4  => iq.error,
    probe5  => stat_code_clk(2),
    probe6  => stat_speed_clk(2),
    probe7  => hdlc.rx_data,
    probe8  => hdlc.rx_data_valid,
    probe9  => hdlc.error,
    probe10 => vendor.rx_xs,
    probe11 => vendor.rx_ns,
    probe12 => vendor.rx_data(15 downto 0),
    probe13 => vendor.vs_error,
    probe14 => eth.rxd,
    probe15 => eth.rx_er,
    probe16 => eth.rx_dv,
    probe17 => eth.error
  );

  -- Calculate R-21 measurement, T14 in master and Toffset in slave. a,b,c & d refer to the steps in the
  -- calculation described in "Performing the Cable Delay Calculation" sub-section in the CPRI product guide.

  -- a is the total transceiver fixed latency.
  p_calc_a : process(aux_clk)
  begin
    if rising_edge(aux_clk) then
      case stat.speed is
        when "100000000000000" | "000010000000000" =>
          a <= FIXED_GTYE4_LATENCY_64B66B + LINE_RATE_LATENCY_24_3G;
        when "010000000000000" | "000001000000000" =>
          a <= FIXED_GTYE4_LATENCY_64B66B + LINE_RATE_LATENCY_12_1G;
        when "001000000000000" | "000000010000000" =>
          a <= FIXED_GTYE4_LATENCY_64B66B + LINE_RATE_LATENCY_10_1G;
        when "000100000000000" | "000000100000000" =>
          a <= FIXED_GTYE4_LATENCY_64B66B + LINE_RATE_LATENCY_8_1G;
        when "000000001000000" =>
          a <= FIXED_GTYE4_LATENCY_8B10B + LINE_RATE_LATENCY_9_8G;
        when "000000000100000" =>
          a <= FIXED_GTYE4_LATENCY_8B10B + LINE_RATE_LATENCY_6_1G;
        when "000000000010000" =>
          a <= FIXED_GTYE4_LATENCY_8B10B + LINE_RATE_LATENCY_4_9G;
        when "000000000001000" =>
          a <= FIXED_GTYE4_LATENCY_8B10B + LINE_RATE_LATENCY_3_0G;
        when others =>
          a <= FIXED_GTYE4_LATENCY_8B10B;
      end case;
    end if;
  end process;

  -- b is the total transceiver variable latency.
  p_calc_b : process(aux_clk)
  begin
    if rising_edge(aux_clk) then
      if stat.speed(14 downto 7) /= "00000000" then
        b <= (x"0000" & "000" & unsigned(s_axi.tx_gb_latency_value(15 downto 3))) + unsigned(s_axi.rx_gb_latency_value(15 downto 3));
      else
        b <= x"000000" & "0" & unsigned(s_axi.barrel_shift_value);
      end if;
    end if;
  end process;

  -- c is the total CDC FIFO transit time.
  p_calc_c : process(aux_clk)
  begin
    if rising_edge(aux_clk) then
      if stat.speed(14 downto 7) /= "00000000" then
        c <= "0000" & ((unsigned(s_axi.fifo_transit_time)*66)/128);
      else
        c <= "0000" & ((unsigned(s_axi.fifo_transit_time)*80)/128);
      end if;
    end if;
  end process;

  -- d is the total delay correction.
  p_calc_d : process(aux_clk)
  begin
    if rising_edge(aux_clk) then
      if stat.speed(14 downto 11) /= "0000" then
        d <= a + b + c + PIPELINE_DELAY_FEC + FIXED_FEC_LATENCY + (unsigned(s_axi.hfec_fifo_latency_value)*66);
      elsif stat.speed(10 downto 7) /= "0000" then
        d <= a + b + c + PIPELINE_DELAY_64B66B;
      else
        d <= a + b + c + PIPELINE_DELAY_8B10B;
      end if;
    end if;
  end process;

  -- r21_timer is T14 for masters or Toffset for slaves.
  p_calc_r21 : process(aux_clk)
  begin
    if rising_edge(aux_clk) then
      if stat.speed(14 downto 7) /= "00000000" then
        if cntrl.core_is_master(1) = '0' then
          r21_timer <= (unsigned(s_axi.coarse_timer_value)*66) + d;
        else
          r21_timer <= (unsigned(s_axi.coarse_timer_value)*66) - d;
        end if;
      else
        if cntrl.core_is_master(1) = '0' then
          r21_timer <= (unsigned(s_axi.coarse_timer_value)*80) + d;
        else
          r21_timer <= (unsigned(s_axi.coarse_timer_value)*80) - d;
        end if;
      end if;
    end if;
  end process;

end rtl;