-- Title      : IIC Controller configuration features package.
-- Project    : CPRI
-----------------------------------------------------------------------
-- File       : iic_controller_pkg.vhd
-- Author     : Xilinx
-----------------------------------------------------------------------
-- Description:
--   This module initializes command ROMs required to configure boards:
--   kcu105/vcu108/zcu102/vcu118
-----------------------------------------------------------------------
-- (c) Copyright 2004 - 2018 Xilinx, Inc. All rights reserved.
--
-- This file contains confidential and proprietary information
-- of Xilinx, Inc. and is protected under U.S. and
-- international copyright and other intellectual property
-- laws.
--
-- DISCLAIMER
-- This disclaimer is not a license and does not grant any
-- rights to the materials distributed herewith. Except as
-- otherwise provided in a valid license issued to you by
-- Xilinx, and to the maximum extent permitted by applicable
-- law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
-- WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
-- AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
-- BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
-- INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
-- (2) Xilinx shall not be liable (whether in contract or tort,
-- including negligence, or under any other theory of
-- liability) for any loss or damage of any kind or nature
-- related to, arising under or in connection with these
-- materials, including for any direct, or any indirect,
-- special, incidental, or consequential loss or damage
-- (including loss of data, profits, goodwill, or any type of
-- loss or damage suffered as a result of any action brought
-- by a third party) even if such damage or loss was
-- reasonably foreseeable or Xilinx had been advised of the
-- possibility of the same.
--
-- CRITICAL APPLICATIONS
-- Xilinx products are not designed or intended to be fail-
-- safe, or for use in any application requiring fail-safe
-- performance, such as life-support or safety devices or
-- systems, Class III medical devices, nuclear facilities,
-- applications related to the deployment of airbags, or any
-- other applications that could lead to death, personal
-- injury, or severe property or environmental damage
-- (individually and collectively, "Critical
-- Applications"). Customer assumes the sole risk and
-- liability of any use of Xilinx products in Critical
-- Applications, subject only to applicable laws and
-- regulations governing limitations on product liability.
--
-- THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
-- PART OF THIS FILE AT ALL TIMES.
-----------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

package iic_controller_pkg is

  -- Demo design list
	type demo_t is ( UNKNOWN
  	             , kcu105
  	             , vcu108, vcu108_ms
  	             , vcu118, vcu118_hfw
  	             , zcu102 );

  -- IIC Controller list of supported parts
  type iic_part is ( UNKNOWN
  	               , kcu105
  	               , vcu108, vcu108_ms
  	               , vcu118, vcu118_hfw
  	               , zcu102 );

  -- Set the depth (address width) of the command ROM.
  -- The data width is fixed but the address width could change to
  -- account for more commands.
  constant LUT_AWIDTH : natural := 7; -- 128 max

  -- Command Look-Up Table - implemented in LUTs (Distributed ROM)
  type t_cmd_rom is array (0 to (2**LUT_AWIDTH)-1) of std_logic_vector(31 downto 0);

  -------------------------------------------------------------------------------
  -- Command look up table format
  -------------------------------------------------------------------------------
  --
  -- The total number of IIC commands is indicated in the first entry.
  -- Each subsequent entry indicates the AXI IIC core target address and target data.
  --
  -- Refer to AXI IIC core DS756 for addresses and tx_fifo format.
  --
  -- for example,
  --  - entry 1 writes 0x0001 to address 0x100
  --  - entry 2 writes 0x01e8 to address 0x108
  --
  -- AXI IIC controller registers :
  -- x0100 - Control Register
  -- x0108 - Tx FIFO          bit 8 = Start, Bit 9 = Stop, 7:0 = Data

  -- Command ROM data for part: KCU105
  constant kcu105_rom : t_cmd_rom := (
        0  => X"0000_003F",  -- Total IIC commands in table
        1  => X"0100_0001",  -- enable AXI IIC core
        -----------------------------------------------
        -- TCA9548 IIC switch                 Addr = 74
        -----------------------------------------------
        2  => X"0108_01e8",  -- addr = 74 (shifted up 1 bit - bit 0 = rw)
        3  => X"0108_0210",  -- channel 4 enabled (Si5328)
        -----------------------------------------------
        -- Si5328 Jitter Attenuator           Addr = 68
        -----------------------------------------------
        4  => X"0108_01d0",  -- Register 0
        5  => X"0108_0000",  -- setup Si5328 for
        6  => X"0108_0254",  -- free run mode
        -- Set N1_HS output divider
        7  => X"0108_01d0",  
        8  => X"0108_0019",  -- Register 25
        9  => X"0108_0280",  -- N1_HS =  8 (b100) (307.20MHz)
        -- Transceiver is clocked from CLKOUT2 from Si5328. 
        -- Set the CLKOUT2 divider here.
        10  => X"0108_01d0",  
        11  => X"0108_0022",  -- Register 34
        12  => X"0108_0200",  -- NC2_LS[19:16] = 2 (0x00)
        13  => X"0108_01d0",  
        14  => X"0108_0023",  -- Register 35
        15  => X"0108_0200",  -- NC2_LS[15:8] = 2 (0x00)
        16  => X"0108_01d0",  
        17  => X"0108_0024",  -- Register 36
        18  => X"0108_0201",  -- NC2_LS[7:0] = 2 (0x01)
        -- Set the PLL divider to 10 * 23904 (307.20MHz & 245.76MHz)
        -- Set the PLL divider to  9 * 27888 (368.64MHz)
        19  => X"0108_01d0",
        20  => X"0108_0028",  -- Register 40
        21  => X"0108_02C0",  -- N2_HS = 10, N2_LS = 23904 (x05D5F) (307.20MHz & 245.76MHz)
        22  => X"0108_01d0",  
        23  => X"0108_0029",  -- Register 41
        24  => X"0108_025D",  -- N2_LS[15:8] = (0x5D) (307.20MHz & 245.76MHz)
        25  => X"0108_01d0",  
        26  => X"0108_002A",  -- Register 42
        27  => X"0108_025F",  -- N2_LS[7:0] = (0x5F) (307.20MHz & 245.76MHz)
        -- The recovered clock from the FPGA to the CLKIN1 input of the Si5328
        -- is set at 15.36MHz. Divide this by 747 to get the correct input
        -- to the PLL.
        28  => X"0108_01d0",  
        29  => X"0108_002B",  -- Register 43
        30  => X"0108_0200",  -- N31[18:16] = 747 (0x00)
        31  => X"0108_01d0",  
        32  => X"0108_002C",  -- Register 44
        33  => X"0108_0202",  -- N31[15:8] = 747 (0x02)
        34  => X"0108_01d0",  
        35  => X"0108_002D",  -- Register 45
        36  => X"0108_02EA",  -- N31[7:0] = 747 (0xEA)
        -- The free running oscillator runs at 114.285MHz. Divide this by
        -- 5558 to get the correct input to the PLL   
        37  => X"0108_01d0",  
        38  => X"0108_002E",  -- Register 46
        39  => X"0108_0200",  -- N32[18:16] = 5558 (0x00)
        40  => X"0108_01d0",
        41  => X"0108_002F",  -- Register 47
        42  => X"0108_0215",  -- N32[18:16] = 5558 (0x15)
        43  => X"0108_01d0",  
        44  => X"0108_0030",  -- Register 48
        45  => X"0108_02B5",  -- N32[7:0] = 5558 (0xB5)
        -- Set the auto-select mode to "Automatic Revertive" and
        -- program LOCK settings
        46  => X"0108_01d0",
        47  => X"0108_0004",  -- Register 4
        48  => X"0108_0292",  -- AUTOSEL_REG[1:0] = 0b10
        49  => X"0108_01d0",
        50  => X"0108_0013",  -- Register 19
        51  => X"0108_0229",  -- LOCKT[2:0] = 0b001
        52  => X"0108_01d0",
        53  => X"0108_0089",  -- Register 137 
        54  => X"0108_0201",  -- FASTLOCK = 1
        55  => X"0108_01d0",
        56  => X"0108_0003",  -- Register 3
        57  => X"0108_0215",  -- SQ_ICAL = 1
        58  => X"0108_01d0",
        59  => X"0108_0002",  -- Register 2
        60  => X"0108_0212",  -- BWSEL_REG = 1
        -- Perform an internal calibration
        61  => X"0108_01d0",
        62  => X"0108_0088",  -- Register 136
        63  => X"0108_0240",  -- ICAL = 1
        others => (others => '0')
    );

  -- Command ROM data for part: VCU108
  constant vcu108_rom : t_cmd_rom := (
        0  => X"0000_0048",  -- Total IIC commands in table
        1  => X"0100_0001",  -- enable AXI IIC core
        -----------------------------------------------
        -- TCA9548 IIC switch                 Addr = 74
        -----------------------------------------------
        2  => X"0108_01e8",  -- addr = 74 (shifted up 1 bit - bit 0 = rw)
        3  => X"0108_0210",  -- channel 4 enabled (Si5328)
        -----------------------------------------------
        -- Si5328 Jitter Attenuator           Addr = 68
        -----------------------------------------------
        4  => X"0108_01d0",  -- Register 0
        5  => X"0108_0000",  -- setup Si5328 for
        6  => X"0108_0254",  -- free run mode
        -- Set N1_HS output divider
        7  => X"0108_01d0",  
        8  => X"0108_0019",  -- Register 25
        9  => X"0108_02C0",  -- N1_HS = 10 (b110) (245.76MHz)
        -- Transceiver is clocked from Si5328/CLKOUT1 when using the CFP.
        -- Set the CLKOUT1 divider here.
        10  => X"0108_01d0",  
        11  => X"0108_001F",  -- Register 31
        12  => X"0108_0200",  -- NC1_LS[19:16] = 2 (0x00)
        13  => X"0108_01d0",  
        14  => X"0108_0020",  -- Register 32
        15  => X"0108_0200",  -- NC1_LS[15:8] = 2 (0x00)
        16  => X"0108_01d0",  
        17  => X"0108_0021",  -- Register 33
        18  => X"0108_0201",  -- NC1_LS[7:0] = 2 (0x01)
        -- Transceiver is clocked from Si5328/CLKOUT2 when using the QSFP.
        -- Set the CLKOUT2 divider here.
        19  => X"0108_01d0",  
        20  => X"0108_0022",  -- Register 34
        21  => X"0108_0200",  -- NC2_LS[19:16] = 2 (0x00)
        22  => X"0108_01d0",  
        23  => X"0108_0023",  -- Register 35
        24  => X"0108_0200",  -- NC2_LS[15:8] = 2 (0x00)
        25  => X"0108_01d0",  
        26  => X"0108_0024",  -- Register 36
        27  => X"0108_0201",  -- NC2_LS[7:0] = 2 (0x01)
        -- Set the PLL divider to 10 * 23904 (307.20MHz & 245.76MHz)
        -- Set the PLL divider to  9 * 27888 (368.64MHz)
        28  => X"0108_01d0",
        29  => X"0108_0028",  -- Register 40
        30  => X"0108_02C0",  -- N2_HS = 10, N2_LS = 23904 (x05D5F) (307.20MHz & 245.76MHz)
        31  => X"0108_01d0",  
        32  => X"0108_0029",  -- Register 41
        33  => X"0108_025D",  -- N2_LS[15:8] = (0x5D) (307.20MHz & 245.76MHz)
        34  => X"0108_01d0",  
        35  => X"0108_002A",  -- Register 42
        36  => X"0108_025F",  -- N2_LS[7:0] = (0x5F) (307.20MHz & 245.76MHz)
        -- The recovered clock from the FPGA to the CLKIN1 input of the Si5328 is
        -- set at 15.36MHz. Divide this by 747 to get the correct input to the PLL.
        37  => X"0108_01d0",  
        38  => X"0108_002B",  -- Register 43
        39  => X"0108_0200",  -- N31[18:16] = 747 (0x00)
        40  => X"0108_01d0",  
        41  => X"0108_002C",  -- Register 44
        42  => X"0108_0202",  -- N31[15:8] = 747 (0x02)
        43  => X"0108_01d0",  
        44  => X"0108_002D",  -- Register 45
        45  => X"0108_02EA",  -- N31[7:0] = 747 (0xEA)
        -- The free running oscillator runs at 114.285MHz. Divide this by
        -- 5558 to get the correct input to the PLL   
        46  => X"0108_01d0",  
        47  => X"0108_002E",  -- Register 46
        48  => X"0108_0200",  -- N32[18:16] = 5558 (0x00)
        49  => X"0108_01d0",
        50  => X"0108_002F",  -- Register 47
        51  => X"0108_0215",  -- N32[18:16] = 5558 (0x15)
        52  => X"0108_01d0",  
        53  => X"0108_0030",  -- Register 48
        54  => X"0108_02B5",  -- N32[7:0] = 5558 (0xB5)
        -- Set the auto-select mode to "Automatic Revertive" and
        -- program LOCK settings
        55  => X"0108_01d0",
        56  => X"0108_0004",  -- Register 4
        57  => X"0108_0292",  -- AUTOSEL_REG[1:0] = 0b10
        58  => X"0108_01d0",
        59  => X"0108_0013",  -- Register 19
        60  => X"0108_0229",  -- LOCKT[2:0] = 0b001
        61  => X"0108_01d0",
        62  => X"0108_0089",  -- Register 137 
        63  => X"0108_0201",  -- FASTLOCK = 1
        64  => X"0108_01d0",
        65  => X"0108_0003",  -- Register 3
        66  => X"0108_0215",  -- SQ_ICAL = 1
        67  => X"0108_01d0",
        68  => X"0108_0002",  -- Register 2
        69  => X"0108_0212",  -- BWSEL_REG = 1
        -- Perform an internal calibration
        70  => X"0108_01d0",
        71  => X"0108_0088",  -- Register 136
        72  => X"0108_0240",  -- ICAL = 1
        others => (others => '0')
    );

  -- Command ROM data for part: VCU108 (Master/Slave)
  constant vcu108_ms_rom : t_cmd_rom := (
        0  => X"0000_005B",  -- Total IIC commands in table
        1  => X"0100_0001",  -- enable AXI IIC core
        -----------------------------------------------
        -- TCA9548 IIC switch                 Addr = 74
        -----------------------------------------------
        2  => X"0108_01e8",  -- addr = 74 (shifted up 1 bit - bit 0 = rw)
        3  => X"0108_0210",  -- channel 4 enabled (Si5328)
        -----------------------------------------------
        -- Si5328 Jitter Attenuator           Addr = 68
        -----------------------------------------------
        4  => X"0108_01d0",  -- Register 0
        5  => X"0108_0000",  -- setup Si5328 for
        6  => X"0108_0254",  -- free run mode
        -- Set N1_HS output divider
        7  => X"0108_01d0",  
        8  => X"0108_0019",  --  Register 25
        9  => X"0108_02C0",  --  N1_HS = 10 (b110) (245.76MHz)
        -- Transceiver is clocked from Si5328/CLKOUT1 when using the CFP.
        -- Set the CLKOUT1 divider here.
        10  => X"0108_01d0",  
        11  => X"0108_001F",  -- Register 31
        12  => X"0108_0200",  -- NC1_LS[19:16] = 2 (0x00)
        13  => X"0108_01d0",  
        14  => X"0108_0020",  -- Register 32
        15  => X"0108_0200",  -- NC1_LS[15:8] = 2 (0x00)
        16  => X"0108_01d0",  
        17  => X"0108_0021",  -- Register 33
        18  => X"0108_0201",  -- NC1_LS[7:0] = 2 (0x01)
        -- Transceiver is clocked from Si5328/CLKOUT2 when using the QSFP. 
        -- Set the CLKOUT2 divider here.
        19  => X"0108_01d0",  
        20  => X"0108_0022",  -- Register 34
        21  => X"0108_0200",  -- NC2_LS[19:16] = 2 (0x00)
        22  => X"0108_01d0",  
        23  => X"0108_0023",  -- Register 35
        24  => X"0108_0200",  -- NC2_LS[15:8] = 2 (0x00)
        25  => X"0108_01d0",  
        26  => X"0108_0024",  -- Register 36
        27  => X"0108_0201",  -- NC2_LS[7:0] = 2 (0x01)
        -- Set the PLL divider to 10 * 23904 (307.20MHz & 245.76MHz)
        -- Set the PLL divider to  9 * 27888 (368.64MHz)
        28  => X"0108_01d0",
        29  => X"0108_0028",  -- Register 40
        30  => X"0108_02C0",  -- N2_HS = 10, N2_LS = 23904 (x05D5F) (307.20MHz & 245.76MHz)
        31  => X"0108_01d0",  
        32  => X"0108_0029",  -- Register 41
        33  => X"0108_025D",  -- N2_LS[15:8] = (0x5D) (307.20MHz & 245.76MHz)
        34  => X"0108_01d0",  
        35  => X"0108_002A",  -- Register 42
        36  => X"0108_025F",  -- N2_LS[7:0] = (0x5F) (307.20MHz & 245.76MHz)
        -- The recovered clock from the FPGA to the CLKIN1 input of the Si5328
        -- is set at 15.36MHz. Divide this by 747 to get the correct input
        -- to the PLL.
        37  => X"0108_01d0",  
        38  => X"0108_002B",  -- Register 43
        39  => X"0108_0200",  -- N31[18:16] = 747 (0x00)
        40  => X"0108_01d0",  
        41  => X"0108_002C",  -- Register 44
        42  => X"0108_0202",  -- N31[15:8] = 747 (0x02)
        43  => X"0108_01d0",  
        44  => X"0108_002D",  -- Register 45
        45  => X"0108_02EA",  -- N31[7:0] = 747 (0xEA)
        -- The free running oscillator runs at 114.285MHz. Divide this by
        -- 5558 to get the correct input to the PLL   
        46  => X"0108_01d0",  
        47  => X"0108_002E",  -- Register 46
        48  => X"0108_0200",  -- N32[18:16] = 5558 (0x00)
        49  => X"0108_01d0",
        50  => X"0108_002F",  -- Register 47
        51  => X"0108_0215",  -- N32[18:16] = 5558 (0x15)
        52  => X"0108_01d0",  
        53  => X"0108_0030",  -- Register 48
        54  => X"0108_02B5",  -- N32[7:0] = 5558 (0xB5)
        -- Set the auto-select mode to "Automatic Revertive" and
        -- program LOCK settings
        55  => X"0108_01d0",
        56  => X"0108_0004",  -- Register 4
        57  => X"0108_0292",  -- AUTOSEL_REG[1:0] = 0b10
        58  => X"0108_01d0",
        59  => X"0108_0013",  -- Register 19
        60  => X"0108_0229",  -- LOCKT[2:0] = 0b001
        61  => X"0108_01d0",
        62  => X"0108_0089",  -- Register 137 
        63  => X"0108_0201",  -- FASTLOCK = 1
        64  => X"0108_01d0",
        65  => X"0108_0003",  -- Register 3
        66  => X"0108_0215",  -- SQ_ICAL = 1
        67  => X"0108_01d0",
        68  => X"0108_0002",  -- Register 2
        69  => X"0108_0212",  -- BWSEL_REG = 1
        -- Perform an internal calibration
        70  => X"0108_01d0",
        71  => X"0108_0088",  -- Register 136
        72  => X"0108_0240",  -- ICAL = 1
        -----------------------------------------------
        -- TCA9548 IIC switch                 Addr = 74
        -----------------------------------------------
        73 => X"0108_01e8",  -- addr = 74 (shifted up 1 bit - bit 0 = rw)
        74 => X"0108_0201",  -- channel 0 enabled (Si570)
        -----------------------------------------------
        -- Si570 Programmable Oscillator U104 Addr = 5D
        -----------------------------------------------
        75 => X"0108_01ba", -- Device address 
        76 => X"0108_0089", -- Register 137 
        77 => X"0108_0210", -- Freeze current freq
        -- Set RFREQ for ~245.76 MHz
        78 => X"0108_01ba",  
        79 => X"0108_0007",  -- Register 7
        80 => X"0108_0020",  -- Data (burst)
        81 => X"0108_00C2",  
        82 => X"0108_00B0",
        83 => X"0108_0021",
        84 => X"0108_00DE",
        85 => X"0108_0277",
        -- Unfreeze the DCO
        86 => X"0108_01ba",
        87 => X"0108_0089",
        88 => X"0108_0200",
        -- Set new freq bit
        89 => X"0108_01ba",
        90 => X"0108_0087",
        91 => X"0108_0240",
        others => (others => '0')
    );

  -- Command ROM data for part: ZCU102
  constant zcu102_rom : t_cmd_rom := (
        0  => X"0000_003F",  -- Total IIC commands in table
        1  => X"0100_0001",  -- enable AXI IIC core
        -----------------------------------------------
        -- TCA9548 IIC switch                 Addr = 74
        -----------------------------------------------
        2  => X"0108_01e8",  -- addr = 74 (shifted up 1 bit - bit 0 = rw)
        3  => X"0108_0210",  -- channel 4 enabled (Si5328)
        -----------------------------------------------
        -- Si5328 Jitter Attenuator           Addr = 69
        -----------------------------------------------
        4  => X"0108_01d2",  -- Register 0
        5  => X"0108_0000",  -- setup Si5328 for
        6  => X"0108_0254",  -- free run mode
        -- Set N1_HS output divider
        7  => X"0108_01d2",  
        8  => X"0108_0019",  -- Register 25
        9  => X"0108_02C0",  -- N1_HS = 10 (b110) (245.76MHz)
        -- Transceiver is clocked from Si5328/CLKOUT1 when using the CFP.
        -- Set the CLKOUT1 divider here.
        10  => X"0108_01d2",  
        11  => X"0108_001F",  -- Register 31
        12  => X"0108_0200",  -- NC1_LS[19:16] = 2 (0x00)
        13  => X"0108_01d2",  
        14  => X"0108_0020",  -- Register 32
        15  => X"0108_0200",  -- NC1_LS[15:8] = 2 (0x00)
        16  => X"0108_01d2",  
        17  => X"0108_0021",  -- Register 33
        18  => X"0108_0201",  -- NC1_LS[7:0] = 2 (0x01)
        -- Set the PLL divider to 10 * 23904 (307.20MHz & 245.76MHz)
        -- Set the PLL divider to  9 * 27888 (368.64MHz)
        19  => X"0108_01d2",
        20  => X"0108_0028",  -- Register 40
        21  => X"0108_02C0",  -- N2_HS = 10, N2_LS = 23904 (x05D5F) (307.20MHz & 245.76MHz)
        22  => X"0108_01d2",  
        23  => X"0108_0029",  -- Register 41
        24  => X"0108_025D",  -- N2_LS[15:8] = (0x5D) (307.20MHz & 245.76MHz)
        25  => X"0108_01d2",  
        26  => X"0108_002A",  -- Register 42
        27  => X"0108_025F",  -- N2_LS[7:0] = (0x5F) (307.20MHz & 245.76MHz)
        -- The recovered clock from the FPGA to the CLKIN1 input of the Si5328
        -- is set at 15.36MHz. Divide this by 747 to get the correct input
        -- to the PLL.
        28  => X"0108_01d2",  
        29  => X"0108_002B",  -- Register 43
        30  => X"0108_0200",  -- N31[18:16] = 747 (0x00)
        31  => X"0108_01d2",  
        32  => X"0108_002C",  -- Register 44
        33  => X"0108_0202",  -- N31[15:8] = 747 (0x02)
        34  => X"0108_01d2",  
        35  => X"0108_002D",  -- Register 45
        36  => X"0108_02EA",  -- N31[7:0] = 747 (0xEA)
        -- The free running oscillator runs at 114.285MHz. Divide this by
        -- 5558 to get the correct input to the PLL   
        37  => X"0108_01d2",  
        38  => X"0108_002E",  -- Register 46
        39  => X"0108_0200",  -- N32[18:16] = 5558 (0x00)
        40  => X"0108_01d2",
        41  => X"0108_002F",  -- Register 47
        42  => X"0108_0215",  -- N32[18:16] = 5558 (0x15)
        43  => X"0108_01d2",  
        44  => X"0108_0030",  -- Register 48
        45  => X"0108_02B5",  -- N32[7:0] = 5558 (0xB5)
        -- Set the auto-select mode to "Automatic Revertive" and
        -- program LOCK settings
        46  => X"0108_01d2",
        47  => X"0108_0004",  -- Register 4
        48  => X"0108_0292",  -- AUTOSEL_REG[1:0] = 0b10
        49  => X"0108_01d2",
        50  => X"0108_0013",  -- Register 19
        51  => X"0108_0229",  -- LOCKT[2:0] = 0b001
        52  => X"0108_01d2",
        53  => X"0108_0089",  -- Register 137 
        54  => X"0108_0201",  -- FASTLOCK = 1
        55  => X"0108_01d2",
        56  => X"0108_0002",  -- Register 2 
        57  => X"0108_0212",  -- BWSEL_REG = 1
        58  => X"0108_01d2",
        59  => X"0108_0003",  -- Register 3
        60  => X"0108_0215",  -- SQ_ICAL = 1
        -- Perform an internal calibration
        61  => X"0108_01d2",
        62  => X"0108_0088",  -- Register 136
        63  => X"0108_0240",  -- ICAL = 1
        others => (others => '0')
    );

  -- Command ROM data for part: VCU118
  constant vcu118_rom : t_cmd_rom := (
        0  => X"0000_003F",  -- Total IIC commands in table
        1  => X"0100_0001",  -- enable AXI IIC core
        -----------------------------------------------
        -- TCA9548 IIC switch (U80)           Addr = 75
        -----------------------------------------------
        2  => X"0108_01ea",  -- addr = 75 (shifted up 1 bit - bit 0 = rw)
        3  => X"0108_0220",  -- channel 5 enabled (Si570)
        -----------------------------------------------
        -- Si570 Programmable Oscillator U38 Addr = 5D
        -----------------------------------------------
        4  => X"0108_01ba", -- Device address 
        5  => X"0108_0089", -- Register 137 
        6  => X"0108_0210", -- Freeze current freq
        -- Set RFREQ for ~245.76 MHz
        7  => X"0108_01ba",  
        8  => X"0108_0007",  -- Register 7
        9  => X"0108_0020",  -- Data (burst)
        10 => X"0108_00C2",  
        11 => X"0108_00B0",
        12 => X"0108_0021",
        13 => X"0108_00DE",
        14 => X"0108_0277",
        -- Unfreeze the DCO
        15 => X"0108_01ba",
        16 => X"0108_0089",
        17 => X"0108_0200",
        -- Set new freq bit
        18 => X"0108_01ba",
        19 => X"0108_0087",
        20 => X"0108_0240",
        others => (others => '0')
    );

  -- Command ROM data for part: VCU118 Hard FEC wrapper demo design
  constant vcu118_hfw_rom : t_cmd_rom := (
        0  => X"0000_0014",  -- Total IIC commands in table
        1  => X"0100_0001",  -- enable AXI IIC core
        -----------------------------------------------
        -- TCA9548 IIC switch (U80)           Addr = 75
        -----------------------------------------------
        2  => X"0108_01ea",  -- addr = 75 (shifted up 1 bit - bit 0 = rw)
        3  => X"0108_0220",  -- channel 5 enabled (Si570)
        -----------------------------------------------
        -- Si570 Programmable Oscillator U38 Addr = 5D
        -----------------------------------------------
        4  => X"0108_01ba", -- Device address 
        5  => X"0108_0089", -- Register 137 
        6  => X"0108_0210", -- Freeze current freq
        -- Set RFREQ for ~245.76 MHz
        7  => X"0108_01ba",  
        8  => X"0108_0007",  -- Register 7
        9  => X"0108_0020",  -- Data (burst)
        10 => X"0108_00C2",  
        11 => X"0108_00B0",
        12 => X"0108_0021",
        13 => X"0108_00DE",
        14 => X"0108_0277",
        -- Unfreeze the DCO
        15 => X"0108_01ba",
        16 => X"0108_0089",
        17 => X"0108_0200",
        -- Set new freq bit
        18 => X"0108_01ba",
        19 => X"0108_0087",
        20 => X"0108_0240",
        others => (others => '0')
    );

  -- Command ROM data for part: UNKNOWN (unrecognized part selected)
  constant unknown_rom : t_cmd_rom := ( others => (others => '1') );

  -- Initialize IIC Controller ROM based on FPGA part
  function init_iic_rom( src : iic_part ) return t_cmd_rom;

end iic_controller_pkg;

package body iic_controller_pkg is

    -- Initialize IIC Controller ROM based on FPGA part
    function init_iic_rom( src : iic_part ) return t_cmd_rom is
        variable tmp : t_cmd_rom := unknown_rom;
    begin

        case src is
            when kcu105     => tmp := kcu105_rom;
            when vcu108     => tmp := vcu108_rom;
            when vcu108_ms  => tmp := vcu108_ms_rom;
            when zcu102     => tmp := zcu102_rom;
            when vcu118     => tmp := vcu118_rom;
            when vcu118_hfw => tmp := vcu118_hfw_rom;
            when others     => tmp := unknown_rom;
        end case;

        return tmp;
    end;

end;
