// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */
#include <linux/dcache.h>
#include <linux/debugfs.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/seq_file.h>
#include <linux/slab.h>
#include "efct_driver.h"
#include "debugfs.h"
#ifdef CONFIG_XILINX_AUX_EFCT
#include "efct_auxbus.h"
#endif

/* Top-level debug directory ([/sys/kernel]/debug/xilinx_efct) */
static struct dentry *efct_debug_root;
/* Maximum length for a name component or symlink target */
#define EFCT_DEBUGFS_NAME_LEN 32

static const char *const evq_type_names[] = {
	[EVQ_T_RX] =	"RX_EVENT_QUEUE",
	[EVQ_T_TX] =	"TX_EVENT_QUEUE",
	[EVQ_T_AUX] =	"AUX_EVENT_QUEUE",
	[EVQ_T_NONE] =	"NONE_EVENT_QUEUE",
};

static const u32 evq_type_max = sizeof(evq_type_names);

static const char *const nic_state_names[] = {
	[STATE_UNINIT] =	"UNINIT",
	[STATE_NET_UP] =	"READY",
	[STATE_NET_DOWN] =	"DOWN",
	[STATE_DISABLED] =	"DISABLED",
};

static const u32 nic_state_max = sizeof(nic_state_names);

static const char *const driver_dist_layout_names[] = {
	[RX_LAYOUT_DISTRIBUTED] =	"RX_LAYOUT_DISTRIBUTED",
	[RX_LAYOUT_SEPARATED] =	"RX_LAYOUT_SEPERATED",
};

static const u32 driver_dist_layout_max = sizeof(driver_dist_layout_names);

#ifdef CONFIG_XILINX_AUX_EFCT
static const char *const client_state_names[] = {
	[STATE_OPEN] =	"OPEN",
	[STATE_CLOSED] = "CLOSED",
};

static const u32 client_state_max = sizeof(client_state_names);

static int efct_debugfs_aux_dev_params_show(struct seq_file *file, void *v)
{
	struct aux_resources *res;

	res = file->private;

	seq_printf(file, "%-16s:%x\n", "evq_base", res->evq_base);
	seq_printf(file, "%-16s:%x\n", "evq_lim", res->evq_lim);
	seq_printf(file, "%-16s:%x\n", "txq_base", res->txq_base);
	seq_printf(file, "%-16s:%x\n", "txq_lim", res->txq_lim);
	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_aux_dev_params);

static int efct_debugfs_aux_client_params_show(struct seq_file *file, void *v)
{
	struct xlnx_efct_client *client;

	client = file->private;

	seq_printf(file, "%-16s:%s\n", "name", client->drvops->name);
	seq_printf(file, "%-16s:%x\n", "index", client->index);
	seq_printf(file, "%-16s:%x => %s\n", "state", client->state,
		   STRING_TABLE_LOOKUP(client->state, client_state));
	seq_printf(file, "%-16s:%lx\n", "active_txq", client->active_txq[0]);
	seq_printf(file, "%-16s:%x\n", "active_hugepages", atomic_read(&client->active_hugepages));
	seq_printf(file, "%-16s:%lx\n", "active_evq", client->active_evq[0]);
	seq_printf(file, "%-16s:%lx\n", "filter_map", client->filter_map[0]);

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_aux_client_params);

static int efct_debugfs_aux_client_rxinfo_show(struct seq_file *file, void *v)
{
	struct efct_rxq_client_meta *meta;
	struct xlnx_efct_client *client;
	int i;

	client = file->private;
	seq_printf(file, "%8s%16s%16s%16s%12s\n", "Index", "active_hp", "added_hp", "removed_hp",
		   "ref_cnt");
	for (i = 0; i < client->efct->rxq_count; i++) {
		meta = &client->rxq_info[i];
		seq_printf(file, "%8u%16x%16x%16u%12x\n", i, atomic_read(&meta->active_hp),
			   meta->added_hp, meta->removed_hp, meta->ref_cnt);
	}

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_aux_client_rxinfo);

static int efct_debugfs_sbl_buf_list_show(struct seq_file *file, void *v)
{
	struct efct_rx_queue *rxq = file->private;
	struct efct_super_buffer *sb;
	int clientidx;
	int i;

	rcu_read_lock();
	sb = rxq->sbl->sb;
	seq_printf(file, "%8s%20s%12s%12s%12s%12s%12s\n", "Index", "DMA addr", "Len", "Sentinel",
		   "Ref count", "flags", "Client id");
	for (i = 0; i < rxq->hpl_alloced * SUPER_BUFFS_PER_HUGE_PAGE; i++) {
		if (sb[i].client_id)
			clientidx = sb[i].client_id->index;
		else
			clientidx = -1;
		seq_printf(file, "%8u%20llx%12x%12u%12x%12x%12x\n", i, sb[i].dma_buffer.dma_addr,
			   sb[i].dma_buffer.len, sb[i].dma_buffer.sentinel,
			   sb[i].ref_count, sb[i].flags, clientidx);
	}
	rcu_read_unlock();

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_sbl_buf_list);

static int efct_debugfs_max_hp_id_show(struct seq_file *file, void *v)
{
	struct efct_rx_queue *rxq = file->private;

	rcu_read_lock();
	seq_printf(file, "%x\n", rxq->hpl->max_hp_id);
	rcu_read_unlock();

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_max_hp_id);

#endif

static int efct_debugfs_card_params_show(struct seq_file *file, void *v)
{
	struct efct_device *efct_dev;

	efct_dev = file->private;

	seq_printf(file, "%-26s: 0x%x\n", "num_ports", efct_dev->num_ports);
	seq_printf(file, "%-26s: 0x%x\n", "mem_bar", efct_dev->mem_bar);
	seq_printf(file, "%-26s: 0x%x\n", "reg_base", efct_dev->reg_base);
	seq_printf(file, "%-26s: 0x%x => %s\n", "dist_layout", efct_dev->dist_layout,
		   STRING_TABLE_LOOKUP(efct_dev->dist_layout, driver_dist_layout));
	seq_printf(file, "%-26s: 0x%x\n", "separated_rx_cpu", efct_dev->separated_rx_cpu);
	seq_printf(file, "%-26s: 0x%x\n", "vec_per_port", efct_dev->vec_per_port);
	seq_printf(file, "%-26s: 0x%x\n", "mcdi_logging", efct_dev->mcdi_logging);

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_card_params);

static int efct_debugfs_nbl_params_show(struct seq_file *file, void *v)
{
	struct efct_nic_buffer_list *nbl;

	nbl = file->private;

	seq_printf(file, "%-26s: 0x%x\n", "active_nic_buffs", nbl->active_nic_buffs);
	seq_printf(file, "%-26s: 0x%x\n", "head_index", nbl->head_index);
	seq_printf(file, "%-26s: 0x%x\n", "tail_index", nbl->tail_index);
	seq_printf(file, "%-26s: 0x%x\n", "meta_offset", nbl->meta_offset);
	seq_printf(file, "%-26s: 0x%x\n", "prev_meta_offset", nbl->prev_meta_offset);
	seq_printf(file, "%-26s: 0x%x\n", "frame_offset_fixed", nbl->frame_offset_fixed);
	seq_printf(file, "%-26s: 0x%x\n", "seq_no", nbl->seq_no);

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_nbl_params);

static int efct_debugfs_rx_queue_params_show(struct seq_file *file, void *v)
{
	struct efct_rx_queue *rxq;

	rxq = file->private;

	seq_printf(file, "%-26s: 0x%x\n", "evq_index", rxq->evq_index);
	seq_printf(file, "%-26s: 0x%x\n", "cpu", rxq->cpu);
	seq_printf(file, "%-26s: 0x%x\n", "label", rxq->label);
	seq_printf(file, "%-26s: 0x%x\n", "buffer_size", rxq->buffer_size);
	seq_printf(file, "%-26s: 0x%x\n", "filter_count", rxq->filter_count);
	seq_printf(file, "%-26s: 0x%x\n", "num_rx_buffs", rxq->num_rx_buffs);
	seq_printf(file, "%-26s: 0x%x\n", "num_entries", rxq->num_entries);
	seq_printf(file, "%-26s: 0x%x\n", "pkt_stride", rxq->pkt_stride);
	seq_printf(file, "%-26s: 0x%llx\n", "n_rx_tcp_udp_chksum_err",
		   rxq->n_rx_tcp_udp_chksum_err);
	seq_printf(file, "%-26s: 0x%llx\n", "n_rx_ip_hdr_chksum_err", rxq->n_rx_ip_hdr_chksum_err);
	seq_printf(file, "%-26s: 0x%llx\n", "n_rx_eth_crc_err", rxq->n_rx_eth_crc_err);
	seq_printf(file, "%-26s: 0x%llx\n", "n_rx_eth_len_err", rxq->n_rx_eth_len_err);
	seq_printf(file, "%-26s: 0x%llx\n", "n_rx_sentinel_drop_count",
		   rxq->n_rx_sentinel_drop_count);
#ifdef CONFIG_XILINX_AUX_EFCT
	seq_printf(file, "%-26s: 0x%lx\n", "active_aux_clients", rxq->active_aux_clients[0]);
	seq_printf(file, "%-26s: 0x%llx\n", "poison_cfg.value", rxq->poison_cfg.value);
	seq_printf(file, "%-26s: 0x%lx\n", "poison_cfg.length", rxq->poison_cfg.length);
	seq_printf(file, "%-26s: 0x%x\n", "exclusive_client", rxq->exclusive_client);
#endif

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_rx_queue_params);

static int efct_debugfs_port_params_show(struct seq_file *file, void *v)
{
	struct efct_nic *efct;

	efct = file->private;

	seq_printf(file, "%-26s: %s\n", "name", efct->name);
	seq_printf(file, "%-26s: 0x%x\n", "port_num", efct->port_num);
	seq_printf(file, "%-26s: 0x%x => %s\n", "nic_state", efct->state,
		   STRING_TABLE_LOOKUP(efct->state, nic_state));
	seq_printf(file, "%-26s: 0x%x\n", "int_error_count", efct->int_error_count);
	seq_printf(file, "%-26s: 0x%lx\n", "int_error_expire", efct->int_error_expire);
	seq_printf(file, "%-26s: 0x%x\n", "max_evq_count", efct->max_evq_count);
	seq_printf(file, "%-26s: 0x%x\n", "max_txq_count", efct->max_txq_count);
	seq_printf(file, "%-26s: 0x%x\n", "rxq_count", efct->rxq_count);
	seq_printf(file, "%-26s: 0x%x\n", "msg_enable", efct->msg_enable);
	seq_printf(file, "%-26s: 0x%x\n", "port_base", efct->port_base);
	seq_printf(file, "%-26s: 0x%x\n", "num_mac_stats", efct->num_mac_stats);
	seq_printf(file, "%-26s: 0x%x\n", "stats_period_ms", efct->stats_period_ms);
	seq_printf(file, "%-26s: 0x%x\n", "stats_initialised", efct->stats_initialised);
	seq_printf(file, "%-26s: 0x%x\n", "stats_enabled", efct->stats_enabled);
	seq_printf(file, "%-26s: 0x%x\n", "wanted_fc", efct->wanted_fc);
	seq_printf(file, "%-26s: 0x%x\n", "irq_mod_step_ns", efct->irq_mod_step_ns);
	seq_printf(file, "%-26s: 0x%x\n", "irq_rx_moderation_ns", efct->irq_rx_moderation_ns);
	seq_printf(file, "%-26s: 0x%x\n", "timer_max_ns", efct->timer_max_ns);
	seq_printf(file, "%-26s: 0x%x\n", "timer_quantum_ns", efct->timer_quantum_ns);
	seq_printf(file, "%-26s: 0x%x\n", "n_link_state_changes", efct->n_link_state_changes);

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_port_params);

static int efct_debugfs_txq_state_show(struct seq_file *file, void *v)
{
	struct efct_tx_queue *txq;

	txq = file->private;

	seq_printf(file, "%-26s: 0x%x\n", "evq_index", txq->evq_index);
	seq_printf(file, "%-26s: 0x%x\n", "label", txq->label);
	seq_printf(file, "%-26s: 0x%x\n", "piobuf_offset", txq->piobuf_offset);
	seq_printf(file, "%-26s: 0x%x\n", "added_sequence", txq->added_sequence);
	seq_printf(file, "%-26s: 0x%x\n", "completed_sequence", txq->completed_sequence);
	seq_printf(file, "%-26s: 0x%x\n", "inuse_fifo_bytes", atomic_read(&txq->inuse_fifo_bytes));
	seq_printf(file, "%-26s: 0x%x\n", "inflight_pkts", atomic_read(&txq->inflight_pkts));
	seq_printf(file, "%-26s: 0x%x\n", "ct_thresh", txq->ct_thresh);

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_txq_state);

static int efct_debugfs_evq_state_show(struct seq_file *file, void *v)
{
	struct efct_ev_queue *evq;

	evq = file->private;

	seq_printf(file, "%-26s: 0x%x => %s\n", "type", evq->type, STRING_TABLE_LOOKUP(evq->type,
										       evq_type));
	seq_printf(file, "%-26s: 0x%x\n", "entries", evq->entries);
	seq_printf(file, "%-26s: 0x%x\n", "msi.irq", evq->msi.irq);
	seq_printf(file, "%-26s: 0x%x\n", "msi.idx", evq->msi.idx);
	seq_printf(file, "%-26s: 0x%x\n", "consumer_index", evq->consumer_index);
	seq_printf(file, "%-26s: 0x%x\n", "queue_count", evq->queue_count);
	seq_printf(file, "%-26s: 0x%x\n", "evq_phase", evq->evq_phase);
	seq_printf(file, "%-26s: 0x%x\n", "unsol_consumer_index", evq->unsol_consumer_index);
	seq_printf(file, "%-26s: 0x%x\n", "irq_moderation_ns", evq->irq_moderation_ns);
	seq_printf(file, "%-26s: 0x%x\n", "irq_count", evq->irq_count);
	seq_printf(file, "%-26s: 0x%x\n", "irq_mod_score", evq->irq_mod_score);
	seq_printf(file, "%-26s: 0x%x\n", "irq_adapt_low_thresh", evq->irq_adapt_low_thresh);
	seq_printf(file, "%-26s: 0x%x\n", "irq_adapt_high_thresh", evq->irq_adapt_high_thresh);
	seq_printf(file, "%-26s: 0x%x\n", "irq_adapt_irqs", evq->irq_adapt_irqs);
	seq_printf(file, "%-26s: 0x%x\n", "tx_merge_timeout_ns", evq->tx_merge_timeout_ns);

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_evq_state);

static int efct_debugfs_link_state_show(struct seq_file *file, void *v)
{
	struct efct_link_state *state;

	state = file->private;

	seq_printf(file, "%-10s: 0x%x\n", "up", state->up);
	seq_printf(file, "%-10s: 0x%x\n", "fd", state->fd);
	seq_printf(file, "%-10s: 0x%x\n", "speed", state->speed);
	seq_printf(file, "%-10s: 0x%x\n", "ld_caps", state->ld_caps);
	seq_printf(file, "%-10s: 0x%x\n", "lp_caps", state->lp_caps);

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_link_state);

static int efct_debugfs_nic_error_show(struct seq_file *file, void *v)
{
	struct efct_nic_errors *error;

	error = file->private;

	seq_printf(file, "%-18s: 0x%x\n", "missing_event", atomic_read(&error->missing_event));
	seq_printf(file, "%-18s: 0x%x\n", "rx_reset", atomic_read(&error->rx_reset));
	seq_printf(file, "%-18s: 0x%x\n", "rx_desc_fetch", atomic_read(&error->rx_desc_fetch));
	seq_printf(file, "%-18s: 0x%x\n", "tx_desc_fetch", atomic_read(&error->tx_desc_fetch));
	seq_printf(file, "%-18s: 0x%x\n", "spurious_tx", atomic_read(&error->spurious_tx));

	return 0;
}

DEFINE_SHOW_ATTRIBUTE(efct_debugfs_nic_error);

static int efct_debugfs_design_param_show(struct seq_file *file, void *v)
{
	struct design_params *params;

	params = file->private;

	seq_printf(file, "%-24s: 0x%x\n", "rx_stride", params->rx_stride);
	seq_printf(file, "%-24s: 0x%x\n", "rx_buffer_len", params->rx_buffer_len);
	seq_printf(file, "%-24s: 0x%x\n", "rx_queues", params->rx_queues);
	seq_printf(file, "%-24s: 0x%x\n", "rx_buf_fifo_size", params->rx_buf_fifo_size);
	seq_printf(file, "%-24s: 0x%x\n", "rx_metadata_len", params->rx_metadata_len);
	seq_printf(file, "%-24s: 0x%x\n", "tx_apertures", params->tx_apertures);
	seq_printf(file, "%-24s: 0x%x\n", "tx_max_reorder", params->tx_max_reorder);
	seq_printf(file, "%-24s: 0x%x\n", "tx_aperture_size", params->tx_aperture_size);
	seq_printf(file, "%-24s: 0x%x\n", "tx_fifo_size", params->tx_fifo_size);
	seq_printf(file, "%-24s: 0x%x\n", "ctpio_stride", params->ctpio_stride);
	seq_printf(file, "%-24s: 0x%x\n", "evq_stride", params->evq_stride);
	seq_printf(file, "%-24s: 0x%x\n", "evq_sizes", params->evq_sizes);
	seq_printf(file, "%-24s: 0x%x\n", "num_evq", params->num_evq);
	seq_printf(file, "%-24s: 0x%x\n", "unsol_credit_seq_mask", params->unsol_credit_seq_mask);
	seq_printf(file, "%-24s: 0x%x\n", "frame_offset_fixed", params->frame_offset_fixed);
	seq_printf(file, "%-24s: 0x%x\n", "ts_subnano_bit", params->ts_subnano_bit);
	seq_printf(file, "%-24s: 0x%x\n", "l4_csum_proto", params->l4_csum_proto);
	seq_printf(file, "%-24s: 0x%x\n", "max_runt", params->max_runt);
	seq_printf(file, "%-24s: 0x%x\n", "num_filter", params->num_filter);

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_design_param);

static int efct_debugfs_nbl_buf_list_show(struct seq_file *file, void *v)
{
	struct efct_nic_buffer *nb = file->private;
	int i;

	seq_printf(file, "%8s%20s%12s%12s%12s%12s\n", "Index", "DMA addr", "Len", "Sentinel",
		   "is_dbl", "id");
	for (i = 0; i < NIC_BUFFS_PER_QUEUE; i++)
		seq_printf(file, "%8u%20llx%12x%12u%12x%12x\n", i, nb[i].dma_buffer.dma_addr,
			   nb[i].dma_buffer.len, nb[i].dma_buffer.sentinel,
			   nb[i].is_dbl, nb[i].id);
	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_nbl_buf_list);

static int efct_debugfs_dbl_buf_list_show(struct seq_file *file, void *v)
{
	struct efct_driver_buffer *db = file->private;
	int i;

	seq_printf(file, "%8s%20s%12s%12s\n", "Index", "DMA addr", "Len", "Sentinel");
	for (i = 0; i < RX_MAX_DRIVER_BUFFS; i++)
		seq_printf(file, "%8u%20llx%12x%12u\n", i, db[i].dma_buffer.dma_addr,
			   db[i].dma_buffer.len, db[i].dma_buffer.sentinel);

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(efct_debugfs_dbl_buf_list);

/* efct_init_debugfs - create debugfs directories for x3 driver
 *
 * Create debugfs directory "xilinx_efct".  This must be
 * called before any of the other functions that create debugfs
 * directories. The directories must be cleaned up using efct_fini_debugfs().
 */
void efct_init_debugfs(const char *module)
{
	/* Create top-level directory */
	efct_debug_root = debugfs_create_dir(module, NULL);
	if (IS_ERR_OR_NULL(efct_debug_root))
		goto err;

	return;
 err:
	pr_err("debugfs mount fail\n");
	efct_debug_root = NULL;
}

/**
 * efct_init_debugfs_func - create debugfs directory for PCIe function
 * @efct_dev:		efct NIC
 * return:		Negative error code or 0 on success.
 *
 * Create debugfs directory containing parameter-files for @efct_dev,
 * and a subdirectory "design_params" containing per-card design parameters.
 * The directories must be cleaned up using efct_fini_debugfs_nic().
 */
void efct_init_debugfs_func(struct efct_device *efct_dev)
{
	if (!efct_debug_root)
		return;
	/* Create directory */
	efct_dev->debug_dir = debugfs_create_dir(pci_name(efct_dev->pci_dev),
						 efct_debug_root);
	if (IS_ERR_OR_NULL(efct_dev->debug_dir))
		goto err;
	debugfs_create_file("design_params", 0444, efct_dev->debug_dir, &efct_dev->params,
			    &efct_debugfs_design_param_fops);
	debugfs_create_file("card_params", 0444, efct_dev->debug_dir, &efct_dev->params,
			    &efct_debugfs_card_params_fops);
	return;

err:
	efct_dev->debug_dir = NULL;
}

/**
 * efct_init_debugfs_nic - create debugfs directory for NIC
 * @efct:		efct NIC
 * return:		Negative error code or 0 on success
 *
 * Create "port" debugfs directory containing parameter-files for @efct,
 * It must be cleaned up using efct_fini_debugfs_nic().
 */
void efct_init_debugfs_nic(struct efct_nic *efct)
{
	char name[EFCT_DEBUGFS_NAME_LEN];

	/* Create directory */
	if (IS_ERR_OR_NULL(efct->efct_dev->debug_dir))
		goto err;
	if (snprintf(name, sizeof(name), "port%d", efct->port_num) >= sizeof(name))
		goto err;

	efct->debug_port_dir = debugfs_create_dir(name, efct->efct_dev->debug_dir);
	if (IS_ERR_OR_NULL(efct->debug_port_dir))
		goto err;
	debugfs_create_file("nic_errors", 0444, efct->debug_port_dir,
			    &efct->errors, &efct_debugfs_nic_error_fops);
	debugfs_create_file("link_state", 0444, efct->debug_port_dir,
			    &efct->link_state, &efct_debugfs_link_state_fops);
	debugfs_create_file("port_params", 0444, efct->debug_port_dir,
			    efct, &efct_debugfs_port_params_fops);
	return;
err:
	efct->debug_port_dir = NULL;
}

#ifdef CONFIG_XILINX_AUX_EFCT
void efct_init_debugfs_aux_dev_client(struct xlnx_efct_client *client)
{
	char name[EFCT_DEBUGFS_NAME_LEN];

	if (IS_ERR_OR_NULL(client->efct->debug_aux_dir))
		goto err;
	/* Create directory */
	if (snprintf(name, sizeof(name), "client@%x", client->index) >= sizeof(name))
		goto err;

	client->debug_dir = debugfs_create_dir(name,
					       client->efct->debug_aux_dir);
	if (IS_ERR_OR_NULL(client->debug_dir))
		goto err;
	debugfs_create_file("client_params", 0444, client->debug_dir,
			    client, &efct_debugfs_aux_client_params_fops);
	debugfs_create_file("client_rxinfo", 0444, client->debug_dir,
			    client, &efct_debugfs_aux_client_rxinfo_fops);
	return;
err:
	client->debug_dir = NULL;
}

/**
 * efct_init_debugfs_aux_dev - create debugfs directory for Aux Device
 * @efct:		efct NIC
 * return:		Negative error code or 0 on success.
 *
 * Create a aux device debugfs directory containing parameter-files
 * for @efct. The directories must be cleaned up using efct_fini_debugfs_aux_dev().
 */
void efct_init_debugfs_aux_dev(struct efct_nic *efct)
{
	char name[EFCT_DEBUGFS_NAME_LEN];

	if (IS_ERR_OR_NULL(efct->debug_port_dir))
		goto err;
	if (snprintf(name, sizeof(name), "%s_%d", efct->aux_dev.adev.name, efct->port_num) >=
	     sizeof(name))
		goto err;

	efct->debug_aux_dir = debugfs_create_dir(name, efct->debug_port_dir);
	if (IS_ERR_OR_NULL(efct->debug_aux_dir))
		goto err;

	debugfs_create_file("aux_dev_params", 0444, efct->debug_aux_dir,
			    &efct->aux_res, &efct_debugfs_aux_dev_params_fops);

	return;
err:
	efct->debug_aux_dir = NULL;
}

void efct_fini_debugfs_aux_dev_client(struct xlnx_efct_client *client)
{
	debugfs_remove_recursive(client->debug_dir);
	client->debug_dir = NULL;
}
#endif

/**
 * efct_init_debugfs_rx_queue - create debugfs directory for RX queue
 * @rx_queue:		efct RX queue
 * return:		Negative error code or 0 on success.
 *
 * Create a debugfs directory containing parameter-files for @rx_queue.
 * The directory must be cleaned up using efct_fini_debugfs_rx_queue().
 */
void efct_init_debugfs_rx_queue(struct efct_rx_queue *rx_queue)
{
	char name[EFCT_DEBUGFS_NAME_LEN];

	if (IS_ERR_OR_NULL(rx_queue->efct->debug_port_dir))
		goto err;
	/* Create directories */
	if (snprintf(name, sizeof(name), "rxq%d", rx_queue->index) >= sizeof(name))
		goto err;

	rx_queue->debug_dir = debugfs_create_dir(name,
						 rx_queue->efct->debug_port_dir);
	if (IS_ERR_OR_NULL(rx_queue->debug_dir))
		goto err;

#ifdef CONFIG_XILINX_AUX_EFCT
	debugfs_create_file("max_hp_id", 0444, rx_queue->debug_dir,
			    rx_queue, &efct_debugfs_max_hp_id_fops);
	debugfs_create_file("sbl_buf_list", 0444, rx_queue->debug_dir,
			    rx_queue, &efct_debugfs_sbl_buf_list_fops);
#endif
	debugfs_create_file("nbl_params", 0444, rx_queue->debug_dir,
			    &rx_queue->nbl, &efct_debugfs_nbl_params_fops);
	debugfs_create_file("nbl_buf_list", 0444, rx_queue->debug_dir,
			    rx_queue->nbl.nb, &efct_debugfs_nbl_buf_list_fops);
	debugfs_create_file("dbl_buf_list", 0444, rx_queue->debug_dir,
			    rx_queue->dbl.db, &efct_debugfs_dbl_buf_list_fops);
	debugfs_create_file("rx_queue_params", 0444, rx_queue->debug_dir,
			    rx_queue, &efct_debugfs_rx_queue_params_fops);

	return;
err:
	rx_queue->debug_dir = NULL;
}

/**
 * efct_fini_debugfs_rx_queue - remove debugfs directory for RX queue
 * @rx_queue:		efct RX queue
 *
 * Remove directory created for @rx_queue by efct_init_debugfs_rx_queue().
 */
void efct_fini_debugfs_rx_queue(struct efct_rx_queue *rx_queue)
{
	debugfs_remove_recursive(rx_queue->debug_dir);
	rx_queue->debug_dir = NULL;
}

/**
 * efct_init_debugfs_tx_queue - create debugfs directory for TX queue
 * @tx_queue:		efct TX queue
 * return:		Negative error code or 0 on success.
 *
 * Create a debugfs file containing parameter-files for @tx_queue.
 */
void efct_init_debugfs_tx_queue(struct efct_tx_queue *tx_queue)
{
	char name[EFCT_DEBUGFS_NAME_LEN];

	if (IS_ERR_OR_NULL(tx_queue->efct->debug_port_dir))
		goto err;
	if (snprintf(name, sizeof(name), "txq%d", tx_queue->txq_index) >= sizeof(name))
		goto err;
	debugfs_create_file(name, 0444, tx_queue->efct->debug_port_dir,
			    tx_queue, &efct_debugfs_txq_state_fops);
err:
	return;
}

/**
 * efct_fini_debugfs_tx_queue - remove debugfs file for TX queue
 * @tx_queue:		efct TX queue
 *
 * Remove file created for @tx_queue by efct_init_debugfs_tx_queue().
 */
void efct_fini_debugfs_tx_queue(struct efct_tx_queue *tx_queue)
{
	char name[EFCT_DEBUGFS_NAME_LEN];

	if (snprintf(name, sizeof(name), "txq%d", tx_queue->txq_index) >= sizeof(name))
		return;
	debugfs_lookup_and_remove(name, tx_queue->efct->debug_port_dir);
}

/**
 * efct_init_debugfs_ev_queue - create debugfs file for EV queue
 * @ev_queue:		efct EV queue
 *
 * Create a debugfs file containing parameters for @ev_queue.
 */
void efct_init_debugfs_ev_queue(struct efct_ev_queue *ev_queue)
{
	char name[EFCT_DEBUGFS_NAME_LEN];

	if (IS_ERR_OR_NULL(ev_queue->efct->debug_port_dir))
		goto err;
	if (snprintf(name, sizeof(name), "evq%d", ev_queue->index) >= sizeof(name))
		goto err;
	debugfs_create_file(name, 0444, ev_queue->efct->debug_port_dir,
			    ev_queue, &efct_debugfs_evq_state_fops);
err:
	return;
}

/**
 * efct_fini_debugfs_ev_queue - remove debugfs file for ev queue
 * @ev_queue:		efct Event queue
 *
 * Remove file created for @ev_queue by efct_init_debugfs_ev_queue().
 */
void efct_fini_debugfs_ev_queue(struct efct_ev_queue *ev_queue)
{
	char name[EFCT_DEBUGFS_NAME_LEN];

	if (snprintf(name, sizeof(name), "evq%d", ev_queue->index) >= sizeof(name))
		return;
	debugfs_lookup_and_remove(name, ev_queue->efct->debug_port_dir);
}

/**
 * efct_fini_debugfs_nic - remove debugfs directories for NIC
 * @efct:		efct NIC
 *
 * Remove debugfs directories created for @efct by efct_init_debugfs_nic().
 */
void efct_fini_debugfs_nic(struct efct_nic *efct)
{
	debugfs_remove_recursive(efct->debug_port_dir);
	efct->debug_port_dir = NULL;
}

#ifdef CONFIG_XILINX_AUX_EFCT
/**
 * efct_fini_debugfs_aux_dev - remove debugfs directories for Aux Device
 * @efct:		efct NIC
 *
 * Remove debugfs directories created for @efct by efct_init_debugfs_aux_dev().
 */
void efct_fini_debugfs_aux_dev(struct efct_nic *efct)
{
	debugfs_remove_recursive(efct->debug_aux_dir);
	efct->debug_aux_dir = NULL;
}
#endif
/**
 * efct_fini_debugfs_func - remove debugfs directories for the PCIe func
 * @efct_dev:		pci device
 *
 * Remove debugfs directories created for @efct_dev by efct_init_debugfs_func().
 */
void efct_fini_debugfs_func(struct efct_device *efct_dev)
{
	debugfs_remove_recursive(efct_dev->debug_dir);
	efct_dev->debug_dir = NULL;
}

/**
 * efct_fini_debugfs - remove debugfs directories for x3 driver
 *
 * Remove directories created by efct_init_debugfs().
 */
void efct_fini_debugfs(void)
{
	debugfs_remove_recursive(efct_debug_root);
	efct_debug_root = NULL;
}
