// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/pci.h>
#include "efct_driver.h"
#include "efct_io.h"
#include "efct_reg.h"
#include "efct_common.h"
#include "efct_netdev.h"
#include "efct_nic.h"
#ifdef CONFIG_XILINX_AUX_EFCT
#include "efct_auxbus.h"
#endif
#include "efct_devlink.h"
#ifdef CONFIG_XILINX_DEBUGFS
#include "debugfs.h"
#endif

/**************************************************************************
 *
 * Configurable values
 *
 *************************************************************************/

#define EFCT_PCI_DEFAULT_BAR	0
#define HOST_NIC_MAGIC_RESET 0xEFC7FA57 //TODO remove when HW reg file is updated
/* Number of bytes at start of vendor specified extended capability that indicate
 * that the capability is vendor specified. i.e. offset from value returned by
 * pci_find_next_ext_capability() to beginning of vendor specified capability
 * header.
 */
#define PCI_EXT_CAP_HDR_LENGTH  4

/* Expected size of a Xilinx continuation
 * address table entry.
 */
#define ESE_GZ_CFGBAR_CONT_CAP_MIN_LENGTH      16

/* Total 19 parameters are there 0 - 18. PAD bit and filter is not mandatory */
#define EFCT_REQRD_DESIGN_PARAMS 0x7FFFE

static int efct_pci_walk_xilinx_table
			(struct efct_device *efct_dev,
			 u64 offset,
			 struct efct_func_ctl_window *result);

/* Number of bytes to offset when reading bit position x with dword accessors. */
#define ROUND_DOWN_TO_DWORD(x) (((x) & (~31)) >> 3)

/* PCIe link bandwidth measure:
 * bw = (width << (speed - 1))
 */
#define EFCT_BW_PCIE_GEN3_X8  (8  << (3 - 1))
#define EFCT_BW_PCIE_GEN3_X16 (16 << (3 - 1))

#define EXTRACT_BITS(x, lbn, width) \
			(((x) >> ((lbn) & (31))) & (((1ull) << (width)) - 1))

enum efct_tlv_state_machine {
	EFCT_TLV_TYPE,
	EFCT_TLV_TYPE_CONT,
	EFCT_TLV_LENGTH,
	EFCT_TLV_VALUE
};

struct efct_tlv_state {
	enum efct_tlv_state_machine state;
	u64 value;
	u32 value_offset;
	u16 type;
	u8 len;
};

static u32 _efct_pci_get_bar_bits_with_width
			(struct efct_device *efct_dev,
			 int structure_start, int lbn, int width)
{
	union efct_dword dword;

	efct_readd(&dword,  efct_dev->membase +
			efct_reg(efct_dev, structure_start + ROUND_DOWN_TO_DWORD(lbn)));

	return EXTRACT_BITS(le32_to_cpu(dword.word32), lbn, width);
}

static int efct_acquire_msix_vectors(struct efct_device *efct_dev, int nvec)
{
	int i, j, rc, index;

	efct_dev->xentries = kmalloc_array(nvec, sizeof(*efct_dev->xentries), GFP_KERNEL);
	if (!efct_dev->xentries)
		return -ENOMEM;

	for (j = 0; j < efct_dev->num_ports; j++) {
		for (i = 0; i < (nvec / efct_dev->num_ports); i++) {
			index = i + (j * nvec / efct_dev->num_ports);
			efct_dev->xentries[index].entry = (EFCT_MSIX_PER_PORT * j) + i;
		}
	}
	rc = pci_enable_msix_range(efct_dev->pci_dev, efct_dev->xentries, nvec, nvec);
	if (rc < 0) {
		pci_disable_msix(efct_dev->pci_dev);
		kfree(efct_dev->xentries);
	}

	return rc;
}

static int efct_alloc_msix(struct efct_device *efct_dev)
{
	int num_vec, min_irq_per_port;
	int nvec_allocated = 0;

	/*Maximum event queues available*/
	num_vec = efct_dev->params.num_evq * efct_dev->num_ports;
	num_vec = min_t(u32, num_vec, pci_msix_vec_count(efct_dev->pci_dev));

	nvec_allocated = efct_acquire_msix_vectors(efct_dev, num_vec);

	if (nvec_allocated == -ENOSPC) {
		/*Minimum irq needed, is the sum of RXQ,one TXQ for net driver and minimum
		 * * of one irq for AUX clients
		 */
		min_irq_per_port = efct_dev->params.rx_queues
				+ EFCT_MAX_CORE_TX_QUEUES + MIN_AUX_IRQ;
		num_vec = efct_dev->num_ports * min_irq_per_port;
		nvec_allocated = efct_acquire_msix_vectors(efct_dev, num_vec);
	}

	if (nvec_allocated < 0) {
		pci_err(efct_dev->pci_dev, "could not enable %d MSI-X\n", num_vec);
		return nvec_allocated;
	}

	/*Number of allocated vectors per port*/
	efct_dev->vec_per_port = (nvec_allocated / efct_dev->num_ports);
	return 0;
}

static int efct_assign_msix(struct efct_nic *efct, int index)
{
	int rc = 0;
#if !defined(EFCT_NOT_UPSTREAM) || !defined(CONFIG_X3_BUSYPOLL)
	int i, j;

	for_each_set_bit(i, &efct->evq_active_mask, efct->max_evq_count) {
		j = (index * efct->efct_dev->vec_per_port) + i;
		efct->evq[i].msi.irq = efct->efct_dev->xentries[j].vector;
	}
#ifdef CONFIG_XILINX_AUX_EFCT
	/*Populate irq resources*/
	rc = efct_make_irq_resources(efct, efct->efct_dev->xentries, index);

#endif
#endif//CONFIG_X3_BUSYPOLL
	return rc;
}

static void efct_free_msix(struct efct_device *efct_dev)
{
	kfree(efct_dev->xentries);
	pci_disable_msix(efct_dev->pci_dev);
}

#define efct_pci_get_bar_bits(efct_dev, entry_location, bitdef) \
	_efct_pci_get_bar_bits_with_width(efct_dev, entry_location, \
		bitdef ## _LBN, bitdef ## _WIDTH)

static int efct_pci_parse_efct_entry
			(struct efct_device *efct_dev, int entry_location,
			 struct efct_func_ctl_window *result)
{
	u16 project_id;
	u8 table_type;
	u8 type_rev;
	u64 offset;
	u32 bar;

	bar = efct_pci_get_bar_bits(efct_dev, entry_location + EFCT_VSEC_TABLE_ENTRY_OFFSET,
				    ERF_HZ_FUNC_CTL_WIN_BAR);
	offset = efct_pci_get_bar_bits(efct_dev, entry_location + EFCT_VSEC_TABLE_ENTRY_OFFSET,
				       ERF_HZ_FUNC_CTL_WIN_OFF_16B);
	project_id = efct_pci_get_bar_bits(efct_dev, entry_location + EFCT_VSEC_TABLE_ENTRY_OFFSET,
					   ERF_HZ_PROJECT_ID);
	type_rev =  efct_pci_get_bar_bits(efct_dev, entry_location + EFCT_VSEC_TABLE_ENTRY_OFFSET,
					  ERF_HZ_TYPE_REVISION);
	table_type = efct_pci_get_bar_bits(efct_dev, entry_location + EFCT_VSEC_TABLE_ENTRY_OFFSET,
					   ERF_HZ_TABLE_TYPE);
	pci_dbg(efct_dev->pci_dev, "Found efct function control window bar=%d offset=0x%llx\n",
		bar, offset);

	if (result->valid) {
		pci_err(efct_dev->pci_dev, "Duplicated efct table entry.\n");
		return -EINVAL;
	}

	if (project_id != EFCT_VSEC_ENTRY_PROJECT_ID_VAL) {
		pci_err(efct_dev->pci_dev, "Bad Project ID value of 0x%x in Xilinx capabilities efct entry.\n",
			project_id);
		return -EINVAL;
	}

	if (bar == ESE_GZ_CFGBAR_EFCT_BAR_NUM_EXPANSION_ROM ||
	    bar == ESE_GZ_CFGBAR_EFCT_BAR_NUM_INVALID) {
		pci_err(efct_dev->pci_dev, "Bad BAR value of %d in Xilinx capabilities efct entry.\n",
			bar);
		return -EINVAL;
	}

	if (type_rev != EFCT_VSEC_ENTRY_TYPE_REV_VAL) {
		pci_err(efct_dev->pci_dev, "Bad Type revision value of 0x%x in Xilinx capabilities efct entry.\n",
			type_rev);
		return -EINVAL;
	}

	if (table_type != EFCT_VSEC_ENTRY_TABLE_TYPE_VAL) {
		pci_err(efct_dev->pci_dev, "Bad Table type value of 0x%x in Xilinx capabilities efct entry.\n",
			table_type);
		return -EINVAL;
	}

	result->bar = bar;
	result->offset = offset;
	result->valid = true;
	return 0;
}

static bool efct_pci_does_bar_overflow
			(struct efct_device *efct_dev, int bar,
			 u64 next_entry)
{
	return next_entry + ESE_GZ_CFGBAR_ENTRY_HEADER_SIZE >
		pci_resource_len(efct_dev->pci_dev, bar);
}

/* Parse a Xilinx capabilities table entry describing a continuation to a new
 * sub-table.
 */
static int efct_pci_parse_continue_entry
			(struct efct_device *efct_dev, int entry_location,
			 struct efct_func_ctl_window *result)
{
	union efct_oword entry;
	u32 previous_bar;
	u64 offset;
	int rc = 0;
	u32 bar;

	efct_reado(efct_dev, &entry, efct_dev->membase + efct_reg(efct_dev, entry_location));

	bar = EFCT_OWORD_FIELD32(entry, ESF_GZ_CFGBAR_CONT_CAP_BAR);

	offset = EFCT_OWORD_FIELD64(entry, ESF_GZ_CFGBAR_CONT_CAP_OFFSET) <<
		ESE_GZ_CONT_CAP_OFFSET_BYTES_SHIFT;

	previous_bar = efct_dev->mem_bar;

	if (bar == ESE_GZ_VSEC_BAR_NUM_EXPANSION_ROM ||
	    bar == ESE_GZ_VSEC_BAR_NUM_INVALID) {
		pci_err(efct_dev->pci_dev, "Bad BAR value of %d in Xilinx capabilities sub-table.\n",
			bar);
		return -EINVAL;
	}

	if (bar != previous_bar) {
		efct_fini_io(efct_dev);

		if (efct_pci_does_bar_overflow(efct_dev, bar, offset)) {
			pci_err(efct_dev->pci_dev, "Xilinx table will overrun BAR[%d] offset=0x%llx\n",
				bar, offset);
			return -EINVAL;
		}

		/* Temporarily map new BAR. */
		rc = efct_init_io(efct_dev, bar, efct_dev->max_dma_mask,
				  pci_resource_len(efct_dev->pci_dev, bar));
		if (rc) {
			pci_err(efct_dev->pci_dev,
				"Mapping new BAR for Xilinx table failed, rc=%d\n", rc);
			return rc;
		}
	}

	rc = efct_pci_walk_xilinx_table(efct_dev, offset, result);
	if (rc) {
		pci_err(efct_dev->pci_dev,
			"Iteration on Xilinx capabilities table failed, rc=%d\n", rc);
		return rc;
	}

	if (bar != previous_bar) {
		efct_fini_io(efct_dev);

		/* Put old BAR back. */
		rc = efct_init_io(efct_dev, previous_bar, efct_dev->max_dma_mask,
				  pci_resource_len(efct_dev->pci_dev, previous_bar));
		if (rc) {
			pci_err(efct_dev->pci_dev, "Putting old BAR back failed, rc=%d\n", rc);
			return rc;
		}
	}

	return 0;
}

/* Iterate over the Xilinx capabilities table in the currently mapped BAR and
 * call efct_pci_parse_efct_entry() on any efct entries and
 * efct_pci_parse_continue_entry() on any table continuations.
 */
static int efct_pci_walk_xilinx_table
			(struct efct_device *efct_dev, u64 offset,
			 struct efct_func_ctl_window *result)
{
	u64 current_entry = offset;
	int rc = 0;

	while (true) {
		u32 id = efct_pci_get_bar_bits(efct_dev, current_entry, ERF_HZ_FORMAT_ID);
		u32 last = efct_pci_get_bar_bits(efct_dev, current_entry, ERF_HZ_LAST_CAPABILITY);
		u32 rev = efct_pci_get_bar_bits(efct_dev, current_entry, ERF_HZ_FORMAT_REV_CODE);
		u32 entry_size;

		if (id == ESE_GZ_CFGBAR_ENTRY_LAST)
			return 0;

		entry_size =
			efct_pci_get_bar_bits(efct_dev, current_entry, ERF_HZ_LENGTH);

		pci_dbg(efct_dev->pci_dev,
			"Seen Xilinx table entry 0x%x size 0x%x at 0x%llx in BAR[%d]\n",
			id, entry_size, current_entry, efct_dev->mem_bar);

		if (entry_size < sizeof(uint32_t) * 2) {
			pci_err(efct_dev->pci_dev,
				"Xilinx table entry too short len=0x%x\n", entry_size);
			return -EINVAL;
		}

		switch (id) {
		case ESE_GZ_CFGBAR_ENTRY_EFCT:
			if (rev != ESE_GZ_CFGBAR_ENTRY_REV_EFCT ||
			    entry_size < ESE_GZ_CFGBAR_ENTRY_SIZE_EFCT) {
				pci_err(efct_dev->pci_dev, "Bad length or rev for efct entry in Xilinx capabilities table. entry_size=%d rev=%d.\n",
					entry_size, rev);
				break;
			}

			rc = efct_pci_parse_efct_entry(efct_dev, current_entry, result);
			if (rc) {
				pci_err(efct_dev->pci_dev,
					"Parsing efct entry failed, rc=%d\n", rc);
				return rc;
			}
			break;
		case ESE_GZ_CFGBAR_ENTRY_CONT_CAP_ADDR:
			if (rev != 0 || entry_size < ESE_GZ_CFGBAR_CONT_CAP_MIN_LENGTH) {
				pci_err(efct_dev->pci_dev, "Bad length or rev for continue entry in Xilinx capabilities table. entry_size=%d rev=%d.\n",
					entry_size, rev);
				return -EINVAL;
			}

			rc = efct_pci_parse_continue_entry(efct_dev, current_entry, result);
			if (rc)
				return rc;
			break;
		default:
			/* Ignore unknown table entries. */
			break;
		}

		if (last)
			return 0;

		current_entry += entry_size;

		if (efct_pci_does_bar_overflow(efct_dev, efct_dev->mem_bar, current_entry)) {
			pci_err(efct_dev->pci_dev, "Xilinx table overrun at position=0x%llx.\n",
				current_entry);
			return -EINVAL;
		}
	}
}

static int _efct_pci_get_config_bits_with_width
			(struct efct_device *efct_dev,
			 int structure_start, int lbn,
			 int width, u32 *result)
{
	int pos = structure_start + ROUND_DOWN_TO_DWORD(lbn);
	u32 temp;
	int rc;

	rc = pci_read_config_dword(efct_dev->pci_dev, pos, &temp);
	if (rc) {
		pci_err(efct_dev->pci_dev, "Failed to read PCI config dword at %d\n",
			pos);
		return rc;
	}

	*result = EXTRACT_BITS(temp, lbn, width);

	return 0;
}

#define efct_pci_get_config_bits(efct_dev, entry_location, bitdef, result) \
	_efct_pci_get_config_bits_with_width(efct_dev, entry_location,  \
		bitdef ## _LBN, bitdef ## _WIDTH, result)

/* Call efct_pci_walk_xilinx_table() for the Xilinx capabilities table pointed
 * to by this PCI_EXT_CAP_ID_VNDR.
 */
static int efct_pci_parse_xilinx_cap
			(struct efct_device *efct_dev, int vndr_cap,
			 bool has_offset_hi,
			 struct efct_func_ctl_window *result)
{
	u32 offset_high = 0;
	u32 offset_lo = 0;
	u64 offset = 0;
	u32 bar = 0;
	int rc = 0;

	rc = efct_pci_get_config_bits(efct_dev, vndr_cap, ESF_GZ_VSEC_TBL_BAR, &bar);
	if (rc) {
		pci_err(efct_dev->pci_dev, "Failed to read ESF_GZ_VSEC_TBL_BAR, rc=%d\n",
			rc);
		return rc;
	}

	if (bar == ESE_GZ_CFGBAR_CONT_CAP_BAR_NUM_EXPANSION_ROM ||
	    bar == ESE_GZ_CFGBAR_CONT_CAP_BAR_NUM_INVALID) {
		pci_err(efct_dev->pci_dev, "Bad BAR value of %d in Xilinx capabilities sub-table.\n",
			bar);
		return -EINVAL;
	}

	rc = efct_pci_get_config_bits(efct_dev, vndr_cap, ESF_GZ_VSEC_TBL_OFF_LO, &offset_lo);
	if (rc) {
		pci_err(efct_dev->pci_dev, "Failed to read ESF_GZ_VSEC_TBL_OFF_LO, rc=%d\n",
			rc);
		return rc;
	}

	/* Get optional extension to 64bit offset. */
	if (has_offset_hi) {
		rc = efct_pci_get_config_bits(efct_dev,
					      vndr_cap, ESF_GZ_VSEC_TBL_OFF_HI, &offset_high);
		if (rc) {
			pci_err(efct_dev->pci_dev, "Failed to read ESF_GZ_VSEC_TBL_OFF_HI, rc=%d\n",
				rc);
			return rc;
		}
	}

	offset = (((u64)offset_lo) << ESE_GZ_VSEC_TBL_OFF_LO_BYTES_SHIFT) |
		 (((u64)offset_high) << ESE_GZ_VSEC_TBL_OFF_HI_BYTES_SHIFT);

	if (offset > pci_resource_len(efct_dev->pci_dev, bar) - sizeof(uint32_t) * 2) {
		pci_err(efct_dev->pci_dev, "Xilinx table will overrun BAR[%d] offset=0x%llx\n",
			bar, offset);
		return -EINVAL;
	}

	/* Temporarily map BAR. */
	rc = efct_init_io(efct_dev, bar, efct_dev->max_dma_mask,
			  pci_resource_len(efct_dev->pci_dev, bar));
	if (rc) {
		pci_err(efct_dev->pci_dev, "efct_init_io failed, rc=%d\n", rc);
		return rc;
	}

	rc = efct_pci_walk_xilinx_table(efct_dev, offset, result);

	/* Unmap temporarily mapped BAR. */
	efct_fini_io(efct_dev);
	return rc;
}

/* Call efct_pci_parse_efct_entry() for each Xilinx PCI_EXT_CAP_ID_VNDR
 * capability.
 */
static int efct_pci_find_func_ctrl_window
			(struct efct_device *efct_dev,
			 struct efct_func_ctl_window *result)
{
	int num_xilinx_caps = 0;
	int cap = 0;

	result->valid = false;

	while ((cap = pci_find_next_ext_capability(efct_dev->pci_dev, cap,
						   PCI_EXT_CAP_ID_VNDR)) != 0) {
		int vndr_cap = cap + PCI_EXT_CAP_HDR_LENGTH;
		u32 vsec_ver = 0;
		u32 vsec_len = 0;
		u32 vsec_id = 0;
		int rc = 0;

		num_xilinx_caps++;

		rc = efct_pci_get_config_bits(efct_dev, vndr_cap,
					      ESF_HZ_PCI_EXPRESS_XCAP_ID, &vsec_id);
		if (rc) {
			pci_err(efct_dev->pci_dev,
				"Failed to read ESF_HZ_PCI_EXPRESS_XCAP_ID, rc=%d\n",
				rc);
			return rc;
		}

		rc = efct_pci_get_config_bits(efct_dev, vndr_cap,
					      ESF_HZ_PCI_EXPRESS_XCAP_VER, &vsec_ver);
		if (rc) {
			pci_err(efct_dev->pci_dev,
				"Failed to read ESF_HZ_PCI_EXPRESS_XCAP_VER, rc=%d\n",
				rc);
			return rc;
		}

		/* Get length of whole capability - i.e. starting at cap */
		rc = efct_pci_get_config_bits(efct_dev, vndr_cap, ESF_HZ_PCI_EXPRESS_XCAP_NEXT,
					      &vsec_len);
		if (rc) {
			pci_err(efct_dev->pci_dev,
				"Failed to read ESF_HZ_PCI_EXPRESS_XCAP_NEXT, rc=%d\n",
				rc);
			return rc;
		}

		if (vsec_id == ESE_GZ_XLNX_VSEC_ID &&
		    vsec_ver == ESE_GZ_VSEC_VER_XIL_CFGBAR &&
		    vsec_len >= ESE_GZ_VSEC_LEN_MIN) {
			bool has_offset_hi = (vsec_len >= ESE_GZ_VSEC_LEN_HIGH_OFFT);

			rc = efct_pci_parse_xilinx_cap(efct_dev, vndr_cap, has_offset_hi, result);
			if (rc) {
				pci_err(efct_dev->pci_dev,
					"Failed to parse xilinx capabilities table, rc=%d", rc);
				return rc;
			}
		}
	}

	if (num_xilinx_caps && !result->valid) {
		pci_err(efct_dev->pci_dev, "Seen %d Xilinx tables, but no efct entry.\n",
			num_xilinx_caps);
		return -EINVAL;
	}

	return 0;
}

static u32 efct_device_check_pcie_link(struct pci_dev *pdev, u32 *actual_width,
				       u32 *max_width, u32 *actual_speed,
				       u32 *nic_bandwidth)
{
	int cap = pci_find_capability(pdev, PCI_CAP_ID_EXP);
	u32 nic_speed;
	u16 lnksta;
	u16 lnkcap;

	*actual_speed = 0;
	*actual_width = 0;
	*max_width = 0;
	*nic_bandwidth = 0;

	if (!cap ||
	    pci_read_config_word(pdev, cap + PCI_EXP_LNKSTA, &lnksta) ||
	    pci_read_config_word(pdev, cap + PCI_EXP_LNKCAP, &lnkcap))
		return 0;

	*actual_width = (lnksta & PCI_EXP_LNKSTA_NLW) >>
			__ffs(PCI_EXP_LNKSTA_NLW);

	*max_width = (lnkcap & PCI_EXP_LNKCAP_MLW) >> __ffs(PCI_EXP_LNKCAP_MLW);
	*actual_speed = (lnksta & PCI_EXP_LNKSTA_CLS);

	nic_speed = 1;
	if (lnkcap & PCI_EXP_LNKCAP_SLS_5_0GB)
		nic_speed = 2;
	/* PCIe Gen3 capabilities are in a different config word. */
	if (!pci_read_config_word(pdev, cap + PCI_EXP_LNKCAP2, &lnkcap)) {
		if (lnkcap & PCI_EXP_LNKCAP2_SLS_8_0GB)
			nic_speed = 3;
	}

	*nic_bandwidth = *max_width << (nic_speed - 1);

	return nic_speed;
}

static void efct_nic_check_pcie_link(struct efct_device *efct_dev, u32 desired_bandwidth,
				     u32 *actual_width, u32 *actual_speed)
{
	struct pci_dev *pdev = efct_dev->pci_dev;
	u32 nic_bandwidth;
	u32 bandwidth = 0;
	u32 nic_width = 0;
	u32 nic_speed = 0;
	u32 width = 0;
	u32 speed = 0;

	nic_speed = efct_device_check_pcie_link(pdev, &width, &nic_width, &speed,
						&nic_bandwidth);

	if (!nic_speed)
		goto out;

	if (width > nic_width)
		pci_dbg(pdev, "PCI Express width is %d, with maximum expected %d. If running on a virtualized platform this is fine, otherwise it indicates a PCI problem.\n",
			width, nic_width);

	bandwidth = width << (speed - 1);

	if (desired_bandwidth > nic_bandwidth)
		/* You can desire all you want, it ain't gonna happen. */
		desired_bandwidth = nic_bandwidth;

	if (desired_bandwidth && bandwidth < desired_bandwidth) {
		pci_warn(pdev,
			 "This Network Adapter requires the equivalent of %d lanes at PCI Express %d speed for full throughput, but is currently limited to %d lanes at PCI Express %d speed.\n",
			 desired_bandwidth > EFCT_BW_PCIE_GEN3_X8 ? 16 : 8,
			 nic_speed, width, speed);
		pci_warn(pdev, "Consult your motherboard documentation to find a more suitable slot\n");
	} else if (bandwidth < nic_bandwidth) {
		pci_warn(pdev,
			 "This Network Adapter requires a slot with %d lanes at PCI Express %d speed for optimal latency, but is currently limited to %d lanes at PCI Express %d speed\n",
			 nic_width, nic_speed, width, speed);
	}

out:
	if (actual_width)
		*actual_width = width;

	if (actual_speed)
		*actual_speed = speed;
}

/**************************************************************************
 *
 * List of NICs supported
 *
 **************************************************************************/

/* PCI device ID table */
static const struct pci_device_id efct_pci_table[] = {
	{PCI_DEVICE(PCI_VENDOR_ID_XILINX, 0x5074),
	.driver_data = (unsigned long)&efct_nic_type},
	{PCI_DEVICE(PCI_VENDOR_ID_XILINX, 0x5075),
	.driver_data = (unsigned long)&efct_nic_type},
	{PCI_DEVICE(PCI_VENDOR_ID_XILINX, 0x5084),
	.driver_data = (unsigned long)&efct_nic_type},
	{0}			/* end of list */
};

static int efct_check_func_ctl_magic(struct efct_device *efct_dev)
{
	union efct_dword reg;

	efct_readd(&reg, efct_dev->membase + efct_reg(efct_dev, ER_HZ_FUN_WIN_REG_HOST_NIC_MAGIC));
	if (EFCT_DWORD_FIELD(reg, EFCT_DWORD_0) != HOST_NIC_MAGIC_RESET)
		return -EIO;

	return 0;
}

static u32 efct_get_num_ports(struct efct_device *efct_dev)
{
	union efct_dword reg;

	efct_readd(&reg, efct_dev->membase + efct_reg(efct_dev, ER_HZ_FUN_WIN_REG_HOST_NUM_PORTS));
	return EFCT_DWORD_FIELD(reg, EFCT_DWORD_0);
}

static void efct_port_free(struct efct_device *efct_dev, u8 port_idx)
{
	efct_close_netdev(efct_dev->efct[port_idx]);
#ifdef CONFIG_XILINX_AUX_EFCT
	efct_auxbus_unregister(efct_dev->efct[port_idx]);
	kfree(efct_dev->efct[port_idx]->irq_resources);
	efct_dev->efct[port_idx]->irq_resources = NULL;
#endif
#if defined(EFCT_USE_KCOMPAT) && (defined(EFCT_USE_DEVLINK) && !defined(EFCT_HAVE_SET_NETDEV_DEVLINK_PORT))
	devlink_port_type_clear(&efct_dev->efct[port_idx]->dl_port);
#endif
	efct_unregister_netdev(efct_dev->efct[port_idx]);
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_USE_DEVLINK)
	devl_port_unregister(&efct_dev->efct[port_idx]->dl_port);
#endif
	efct_remove_netdev(efct_dev->efct[port_idx]);
	efct_dev->efct[port_idx]->type->remove(efct_dev->efct[port_idx]);
	efct_flush_reset_workqueue();
	efct_unmap_membase(efct_dev->efct[port_idx]);
	free_netdev(efct_dev->efct[port_idx]->net_dev);
}

static int efct_tlv_feed(struct efct_tlv_state *state, u8 byte)
{
	switch (state->state) {
	case EFCT_TLV_TYPE:
		state->type = byte & 0x7f;
		state->state = (byte & 0x80) ? EFCT_TLV_TYPE_CONT
					     : EFCT_TLV_LENGTH;
		/* Clear ready to read in a new entry */
		state->value = 0;
		state->value_offset = 0;
		return 0;
	case EFCT_TLV_TYPE_CONT:
		state->type |= byte << 7;
		state->state = EFCT_TLV_LENGTH;
		return 0;
	case EFCT_TLV_LENGTH:
		state->len = byte;
		/* We only handle TLVs that fit in a u64 */
		if (state->len > sizeof(state->value))
			return -EOPNOTSUPP;
		/* len may be zero, implying a value of zero */
		state->state = state->len ? EFCT_TLV_VALUE : EFCT_TLV_TYPE;
		return 0;
	case EFCT_TLV_VALUE:
		state->value |= ((u64)byte) << (state->value_offset * 8);
		state->value_offset++;
		if (state->value_offset >= state->len)
			state->state = EFCT_TLV_TYPE;
		return 0;
	default: /* state machine error, can't happen */
		WARN_ON_ONCE(1);
		return -EIO;
	}
}

static int efct_process_design_param
			(struct efct_device *efct_dev,
			 const struct efct_tlv_state *reader)
{
	switch (reader->type) {
	case ESE_EFCT_DP_GZ_PAD: /* padding, skip it */
		return 0;
	case ESE_EFCT_DP_GZ_RX_STRIDE:
		efct_dev->params.rx_stride = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_EVQ_STRIDE:
		efct_dev->params.evq_stride = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_CTPIO_STRIDE:
		efct_dev->params.ctpio_stride = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_RX_BUFFER_SIZE:
		/*This value shall be multiplied by 4096 to get exact buffer size*/
		efct_dev->params.rx_buffer_len = min_t(u16, reader->value, 0x100);
		return 0;
	case ESE_EFCT_DP_GZ_RX_QUEUES:
		efct_dev->params.rx_queues = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_TX_CTPIO_APERTURES:
		efct_dev->params.tx_apertures = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_RX_BUFFER_FIFO_SIZE:
		/*Number of RX buffer that can be posted at any time*/
		efct_dev->params.rx_buf_fifo_size = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_FRAME_OFFSET_FIXED:
		efct_dev->params.frame_offset_fixed = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_RX_METADATA_LENGTH:
		efct_dev->params.rx_metadata_len = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_TX_MAXIMUM_REORDER:
		efct_dev->params.tx_max_reorder = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_TX_CTPIO_APERTURE_SIZE:
		efct_dev->params.tx_aperture_size = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_TX_PACKET_FIFO_SIZE:
		efct_dev->params.tx_fifo_size = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_PARTIAL_TSTAMP_SUB_NANO_BITS:
		efct_dev->params.ts_subnano_bit = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_EVQ_UNSOL_CREDIT_SEQ_BITS:
		 efct_dev->params.unsol_credit_seq_mask = (1 << reader->value) - 1;
		return 0;
	case ESE_EFCT_DP_GZ_RX_L4_CSUM_PROTOCOLS:
		efct_dev->params.l4_csum_proto = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_RX_MAX_RUNT:
		efct_dev->params.max_runt = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_EVQ_SIZES:
		efct_dev->params.evq_sizes = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_EV_QUEUES:
		efct_dev->params.num_evq = reader->value;
		return 0;
	case ESE_EFCT_DP_GZ_NUM_FILTER:
		efct_dev->params.num_filter = reader->value;
		return 0;
	default:
		/* Host interface says "Drivers should ignore design parameters
		 * that they do not recognise."
		 */
		pr_info("Ignoring unrecognised design parameter %u\n", reader->type);
		return 0;
	}
}

static int efct_check_design_params(struct efct_device *efct_dev)
{
	u32 data, design_params_present_bitmask = 0x0;
	struct efct_tlv_state reader = {};
	u32 total_len, offset = 0;
	union efct_dword reg;
	int rc = 0, i;

	efct_readd(&reg, efct_dev->membase +
		  efct_reg(efct_dev, ER_HZ_FUN_WIN_REG_HOST_PARAMS_TLV_LEN));
	total_len = EFCT_DWORD_FIELD(reg, EFCT_DWORD_0);
	pci_dbg(efct_dev->pci_dev, "%u bytes of design parameters\n", total_len);
	while (offset < total_len) {
		efct_readd(&reg, efct_dev->membase +
			  efct_reg(efct_dev, ER_HZ_FUN_WIN_REG_HOST_PARAMS_TLV + offset));
		data = EFCT_DWORD_FIELD(reg, EFCT_DWORD_0);
		for (i = 0; i < sizeof(data); i++) {
			rc = efct_tlv_feed(&reader, data);
			/* Got a complete value? */
			if (!rc && reader.state == EFCT_TLV_TYPE) {
				rc = efct_process_design_param(efct_dev, &reader);
				if (rc) {
					pci_err(efct_dev->pci_dev, "Processing Design Parameter for type %d failed\n",
						reader.type);
					goto out;
				}
				design_params_present_bitmask |= (1 << reader.type);
			}

			if (rc) {
				pci_err(efct_dev->pci_dev,
					"Unable to read design parameters, rc=%d", rc);
				goto out;
			}
			data >>= 8;
			offset++;
		}
	}
	if (!efct_dev->params.num_filter)
		efct_dev->params.num_filter = EFCT_DEFAULT_FILTER_TBL_ROWS;
	if ((design_params_present_bitmask & EFCT_REQRD_DESIGN_PARAMS) !=
			EFCT_REQRD_DESIGN_PARAMS) {
		pr_err("Design parameters are missing in the hardware\n");
		rc = -EINVAL;
	}
out:
	return rc;
}

static int efct_queues_size(struct efct_device *efct_dev)
{
	int sizeq;
	u32 nevq;

	nevq = min(efct_dev->params.num_evq,
		   (efct_dev->params.rx_queues + efct_dev->params.tx_apertures));
	sizeq = sizeof(struct efct_tx_queue) * efct_dev->params.tx_apertures;
	sizeq += sizeof(struct efct_rx_queue) * efct_dev->params.rx_queues;
	sizeq += sizeof(struct efct_ev_queue) * nevq;

	return sizeq;
}

static void efct_init_queue_addr(struct efct_device *efct_dev, struct efct_nic *efct)
{
	/*Increment efct variable by sizeof(struct efct_nic)*/
	efct->txq = (struct efct_tx_queue *)(efct + 1);
	/*Increment efct->txq variable by sizeof(struct efct_tx_queue) * no of Tx queues*/
	efct->rxq = (struct efct_rx_queue *)(efct->txq + efct_dev->params.tx_apertures);
	efct->evq = (struct efct_ev_queue *)(efct->rxq + efct_dev->params.rx_queues);
}

static int efct_port_alloc(struct efct_device *efct_dev, u8 port_idx)
{
	struct pci_dev *pdev = efct_dev->pci_dev;
	struct net_device *net_dev;
	int rc = 0, port_base = 0;
	struct efct_nic *efct;

	/* Allocate and initialise a struct net_device and struct efct_nic */
	net_dev = alloc_etherdev_mq(sizeof(*efct) + efct_queues_size(efct_dev),
				    EFCT_MAX_CORE_TX_QUEUES);
	if (!net_dev) {
		pci_err(pdev, "Unable to allocate net device\n");
		return -ENOMEM;
	}
	efct = netdev_priv(net_dev);
	efct_init_queue_addr(efct_dev, efct);
	efct_dev->efct[port_idx] = efct;
	efct->efct_dev = efct_dev;
	efct->port_idx = port_idx;
	efct->type = efct_dev->eth_type;
	efct->net_dev = net_dev;
	/*initializing window offsets*/
	port_base = (port_idx * EFCT_PORT_OFFSET);

	efct->port_base = port_base;

	/* Map uc and mc regions */
	rc = efct_map_membase(efct, efct_dev->reg_base, port_idx);
	if (rc) {
		pci_err(pdev, "efct_map_membase failed, rc=%d\n", rc);
		goto fail1;
	}

	efct_init_struct(efct);

	rc = efct->type->probe(efct);
	if (rc) {
		pci_err(pdev, "HW Probe failed, rc=%d\n", rc);
		goto fail2;
	}

	/* Helps biosdevname tool to differentiate ports attached to same PF */
	efct->net_dev->dev_port = efct->port_num;
	snprintf(efct->name, sizeof(efct->name), "%s#%d",
			pci_name(pdev), efct->port_num);
	mutex_init(&efct->state_lock);
	init_rwsem(&efct->hpl_mutation_lock);

	rc = efct_probe_netdev(efct);
	if (rc) {
		pci_err(pdev, "Unable to probe net device, rc=%d\n", rc);
		goto fail3;
	}
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_USE_DEVLINK)
	/*Devlink port register*/
	rc = efct_devlink_port_register(efct);
	if (rc) {
		pci_err(pdev, "Unable to register devlink port, rc=%d\n", rc);
		goto fail4;
	}
#endif
	rc = efct_register_netdev(efct);
	if (rc) {
		pci_err(pdev, "Unable to register net device, rc=%d\n", rc);
		goto fail5;
	}
#if defined(EFCT_USE_KCOMPAT) && (defined(EFCT_USE_DEVLINK) && !defined(EFCT_HAVE_SET_NETDEV_DEVLINK_PORT))
	devlink_port_type_eth_set(&efct->dl_port, efct->net_dev);
#endif
	/*Populating IRQ numbers in aux resources*/
	rc = efct_assign_msix(efct, port_idx);
	if (rc) {
		pci_err(pdev, "Unable to assign IRQ numbers (%d)\n", rc);
		goto fail6;
	}
#ifdef CONFIG_XILINX_AUX_EFCT
	rc = efct_auxbus_register(efct);
	if (rc) {
		pci_warn(pdev, "Unable to register aux bus driver (%d)\n", rc);
		goto fail7;
	}
#endif
	return rc;
#ifdef CONFIG_XILINX_AUX_EFCT
fail7:
	kfree(efct_dev->efct[port_idx]->irq_resources);
	efct_dev->efct[port_idx]->irq_resources = NULL;
#endif
fail6:
	efct_unregister_netdev(efct_dev->efct[port_idx]);
fail5:
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_USE_DEVLINK)
	devl_port_unregister(&efct->dl_port);
fail4:
#endif
	efct_remove_netdev(efct_dev->efct[port_idx]);
fail3:
	efct->type->remove(efct);
fail2:
	efct_unmap_membase(efct_dev->efct[port_idx]);
fail1:
	free_netdev(net_dev);
	efct_dev->efct[port_idx] = NULL;
	return rc;
}

int efct_unload(struct efct_device *efct_dev)
{
	int i;

	if (!efct_dev)
		return -EINVAL;

	efct_fini_stats_wq(efct_dev);
	for (i = 0; i < efct_dev->num_ports; i++)
		efct_port_free(efct_dev, i);

	efct_free_msix(efct_dev);
	return 0;
}

int efct_load(struct efct_device *efct_dev)
{
	int i, j, rc = 0;

	if (!efct_dev)
		return -EINVAL;

	rc = efct_alloc_msix(efct_dev);
	if (rc) {
		pci_err(efct_dev->pci_dev, "Unable to assign irq's, rc=%d\n", rc);
		return rc;
	}

	/* Initialize netdev */
	for (i = 0; i < efct_dev->num_ports ; i++) {
		rc = efct_port_alloc(efct_dev, i);
		if (rc) {
			pci_err(efct_dev->pci_dev, "port allocation failed, rc=%d\n", rc);
			goto err_port_alloc;
		}
	}

	rc = efct_init_stats_wq(efct_dev);
	if (rc) {
		pci_info(efct_dev->pci_dev, "Workqueue init failed\n");
		goto err_port_alloc;
	}

	/* Update state to allow reset scheduling*/
	for (i = 0; i < efct_dev->num_ports; i++) {
		mutex_lock(&efct_dev->efct[i]->state_lock);
		efct_dev->efct[i]->state = STATE_NET_DOWN;
		mutex_unlock(&efct_dev->efct[i]->state_lock);
	}

	return 0;

err_port_alloc:
	for (j = 0; j < i; j++)
		efct_port_free(efct_dev, j);

	efct_free_msix(efct_dev);
	return rc;
}

/* NIC initialisation
 *
 * This is called at module load (or hotplug insertion,
 * theoretically).  It sets up PCI mappings, resets the NIC,
 * sets up and registers the network devices with the kernel and hooks
 * the interrupt service routine.  It does not prepare the device for
 * transmission; this is left to the first time one of the network
 * interfaces is brought up (i.e. efct_net_open).
 */
static int efct_pci_probe
			(struct pci_dev *pci_dev,
			 const struct pci_device_id *entry)
{
	struct efct_func_ctl_window fcw = { 0 };
	struct efct_device *efct_dev;
	int rc = 0;

	efct_dev = kzalloc(sizeof(*efct_dev), GFP_KERNEL);
	if (!efct_dev)
		return -ENOMEM;

	efct_dev->xentries = NULL;
	efct_dev->pci_dev = pci_dev;
	efct_dev->eth_type = (const struct efct_nic_type *)entry->driver_data;
	pci_set_drvdata(pci_dev, efct_dev);
	efct_dev->max_dma_mask = DMA_BIT_MASK(ESF_GZ_TX_SEND_ADDR_WIDTH);

	/*Finding function control window*/
	rc = efct_pci_find_func_ctrl_window(efct_dev, &fcw);
	if (rc) {
		pci_err(pci_dev,
			"Error looking for efct function control window, rc=%d\n", rc);
		goto fail;
	}

	if (!fcw.valid) {
		/* Extended capability not found - use defaults. */
		fcw.bar = EFCT_PCI_DEFAULT_BAR;
		fcw.offset = 0;
	}

	/* Set up basic I/O (BAR mappings etc) */
	rc = efct_init_io(efct_dev, fcw.bar, efct_dev->max_dma_mask,
			  pci_resource_len(efct_dev->pci_dev, fcw.bar));
	if (rc) {
		pci_err(pci_dev,
			"Basic I/O initialization of efct_dev failed, rc=%d\n", rc);
		goto fail;
	}

	/* Set default layout to distributed */
	efct_dev->dist_layout = RX_LAYOUT_DISTRIBUTED;
	efct_dev->separated_rx_cpu = DEFAULT_SEPARATED_RX_CPU;

	efct_dev->reg_base = fcw.offset;
	efct_dev->num_ports = efct_get_num_ports(efct_dev);
	if (fcw.offset > pci_resource_len(efct_dev->pci_dev, fcw.bar)
		- (((efct_dev->num_ports - 1) * EFCT_PORT_OFFSET) + EFCT_PORT_LEN)) {
		pci_err(pci_dev, "Func control window overruns BAR\n");
		goto fail1;
	}

	rc = efct_check_func_ctl_magic(efct_dev);
	if (rc) {
		pci_err(pci_dev, "Func control window magic is wrong, rc=%d\n", rc);
		goto fail1;
	}

	/* Read design parameters */
	rc = efct_check_design_params(efct_dev);
	if (rc) {
		pci_err(efct_dev->pci_dev, "Unsupported design parameters, rc=%d\n", rc);
		goto fail1;
	}
#ifdef CONFIG_XILINX_DEBUGFS
	/*Create debugfs directory for the PCI func */
	efct_init_debugfs_func(efct_dev);
#endif

#ifdef CONFIG_XILINX_MCDI_LOGGING
	efct_init_mcdi_logging(efct_dev);
#endif
	/*Initialize the BIU lock*/
	spin_lock_init(&efct_dev->biu_lock);

	rc = efct_probe_devlink(efct_dev);
	if (rc)
		goto fail2;

	efct_nic_check_pcie_link(efct_dev, EFCT_BW_PCIE_GEN3_X8, NULL, NULL);
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_DEVL_API)
	devl_lock(efct_dev->devlink);
#endif
	rc  = efct_load(efct_dev);
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_DEVL_API)
	devl_unlock(efct_dev->devlink);
#endif
	if (rc)
		goto fail3;
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_USE_DEVLINK)
	rc = efct_devlink_register(efct_dev);
	if (rc)
		goto fail4;
#endif
	pr_info("Xilinx X3 NIC detected: device %04x:%04x subsys %04x:%04x\n",
		pci_dev->vendor, pci_dev->device,
		pci_dev->subsystem_vendor,
		pci_dev->subsystem_device);
	return 0;
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_USE_DEVLINK)
fail4:
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_DEVL_API)
		devl_lock(efct_dev->devlink);
#endif
		efct_unload(efct_dev);
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_DEVL_API)
		devl_unlock(efct_dev->devlink);
#endif
#endif
fail3:
	efct_fini_devlink(efct_dev);
fail2:
#ifdef CONFIG_XILINX_MCDI_LOGGING
	efct_fini_mcdi_logging(efct_dev);
#endif
#ifdef CONFIG_XILINX_DEBUGFS
	efct_fini_debugfs_func(efct_dev);
#endif
fail1:
	efct_fini_io(efct_dev);
fail:
	kfree(efct_dev);
	pci_set_drvdata(pci_dev, NULL);
	pr_err("Driver Probe failed. rc=%d\n", rc);
	return rc;
}

/* Final NIC shutdown
 * This is called only at module unload (or hotplug removal).
 */
static void efct_pci_remove(struct pci_dev *pci_dev)
{
	struct efct_device *efct_dev;
	int i;

	efct_dev = pci_get_drvdata(pci_dev);
	if (!efct_dev)
		return;

	for (i = 0; i < efct_dev->num_ports; i++) {
		mutex_lock(&efct_dev->efct[i]->state_lock);
		efct_dev->efct[i]->state = STATE_UNINIT;
		mutex_unlock(&efct_dev->efct[i]->state_lock);
	}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_USE_DEVLINK)
	efct_devlink_unregister(efct_dev);
#endif
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_DEVL_API)
	devl_lock(efct_dev->devlink);
#endif
	efct_unload(efct_dev);
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_DEVL_API)
	devl_unlock(efct_dev->devlink);
#endif
	efct_fini_devlink(efct_dev);
#ifdef CONFIG_XILINX_MCDI_LOGGING
	efct_fini_mcdi_logging(efct_dev);
#endif
#ifdef CONFIG_XILINX_DEBUGFS
	efct_fini_debugfs_func(efct_dev);
#endif
	efct_fini_io(efct_dev);

	pci_set_drvdata(pci_dev, NULL);
	kfree(efct_dev);
	pr_info("Xilinx X3 NIC removed successfully\n");
}

#ifdef EFCT_NOT_UPSTREAM
#if defined(EFCT_HAVE_PCI_FLR_RESET_PREPARE) || defined(EFCT_HAVE_PCI_FLR_RESET_NOTIFY)
static void efct_pci_reset_prepare(struct pci_dev *pdev)
{
	/*Nothing to be done here*/
}

static void efct_pci_reset_done(struct pci_dev *pdev)
{
	struct efct_device *efct_dev = pci_get_drvdata(pdev);
	struct efct_nic *efct;
	int i;

	if (!efct_dev) {
		pr_err("%s: efct_dev is NULL\n", __func__);
		return;
	}

	for (i = 0; i < efct_dev->num_ports; i++) {
		efct = efct_dev->efct[i];
		efct_schedule_reset(efct, RESET_TYPE_MC_FAILURE);
	}
}
#endif

#if defined(EFCT_HAVE_PCI_FLR_RESET_NOTIFY)
static void efct_pci_reset_notify(struct pci_dev *pdev, bool prepare)
{
	if (prepare)
		efct_pci_reset_prepare(pdev);
	else
		efct_pci_reset_done(pdev);
}
#endif

/* PCIe Error handlers */
static const struct pci_error_handlers efct_pci_err_handler = {
#if defined(EFCT_HAVE_PCI_FLR_RESET_PREPARE)
	.reset_prepare = efct_pci_reset_prepare,
	.reset_done = efct_pci_reset_done,
#elif defined(EFCT_HAVE_PCI_FLR_RESET_NOTIFY)
	.reset_notify = efct_pci_reset_notify,
#endif
};
#endif

static void efct_pci_shutdown(struct pci_dev *pci_dev)
{
	struct efct_device *efct_dev = pci_get_drvdata(pci_dev);
	int i;

	if (!efct_dev)
		return;

	for (i = 0; i < efct_dev->num_ports; i++)
		efct_shutdown(efct_dev->efct[i]);
	efct_free_msix(efct_dev);
	pci_disable_device(pci_dev);
}

static struct pci_driver efct_pci_driver = {
	.name		= KBUILD_MODNAME,
	.id_table	= efct_pci_table,
	.probe		= efct_pci_probe,
	.remove		= efct_pci_remove,
	.shutdown	= efct_pci_shutdown,
#ifdef EFCT_NOT_UPSTREAM
	.err_handler    = &efct_pci_err_handler
#endif
};

static int __init efct_init_module(void)
{
	int rc;

	pr_info("Xilinx X3 Net driver v" EFCT_DRIVER_VERSION "\n");

#ifdef CONFIG_XILINX_DEBUGFS
	efct_init_debugfs(KBUILD_MODNAME);
#endif
	rc = efct_create_reset_workqueue();
	if (rc)
		goto err1;

	rc = pci_register_driver(&efct_pci_driver);
	if (rc < 0) {
		pr_err("xilixn_efct driver register fail, rc=%d\n", rc);
		goto err2;
	}
	return 0;
err2:
	efct_destroy_reset_workqueue();
err1:
#ifdef CONFIG_XILINX_DEBUGFS
	efct_fini_debugfs();
#endif
	return rc;
}

static void __exit efct_exit_module(void)
{
	pr_info("Xilinx X3 Net driver unloading\n");
	pci_unregister_driver(&efct_pci_driver);
	efct_destroy_reset_workqueue();
#ifdef CONFIG_XILINX_DEBUGFS
	efct_fini_debugfs();
#endif
#ifdef CONFIG_EFCT_AUX
	rcu_barrier();
#endif
}

module_init(efct_init_module);
module_exit(efct_exit_module);

MODULE_AUTHOR("Xilinx Inc.");
MODULE_DESCRIPTION("Xilinx X3 Network driver");
MODULE_LICENSE("GPL");
MODULE_DEVICE_TABLE(pci, efct_pci_table);
MODULE_VERSION(EFCT_DRIVER_VERSION);
