// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include <linux/auxiliary_bus.h>
#include <linux/nsproxy.h>
#include <linux/idr.h>
#include "xlnx_efct.h"
#include "efct_driver.h"
#include "mcdi.h"
#include "mcdi_functions.h"
#include "efct_auxbus.h"
#include "efct_evq.h"
#ifdef CONFIG_XILINX_DEBUGFS
#include "debugfs.h"
#endif
#include "efct_reg.h"
#include "efct_rx.h"
#include "efct_ptp.h"
#include "efct_common.h"
#ifdef CONFIG_XILINX_EFCT_TRACING
#include <trace/events/xilinx_efct.h>
#endif

/* For allocating unique device IDs */
static DEFINE_IDA(aux_ida);

static int aux_get_free_client_index(struct efct_nic *efct)
{
	unsigned long mask =  (1UL << MAX_AUX_CLIENTS) - 1;
	int index = 0;

	if (!efct)
		return -EINVAL;

	/*No free index present, all aux clients are in use*/
	if ((efct->aux_client_map[0] & mask) == mask)
		return -EINVAL;

	index = ffz(efct->aux_client_map[0]);
	if (index < MAX_AUX_CLIENTS)
		return index;
	else
		return -1;
}

static struct efct_nic *client_to_efct(struct xlnx_efct_client *client)
{
	if (client)
		return client->efct;
	pr_err("Aux client passed by the application is NULL\n");
	return NULL;
}

static struct device *aux_client_to_dev(struct xlnx_efct_client *client)
{
	if (client && client->efct)
		return &client->efct->aux_dev.adev.dev;
	pr_err("Aux client passed by the application is NULL\n");
	return NULL;
}

static struct xlnx_efct_client *efct_aux_open(struct auxiliary_device *adev,
					      const struct xlnx_efct_drvops *ops,
					      void *driver_data)
{
	struct efct_nic *efct;
	int client_index;
	int i;

	if (!adev || !ops || !driver_data)
		return ERR_PTR(-EINVAL);

	efct = container_of(adev, struct efct_nic, aux_dev.adev);
	mutex_lock(&efct->client_list_lock);
	client_index = aux_get_free_client_index(efct);
	if (client_index < 0) {
		netif_err(efct, probe, efct->net_dev,
			  "Failed to find the free Client index\n");
		mutex_unlock(&efct->client_list_lock);
		return ERR_PTR(-ENOMEM);
	}

	efct->aux_clients[client_index].efct = efct;
	efct->aux_clients[client_index].drvops = ops;
	efct->aux_clients[client_index].driver_priv = driver_data;
	efct->aux_clients[client_index].index = client_index;
	atomic_set(&efct->aux_clients[client_index].active_hugepages, 0);
	efct->aux_clients[client_index].state = STATE_OPEN;
	bitmap_zero(efct->aux_clients[client_index].active_evq, EFCT_MAX_EV_QUEUES);
	bitmap_zero(efct->aux_clients[client_index].active_txq, EFCT_MAX_TX_QUEUES);
	bitmap_zero(efct->aux_clients[client_index].filter_map, EFCT_MAX_FILTER_TBL_ROWS);
	for (i = 0; i < efct->rxq_count; i++) {
		atomic_set(&efct->aux_clients[client_index].rxq_info[i].active_hp, 0);
		efct->aux_clients[client_index].rxq_info[i].added_hp = 0;
		efct->aux_clients[client_index].rxq_info[i].removed_hp = 0;
		efct->aux_clients[client_index].rxq_info[i].ref_cnt = 0;
	}

#ifdef CONFIG_XILINX_DEBUGFS
	efct_init_debugfs_aux_dev_client(&efct->aux_clients[client_index]);
#endif
	/* NB: setting aux_client_map must be last, since it's read locklessly: */
	set_bit(client_index, &efct->aux_client_map[0]);

	mutex_unlock(&efct->client_list_lock);
	return &efct->aux_clients[client_index];
}

static void efct_fill_design_param(struct efct_nic *efct,
				   struct xlnx_efct_design_params *params)
{
	struct efct_device *efct_dev;

	efct_dev = efct->efct_dev;
	params->evq_sizes = efct_dev->params.evq_sizes;
	params->frame_offset_fixed = efct_dev->params.frame_offset_fixed;
	params->l4_csum_proto = efct_dev->params.l4_csum_proto;
	params->max_runt = efct_dev->params.max_runt;
	params->num_filter = efct_dev->params.num_filter;
	params->rx_buffer_len = efct_dev->params.rx_buffer_len;
	params->rx_buf_fifo_size = efct_dev->params.rx_buf_fifo_size;
	params->rx_metadata_len = efct_dev->params.rx_metadata_len;
	params->rx_queues = efct_dev->params.rx_queues;
	params->rx_stride = efct_dev->params.rx_stride;
	params->ts_subnano_bit = efct_dev->params.ts_subnano_bit;
	params->tx_apertures = efct_dev->params.tx_apertures;
	params->tx_aperture_size = efct_dev->params.tx_aperture_size;
	params->tx_fifo_size = efct_dev->params.tx_fifo_size;
	params->tx_max_reorder = efct_dev->params.tx_max_reorder;
	params->unsol_credit_seq_mask = efct_dev->params.unsol_credit_seq_mask;
}

static int efct_aux_get_param(struct xlnx_efct_client *client,
			      enum xlnx_efct_param param,
			      union xlnx_efct_param_value *data)
{
	struct efct_nic *efct;
	int rc = 0;

	if (!client || !data)
		return -EINVAL;

	efct = client_to_efct(client);

	if (!efct)
		return -ENODEV;

	switch (param) {
	case XLNX_EFCT_CONFIG_MEM:
		data->config_mem = efct->membase;
		break;
	case XLNX_EFCT_NETDEV:
		data->net_dev = efct->net_dev;
		break;
	case XLNX_EFCT_NETNS:
		data->net_ns = current->nsproxy->net_ns;
		break;
	case XLNX_EFCT_VARIANT:
		data->variant = 'A';//TODO
		break;
	case XLNX_EFCT_REVISION:
		data->value = 0;//TODO
		break;
	case XLNX_EFCT_NIC_RESOURCES:
		data->nic_res.evq_min = efct->aux_res.evq_base;
		data->nic_res.evq_lim = efct->aux_res.evq_lim;
		data->nic_res.txq_min = efct->aux_res.txq_base;
		data->nic_res.txq_lim = efct->aux_res.txq_lim;
		break;
	case XLNX_EFCT_IRQ_RESOURCES:
		data->irq_res = efct->irq_resources;
		break;
	case XLNX_EFCT_POISON_CONFIG:
		//TBD may be only set_param
		break;
	case XLNX_EFCT_EVQ_WINDOW:
		data->evq_window.stride =
			ER_HZ_PORT0_REG_HOST_EVQ_UNSOL_CREDIT_GRANT_STEP;
		data->evq_window.base = efct->efct_dev->membase_phys +
			efct->efct_dev->reg_base + efct->port_base +
			ER_HZ_PORT0_REG_HOST_EVQ_UNSOL_CREDIT_GRANT +
			(efct->aux_res.evq_base * data->evq_window.stride);
		break;
	case XLNX_EFCT_DESIGN_PARAM:
		efct_fill_design_param(efct, &data->design_params);
		break;
	default:
		dev_info(aux_client_to_dev(client),
			 "Unsupported parameter %u\n", param);
		rc = -EOPNOTSUPP;
	}
	return rc;
}

static int efct_aux_set_param(struct xlnx_efct_client *client,
			      enum xlnx_efct_param param,
			      union xlnx_efct_param_value *data)
{
	struct efct_nic *efct = client_to_efct(client);
	int rc = 0;

	if (!efct) {
		pr_err("Invalid client passed\n");
		return -ENODEV;
	}
	if (!data || data->poison.qid < 0 || data->poison.qid >= efct->rxq_count) {
		netif_err(efct, probe, efct->net_dev, "Invalid input parameters\n");
		return -EINVAL;
	}

	switch (param) {
	case XLNX_EFCT_POISON_CONFIG:
		if (client->rxq_info[data->poison.qid].ref_cnt <= 0) {
			netif_dbg(efct, drv, efct->net_dev,
				  "%s:Client is not bound to the queue %d\n",
				  __func__, data->poison.qid);
			return -EACCES;
		}
		efct->rxq[data->poison.qid].poison_cfg.length = data->poison.length;
		efct->rxq[data->poison.qid].poison_cfg.value = data->poison.value;
		break;
	default:
		dev_info(aux_client_to_dev(client),
			 "Unknown parameter %u\n", param);
		rc = -EOPNOTSUPP;
	}
	return rc;
}

static int efct_aux_fw_rpc(struct xlnx_efct_client *client,
			   struct xlnx_efct_rpc *rpc)
{
	struct efct_nic *efct = client_to_efct(client);
	int rc;

	if (!efct || !rpc)
		return -ENODEV;

	rc = efct_mcdi_rpc_quiet(efct, rpc->cmd,
				 (const union efct_dword *)rpc->inbuf, rpc->inlen,
				 (union efct_dword *)rpc->outbuf, rpc->outlen,
				 rpc->outlen_actual);
	return rc;
}

static int efct_find_msix_index(struct efct_nic *efct,
				struct efct_ev_queue *eventq, int irq)
{
	int base = efct->port_idx * efct->efct_dev->vec_per_port;
	int end_idx = base + efct->efct_dev->vec_per_port;
	int st_idx = base + efct->rxq_count + 1;
	int i;

	for (i = st_idx; i < end_idx; i++) {
		if (efct->efct_dev->xentries[i].vector == irq) {
			eventq->msi.idx = efct->efct_dev->xentries[i].entry - base;
			return 0;
		}
	}
	return -EINVAL;
}

static int efct_aux_init_evq(struct xlnx_efct_client *client,
			     struct xlnx_efct_evq_params *params)
{
	struct efct_nic *efct = client_to_efct(client);
	struct efct_ev_queue *eventq;
	int rc;

	if (!efct) {
		pr_err("Invalid client specified\n");
		return -ENODEV;
	}

	if (!params || !params->entries || !params->q_page) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid input params received\n");
		return -EINVAL;
	}

	if (params->qid < efct->aux_res.evq_base || params->qid >= efct->aux_res.evq_lim) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid eventq id passed. Shall be in range: %d - %d\n",
			  efct->aux_res.evq_base, efct->aux_res.evq_lim);
		return -EINVAL;
	}

	rc = validate_evq_size(efct, params->entries);
	if (rc < 1) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid event queue size: %d\n",
			  params->entries);
		return -EINVAL;
	}

	mutex_lock(&efct->state_lock);

	if (test_bit(params->qid, &efct->evq_active_mask)) {
		netif_err(efct, probe, efct->net_dev,
			  "Event queue index %d is already in use\n", params->qid);
		rc = -EINVAL;
		goto exit;
	}

	eventq = &efct->evq[params->qid];
	eventq->evq_base =
			efct->membase + ER_HZ_PORT0_REG_HOST_EVQ_UNSOL_CREDIT_GRANT
			+ (eventq->index * efct->efct_dev->params.evq_stride);
	eventq->entries = params->entries;
	eventq->buf.len = params->q_size;
	eventq->buf.dma_addr = dma_map_page(&efct->efct_dev->pci_dev->dev, params->q_page,
					    params->page_offset, params->q_size,
					    DMA_BIDIRECTIONAL);
	if (unlikely(dma_mapping_error(&efct->efct_dev->pci_dev->dev, eventq->buf.dma_addr))) {
		dev_err(&efct->efct_dev->pci_dev->dev, "page 0x%p mapping error 0x%llx.\n",
			params->q_page, (unsigned long long)eventq->buf.dma_addr);
		rc = -EINVAL;
		goto exit;
	}

	eventq->buf.addr = page_address(params->q_page) + params->page_offset;
	eventq->type = EVQ_T_AUX;
	eventq->subscribe_time_sync = params->subscribe_time_sync;
	eventq->unsol_credit = params->unsol_credit;

	if (params->irq) {
		rc = efct_find_msix_index(efct, eventq, params->irq);
		if (rc < 0) {
			dev_err(&efct->efct_dev->pci_dev->dev,
					"Failed to map msix idx from irq %d\n",
					params->irq);
			rc = -EINVAL;
			goto exit;
		}
	} else {
		eventq->msi.idx = eventq->index;
	}
	rc = efct->type->ev_init(eventq);
	if (rc) {
		if (eventq->buf.dma_addr) {
			dma_unmap_page(&efct->efct_dev->pci_dev->dev, eventq->buf.dma_addr,
				       eventq->buf.len, DMA_BIDIRECTIONAL);
			eventq->buf.dma_addr = 0UL;
		}
		netif_err(efct, probe, efct->net_dev,
			  "Event queue initialization failed\n");
		goto exit;
	}
#ifdef CONFIG_XILINX_PTP
	/*subscribe for time sync events*/
	if (eventq->subscribe_time_sync) {
		rc = efct_ptp_subscribe_timesync(eventq);
		if (rc) {
			efct->type->ev_fini(eventq);
			efct->type->ev_purge(eventq);
			if (eventq->buf.dma_addr) {
				dma_unmap_page(&efct->efct_dev->pci_dev->dev, eventq->buf.dma_addr,
					       eventq->buf.len, DMA_BIDIRECTIONAL);
				eventq->buf.dma_addr = 0UL;
			}
			netif_err(efct, probe, efct->net_dev,
				  "Time sync event subscription failed for queue %d\n",
				  eventq->index);
			goto exit;
		}
	}
#endif
	set_bit(params->qid, &efct->evq_active_mask);
	set_bit(params->qid, client->active_evq);

exit:
	mutex_unlock(&efct->state_lock);

	return rc;
}

static void efct_aux_free_evq(struct xlnx_efct_client *client, int evq)
{
	struct efct_nic *efct = client_to_efct(client);
	struct efct_ev_queue *eventq;
#ifdef CONFIG_XILINX_PTP
	int rc;
#endif

	if (!efct || evq < 0)
		return;

	mutex_lock(&efct->state_lock);

	eventq = &efct->evq[evq];

	if (!test_bit(eventq->index, client->active_evq)) {
		netif_err(efct, probe, efct->net_dev,
			  "Event queue index %d is not added by this client\n", evq);
		goto exit;
	}

	if (!test_bit(evq, &efct->evq_active_mask)) {
		netif_err(efct, probe, efct->net_dev,
			  "Event queue index %d is not setup\n", evq);
		goto exit;
	}
#ifdef CONFIG_XILINX_PTP
	/*Unsubscribe for time sync events*/
	if (eventq->subscribe_time_sync) {
		rc = efct_ptp_unsubscribe_timesync(eventq);
		if (rc) {
			netif_err(efct, probe, efct->net_dev,
				  "Time sync event unsubscription failed for queue %d\n",
				  eventq->index);
		}
	}
#endif
	efct->type->ev_fini(eventq);
	efct->type->ev_purge(eventq);

	if (eventq->buf.dma_addr) {
		dma_unmap_page(&efct->efct_dev->pci_dev->dev, eventq->buf.dma_addr, eventq->buf.len,
			       DMA_BIDIRECTIONAL);
		eventq->buf.dma_addr = 0UL;
	}

	eventq->buf.addr = NULL;

	clear_bit(eventq->index, &efct->evq_active_mask);
	clear_bit(eventq->index, client->active_evq);

exit:
	mutex_unlock(&efct->state_lock);
}

static int efct_aux_bind_rxq(struct xlnx_efct_client *client,
			     struct xlnx_efct_rxq_params *params)
{
	struct efct_nic *efct = client_to_efct(client);
	struct xlnx_efct_client *handle;
	struct efct_rx_queue *rxq;
	int qid, i, rc;

	if (!efct) {
		pr_err("Invalid client passed\n");
		return -ENODEV;
	}
	if (!params) {
		netif_err(efct, probe, efct->net_dev, "Invalid input params received\n");
		return -EINVAL;
	}

	qid = params->qid;
	if (qid < 0 || qid >= efct->rxq_count) {
		netif_err(efct, probe, efct->net_dev, "Invalid queue id specified\n");
		return -EINVAL;
	}

	mutex_lock(&efct->state_lock);

	/*Set the client bit in per queue client list*/
	rxq = &efct->rxq[qid];
	if (rxq->exclusive_client >= 0 && rxq->exclusive_client != client->index) {
		netif_err(efct, probe, efct->net_dev,
			  "Trying to bind to an exclusive queue (%d)\n", qid);
		mutex_unlock(&efct->state_lock);
		return -EPERM;
	}
	efct_disable_napi(&efct->evq[rxq->evq_index]);
	client->rxq_info[qid].ref_cnt++;
	if (test_and_set_bit(client->index, &efct->rxq[qid].active_aux_clients[0])) {
		netif_dbg(efct, probe, efct->net_dev, "Recurring addition of RX queue index %d, ref count :%d\n",
			  qid, client->rxq_info[qid].ref_cnt);
	}

	if (params->n_hugepages) {
		int n_added_hp = params->n_hugepages;

		if (client->rxq_info[qid].ref_cnt == 1)
			n_added_hp += ((RX_MIN_DRIVER_BUFFS * rxq->buffer_size) / HUGE_PAGE_SZ);

		/*Add huge pages to the huge page list*/
		down_write(&efct->hpl_mutation_lock);
		rc = hpl_add_hp_for_client(rxq, client, &n_added_hp);
		up_write(&efct->hpl_mutation_lock);
		if (rc) {
			int n_removed_hp;

			netif_err(efct, probe, efct->net_dev,
				  "Failed to add hugepage to the list, Queue: %d\n", qid);
			client->rxq_info[qid].ref_cnt--;
			if (client->rxq_info[qid].ref_cnt == 0)
				clear_bit(client->index, &efct->rxq[qid].active_aux_clients[0]);

			n_removed_hp = hpl_remove_hp_for_client(rxq, client, n_added_hp);
			if (n_removed_hp != n_added_hp) {
				netif_err(efct, probe, efct->net_dev,
					  "hpl_remove_hp_for_client failed, Queue: %d\n", qid);
			}
			efct_enable_napi(&efct->evq[rxq->evq_index]);
			mutex_unlock(&efct->state_lock);
			return rc;
		}

		client->rxq_info[qid].added_hp += n_added_hp;
		/*Notify all clients of this queue about new huge page list*/
		for_each_set_bit(i, efct->rxq[qid].active_aux_clients,
				 sizeof(efct->rxq[qid].active_aux_clients) * BITS_PER_BYTE) {
			handle = &efct->aux_clients[i];
			handle->drvops->hugepage_list_changed(handle->driver_priv, qid);
		}
	}
	efct_enable_napi(&efct->evq[rxq->evq_index]);
	mutex_unlock(&efct->state_lock);

	return qid;
}

static void efct_aux_free_rxq_locked(struct xlnx_efct_client *client, int rxq_id,
				     size_t n_hugepages, bool *is_ro)
{
	struct efct_nic *efct = client_to_efct(client);
	struct efct_rx_queue *rxq;
	int rc = n_hugepages;

	rxq = &efct->rxq[rxq_id];

	if (n_hugepages) {
		if (client->rxq_info[rxq_id].added_hp <
				(client->rxq_info[rxq_id].removed_hp + n_hugepages))
			n_hugepages =
			client->rxq_info[rxq_id].added_hp - client->rxq_info[rxq_id].removed_hp;
		rc = hpl_remove_hp_for_client(rxq, client, n_hugepages);
		if (rc < n_hugepages) {
			netif_dbg(efct, probe, efct->net_dev,
				  "%zu huge pages marked for lazy free, Queue: %d\n",
				  (n_hugepages - rc), rxq_id);
		}
	}

	client->rxq_info[rxq_id].removed_hp += n_hugepages;

	if (rc < n_hugepages && client->rxq_info[rxq_id].ref_cnt == 1)
		*is_ro = efct_rx_rollover(rxq) == 0;
	else
		*is_ro = false;
}

static void efct_aux_free_rxq(struct xlnx_efct_client *client, int rxq_id, size_t n_hugepages)
{
	struct efct_nic *efct = client_to_efct(client);
	struct efct_rx_queue *rxq;
	bool is_ro;

	if (!efct) {
		pr_err("Invalid input params received\n");
		return;
	}

	if (rxq_id < 0 || rxq_id >= efct->rxq_count) {
		netif_err(efct, probe, efct->net_dev, "Invalid queue id specified\n");
		return;
	}

	mutex_lock(&efct->state_lock);

	if (client->rxq_info[rxq_id].ref_cnt <= 0) {
		mutex_unlock(&efct->state_lock);
		netif_dbg(efct, drv, efct->net_dev,
			  "%s:Client is not bound to the queue %d\n", __func__, rxq_id);
		return;
	}

	rxq = &efct->rxq[rxq_id];

	if (client->rxq_info[rxq_id].ref_cnt == 1)
		n_hugepages =
			client->rxq_info[rxq_id].added_hp - client->rxq_info[rxq_id].removed_hp;

	efct_disable_napi(&efct->evq[rxq->evq_index]);

	efct_aux_free_rxq_locked(client, rxq_id, n_hugepages, &is_ro);

	if (!rxq->enable) {
		/* aux client might have some cleanup to do, but if the rxq is down
		 * then it's never going to get a chance. Give it a poll while we're
		 * (pseudo-) in NAPI context
		 */
		efct_aux_poll(efct, rxq_id, INT_MAX);
	}

	efct_enable_napi(&efct->evq[rxq->evq_index]);

	if (is_ro) {
		wait_event_timeout(efct->hp_cleanup_wq,
				   atomic_read(&client->rxq_info[rxq_id].active_hp) == 0,
				   msecs_to_jiffies(EFCT_AUX_RO_CLEANUP_TIME_MS));

		n_hugepages = atomic_read(&client->rxq_info[rxq_id].active_hp);
		if (n_hugepages)
			netif_err(efct, probe, efct->net_dev,
				  "Failed to free %zu HugePages\n", n_hugepages);
	}

	client->rxq_info[rxq_id].ref_cnt--;
	if (client->rxq_info[rxq_id].ref_cnt == 0)
		test_and_clear_bit(client->index, &efct->rxq[rxq_id].active_aux_clients[0]);

	mutex_unlock(&efct->state_lock);
}

static int efct_aux_init_txq(struct xlnx_efct_client *client,
			     struct xlnx_efct_txq_params *params)
{
	struct efct_nic *efct = client_to_efct(client);
	int txq_id, rc;

	if (!efct) {
		pr_err("Invalid client passed\n");
		return -ENODEV;
	}
	if (!params) {
		netif_err(efct, probe, efct->net_dev, "Invalid input params received\n");
		return -EINVAL;
	}

	if (test_bit(params->evq, &efct->evq_active_mask) == 0) {
		netif_err(efct, probe, efct->net_dev,
			  "Event queue index %d is not setup\n", params->evq);
		return -EINVAL;
	}
	if (params->qid < efct->aux_res.txq_base || params->qid >= efct->aux_res.txq_lim) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid Txq id %d passed. Shall be in range: %d - %d\n",
			  params->qid, efct->aux_res.txq_base, efct->aux_res.txq_lim);
		return -EINVAL;
	}
	if (test_bit(params->qid, &efct->txq_active_mask)) {
		netif_err(efct, probe, efct->net_dev,
			  "Tx queue index %d already in use\n", params->qid);
		return -EINVAL;
	}

	txq_id = efct->type->tx_probe(efct, params->evq, params->qid);
	if (txq_id != params->qid) {
		netif_err(efct, probe, efct->net_dev,
			  "Failed to init requested Tx queue %d\n", params->qid);
		return -EINVAL;
	}

	mutex_lock(&efct->state_lock);

	efct->txq[txq_id].label = params->label >= 0 ? params->label : txq_id;
	rc = efct->type->tx_init(&efct->txq[txq_id]);
	if (rc) {
		efct->type->tx_remove(&efct->txq[txq_id]);
		netif_err(efct, probe, efct->net_dev,
			  "Tx initialization failed\n");
		mutex_unlock(&efct->state_lock);
		return rc;
	}

	set_bit(txq_id, &client->active_txq[0]);
	mutex_unlock(&efct->state_lock);

	return txq_id;
}

static void efct_aux_free_txq(struct xlnx_efct_client *client, int txq)
{
	struct efct_nic *efct = client_to_efct(client);
	struct efct_tx_queue *tx_queue;

	if (!efct) {
		pr_err("Invalid input params received\n");
		return;
	}
	if (txq < efct->aux_res.txq_base || txq >= efct->aux_res.txq_lim) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid txq 0x%x passed by the application\n", txq);
		return;
	}

	mutex_lock(&efct->state_lock);

	if (test_bit(txq, client->active_txq) == 0) {
		netif_err(efct, probe, efct->net_dev,
			  "Tx queue index %d is not added by this client\n", txq);
		goto exit;
	}

	if (test_bit(txq, &efct->txq_active_mask) == 0) {
		netif_err(efct, probe, efct->net_dev,
			  "TX queue index %d is not setup\n", txq);
		goto exit;
	}
	tx_queue = &efct->txq[txq];
	efct->type->tx_fini(tx_queue);
	efct->type->tx_purge(tx_queue);
	efct->type->tx_remove(tx_queue);
	clear_bit(txq, client->active_txq);

exit:
	mutex_unlock(&efct->state_lock);
}

static int efct_aux_ctpio_addr(struct xlnx_efct_client *client, int txq,
			       resource_size_t *addr, size_t *size)
{
	struct efct_nic *efct = client_to_efct(client);

	if (!efct) {
		pr_err("Invalid client passed\n");
		return -ENODEV;
	}
	if (!addr || !size) {
		netif_err(efct, probe, efct->net_dev, "Invalid input params received\n");
		return -EINVAL;
	}

	if (txq < efct->aux_res.txq_base || txq >= efct->aux_res.txq_lim) {
		netif_err(efct, probe, efct->net_dev, "Invalid queue id %d specified\n", txq);
		return -EINVAL;
	}

	if (test_bit(txq, &client->active_txq[0]) == 0) {
		netif_err(efct, probe, efct->net_dev,
			  "Tx queue index %d is not added by this client\n", txq);
		return -EINVAL;
	}
	*addr = efct->efct_dev->membase_phys + efct->efct_dev->reg_base + efct->port_base +
		ER_HZ_PORT0_REG_HOST_CTPIO_REGION0 + (txq * efct->efct_dev->params.ctpio_stride);
	*size = efct->efct_dev->params.tx_aperture_size;

	return 0;
}

static int efct_aux_get_hugepages(struct xlnx_efct_client *client, int rxq_id,
				  struct xlnx_efct_hugepage *pages, size_t n_pages)
{
	struct efct_nic *efct = client_to_efct(client);
	struct efct_rx_queue *rxq;
	size_t i;
	int rc;

	if (!efct) {
		pr_err("Invalid client passed\n");
		return -ENODEV;
	}
	if (!pages) {
		netif_err(efct, probe, efct->net_dev, "Invalid parameters for Rx queue %d\n", rxq_id);
		return -EINVAL;
	}

	if (rxq_id < 0 || rxq_id >= efct->rxq_count) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid queue id specified %d\n", rxq_id);
		return -EINVAL;
	}
	if (client->rxq_info[rxq_id].ref_cnt <= 0) {
		netif_dbg(efct, drv, efct->net_dev,
			  "%s:Client is not bound to the queue %d\n", __func__, rxq_id);
		return -EACCES;
	}

	rxq = &efct->rxq[rxq_id];
	down_read(&efct->hpl_mutation_lock);
	rcu_read_lock();
	rc = hpl_get_hugepages(rxq, pages, n_pages);
	if (rc) {
		rcu_read_unlock();
		netif_err(efct, probe, efct->net_dev, "Number of entries in hugepage array not sufficient, expected > %d, received %ld\n",
			  rxq->hpl->max_hp_id, n_pages);
	} else {
		// Incref each file and page while NAPI is stopped so that concurrent
		// free can't happen
		for (i = 0; i < n_pages; i++) {
			if (pages[i].page) {
				get_page(pages[i].page);
				if (pages[i].file)
					get_file(pages[i].file);
			}
		}
		rcu_read_unlock();
	}
	up_read(&efct->hpl_mutation_lock);

	return rc;
}

static void efct_aux_release_superbuf(struct xlnx_efct_client *client, int rxq_id, int sbid)
{
	struct efct_nic *efct = client_to_efct(client);
	struct efct_rx_queue *rxq;

	if (!efct) {
		pr_err("Invalid client passed\n");
		return;
	}
	if (sbid < 0) {
		netif_err(efct, probe, efct->net_dev, "Invalid parameters for Rx queue %d\n", rxq_id);
		return;
	}

	if (rxq_id < 0 || rxq_id >= efct->rxq_count) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid queue id specified, queue %d\n", rxq_id);
		return;
	}

	rxq = &efct->rxq[rxq_id];
	sbl_release_buff(rxq, sbid);
}

static int efct_aux_rollover_rxq(struct xlnx_efct_client *client, int rxq_id)
{
	struct efct_nic *efct = client_to_efct(client);
	struct efct_rx_queue *rxq;
	int rc;

	if (!efct) {
		pr_err("Invalid parameters for Rx queue %d\n", rxq_id);
		return -EINVAL;
	}

	if (rxq_id < 0 || rxq_id >= efct->rxq_count) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid queue id %d\n", rxq_id);
		return -EINVAL;
	}
	if (client->rxq_info[rxq_id].ref_cnt <= 0) {
		netif_dbg(efct, drv, efct->net_dev,
			  "%s:Client is not bound to the queue %d\n", __func__, rxq_id);
		return -EACCES;
	}

	mutex_lock(&efct->state_lock);
	if (efct->state != STATE_NET_UP) {
		netif_dbg(efct, probe, efct->net_dev, "Interface is not up\n");
		mutex_unlock(&efct->state_lock);
		return -ENETDOWN;
	}

	rxq = &efct->rxq[rxq_id];
	efct_disable_napi(&efct->evq[rxq->evq_index]);
	rc = efct_rx_rollover(rxq);
	if (rc < 0) {
		netif_err(efct, probe, efct->net_dev,
			  "efct_rx_rollover failed, Queue: %d\n", rxq_id);
	}
	efct_enable_napi(&efct->evq[rxq->evq_index]);
	mutex_unlock(&efct->state_lock);

	return rc;
}

/*supporting functions for RX datapath*/
int efct_aux_buffer_start(struct efct_nic *efct, int rxq_id, u32 sbseq,
			  int sbid, bool sentinel)
{
	struct xlnx_efct_client *handle;
	int i, rc, cnt;

	if (!efct) {
		pr_err("Invalid parameters for Rx queue %d\n", rxq_id);
		return -EINVAL;
	}
	if (rxq_id < 0 || rxq_id >= efct->rxq_count) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid queue id %d\n", rxq_id);
		return -EINVAL;
	}
	cnt = 0;
	/*Notify all clients of this queue about buffer posting*/
	for_each_set_bit(i, efct->aux_client_map, MAX_AUX_CLIENTS) {
		handle = &efct->aux_clients[i];
		if (handle->drvops->buffer_start) {
			rc = handle->drvops->buffer_start(handle->driver_priv, rxq_id,
							  sbseq, sbid, sentinel);
			if (rc > 0)
				++cnt;
		}
	}

	return cnt;
}

int efct_aux_buffer_end(struct efct_nic *efct, int rxq_id, int sbid, bool force)
{
	struct xlnx_efct_client *handle;
	int i, rc, cnt;

	if (!efct) {
		pr_err("Invalid parameters for Rx queue %d\n", rxq_id);
		return -EINVAL;
	}
	if (rxq_id < 0 || rxq_id >= efct->rxq_count) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid queue id %d\n", rxq_id);
		return -EINVAL;
	}
	cnt = 0;
	/*Notify all clients of this queue about buffer having been concluded*/
	for_each_set_bit(i, efct->aux_client_map, MAX_AUX_CLIENTS) {
		handle = &efct->aux_clients[i];
		if (handle->drvops->buffer_end) {
			rc = handle->drvops->buffer_end(handle->driver_priv, rxq_id, sbid, force);
			/* As specified in the API client who rejected buffer on buffer_start() or
			 * already called release_superbuf() will not return 1
			 */
			if (rc > 0)
				++cnt;
		}
	}

	return cnt;
}

int efct_aux_poll(struct efct_nic *efct, int rxq_id, int budget)
{
	struct xlnx_efct_client *handle;
	int i, rc;

	if (!efct) {
		pr_err("Invalid parameters for Rx queue %d\n", rxq_id);
		return -EINVAL;
	}

	if (rxq_id < 0 || rxq_id >= efct->rxq_count) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid queue id %d\n", rxq_id);
		return -EINVAL;
	}
	/*Call poll for all clients of the given queue*/
	for_each_set_bit(i, efct->rxq[rxq_id].active_aux_clients,
			 sizeof(efct->rxq[rxq_id].active_aux_clients) * BITS_PER_BYTE) {
		handle = &efct->aux_clients[i];
		rc = handle->drvops->poll(handle->driver_priv, rxq_id, budget);
		if (rc < 0) {
			netif_err(efct, probe, efct->net_dev,
				  "Unable to call poll of RXQ %d to the client %d\n",
				  rxq_id, i);
		}
	}

	return 0;
}

static int check_queue_exclusivity(struct efct_nic *efct, u8 client_index,
				   int qid, bool exclusive)
{
	u8 i;

	if (efct->rxq[qid].exclusive_client >= 0 &&
	    efct->rxq[qid].exclusive_client != client_index)
		return -EINVAL;
	if (exclusive) {
		if (qid == EFCT_DEFAULT_QUEUE)
			return -EINVAL;
		for (i = 0; i < MAX_AUX_CLIENTS; i++)
			if (i != client_index && efct->rxq[qid].aux_filter_count[i])
				return -EINVAL;
	}

	return 0;
}

static int get_rxq_from_mask(struct efct_nic *efct, const struct cpumask *mask,
			     u8 client_index, bool exclusive)
{
	int rc = -1;
	int i = 0;
	int qid;

	if (!efct)
		return -EINVAL;

	/* Skip iterating over default queue when exclusive flag is set.
	 * start loop from default queue to make sure, selected queue has least no. of filters
	 * programmed
	 */
	for (i = !!exclusive; i < efct->rxq_count; i++) {
		qid = (i + EFCT_DEFAULT_QUEUE) % efct->rxq_count;
		if (cpumask_test_cpu(efct->rxq[qid].cpu, mask)) {
			if (check_queue_exclusivity(efct, client_index, qid, exclusive) < 0)
				continue;
			if (rc >= 0) {
				if (efct->rxq[qid].filter_count < efct->rxq[rc].filter_count ||
				    rc == EFCT_DEFAULT_QUEUE)
					rc = qid;
			} else {
				rc = qid;
			}
		}
	}

	return rc;
}

static int efct_aux_allow_dup_filter(struct efct_nic *efct,
				     struct xlnx_efct_client *client,
				     struct efct_filter_spec *spec_in_table,
				     struct efct_filter_spec *spec,
				     struct xlnx_efct_filter_params *params)
{
	bool anyqueue, loose, exclusive;

	if (spec_in_table->queue_id == RX_CLS_FLOW_DISC && spec->queue_id == RX_CLS_FLOW_DISC)
		return 0;
	if (spec_in_table->queue_id == RX_CLS_FLOW_DISC || spec->queue_id == RX_CLS_FLOW_DISC)
		return -EEXIST;

	exclusive = params->flags & XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	if (check_queue_exclusivity(efct, client->index, spec_in_table->queue_id, exclusive))
		return -EEXIST;
	if (spec_in_table->queue_id == spec->queue_id)
		return 0;
	loose = ((params->flags & XLNX_EFCT_FILTER_F_PREF_QUEUE) ||
		 (params->flags & XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE));
	if (loose)
		return 0;
	anyqueue = ((params->flags & XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT) ||
		    (params->flags & XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE));
	if (anyqueue && cpumask_test_cpu(efct->rxq[spec_in_table->queue_id].cpu, params->mask))
		return 0;

	return -EEXIST;
}

static int efct_aux_filter_insert(struct xlnx_efct_client *client,
				  struct xlnx_efct_filter_params *params)
{
	struct efct_filter_spec spec, *spec_in_table = NULL;
	struct efct_nic *efct = client_to_efct(client);
	struct efct_rx_queue *rx_queue = NULL;
	struct efct_mcdi_filter_table *table;
	int rc = 0, ins_index = -1, rxq = 0;
	bool anyqueue, loose, exclusive;
	u64 handle = 0;
	u32 flags;

	if (!efct) {
		pr_err("Invalid client passed\n");
		return -ENODEV;
	}

	if (!params || !params->spec) {
		netif_err(efct, probe, efct->net_dev, "Invalid params passed\n");
		return -EINVAL;
	}

	memset(&spec, 0, sizeof(spec));
	flags = params->flags;

	anyqueue = ((flags & XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT) ||
			(flags & XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE));
	loose = ((flags & XLNX_EFCT_FILTER_F_PREF_QUEUE) ||
			(flags & XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE));
	exclusive = flags & XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;

	if (!anyqueue) {
		if (params->spec->ring_cookie >= efct->rxq_count &&
		    params->spec->ring_cookie != RX_CLS_FLOW_DISC) {
			netif_err(efct, probe, efct->net_dev,
				  "Invalid queue id %lld\n", params->spec->ring_cookie);
			return -EINVAL;
		}
		if (params->spec->ring_cookie == RX_CLS_FLOW_DISC) {
			spec.queue_id = params->spec->ring_cookie;
			goto insert;
		}
		rxq = params->spec->ring_cookie;
		if (check_queue_exclusivity(efct, client->index, rxq, exclusive) < 0)
			rxq = -1;
	} else {
		if (!params->mask) {
			netif_err(efct, probe, efct->net_dev, "Invalid CPU mask\n");
			rc = -EINVAL;
			goto out;
		}

		rxq = get_rxq_from_mask(efct, params->mask, client->index, exclusive);
	}

	if (rxq < 0 && loose)
		rxq = get_rxq_from_mask(efct, cpu_online_mask, client->index, exclusive);

	if (rxq < 0) {
		netif_err(efct, probe, efct->net_dev,
			  "Unable to find the queue ID for given mask, flag= %d\n",
			  flags);
		rc = -EINVAL;
		goto out;
	}
	spec.queue_id = rxq;

insert:
	table = efct->filter_table;
	if (!table || !table->entry) {
		netif_err(efct, probe, efct->net_dev, "Invalid filter table\n");
		return -ENODEV;
	}

	down_write(&table->lock);

	rc = efct_fill_spec(efct, params->spec, &spec, true);
	if (rc < 0)
		goto unlock;

	rc = efct_get_spec_idx(efct, &spec);
	if (rc < 0)
		goto unlock;
	ins_index = rc;
	spec_in_table = (struct efct_filter_spec *)table->entry[ins_index].spec;
	if (!spec_in_table) {
		spec_in_table = kmalloc(sizeof(*spec_in_table), GFP_ATOMIC);
		if (!spec_in_table) {
			rc = -ENOMEM;
			goto unlock;
		}

		*spec_in_table = spec;
	} else {
		netif_dbg(efct, drv, efct->net_dev,
			  "The given spec already exists on the queue %lld\n",
			  spec_in_table->queue_id);

		params->rxq_out = spec_in_table->queue_id;
		if (test_bit(ins_index, &client->filter_map[0])) {
			rc = -EEXIST;
			goto unlock;
		}
		rc = efct_aux_allow_dup_filter(efct, client, spec_in_table, &spec, params);
		if (!rc) {
			params->filter_id_out = ins_index;
			params->filter_handle = table->entry[ins_index].handle;
			set_bit(ins_index, &client->filter_map[0]);
			if (spec_in_table->queue_id != RX_CLS_FLOW_DISC) {
				rx_queue = &efct->rxq[spec_in_table->queue_id];
				rx_queue->aux_filter_count[client->index]++;
			}
			table->entry[ins_index].ref_cnt++;
		}
		goto unlock;
	}

	table->entry[ins_index].spec = (unsigned long)spec_in_table;
	rc = efct_mcdi_filter_insert(efct, &spec, &handle);
	if (rc) {
		netif_dbg(efct, drv, efct->net_dev,
			  "efct_mcdi_filter_insert failed, rc: %d\n", rc);
		kfree(spec_in_table);
		table->entry[ins_index].spec = (unsigned long)NULL;
		goto unlock;
	}

	if (spec.queue_id != RX_CLS_FLOW_DISC) {
		efct->rxq[spec.queue_id].filter_count++;
		efct->rxq[spec.queue_id].aux_filter_count[client->index]++;
		if (exclusive)
			efct->rxq[spec.queue_id].exclusive_client = client->index;
	}
	params->rxq_out = spec.queue_id;
	table->entry[ins_index].handle = handle;
	table->entry[ins_index].ref_cnt = 1;
	params->filter_id_out = ins_index;
	params->filter_handle = (u32)(handle & 0xFFFFFFFF);

	set_bit(ins_index, &client->filter_map[0]);
unlock:
	up_write(&table->lock);
out:
#ifdef CONFIG_XILINX_EFCT_TRACING
	if (!rc)
		trace_xilinx_efct_filter_entry(params->filter_handle, ins_index);
#endif
	return rc;
}

static int efct_aux_filter_remove(struct xlnx_efct_client *client, u32 filter_id)
{
	struct efct_nic *efct = client_to_efct(client);
	int rc = -EINVAL;
	u64 qid;

	if (!efct) {
		pr_err("Invalid client passed\n");
		return -ENODEV;
	}

	if (filter_id >= efct->efct_dev->params.num_filter ||
	    !test_bit(filter_id, &client->filter_map[0]))
		return rc;
	rc = efct_delete_rule(efct, filter_id, &qid);

	if (!rc) {
		if (qid != RX_CLS_FLOW_DISC)
			efct->rxq[qid].aux_filter_count[client->index]--;
		clear_bit(filter_id, &client->filter_map[0]);
	}

	return rc;
}

bool efct_aux_packet_handled(struct efct_nic *efct, int rxq_id, bool flow_lookup,
			     void *meta, void *payload)
{
	struct xlnx_efct_client *handle;
	int i, rc;

	/*Notify all clients of this queue about packet*/
	for_each_set_bit(i, efct->rxq[rxq_id].active_aux_clients,
			 sizeof(efct->rxq[rxq_id].active_aux_clients) * BITS_PER_BYTE) {
		handle = &efct->aux_clients[i];
		if (handle->drvops->packet_handled) {
			rc = handle->drvops->packet_handled(handle->driver_priv, rxq_id,
							    flow_lookup, meta, payload);
			if (rc)
				return true;
		}
	}

	return false;
}

static int efct_aux_close(struct xlnx_efct_client *client)
{
	struct efct_nic *efct = client_to_efct(client);
	struct efct_mcdi_filter_table *table;
	struct efct_rx_queue *rxq;
	int i, pending_hp;
	bool is_ro = true;

	if (!efct)
		return -EINVAL;

	if (client->index >= MAX_AUX_CLIENTS) {
		netif_err(efct, probe, efct->net_dev, "Invalid client passed\n");
		return -EINVAL;
	}

	table = efct->filter_table;
	if (!table || !table->entry) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid filter table\n");
		return -EINVAL;
	}

	mutex_lock(&efct->client_list_lock);
	/* NB: clearing aux_client_map must be first, since it's read locklessly */
	if (!test_and_clear_bit(client->index, efct->aux_client_map)) {
		netif_err(efct, probe, efct->net_dev,
			  "Trying to close the client which is already closed, client :%d\n",
			  client->index);
		mutex_unlock(&efct->client_list_lock);
		return -EINVAL;
	}

#ifdef CONFIG_XILINX_DEBUGFS
	efct_fini_debugfs_aux_dev_client(client);
#endif
	/*Lazy free the huge pages memory*/
	for (i = 0; i < efct->rxq_count; i++) {
		rxq = &efct->rxq[i];
		pending_hp = atomic_read(&client->rxq_info[i].active_hp);

		if (pending_hp != 0) {
			netif_warn(efct, probe, efct->net_dev, "%u pending HugePages before conn close\n",
				   pending_hp);

			pending_hp = client->rxq_info[i].added_hp - client->rxq_info[i].removed_hp;
			mutex_lock(&efct->state_lock);
			efct_disable_napi(&efct->evq[rxq->evq_index]);

			if (pending_hp)
				/* Try freeing any pending HP */
				efct_aux_free_rxq_locked(client, rxq->index, pending_hp, &is_ro);

			efct_enable_napi(&efct->evq[rxq->evq_index]);

			if (is_ro) {
				wait_event_timeout
					(efct->hp_cleanup_wq,
					atomic_read(&client->rxq_info[rxq->index].active_hp) == 0,
					msecs_to_jiffies(EFCT_AUX_RO_CLEANUP_TIME_MS));

				pending_hp = atomic_read(&client->rxq_info[rxq->index].active_hp);
				if (pending_hp)
					netif_err(efct, probe, efct->net_dev,
						  "Failed to free %u HugePages\n", pending_hp);
			}

			mutex_unlock(&efct->state_lock);
		}

		/* Make sure NAPI is in sync with updated aux_client_map */
		efct_synchronize_napi(&efct->evq[rxq->evq_index]);

		/*Clearing client reference in the queue if it is not cleared already*/
		clear_bit(client->index, &efct->rxq[i].active_aux_clients[0]);

		client->rxq_info[i].ref_cnt = 0;
		atomic_set(&client->rxq_info[i].active_hp, 0);
		client->rxq_info[i].added_hp = 0;
		client->rxq_info[i].removed_hp = 0;
	}

	for_each_set_bit(i, client->active_txq, sizeof(client->active_txq) * BITS_PER_BYTE)
		efct_aux_free_txq(client, i);
	for_each_set_bit(i, client->active_evq, sizeof(client->active_evq) * BITS_PER_BYTE)
		efct_aux_free_evq(client, i);
	/*Remove pending filters for this client*/
	for_each_set_bit(i, client->filter_map, sizeof(client->filter_map) * BITS_PER_BYTE)
		if (table->entry[i].ref_cnt)
			efct_aux_filter_remove(client, i);
		else
			clear_bit(i, client->filter_map);

	client->state = STATE_CLOSED;
	mutex_unlock(&efct->client_list_lock);

	return 0;
}

static bool efct_aux_filter_supported(struct xlnx_efct_client *client,
				      const struct ethtool_rx_flow_spec *flow)
{
	struct efct_nic *efct = client_to_efct(client);
	struct efct_filter_spec spec;
	int rc;

	if (!efct) {
		pr_err("Invalid client passed\n");
		return false;
	}

	if (!flow) {
		netif_err(efct, probe, efct->net_dev, "rx_flow_spec is null\n");
		return false;
	}
	memset(&spec, 0, sizeof(spec));
	rc = efct_fill_spec(efct, flow, &spec, false);
	if (rc < 0)
		return false;
	return true;
}

static const struct xlnx_efct_devops efct_devops = {
	.open = efct_aux_open,
	.close = efct_aux_close,
	.get_param = efct_aux_get_param,
	.set_param = efct_aux_set_param,
	.init_evq = efct_aux_init_evq,
	.free_evq = efct_aux_free_evq,
	.bind_rxq = efct_aux_bind_rxq,
	.free_rxq = efct_aux_free_rxq,
	.init_txq = efct_aux_init_txq,
	.free_txq = efct_aux_free_txq,
	.fw_rpc = efct_aux_fw_rpc,
	.ctpio_addr = efct_aux_ctpio_addr,
	.get_hugepages = efct_aux_get_hugepages,
	.release_superbuf = efct_aux_release_superbuf,
	.rollover_rxq = efct_aux_rollover_rxq,
	.filter_insert = efct_aux_filter_insert,
	.filter_remove = efct_aux_filter_remove,
	.is_filter_supported = efct_aux_filter_supported,
};

static void efct_auxbus_release(struct device *dev)
{
	/*Dummy function for aux bus registration*/
}

int efct_auxbus_register(struct efct_nic *efct)
{
	int dev_id;
	int rc;

	if (!efct) {
		pr_err("Invalid device, caught NULL pointer\n");
		return -EINVAL;
	}
	dev_id = ida_alloc_range(&aux_ida, 0, INT_MAX, GFP_KERNEL);
	if (dev_id < 0) {
		netif_err(efct, probe, efct->net_dev,
			  "Fetching device ID for Aux device failed, rc=%d\n", dev_id);
		return dev_id;
	}

	efct->aux_dev.adev.name = XLNX_EFCT_DEVNAME;
	efct->aux_dev.adev.id = dev_id;
	efct->aux_dev.adev.dev.parent = &efct->efct_dev->pci_dev->dev;
	efct->aux_dev.adev.dev.release = efct_auxbus_release;
	efct->aux_dev.version = XLNX_EFCT_AUX_VERSION;
	efct->aux_dev.ops = &efct_devops;

	rc = auxiliary_device_init(&efct->aux_dev.adev);
	if (rc) {
		netif_err(efct, probe, efct->net_dev,
			  "Initialization of Aux device failed, rc=%d\n", rc);
		return rc;
	}

#ifdef CONFIG_XILINX_DEBUGFS
	efct_init_debugfs_aux_dev(efct);
#endif
	rc = auxiliary_device_add(&efct->aux_dev.adev);
	if (rc) {
		netif_err(efct, probe, efct->net_dev, "Addition of Aux device failed, rc=%d\n", rc);
#ifdef CONFIG_XILINX_DEBUGFS
		efct_fini_debugfs_aux_dev(efct);
#endif
		auxiliary_device_uninit(&efct->aux_dev.adev);
		return rc;
	}

	rc = sysfs_create_link(&efct->net_dev->dev.kobj, &efct->aux_dev.adev.dev.kobj,
			       "aux_device");
	if (rc) {
		netif_err(efct, probe, efct->net_dev, "Failed to create link aux_device, rc = %d\n",
			  rc);
		auxiliary_device_delete(&efct->aux_dev.adev);
#ifdef CONFIG_XILINX_DEBUGFS
		efct_fini_debugfs_aux_dev(efct);
#endif
		auxiliary_device_uninit(&efct->aux_dev.adev);
		return rc;
	}

	return 0;
}

void efct_auxbus_unregister(struct efct_nic *efct)
{
	struct xlnx_efct_client *client;
	int i;

	if (!efct)
		return;

	sysfs_remove_link(&efct->net_dev->dev.kobj, "aux_device");
	/* Free ID before deleting device */
	ida_free(&aux_ida, efct->aux_dev.adev.id);
	auxiliary_device_delete(&efct->aux_dev.adev);

	for_each_set_bit(i, efct->aux_client_map, sizeof(efct->aux_client_map) * BITS_PER_BYTE) {
		client = &efct->aux_clients[i];
		efct_aux_close(client);
	}

#ifdef CONFIG_XILINX_DEBUGFS
	efct_fini_debugfs_aux_dev(efct);
#endif
	auxiliary_device_uninit(&efct->aux_dev.adev);
}

int efct_aux_handle_event(struct efct_nic *efct, enum xlnx_event_type event,
			  int rxq, u64 value, int budget)
{
	struct xlnx_efct_client *client;
	struct xlnx_efct_event ev;
	int spent = 0;
	int i;

	ev.type = event;
	ev.rxq = rxq;
	ev.value = value;

	/* Notify all users */
	for_each_set_bit(i, efct->aux_client_map, sizeof(efct->aux_client_map) * BITS_PER_BYTE) {
		client = &efct->aux_clients[i];
		if (client->drvops->handle_event) {
			int rc = client->drvops->handle_event(client->driver_priv, &ev,
							      budget - spent);
			if (rc > 0)
				spent += rc;
		}
	}
	return spent;
}

void efct_aux_on_dev_reset(struct efct_nic *efct)
{
	struct xlnx_efct_client *client;
	struct efct_ev_queue *evq;
	int i, j;

	mutex_lock(&efct->state_lock);

	for_each_set_bit(i, efct->aux_client_map, MAX_AUX_CLIENTS) {
		client = &efct->aux_clients[i];

		for_each_set_bit(j, client->active_txq, efct->max_txq_count) {
			efct->type->tx_purge(&efct->txq[j]);
			clear_bit(j, client->active_txq);
			clear_bit(j, &efct->txq_active_mask);
		}

		for_each_set_bit(j, client->active_evq, efct->max_evq_count) {
			evq = &efct->evq[j];
			efct->type->ev_purge(evq);
			if (evq->buf.dma_addr) {
				dma_unmap_page(&efct->efct_dev->pci_dev->dev,
					       evq->buf.dma_addr, evq->buf.len,
					       DMA_BIDIRECTIONAL);
				evq->buf.dma_addr = 0UL;
				evq->buf.addr = NULL;
			}

			clear_bit(j, client->active_evq);
			clear_bit(j, &efct->evq_active_mask);
			evq->type = EVQ_T_NONE;
		}
	}

	mutex_unlock(&efct->state_lock);
}
