// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include <linux/module.h>
#include <linux/netdevice.h>
#include "efct_common.h"
#include "mcdi.h"
#include "mcdi_functions.h"
#include "efct_io.h"
#include "mcdi_pcol.h"
#include "efct_reg.h"
#include "efct_evq.h"
#ifdef CONFIG_XILINX_AUX_EFCT
#include "efct_auxbus.h"
#endif
/*Use NETIF_MSG_DRV is for debug logs*/
static u32 debug = (NETIF_MSG_PROBE |
		NETIF_MSG_LINK | NETIF_MSG_IFDOWN |
		NETIF_MSG_IFUP | NETIF_MSG_RX_ERR |
		NETIF_MSG_TX_ERR | NETIF_MSG_HW);
#ifdef EFCT_NOT_UPSTREAM
module_param(debug, uint, 0);
MODULE_PARM_DESC(debug, "Bitmapped debugging message enable value");
#endif
static u32 rxq_entries = 0;
#ifdef EFCT_NOT_UPSTREAM
module_param(rxq_entries, uint, 0444);
MODULE_PARM_DESC(rxq_entries, "Maximum entries in RX queue");
#endif
static int efct_probe_queues(struct efct_nic *efct);
static void efct_remove_queues(struct efct_nic *efct);
static void efct_reset_queue_work(struct work_struct *work);

static const u32 efct_reset_type_max = RESET_TYPE_MAX;
static const char *const efct_reset_type_names[] = {
	[RESET_TYPE_ALL]                = "ALL",
	[RESET_TYPE_WORLD]              = "WORLD",
	[RESET_TYPE_DATAPATH]           = "DATAPATH",
	[RESET_TYPE_MC_BIST]            = "MC_BIST",
	[RESET_TYPE_DISABLE]            = "DISABLE",
	[RESET_TYPE_TX_WATCHDOG]        = "TX_WATCHDOG",
	[RESET_TYPE_MC_FAILURE]         = "MC_FAILURE",
	[RESET_TYPE_MCDI_TIMEOUT]       = "MCDI_TIMEOUT (FLR)",
};

#define RESET_TYPE(type) STRING_TABLE_LOOKUP(type, efct_reset_type)

/* How often and how many times to poll for a reset while waiting for a
 * BIST that another function started to complete.
 */
#define BIST_WAIT_DELAY_MS  100
#define BIST_WAIT_DELAY_COUNT   100

static struct workqueue_struct *reset_workqueue;

int efct_create_reset_workqueue(void)
{
	reset_workqueue = alloc_ordered_workqueue("efct_reset", 0);
	if (!reset_workqueue) {
		pr_err("Failed to create reset workqueue\n");
		return -ENOMEM;
	}

	return 0;
}

void efct_destroy_reset_workqueue(void)
{
	if (reset_workqueue) {
		destroy_workqueue(reset_workqueue);
		reset_workqueue = NULL;
	}
}

void efct_flush_reset_workqueue(void)
{
	if (reset_workqueue)
		flush_workqueue(reset_workqueue);
}

int efct_nic_alloc_buffer(struct efct_nic *efct, struct efct_buffer *buffer,
			  u32 len, gfp_t gfp_flags)
{
	buffer->addr = dma_alloc_coherent(&efct->efct_dev->pci_dev->dev, len,
					  &buffer->dma_addr, gfp_flags);
	if (!buffer->addr)
		return -ENOMEM;
	buffer->len = len;
	memset(buffer->addr, 0, len);
	return 0;
}

void efct_nic_free_buffer(struct efct_nic *efct, struct efct_buffer *buffer)
{
	if (buffer->addr) {
		dma_free_coherent(&efct->efct_dev->pci_dev->dev, buffer->len,
				  buffer->addr, buffer->dma_addr);
		buffer->addr = NULL;
	}
}

static int evq_get_free_index(struct efct_nic *efct)
{
	unsigned long evq_active;
	unsigned long mask;
	int index;

	if (!efct)
		return -EINVAL;

	mask  = (1UL << efct->max_evq_count) - 1;

	do {
		evq_active = efct->evq_active_mask;
		/*No free index present, all event queues are in use. ffz()
		 *behavior is undefined if at least one zero is not present.
		 * This check will make sure at least 1 zero bit is there in evq_active
		 */
		if ((evq_active & mask) == mask)
			return -EAGAIN;
		index = ffz(evq_active);
		/* In case queue is already allocated because of contention. It will
		 * retry for next available index
		 */
		if (test_and_set_bit(index, &efct->evq_active_mask)) {
			netif_dbg(efct, drv, efct->net_dev,
				  "Event queue index %d is already in use\n", index);
			continue;
		}

		if (index < efct->max_evq_count)
			return index;
		else
			return -ENOSPC;
	} while (1);

	return -ENOSPC;
}

void efct_schedule_reset(struct efct_nic *efct, enum reset_type type)
{
	unsigned long last_reset = READ_ONCE(efct->last_reset);
	static const u32 RESETS_BEFORE_DISABLE = 5;
	enum reset_type method;

	if (!efct_net_active(READ_ONCE(efct->state)))
		return;
	method = efct->type->map_reset_reason(type);

	/* check we're scheduling a new reset and if so check we're
	 * not scheduling resets too often.
	 * this part is not atomically safe, but is also ultimately a
	 * heuristic; if we lose increments due to dirty writes
	 * that's fine and if we falsely increment or reset due to an
	 * inconsistent read of last_reset on 32-bit arch it's also ok.
	 */
	if (time_after(jiffies, last_reset + HZ))
		efct->reset_count = 0;

	if (!(efct->reset_pending & (1 << method)) && ++efct->reset_count > RESETS_BEFORE_DISABLE) {
		method = RESET_TYPE_DISABLE;

		netif_err(efct, probe, efct->net_dev,
			  "too many resets, scheduling %s\n",
			  RESET_TYPE(method));
	}
	netif_dbg(efct, probe, efct->net_dev, "scheduling %s reset for %s\n",
		  RESET_TYPE(method), RESET_TYPE(type));

	set_bit(method, &efct->reset_pending);
	smp_mb(); /* ensure we change reset_pending before checking state */

	queue_work(reset_workqueue, &efct->reset_work);
}

void efct_schedule_queue_reset(struct efct_nic *efct, bool is_txq, int qid)
{
	struct queue_reset_work *reset_work;

	reset_work = kzalloc(sizeof(*reset_work), GFP_ATOMIC);
	if (!reset_work)
		return;

	INIT_WORK(&reset_work->work, efct_reset_queue_work);
	reset_work->efct = efct;
	reset_work->is_txq = is_txq;
	reset_work->qid = qid;

	netif_dbg(efct, probe, efct->net_dev,
		  "scheduling %s reset for queue %d\n", reset_work->is_txq ? "Txq" : "Rxq",
		  reset_work->qid);

	queue_work(reset_workqueue, &reset_work->work);
}

static void efct_wait_for_bist_end(struct efct_nic *efct)
{
	int i;

	for (i = 0; i < BIST_WAIT_DELAY_COUNT; ++i) {
		if (efct->type->mcdi_poll_bist_end(efct))
			goto out;
		msleep(BIST_WAIT_DELAY_MS);
	}

	netif_err(efct, probe, efct->net_dev, "Warning: No MC reboot after BIST mode\n");
out:
	/* Either way unset the BIST flag. If we found no reboot we probably
	 * won't recover, but we should try.
	 */
	efct->mc_bist_for_other_fn = false;
}

/* The worker thread exists so that code that cannot sleep can
 * schedule a reset for later.
 */
static void efct_reset_work(struct work_struct *data)
{
	struct efct_nic *efct = container_of(data, struct efct_nic, reset_work);
	enum reset_type method;
	unsigned long pending;

	pending = READ_ONCE(efct->reset_pending);
	method = fls(pending) - 1;

	if (method == RESET_TYPE_MC_BIST)
		/* TODO: Check MC config for BIST support. */
		efct_wait_for_bist_end(efct);

	rtnl_lock();
	/* We checked the state in efct_schedule_reset() but it may
	 * have changed by now.  Now that we have the RTNL lock,
	 * it cannot change again.
	 */
	(void)efct_reset(efct, method);
	rtnl_unlock();
}

static void efct_device_detach_sync(struct efct_nic *efct)
{
	struct net_device *dev = efct->net_dev;

	/* Lock/freeze all TX queues so that we can be sure the
	 * TX scheduler is stopped when we're done and before
	 * netif_device_present() becomes false.
	 */
	netif_tx_lock_bh(dev);
	netif_device_detach(dev);
	netif_tx_stop_all_queues(dev);
	netif_tx_unlock_bh(dev);
}

static void efct_device_attach_if_not_resetting(struct efct_nic *efct)
{
	if (efct->state != STATE_DISABLED && !efct->reset_pending) {
		netif_tx_start_all_queues(efct->net_dev);
		netif_device_attach(efct->net_dev);
	}
}

/* Do post-processing after the reset.
 * Returns whether the reset was completed and the device is back up.
 */
static int efct_reset_complete(struct efct_nic *efct, enum reset_type method,
			       bool retry, bool disabled)
{
	if (disabled) {
		netif_err(efct, probe, efct->net_dev, "has been disabled\n");
		efct->state = STATE_DISABLED;
		return false;
	}

	if (retry) {
		netif_info(efct, probe, efct->net_dev, "scheduling retry of reset\n");
		if (method == RESET_TYPE_MC_BIST)
			method = RESET_TYPE_DATAPATH;
		efct_schedule_reset(efct, method);
		return false;
	}

	netif_dbg(efct, probe, efct->net_dev, "reset complete\n");
	return true;
}

static int efct_do_reset(struct efct_nic *efct, enum reset_type method)
{
	int rc;

	rc = efct->type->reset(efct, method);
	if (rc) {
		netif_err(efct, probe, efct->net_dev, "failed to reset hardware\n");
		return rc;
	}

	/* Clear flags for the scopes we covered.  We assume the NIC and
	 * driver are now quiescent so that there is no race here.
	 */
	if (method < RESET_TYPE_MAX_METHOD)
		efct->reset_pending &= -(1 << (method + 1));
	else /* it doesn't fit into the well-ordered scope hierarchy */
		__clear_bit(method, &efct->reset_pending);

	/* Reinitialise bus-mastering, which may have been turned off before
	 * the reset was scheduled. This is still appropriate, even in the
	 * RESET_TYPE_DISABLE since this driver generally assumes the hardware
	 * can respond to requests.
	 */
	pci_set_master(efct->efct_dev->pci_dev);

	return 0;
}

/* Reset the NIC using the specified method.  Note that the reset may
 * fail, in which case the card will be left in an unusable state.
 *
 * Caller must hold the rtnl_lock.
 */
int efct_reset(struct efct_nic *efct, enum reset_type method)
{
	bool disabled, retry;
	int rc;

	ASSERT_RTNL();

	netif_info(efct, probe, efct->net_dev, "resetting (%s)\n", RESET_TYPE(method));

	efct_device_detach_sync(efct);

#ifdef CONFIG_XILINX_AUX_EFCT
	efct_aux_handle_event(efct, XLNX_EVENT_RESET_DOWN, -1, true, INT_MAX);
#endif
	rc = efct_do_reset(efct, method);
	retry = rc == -EAGAIN;

	/* Leave device stopped if necessary */
	disabled = (rc && !retry) || method == RESET_TYPE_DISABLE;

	if (disabled)
		dev_close(efct->net_dev);

	if (efct_reset_complete(efct, method, retry, disabled))
		efct_device_attach_if_not_resetting(efct);
#ifdef CONFIG_XILINX_AUX_EFCT
	efct_aux_handle_event(efct, XLNX_EVENT_RESET_UP, -1, !disabled, INT_MAX);
#endif

	return rc;
}

/* Reset the specific queue */
static void efct_reset_queue_work(struct work_struct *work)
{
	struct queue_reset_work *reset_work = (struct queue_reset_work *)work;
	struct efct_nic *efct = reset_work->efct;
	int rc;

	rtnl_lock();

	if (reset_work->is_txq) {
		struct efct_tx_queue *txq = &efct->txq[reset_work->qid];

		efct_device_detach_sync(efct);

		efct->type->tx_purge(txq);

		rc = efct->type->tx_init(txq);
		if (!rc)
			efct_device_attach_if_not_resetting(efct);
	} else {
		struct efct_rx_queue *rxq = &efct->rxq[reset_work->qid];

		efct->type->rx_purge(rxq);
		efct_disable_napi(&efct->evq[rxq->evq_index]);
		rc = efct->type->rx_init(rxq);
		efct_enable_napi(&efct->evq[rxq->evq_index]);
	}

	rtnl_unlock();

	if (rc)
		netif_err(efct, probe, efct->net_dev, "Warning: %s:%d queue is disabled\n",
			  reset_work->is_txq ? "Txq" : "Rxq", reset_work->qid);
	else
		netif_dbg(efct, probe, efct->net_dev, "%s:%d reset completed\n",
			  reset_work->is_txq ? "Txq" : "Rxq", reset_work->qid);

	kfree(reset_work);
}

/* Initializes the "struct efct_nic" structure
 */
/* Default stats update time */
#define STAT_INTERVAL_MS 990

void efct_set_evq_names(struct efct_nic *efct)
{
	struct efct_ev_queue *evq;
	struct efct_rx_queue *rxq;
	struct efct_tx_queue *txq;
	char *type;
	int i;

	evq = efct->evq;
	for (i = 0; i < efct->max_evq_count; i++) {
		if (evq[i].type == EVQ_T_RX) {
			type = "-rx";
			rxq = (struct efct_rx_queue *)evq[i].queue;
			snprintf(evq[i].msi.name, sizeof(evq[0].msi.name), "%s%s-%d", efct->name,
				 type, rxq->index);
		} else if (evq[i].type == EVQ_T_TX) {
			type = "-tx";
			txq = (struct efct_tx_queue *)evq[i].queue;
			snprintf(evq[i].msi.name, sizeof(evq[0].msi.name), "%s%s-%d", efct->name,
				 type, txq->txq_index);
		} else {
			continue;
		}
	}
}

static int validate_rxq_entries(u32 entries_per_buff, u32 *entries)
{
	u32 max = entries_per_buff * RX_MAX_DRIVER_BUFFS;
	u32 min = entries_per_buff * RX_MIN_DRIVER_BUFFS;

	if (*entries < min || *entries > max) {
		*entries = clamp_val(*entries, min, max);
		return -ERANGE;
	} else if (*entries % entries_per_buff) {
		*entries = rounddown(*entries, entries_per_buff);
		return -EINVAL;
	}
	return 0;
}

void efct_init_struct(struct efct_nic *efct)
{
	u32 buffer_size, entries_per_buff, num_rx_buffs;
	int pkt_stride = 0;
	int j = 0;
	int rc;

	efct->state = STATE_UNINIT;
	efct->mtu = EFCT_MAX_MTU;
	efct->num_mac_stats = MC_CMD_MAC_NSTATS_V4;
	efct->stats_period_ms = 0;
	efct->stats_initialised = false;
	spin_lock_init(&efct->stats_lock);

	/* Update reset info */
	INIT_WORK(&efct->reset_work, efct_reset_work);
	efct->reset_pending = 0;
	efct->stats_enabled = false;
	/*Updating bitmap of active queue count*/
	efct->evq_active_mask = 0;
	efct->rxq_active_mask = 0;
	efct->txq_active_mask = 0;
	efct->msg_enable = debug;
	efct->irq_rx_adaptive = true;
	/*updating number of queues*/
	efct->max_txq_count = efct->efct_dev->params.tx_apertures;
	efct->rxq_count = min(efct->efct_dev->params.rx_queues, num_online_cpus());
	efct->max_evq_count = efct->max_txq_count + efct->rxq_count;
	if (efct->max_evq_count > efct->efct_dev->params.num_evq)
		efct->max_evq_count = efct->efct_dev->params.num_evq;
#ifdef CONFIG_XILINX_AUX_EFCT
	/*Updating auxiliary device resources*/
	efct->aux_res.evq_base =
		efct->rxq_count + EFCT_MAX_CORE_TX_QUEUES;
	efct->aux_res.evq_lim = efct->max_evq_count;
	efct->aux_res.txq_base = EFCT_MAX_CORE_TX_QUEUES;
	efct->aux_res.txq_lim = efct->max_txq_count;
#endif
	/*Updating indexes and efct*/
	for (j = 0; j < efct->max_evq_count; j++) {
		efct->evq[j].index = j;
		efct->evq[j].efct = efct;
		efct->evq[j].unsol_credit = EFCT_EV_UNSOL_MAX;
	}
	for (j = 0; j < efct->max_txq_count; j++) {
		efct->txq[j].txq_index = j;
		efct->txq[j].efct = efct;
	}
	/*updating queue size*/
	for (j = 0; j < EFCT_MAX_CORE_TX_QUEUES; j++)
		efct->txq[j].num_entries =
		roundup_pow_of_two(DIV_ROUND_UP(efct->efct_dev->params.tx_fifo_size, EFCT_MIN_MTU));

	pkt_stride = roundup_pow_of_two(efct->mtu);
	buffer_size = efct->efct_dev->params.rx_buffer_len * EFCT_RX_BUF_SIZE_MULTIPLIER;
	entries_per_buff = DIV_ROUND_UP(buffer_size, pkt_stride);
	if (!rxq_entries)
		rxq_entries = RX_BUFFS_PER_QUEUE * entries_per_buff;
	rc = validate_rxq_entries(entries_per_buff, &rxq_entries);
	if (rc == -ERANGE)
		pci_warn(efct->efct_dev->pci_dev,
			 "rxq_entries parameter clamped to %u\n", rxq_entries);
	else if (rc == -EINVAL)
		pci_warn(efct->efct_dev->pci_dev,
			 "Invalid rxq_entries parameter value. rxq_entries set to %u\n",
                         rxq_entries);
	num_rx_buffs = DIV_ROUND_UP(rxq_entries, entries_per_buff);

	for (j = 0; j < efct->rxq_count; j++) {
		efct->rxq[j].efct = efct;
		efct->rxq[j].index = j;
		efct->rxq[j].enable = false;
		efct->rxq[j].num_rx_buffs = num_rx_buffs;
		efct->rxq[j].buffer_size = buffer_size;
		efct->rxq[j].pkt_stride = pkt_stride;
		efct->rxq[j].num_entries = entries_per_buff * num_rx_buffs;
#ifdef CONFIG_XILINX_AUX_EFCT
		/*Initialize the Aux lock*/
		bitmap_zero(efct->rxq[j].active_aux_clients, MAX_AUX_CLIENTS);
		efct->rxq[j].filter_count = 0;
		efct->rxq[j].exclusive_client = -1;
#endif
	}
	efct->ct_thresh = CT_DEFAULT_THRESHOLD;
#ifdef CONFIG_XILINX_AUX_EFCT
	/*Initialize per port lock for aux API's*/
	mutex_init(&efct->client_list_lock);
	bitmap_zero(efct->aux_client_map, MAX_AUX_CLIENTS);
	init_waitqueue_head(&efct->hp_cleanup_wq);
#endif
	mutex_init(&efct->reflash_mutex);
	mutex_init(&efct->mac_lock);

	init_waitqueue_head(&efct->flush_wq);
}

int efct_map_membase(struct efct_nic *efct, u32 fcw_offset, u8 port_id)
{
	struct pci_dev *pci_dev = efct->efct_dev->pci_dev;
	u32 uc_mem_map_size, wc_mem_map_size;
	resource_size_t membase_phy;

	if (port_id == 0) {
		/* Unmap previous map */
		if (efct->efct_dev->membase) {
			iounmap(efct->efct_dev->membase);
			efct->efct_dev->membase = NULL;
		}
		membase_phy = efct->efct_dev->membase_phys;
		uc_mem_map_size = PAGE_ALIGN(fcw_offset + ER_HZ_PORT0_REG_HOST_CTPIO_REGION0);
	} else {
		membase_phy = efct->efct_dev->membase_phys + fcw_offset + EFCT_PORT_OFFSET;
		uc_mem_map_size = PAGE_ALIGN(ER_HZ_PORT0_REG_HOST_CTPIO_REGION0);
	}

	wc_mem_map_size = PAGE_ALIGN(EFCT_PORT_LEN - ER_HZ_PORT0_REG_HOST_CTPIO_REGION0);

	efct->membase = ioremap(membase_phy, uc_mem_map_size);
	if (!efct->membase) {
		pci_err(efct->efct_dev->pci_dev,
			"could not map BAR at %llx+%x\n",
			(unsigned long long)membase_phy, uc_mem_map_size);
		return -ENOMEM;
	}

	efct->wc_membase = ioremap_wc(membase_phy + uc_mem_map_size, wc_mem_map_size);
	if (!efct->wc_membase) {
		pci_err(efct->efct_dev->pci_dev, "could not map wc to BAR at %llx+%x\n",
			((unsigned long long)membase_phy + uc_mem_map_size), wc_mem_map_size);
		iounmap(efct->membase);
		efct->membase = NULL;
		return -ENOMEM;
	}

	if (port_id == 0) {
		efct->efct_dev->membase = efct->membase;
		efct->membase += fcw_offset;
	}

	pci_dbg(pci_dev, "Port : %u, UC at %llx+%x (virtual %p), MC at %llx+%x (virtual %p)\n",
		port_id, (unsigned long long)membase_phy, uc_mem_map_size,  efct->membase,
		(unsigned long long)membase_phy + uc_mem_map_size, wc_mem_map_size,
		efct->wc_membase);

	return 0;
}

void efct_unmap_membase(struct efct_nic *efct)
{
	if (efct->wc_membase) {
		iounmap(efct->wc_membase);
		efct->wc_membase = NULL;
	}

	if (efct->membase) {
		if (efct->port_base == 0u) {
			iounmap(efct->efct_dev->membase);
			efct->efct_dev->membase = NULL;
		} else {
			iounmap(efct->membase);
		}

		efct->membase = NULL;
	}
}

/* This configures the PCI device to enable I/O and DMA. */
int efct_init_io(struct efct_device *efct_dev,
		 int bar, dma_addr_t dma_mask, u32 mem_map_size)
{
	struct pci_dev *pci_dev = efct_dev->pci_dev;
	int rc;

	efct_dev->mem_bar = UINT_MAX;

	rc = pci_enable_device(pci_dev);
	if (rc) {
		pci_err(pci_dev, "failed to enable PCI device\n");
		goto fail1;
	}

	pci_set_master(pci_dev);

	/* Set the PCI DMA mask.  Try all possibilities from our
	 * genuine mask down to 32 bits, because some architectures
	 * (e.g. x86_64 with iommu_sac_force set) will allow 40 bit
	 * masks event though they reject 46 bit masks.
	 */
	while (dma_mask > 0x7fffffffUL) {
		rc = dma_set_mask_and_coherent(&pci_dev->dev, dma_mask);
		if (rc == 0)
			break;
		dma_mask >>= 1;
	}
	if (rc) {
		pci_err(pci_dev, "could not find a suitable DMA mask\n");
		goto fail2;
	}

	efct_dev->membase_phys = pci_resource_start(efct_dev->pci_dev, bar);
	if (!efct_dev->membase_phys) {
		pci_err(pci_dev, "ERROR: No BAR%d mapping from the BIOS. Try pci=realloc on the kernel command line\n",
			bar);
		rc = -ENODEV;
		goto fail3;
	}
	rc = pci_request_region(pci_dev, bar, "xilinx");
	if (rc) {
		pci_err(pci_dev, "request for memory BAR[%d] failed\n", bar);
		goto fail3;
	}
	efct_dev->mem_bar = bar;
	efct_dev->membase = ioremap(efct_dev->membase_phys, mem_map_size);
	if (!efct_dev->membase) {
		pci_err(pci_dev, "could not map memory BAR[%d] at %llx+%x\n",
			bar, (unsigned long long)efct_dev->membase_phys, mem_map_size);
		rc = -ENOMEM;
		goto fail4;
	}
	pci_dbg(pci_dev, "memory BAR[%d] at %llx+%x (virtual %p)\n", bar,
		(unsigned long long)efct_dev->membase_phys, mem_map_size,	efct_dev->membase);

	return 0;

fail4:
	pci_release_region(efct_dev->pci_dev, bar);
fail3:
	efct_dev->membase_phys = 0;
fail2:
	pci_disable_device(efct_dev->pci_dev);
fail1:
	return rc;
}

void efct_fini_io(struct efct_device *efct_dev)
{

	if (efct_dev->membase) {
		iounmap(efct_dev->membase);
		efct_dev->membase = NULL;
	}

	if (efct_dev->membase_phys) {
		pci_release_region(efct_dev->pci_dev, efct_dev->mem_bar);
		efct_dev->membase_phys = 0;
		efct_dev->mem_bar = UINT_MAX;

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_PCI_DEV_FLAGS_ASSIGNED)
		/* Don't disable bus-mastering if VFs are assigned */
		if (!pci_vfs_assigned(efct_dev->pci_dev))
#endif
			pci_disable_device(efct_dev->pci_dev);
	}
}

#ifdef CONFIG_XILINX_AUX_EFCT
static u16 efct_make_irq_resources_(struct efct_nic *efct, struct msix_entry *xe, u8 idx)
{
	u16 n_ranges = 0, n_ports;
	size_t i, j;
	u8 max_irq;

	n_ports = efct->efct_dev->num_ports;
	i = efct->aux_res.evq_base + (efct->efct_dev->vec_per_port * idx);
	max_irq = min(efct->efct_dev->vec_per_port, efct->aux_res.evq_lim) * n_ports;
	while (i < (max_irq / n_ports) * (idx + 1)) {
		j = i + 1;

		/* Get the range of continuous vectors by traversing allocated vectors list.
		 * Increment the current vector by 1 and compare with next vector to
		 * check the continuation. Starts from evq_base to max_irq.
		 */
		while (j < (max_irq / n_ports * (idx + 1)) && xe[j - 1].vector + 1 == xe[j].vector)
			++j;

		if (efct->irq_resources) {
			efct->irq_resources->irq_ranges[n_ranges].vector = xe[i].vector;
			efct->irq_resources->irq_ranges[n_ranges].range = j - i;
		}
		n_ranges++;
		i = j;
	}

	return n_ranges;
}

int efct_make_irq_resources(struct efct_nic *efct, struct msix_entry *xentries, u8 index)
{
	u16 n_ranges;

	efct->irq_resources = NULL;
	n_ranges = efct_make_irq_resources_(efct, xentries, index);
	if (n_ranges) {
		efct->irq_resources =
			kzalloc(sizeof(struct xlnx_efct_irq_resources) +
				(sizeof(struct xlnx_efct_irq_range) * n_ranges),
				GFP_KERNEL);
		if (!efct->irq_resources) {
			netif_warn(efct, probe, efct->net_dev,
				   "Out of memory exporting interrupts to aux client\n");
			return -ENOMEM;
		}
	}
	if (efct->irq_resources) {
		efct->irq_resources->n_ranges = n_ranges;
		efct_make_irq_resources_(efct, xentries, index);
		efct->irq_resources->int_prime = (void __iomem *)(efct->membase +
				ER_HZ_PORT0_REG_HOST_EVQ_INT_PRIME);
		netif_dbg(efct, drv, efct->net_dev,
			  "Exporting %d IRQ range(s) to aux client, IRQ[0]=%d.\n",
			  efct->irq_resources->n_ranges,
			  efct->irq_resources->irq_ranges[0].vector);
	}
	return 0;
}
#endif
/* This ensures that the kernel is kept informed (via
 * netif_carrier_on/off) of the link status.
 */
void efct_link_status_changed(struct efct_nic *efct)
{
	struct efct_link_state *link_state = &efct->link_state;
	bool kernel_link_up;

	if (!netif_running(efct->net_dev))
		return;

	kernel_link_up = netif_carrier_ok(efct->net_dev);
	if (link_state->up != kernel_link_up) {
		efct->n_link_state_changes++;

		if (link_state->up)
			netif_carrier_on(efct->net_dev);
		else
			netif_carrier_off(efct->net_dev);
	}

	/* Status message for kernel log */
	if (!net_ratelimit())
		return;

	if (link_state->up) {
		netif_info(efct, link, efct->net_dev,
			   "link up at %uMbps %s-duplex (MTU %d)\n",
			   link_state->speed, link_state->fd ? "full" : "half",
			   efct->net_dev->mtu);

	} else if (kernel_link_up) {
		netif_info(efct, link, efct->net_dev, "link down\n");
	}
}

int efct_probe_common(struct efct_nic *efct)
{
	int rc = efct_mcdi_init(efct);

	if (rc) {
		pci_err(efct->efct_dev->pci_dev, "MCDI initialization failed, rc=%d\n", rc);
		goto fail;
	}

	/* Reset (most) configuration for this function */
	rc = efct_mcdi_reset(efct, RESET_TYPE_ALL);
	if (rc) {
		pci_err(efct->efct_dev->pci_dev, "MCDI reset failed, rc=%d\n", rc);
		goto fail;
	}

	/* Enable event logging */
	rc = efct_mcdi_log_ctrl(efct, true, false, 0);
	if (rc) {
		pci_err(efct->efct_dev->pci_dev, "Enabling MCDI Log control failed, rc=%d\n", rc);
		goto fail;
	}

	rc = efct_mcdi_filter_table_probe(efct);
	if (rc) {
		pci_err(efct->efct_dev->pci_dev, "efct_mcdi_filter_table_probe failed, rc=%d\n",
			rc);
		goto fail;
	}

	/*Initialize structure for all the RXQs and TXQ#0*/
	rc = efct_probe_queues(efct);
	if (rc) {
		pci_err(efct->efct_dev->pci_dev, "Queues initialization(efct_probe_queues) failed\n");
		goto free_filter;
	}
	return rc;
free_filter:
	efct_mcdi_filter_table_remove(efct);
fail:
	return rc;
}

void efct_remove_common(struct efct_nic *efct)
{
	efct_remove_queues(efct);
	efct_mcdi_filter_table_remove(efct);
	efct_mcdi_detach(efct);
	efct_mcdi_fini(efct);
}

static int efct_probe_queues(struct efct_nic *efct)
{
	int txq_index[EFCT_MAX_CORE_TX_QUEUES];
	int i = 0, j = 0, k = 0, rc;
	int evq_index;

	if (!efct)
		return -EINVAL;
	/*RXQ init*/
	for (i = 0; i < efct->rxq_count; i++) {
		evq_index = evq_get_free_index(efct);
		if (evq_index < 0) {
			netif_err(efct, probe, efct->net_dev,
				  "Got invalid evq_index for Rx, index %d\n", evq_index);
			rc = evq_index;
			goto fail;
		}

		rc = efct->type->ev_probe(efct, evq_index, efct->rxq[i].num_entries);
		if (rc) {
			netif_err(efct, probe, efct->net_dev,
				  "Event queue probe failed, index %d\n", evq_index);
			if (!test_and_clear_bit(evq_index, &efct->evq_active_mask)) {
				netif_err(efct, probe, efct->net_dev,
					  "Event queue is already removed\n");
			}
			goto fail;
		}
#if !defined(EFCT_NOT_UPSTREAM) || !defined(CONFIG_X3_BUSYPOLL)
		efct->evq[evq_index].msi.idx = i;
#endif
		/*Borrowed from sfc ef10 need to modify*/
		// request irq here.
		efct->evq[evq_index].irq_moderation_ns = EFCT_IRQ_RX_MOD_DEFAULT_VAL;
		efct->evq[evq_index].irq_adapt_low_thresh = EFCT_IRQ_ADAPT_LOW_THRESH_VAL;
		efct->evq[evq_index].irq_adapt_high_thresh = EFCT_IRQ_ADAPT_HIGH_THRESH_VAL;
		efct->evq[evq_index].irq_adapt_irqs = EFCT_IRQ_ADAPT_IRQS;
		/* Setting rx_merge_timeout val as 0 to use the firmware's default value */
		efct->evq[evq_index].rx_merge_timeout_ns = 0;

		/* Add one rxq for an event queue */
		efct->evq[evq_index].queue_count = 1;
		efct->evq[evq_index].queue = (void *)&efct->rxq[i];

		rc = efct->type->rx_probe(efct, i, evq_index);
		if (rc) {
			netif_err(efct, probe, efct->net_dev, "RX queue probe failed, index %d\n", i);
			efct->type->ev_remove(&efct->evq[evq_index]);
			goto fail;
		}
	}

	/*TXQ init*/
	for (j = 0; j < EFCT_MAX_CORE_TX_QUEUES; j++) {
		evq_index = evq_get_free_index(efct);
		if (evq_index < 0) {
			netif_err(efct, probe, efct->net_dev,
				  "Got invalid evq_index for Tx, index %d\n", evq_index);
			rc = evq_index;
			goto fail;
		}

		rc = efct->type->ev_probe(efct, evq_index, efct->txq[j].num_entries);
		if (rc) {
			netif_err(efct, probe, efct->net_dev,
				  "Event queue probe failed, index %d\n", evq_index);
			if (!test_and_clear_bit(evq_index, &efct->evq_active_mask)) {
				netif_err(efct, probe, efct->net_dev,
					  "Event queue is already removed\n");
			}
			goto fail;
		}
#if !defined(EFCT_NOT_UPSTREAM) || !defined(CONFIG_X3_BUSYPOLL)
		efct->evq[evq_index].msi.idx = j + i;
#endif
		efct->evq[evq_index].type = EVQ_T_TX;
		efct->evq[evq_index].irq_moderation_ns = EFCT_IRQ_TX_MOD_DEFAULT_VAL;
		/* Setting tx_merge_timeout val as 0 to use the firmware's default value */
		efct->evq[evq_index].tx_merge_timeout_ns = 0;

		/* Add one txq for an event queue */
		efct->evq[evq_index].queue_count = 1;
		efct->evq[evq_index].queue = (void *)&efct->txq[j];
		txq_index[j] = efct->type->tx_probe(efct, evq_index, -1);
		if (txq_index[j] < 0) {
			netif_err(efct, probe, efct->net_dev,
				  "TX queue probe failed, index %d\n", txq_index[j]);
			efct->type->ev_remove(&efct->evq[evq_index]);
			goto fail;
		}
	}

	return 0;

fail:
	for (k = 0; k < i ; k++) {
		efct->type->rx_remove(&efct->rxq[k]);
		efct->type->ev_remove(&efct->evq[efct->rxq[k].evq_index]);
	}

	for (k = 0; k < j ; k++) {
		efct->type->tx_remove(&efct->txq[txq_index[k]]);
		efct->type->ev_remove(&efct->evq[efct->txq[txq_index[k]].evq_index]);
	}

	return rc;
}

static void efct_remove_queues(struct efct_nic *efct)
{
	int i;

	for (i = 0; i < efct->rxq_count; i++) {
		efct->type->rx_remove(&efct->rxq[i]);
		efct->type->ev_remove(&efct->evq[efct->rxq[i].evq_index]);
	}

	for (i = 0; i < EFCT_MAX_CORE_TX_QUEUES; i++) {
		efct->type->tx_remove(&efct->txq[i]);
		efct->type->ev_remove(&efct->evq[efct->txq[i].evq_index]);
	}
}

void efct_reset_sw_stats(struct efct_nic *efct)
{
	struct efct_rx_queue *rxq;
	int i;

	/*Rx stats reset*/
	for (i = 0; i < efct->rxq_count; i++) {
		rxq = &efct->rxq[i];

		rxq->n_rx_eth_crc_err = 0;
		rxq->n_rx_eth_len_err = 0;
		rxq->n_rx_ip_hdr_chksum_err = 0;
		rxq->n_rx_tcp_udp_chksum_err = 0;
		rxq->n_rx_sentinel_drop_count = 0;
		rxq->n_rx_mcast_mismatch = 0;
		rxq->n_rx_merge_events = 0;
		rxq->n_rx_merge_packets = 0;
		rxq->n_rx_alloc_skb_fail = 0;
		rxq->n_rx_broadcast_drop = 0;
		rxq->n_rx_other_host_drop = 0;
		rxq->n_rx_nbl_empty = 0;
		rxq->n_rx_buffers_posted = 0;
		rxq->n_rx_rollover_events = 0;
		rxq->n_rx_aux_pkts = 0;
		rxq->rx_packets = 0;
	}

	/*Tx stats reset*/
	for (i = 0; i < EFCT_MAX_CORE_TX_QUEUES; i++) {
		efct->txq[i].n_tx_stop_queue = 0;
		efct->txq[i].tx_packets = 0;
	}

	/*EVQ stats reset*/
	for (i = 0; i < (efct->rxq_count + EFCT_MAX_CORE_TX_QUEUES); i++) {
		efct->evq[i].n_evq_time_sync_events = 0;
		efct->evq[i].n_evq_error_events = 0;
		efct->evq[i].n_evq_flush_events = 0;
		efct->evq[i].n_evq_unsol_overflow = 0;
		efct->evq[i].n_evq_unhandled_events = 0;
	}
	/*Resetting generic stats*/
	atomic64_set(&efct->n_rx_sw_drops, 0);
	atomic64_set(&efct->n_tx_sw_drops, 0);
}

int efct_mac_reconfigure(struct efct_nic *efct)
{
	int rc = 0;

	if (efct->type->reconfigure_mac)
		rc = efct->type->reconfigure_mac(efct);

	return rc;
}

int efct_set_mac_address(struct net_device *net_dev, void *data)
{
	struct efct_nic *efct = efct_netdev_priv(net_dev);
	struct sockaddr *addr = data;
	u8 *new_addr = addr->sa_data;
	u8 old_addr[6];
	int rc;

	rc = eth_prepare_mac_addr_change(net_dev, data);
	if (rc)
		return rc;

	ether_addr_copy(old_addr, net_dev->dev_addr); /* save old address */
	eth_hw_addr_set(net_dev, new_addr);

	/* Reconfigure the MAC */
	mutex_lock(&efct->mac_lock);
	rc = efct_mac_reconfigure(efct);
	if (rc)
		eth_hw_addr_set(net_dev, old_addr);
	mutex_unlock(&efct->mac_lock);

	return rc;
}

/* Push loopback/power/transmit disable settings to the PHY, and reconfigure
 * the MAC appropriately. All other PHY configuration changes are pushed
 * through efct_mcdi_phy_set_settings(), and pushed asynchronously to the MAC
 * through efct_monitor().
 *
 * Callers must hold the mac_lock
 */
int __efct_reconfigure_port(struct efct_nic *efct)
{
	enum efct_phy_mode phy_mode;
	int rc = 0;

	WARN_ON(!mutex_is_locked(&efct->mac_lock));

	/* Disable PHY transmit in mac level loopbacks */
	phy_mode = efct->phy_mode;
	if (LOOPBACK_INTERNAL(efct))
		efct->phy_mode |= PHY_MODE_TX_DISABLED;
	else
		efct->phy_mode &= ~PHY_MODE_TX_DISABLED;

	rc = efct_mcdi_port_reconfigure(efct);
	if (rc)
		efct->phy_mode = phy_mode;

	return rc;
}

int efct_change_mtu(struct net_device *net_dev, int new_mtu)
{
	struct efct_nic *efct = efct_netdev_priv(net_dev);
	int old_mtu;
	int rc;

	rc = efct_check_disabled(efct);
	if (rc)
		return rc;

#if defined(EFCT_USE_KCOMPAT) && !(defined(EFCT_HAVE_NETDEV_MTU_LIMITS) || defined(EFCT_HAVE_NETDEV_EXT_MTU_LIMITS))
	if (new_mtu > EFCT_MAX_MTU) {
		netif_err(efct, probe, efct->net_dev,
			  "Requested MTU of %d too big (max: %d)\n",
			  new_mtu, EFCT_MAX_MTU);
		return -EINVAL;
	}
	if (new_mtu < EFCT_MIN_MTU) {
		netif_err(efct, probe, efct->net_dev,
			  "Requested MTU of %d too small (min: %d)\n",
			  new_mtu, EFCT_MIN_MTU);
		return -EINVAL;
	}
#endif

	netif_dbg(efct, drv, efct->net_dev, "changing MTU to %d\n", new_mtu);
	//TODO is device reconfigure required?
	mutex_lock(&efct->mac_lock);
	old_mtu = net_dev->mtu;
	net_dev->mtu = new_mtu;
	rc = efct_mac_reconfigure(efct);
	if (rc) {
		netif_err(efct, probe, efct->net_dev, "efct_mac_reconfigure failed\n");
		net_dev->mtu = old_mtu;
	}
	mutex_unlock(&efct->mac_lock);

	return rc;
}

/* Hook interrupt handler(s)
 */

void efct_set_interrupt_affinity(struct efct_nic *efct)
{
	struct efct_device *efct_dev;
	struct pci_dev *pci_dev;
	u32 cpu;
	int i;

	efct_dev = efct->efct_dev;
	pci_dev = efct->efct_dev->pci_dev;
	for (i = 0; i < efct->max_evq_count; ++i) {
		if (!efct->evq[i].msi.irq)
			continue;
		if (efct_dev->dist_layout == RX_LAYOUT_DISTRIBUTED)
			cpu = cpumask_local_spread(efct->evq[i].msi.idx,
						   pcibus_to_node(pci_dev->bus));
		else
			cpu = cpumask_local_spread(efct_dev->separated_rx_cpu,
						   pcibus_to_node(pci_dev->bus));
		irq_set_affinity_hint(efct->evq[i].msi.irq, cpumask_of(cpu));
		netif_dbg(efct, drv, efct->net_dev,
			  "EVQ : %u IRQ : %u IDX : %u CPU : %u\n",
			  i, efct->evq[i].msi.irq, efct->evq[i].msi.idx, cpu);
	}
}

void efct_clear_interrupt_affinity(struct efct_nic *efct)
{
	int i;

	for (i = 0; i < efct->max_evq_count; ++i) {
		if (!efct->evq[i].msi.irq)
			continue;
		irq_set_affinity_hint(efct->evq[i].msi.irq, NULL);
	}
}

static int efct_is_filter_supported(struct efct_nic *efct,
				    struct efct_mcdi_filter_table *table,
				    struct efct_filter_spec *spec)
{
	unsigned int idx;

	for (idx = 0; idx < table->rx_match_count; idx++)
		if (table->rx_match_fields[idx] == spec->match_fields)
			return true;
	return false;
}

int efct_get_spec_idx(struct efct_nic *efct, struct efct_filter_spec *spec)
{
	struct efct_filter_spec *spec_in_table = NULL;
	struct efct_mcdi_filter_table *table;
	int ins_index = -1;
	u32 num_filter;
	int index;

	num_filter = efct->efct_dev->params.num_filter;
	table = efct->filter_table;
	if (efct_is_unknown_mcast(spec->match_fields)) {
		ins_index = num_filter - 1;
		if (table->entry[ins_index].spec) {
			netif_err(efct, ifup, efct->net_dev,
				  "Filter already exist at Multicast location %u\n", ins_index);
			return -EBUSY;
		}
	}
	for (index = 0;  index < num_filter; index++) {
		//TODO hash table implementation
		spec_in_table = (struct efct_filter_spec *)table->entry[index].spec;
		if (!spec_in_table) {
			if (ins_index < 0)
				ins_index = index;
		} else if (efct_filter_spec_equal(spec, spec_in_table)) {
			ins_index = index;
			break;
		}
	}

	if (ins_index < 0) {
		netif_err(efct, probe, efct->net_dev, "No free index found\n");
		ins_index = -EBUSY;
	}
	return ins_index;
}

int efct_fill_spec(struct efct_nic *efct, const struct ethtool_rx_flow_spec *rule,
		   struct efct_filter_spec *spec, bool logmsg)
{
	const struct ethtool_usrip4_spec *uip_entry = &rule->h_u.usr_ip4_spec;
	const struct ethtool_tcpip4_spec *ip_entry = &rule->h_u.tcp_ip4_spec;
	const struct ethtool_usrip4_spec *uip_mask = &rule->m_u.usr_ip4_spec;
	const struct ethtool_tcpip4_spec *ip_mask = &rule->m_u.tcp_ip4_spec;
	u8 ipv4_eth_mask[ETH_ALEN] = {0xff, 0xff, 0xff, 0x80, 0x00, 0x00};
	u8 eth_mask[ETH_ALEN] = {0x01, 0x00, 0x00, 0x00, 0x00, 0x00};
	const struct ethhdr *mac_entry = &rule->h_u.ether_spec;
	const struct ethhdr *mac_mask = &rule->m_u.ether_spec;
	struct efct_mcdi_filter_table *table;
	u32 num_filter;
	u32 flow_type;
	int rc;

#define LOG_ERROR(flag, msg, ...) \
		do { \
			if (flag)	\
				netif_err(efct, probe, efct->net_dev, msg, ##__VA_ARGS__);\
		} while (0)
	/* Check that user wants us to choose the location */
	rc = -EINVAL;
	if (rule->location != RX_CLS_LOC_ANY) {
		LOG_ERROR(logmsg, "Not supported: Location/ID to insert the rule\n");
		goto err;
	}
	if ((rule->flow_type & FLOW_EXT) && (rule->m_ext.vlan_etype || rule->m_ext.data[0] ||
					     rule->m_ext.data[1])) {
		LOG_ERROR(logmsg, "vlan_etype, data extension unsupported\n");
		goto err;
	}
	flow_type = rule->flow_type & ~FLOW_EXT;
	table = efct->filter_table;
	num_filter = efct->efct_dev->params.num_filter;
	switch (flow_type) {
	case TCP_V4_FLOW:
	case UDP_V4_FLOW:
		if (ip_mask->psrc || ip_mask->ip4src || ip_mask->tos) {
			LOG_ERROR(logmsg, "Source IP, port, tos not supported\n");
			goto err;
		}
		if (rule->flow_type & FLOW_EXT) {
			LOG_ERROR(logmsg, "FLOW_EXT with TCP/UDP not supported\n");
			goto err;
		}
		if (ip_mask->ip4dst == IP4_ADDR_MASK && ip_mask->pdst == PORT_MASK) {
			spec->match_fields = BIT(MC_CMD_FILTER_OP_V3_IN_MATCH_ETHER_TYPE_LBN) |
					     BIT(MC_CMD_FILTER_OP_V3_IN_MATCH_IP_PROTO_LBN) |
					     BIT(MC_CMD_FILTER_OP_V3_IN_MATCH_DST_PORT_LBN) |
					     BIT(MC_CMD_FILTER_OP_V3_IN_MATCH_DST_IP_LBN);

			spec->ether_type = htons(ETH_P_IP);
			spec->ip_proto = flow_type == TCP_V4_FLOW ? IPPROTO_TCP :
									  IPPROTO_UDP;
			spec->dst_ip = ip_entry->ip4dst;
			spec->dst_port = ip_entry->pdst;
		} else {
			LOG_ERROR(logmsg,
				  "Exact match value required for destination IP and port\n");
			goto err;
		}
		break;
	case IP_USER_FLOW:
		if (uip_mask->ip4dst) {
			LOG_ERROR(logmsg, "Destination IP not supported\n");
			return -EINVAL;
		}
		if (uip_mask->ip4src) {
			LOG_ERROR(logmsg, "Source IP not supported\n");
			return -EINVAL;
		}
		if (rule->flow_type & FLOW_EXT) {
			LOG_ERROR(logmsg, "FLOW_EXT with IP proto is not supported\n");
			return -EINVAL;
		}
		spec->match_fields = BIT(MC_CMD_FILTER_OP_V3_IN_MATCH_ETHER_TYPE_LBN);
		spec->ether_type = htons(ETH_P_IP);
		if (uip_mask->proto) {
			if (uip_mask->proto != IP_PROTO_FULL_MASK)
				return -EINVAL;
			spec->match_fields |= BIT(MC_CMD_FILTER_OP_V3_IN_MATCH_IP_PROTO_LBN);
			spec->ip_proto = uip_entry->proto;
		}
		break;

	case ETHER_FLOW:
		if (mac_mask->h_proto) {
			LOG_ERROR(logmsg, "proto not supported for flow-type ether\n");
			goto err;
		}
		if (!is_zero_ether_addr(mac_mask->h_source)) {
			LOG_ERROR(logmsg, "Source mac not supported for flow-type ether\n");
			goto err;
		}
		if (is_multicast_ether_addr(mac_entry->h_dest) &&
		    ether_addr_equal(mac_mask->h_dest, eth_mask)) {
			spec->match_fields =
				BIT(MC_CMD_FILTER_OP_V3_IN_MATCH_UNKNOWN_MCAST_DST_LBN);
		} else if (ether_addr_is_ipv4_mcast(mac_entry->h_dest) &&
			ether_addr_equal(mac_mask->h_dest, ipv4_eth_mask)) {
			spec->match_fields =
				BIT(MC_CMD_FILTER_OP_V3_IN_MATCH_UNKNOWN_IPV4_MCAST_DST_LBN);
		} else if (is_broadcast_ether_addr(mac_mask->h_dest)) {
			spec->match_fields = BIT(MC_CMD_FILTER_OP_V3_IN_MATCH_DST_MAC_LBN);
			memcpy(spec->loc_mac, mac_entry->h_dest, sizeof(spec->loc_mac));
			if ((rule->flow_type & FLOW_EXT) && rule->m_ext.vlan_tci) {
				if (rule->m_ext.vlan_tci != htons(0xfff)) {
					LOG_ERROR(logmsg, "Exact match required for vlan tci\n");
					goto err;
				}
				spec->match_fields |=
					BIT(MC_CMD_FILTER_OP_V3_IN_MATCH_OUTER_VLAN_LBN);
				spec->outer_vid = rule->h_ext.vlan_tci;
			}
		} else {
			LOG_ERROR(logmsg, "Exact match required for ether addr\n");
			goto err;
		}
		break;
	default:
		LOG_ERROR(logmsg, "Flow type 0X%x not supported\n", rule->flow_type);
		goto err;
	}
	if (!efct_is_filter_supported(efct, table, spec))
		return -EPROTONOSUPPORT;
#undef LOG_ERROR
	rc = 0;
err:
	return rc;
}

int efct_delete_rule(struct efct_nic *efct, u32 id, u64 *qid)
{
	struct efct_filter_spec *spec_in_table;
	struct efct_mcdi_filter_table *table;
	int rc;

	if (id >= efct->efct_dev->params.num_filter)
		return -EINVAL;
	table = efct->filter_table;
	if (!table || !table->entry) {
		netif_err(efct, probe, efct->net_dev,
			  "Invlid filter table\n");
		return -EINVAL;
	}

	down_write(&table->lock);

	if (table->entry[id].handle == EFCT_HANDLE_INVALID) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid filter is passed, id: %d\n", id);
		rc = -EINVAL;
		goto out;
	}
	spec_in_table = (struct efct_filter_spec *)table->entry[id].spec;
	table->entry[id].ref_cnt--;
	if (table->entry[id].ref_cnt) {
		netif_dbg(efct, drv, efct->net_dev,
			  "There are other active clients for this filter, id: %d\n", id);
		rc = 0;
		if (qid)
			*qid = spec_in_table->queue_id;
		goto out;
	}
	rc = efct_mcdi_filter_remove(efct, table->entry[id].handle);
	if (rc) {
		netif_err(efct, probe, efct->net_dev,
			  "efct_mcdi_filter_remove failed, rc: %d\n", rc);
		goto out;
	}
	if (spec_in_table->queue_id != RX_CLS_FLOW_DISC) {
		efct->rxq[spec_in_table->queue_id].filter_count--;
	/*Removing exclusive client if filter count is zero*/
#ifdef CONFIG_XILINX_AUX_EFCT
		if (!efct->rxq[spec_in_table->queue_id].filter_count)
			efct->rxq[spec_in_table->queue_id].exclusive_client = -1;
#endif
	}
	if (qid)
		*qid = spec_in_table->queue_id;
	kfree(spec_in_table);
	table->entry[id].spec = (unsigned long)NULL;
	table->entry[id].handle = EFCT_HANDLE_INVALID;
out:
	up_write(&table->lock);
	return rc;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_NDO_GET_PHYS_PORT_ID)
int efct_get_phys_port_id(struct net_device *net_dev, struct netdev_phys_item_id *ppid)
{
	struct efct_nic *efct = efct_netdev_priv(net_dev);

	if (efct->type->get_phys_port_id)
		return efct->type->get_phys_port_id(efct, ppid);
	else
		return -EOPNOTSUPP;
}
#endif

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_NDO_GET_PHYS_PORT_NAME)
int efct_get_phys_port_name(struct net_device *net_dev, char *name, size_t len)
{
	struct efct_nic *efct = efct_netdev_priv(net_dev);

	if (snprintf(name, len, "p%u", efct->port_num) >= len)
		return -EINVAL;

	return 0;
}
#endif

static void efct_stats_collector(struct work_struct *work)
{
	struct efct_device *efct_dev;
	struct delayed_work *dwork;
	struct efct_nic *efct;
	int i;

	dwork = to_delayed_work(work);
	efct_dev = container_of(dwork, struct efct_device, stats_work);
	for (i = 0; i < efct_dev->num_ports; i++) {
		efct = efct_dev->efct[i];
		efct->type->update_stats(efct, false);
	}
	queue_delayed_work(efct_dev->stats_wq, &efct_dev->stats_work,
			   msecs_to_jiffies(STAT_INTERVAL_MS));
}

int efct_init_stats_wq(struct efct_device *efct_dev)
{
	struct pci_dev *pci_dev = efct_dev->pci_dev;
	struct efct_nic *efct;
	u8 bus, dev, fun;
	int i;

	bus = pci_dev->bus->number;
	dev = PCI_SLOT(pci_dev->devfn);
	fun = PCI_FUNC(pci_dev->devfn);

	/* Create ktrhead for mac stat pulling */
	INIT_DELAYED_WORK(&efct_dev->stats_work, efct_stats_collector);
	efct_dev->stats_wq = alloc_ordered_workqueue("efctsc/%02hhx:%02hhx.%hhx", 0, bus, dev, fun);
	if (!efct_dev->stats_wq) {
		pci_err(pci_dev, "Failed to create stats workqueue\n");
		return -EINVAL;
	}

	/* Pull initial stats */
	for (i = 0; i < efct_dev->num_ports; i++) {
		efct = efct_dev->efct[i];
		efct->type->pull_stats(efct);
		efct->type->update_stats(efct, true);
	}

	queue_delayed_work(efct_dev->stats_wq, &efct_dev->stats_work,
			   msecs_to_jiffies(STAT_INTERVAL_MS));

	return 0;
}

void efct_fini_stats_wq(struct efct_device *efct_dev)
{
	if (efct_dev->stats_wq) {
		cancel_delayed_work_sync(&efct_dev->stats_work);
		destroy_workqueue(efct_dev->stats_wq);
		efct_dev->stats_wq = NULL;
	}
}

int efct_filter_table_down(struct efct_nic *efct)
{
	struct efct_mcdi_filter_table *table = efct->filter_table;
	struct efct_filter_spec *spec_in_table;
	int index, rc;

	if (!table)
		return 0;
	down_write(&table->lock);

	for (index = 0; index < efct->efct_dev->params.num_filter; index++) {
		spec_in_table = (struct efct_filter_spec *)table->entry[index].spec;
		if (!spec_in_table) {
			continue;
		} else {
			rc = efct_mcdi_filter_remove(efct, table->entry[index].handle);
			if (rc) {
				up_write(&table->lock);
				netif_err(efct, probe, efct->net_dev,
					  "efct_mcdi_filter_remove failed, rc: %d\n", rc);
				return rc;
			}
			table->entry[index].handle = EFCT_HANDLE_INVALID;
		}
	}

	up_write(&table->lock);

	return 0;
}

int efct_filter_table_up(struct efct_nic *efct)
{
	struct efct_mcdi_filter_table *table = efct->filter_table;
	struct efct_filter_spec *spec_in_table;
	u64 handle = 0;
	int index, rc;

	if (!table)
		return 0;

	down_write(&table->lock);

	for (index = 0; index < efct->efct_dev->params.num_filter; index++) {
		spec_in_table = (struct efct_filter_spec *)table->entry[index].spec;
		if (!spec_in_table) {
			continue;
		} else {
			rc = efct_mcdi_filter_insert(efct, spec_in_table, &handle);
			if (rc) {
				up_write(&table->lock);
				netif_err(efct, probe, efct->net_dev,
					  "efct_mcdi_filter_remove failed, rc: %d\n", rc);
				return rc;
			}

			table->entry[index].handle = handle;
		}
	}

	up_write(&table->lock);

	return 0;
}

