/* SPDX-License-Identifier: GPL-2.0
 ****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#ifndef EFCT_COMMON_H
#define EFCT_COMMON_H

#include <linux/workqueue.h>
#include <linux/rwsem.h>
#include "efct_driver.h"
#include "mcdi_pcol.h"

struct efct_func_ctl_window {
	bool valid;
	u32 bar;
	u64 offset;
};

struct queue_reset_work {
	struct work_struct work;
	struct efct_nic *efct;
	bool is_txq;
	int qid;
};

#define IP4_ADDR_MASK	((__force __be32)~0)
#define PORT_MASK		((__force __be16)~0)
#define MULTICAST_DST_MASK htonl(0xf0000000)
#define MULTICAST_ADDR_START htonl(0xe0000000)
#define EFCT_HANDLE_INVALID 0xFFFFFFFFFFFFFFFF
#define MAC_MULTICAST_DST_MASK   0x0000010000000000
#define MAC_MULTICAST_DST_START  0x0000010000000000
#define IPV4_MCAST_DST_START	 0x000001005E000000
#define IP_PROTO_FULL_MASK	0xFF

struct efct_filter_spec {
	u32 match_fields;
	u64 queue_id;
	__be16	ether_type;
	__be32 dst_ip;
	__be16	dst_port;
	__be16  outer_vid;
	u8 loc_mac[ETH_ALEN];
	u8 ip_proto;
};

struct efct_mcdi_filter_table {
	/* The MCDI match masks supported by this fw & hw, in order of priority */
	u32 rx_match_fields[MC_CMD_GET_PARSER_DISP_INFO_OUT_SUPPORTED_MATCHES_MAXNUM];
	unsigned int rx_match_count;
	struct rw_semaphore lock; /* Protects entries */
	struct {
		unsigned long spec;     /* pointer to spec */
		u64 handle;	     /* firmware handle */
		u64 ref_cnt; /*ref count for clients*/
	} *entry;
};

/* Global Resources */
int efct_nic_alloc_buffer(struct efct_nic *efct, struct efct_buffer *buffer,
			  u32 len, gfp_t gfp_flags);
void efct_nic_free_buffer(struct efct_nic *efct, struct efct_buffer *buffer);

static inline bool efct_nic_mcdi_ev_pending(struct efct_nic *efct, u16 index)
{
	return efct->type->ev_mcdi_pending(efct, index);
}

static inline bool efct_is_unknown_mcast(u32 match_field)
{
	return (match_field == BIT(MC_CMD_FILTER_OP_V3_IN_MATCH_UNKNOWN_MCAST_DST_LBN) ||
		match_field == BIT(MC_CMD_FILTER_OP_V3_IN_MATCH_UNKNOWN_IPV4_MCAST_DST_LBN));
}

static inline bool efct_nic_has_dynamic_sensors(struct efct_nic *efct)
{
	if (efct->type->has_dynamic_sensors)
		return efct->type->has_dynamic_sensors(efct);

	return false;
}

int efct_init_io(struct efct_device *efct_dev,
		 int bar, dma_addr_t dma_mask, u32 mem_map_size);
void efct_fini_io(struct efct_device *efct_dev);
void efct_init_struct(struct efct_nic *efct);
int efct_map_membase(struct efct_nic *efct, u32 fcw_offset, u8 port_id);
void efct_unmap_membase(struct efct_nic *efct);
#ifdef CONFIG_XILINX_AUX_EFCT
int efct_make_irq_resources(struct efct_nic *efct, struct msix_entry *xentries, u8 index);
#endif
int efct_probe_common(struct efct_nic *efct);
void efct_remove_common(struct efct_nic *efct);
void efct_schedule_reset(struct efct_nic *efct, enum reset_type type);
int efct_create_reset_workqueue(void);
void efct_destroy_reset_workqueue(void);
void efct_flush_reset_workqueue(void);
void efct_reset_sw_stats(struct efct_nic *efct);
void efct_set_evq_names(struct efct_nic *efct);
void efct_link_status_changed(struct efct_nic *efct);
int efct_mac_reconfigure(struct efct_nic *efct);
int efct_set_mac_address(struct net_device *net_dev, void *data);
int __efct_reconfigure_port(struct efct_nic *efct);
int efct_reset(struct efct_nic *efct, enum reset_type method);
void efct_schedule_queue_reset(struct efct_nic *efct, bool is_txq, int qid);
int efct_change_mtu(struct net_device *net_dev, int new_mtu);
void efct_set_interrupt_affinity(struct efct_nic *efct);
void efct_clear_interrupt_affinity(struct efct_nic *efct);
int efct_fill_spec(struct efct_nic *efct, const struct ethtool_rx_flow_spec *rule,
		   struct efct_filter_spec *spec, bool logmsg);
int efct_get_spec_idx(struct efct_nic *efct, struct efct_filter_spec *spec);
int efct_delete_rule(struct efct_nic *efct, u32 id, u64 *qid);
int efct_init_stats_wq(struct efct_device *efct_dev);
void efct_fini_stats_wq(struct efct_device *efct_dev);
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_NDO_GET_PHYS_PORT_ID)
int efct_get_phys_port_id(struct net_device *net_dev, struct netdev_phys_item_id *ppid);
#endif
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_NDO_GET_PHYS_PORT_NAME)
int efct_get_phys_port_name(struct net_device *net_dev, char *name, size_t len);
#endif
int efct_filter_table_down(struct efct_nic *efct);
int efct_filter_table_up(struct efct_nic *efct);
static inline bool efct_filter_spec_equal(const struct efct_filter_spec *left,
					  const struct efct_filter_spec *right)
{
	return memcmp(&left->dst_ip, &right->dst_ip,
		      sizeof(struct efct_filter_spec) -
		      offsetof(struct efct_filter_spec, dst_ip)) == 0;
}

static inline int efct_check_disabled(struct efct_nic *efct)
{
	if (efct->state == STATE_DISABLED) {
		netif_err(efct, probe, efct->net_dev,
			  "device is disabled due to earlier errors\n");
		return -EIO;
	}
	return 0;
}
#endif
