// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include <linux/rtc.h>
#include "efct_devlink.h"
#include "mcdi.h"
#include "mcdi_functions.h"
#include "mcdi_pcol.h"
#include "efct_reflash.h"
#include "efct_netdev.h"

/* Custom devlink-info version object names for details that do not map to the
 * generic standardized names.
 */
#define EFCT_DEVLINK_INFO_VERSION_FW_MGMT_SUC	"fw.mgmt.suc"
#define EFCT_DEVLINK_INFO_VERSION_FW_MGMT_CMC	"fw.mgmt.cmc"
#define EFCT_DEVLINK_INFO_VERSION_FPGA_REV	"fpga.rev"
#define EFCT_DEVLINK_INFO_VERSION_DATAPATH_HW	"fpga.app"
#define EFCT_DEVLINK_INFO_VERSION_DATAPATH_FW	DEVLINK_INFO_VERSION_GENERIC_FW_APP
#define EFCT_DEVLINK_INFO_VERSION_SOC_BOOT	"coproc.boot"
#define EFCT_DEVLINK_INFO_VERSION_SOC_UBOOT	"coproc.uboot"
#define EFCT_DEVLINK_INFO_VERSION_SOC_MAIN	"coproc.main"
#define EFCT_DEVLINK_INFO_VERSION_SOC_RECOVERY	"coproc.recovery"
#define EFCT_DEVLINK_INFO_VERSION_FW_EXPROM	"fw.exprom"
#define EFCT_DEVLINK_INFO_VERSION_FW_UEFI	"fw.uefi"

static int efct_devlink_info_user_cfg(struct efct_nic *efct,
				      struct devlink_info_req *req)
{
	char *buf, *desc;
	u16 version[4];
	int offset;
	int rc;

	offset = 0;
	buf = kmalloc(MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_MAXNUM_MCDI2 +
		      EFCT_MAX_VERSION_INFO_LEN, GFP_KERNEL);
	if (!buf)
		return -ENOMEM;

	desc = kmalloc(MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_MAXNUM_MCDI2, GFP_KERNEL);
	if (!desc) {
		kfree(buf);
		return -ENOMEM;
	}

	rc = efct_mcdi_nvram_metadata(efct, NVRAM_PARTITION_TYPE_USER_CONFIG, NULL, version,
				      desc, MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_MAXNUM_MCDI2);
	if (rc)
		goto out;
	if (version[0] || version[1] || version[2] || version[3])
		offset = snprintf(buf, (EFCT_MAX_VERSION_INFO_LEN +
				  MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_MAXNUM_MCDI2),
				  "%u.%u.%u.%u ", version[0], version[1], version[2], version[3]);

	snprintf(buf + offset, (EFCT_MAX_VERSION_INFO_LEN +
				MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_MAXNUM_MCDI2),
		 "%s", desc);

	devlink_info_version_stored_put(req, DEVLINK_INFO_VERSION_GENERIC_FW_PSID, buf);

out:
	kfree(buf);
	kfree(desc);
	return rc;
}

static int efct_devlink_info_nvram_partition(struct efct_nic *efct,
					     struct devlink_info_req *req,
					    u32 partition_type,
					    const char *version_name)
{
	char buf[EFCT_MAX_VERSION_INFO_LEN];
	u16 version[4];
	int rc;

	rc = efct_mcdi_nvram_metadata(efct, partition_type, NULL, version, NULL, 0);
	if (rc)
		return rc;

	snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u.%u.%u.%u", version[0],
		 version[1], version[2], version[3]);
	devlink_info_version_stored_put(req, version_name, buf);

	return 0;
}

static void efct_devlink_info_stored_versions(struct efct_nic *efct,
					      struct devlink_info_req *req)
{
	efct_devlink_info_nvram_partition(efct, req, NVRAM_PARTITION_TYPE_BUNDLE,
					  DEVLINK_INFO_VERSION_GENERIC_FW_BUNDLE_ID);
	efct_devlink_info_nvram_partition(efct, req,
					  NVRAM_PARTITION_TYPE_MC_FIRMWARE,
					 DEVLINK_INFO_VERSION_GENERIC_FW_MGMT);
	efct_devlink_info_nvram_partition(efct, req,
					  NVRAM_PARTITION_TYPE_SUC_FIRMWARE,
					 EFCT_DEVLINK_INFO_VERSION_FW_MGMT_SUC);
	efct_devlink_info_nvram_partition(efct, req,
					  NVRAM_PARTITION_TYPE_EXPANSION_ROM,
					 EFCT_DEVLINK_INFO_VERSION_FW_EXPROM);
	efct_devlink_info_nvram_partition(efct, req,
					  NVRAM_PARTITION_TYPE_EXPANSION_UEFI,
					 EFCT_DEVLINK_INFO_VERSION_FW_UEFI);
	efct_devlink_info_user_cfg(efct, req);
}

#define EFCT_MAX_SERIALNUM_LEN	(ETH_ALEN * 2 + 1)

static void efct_devlink_info_board_cfg(struct efct_nic *efct,
					struct devlink_info_req *req)
{
	char sn[EFCT_MAX_SERIALNUM_LEN];
	u8 maddr[ETH_ALEN];
	u32 mac;
	int rc;

	mac = 0;
	rc = efct_get_mac_address(efct, maddr);
	if (rc)
		return;
	mac  = maddr[3];
	mac = (mac << 8) | maddr[4];
	mac = (mac << 8) | maddr[5];
	/* Calculate base mac address of device*/
	mac -= efct->port_num;
	maddr[3] = (mac >> 16) & 0xff;
	maddr[4] = (mac >> 8) & 0xff;
	maddr[5] = mac & 0xff;

	snprintf(sn, EFCT_MAX_SERIALNUM_LEN, "%pm", maddr);
	devlink_info_serial_number_put(req, sn);
}

#define EFCT_VER_FLAG(_f)	\
	(MC_CMD_GET_VERSION_V5_OUT_ ## _f ## _PRESENT_LBN)

static void efct_devlink_info_running_versions(struct efct_nic *efct,
					       struct devlink_info_req *req)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_VERSION_V5_OUT_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_GET_VERSION_EXT_IN_LEN);
	char buf[EFCT_MAX_VERSION_INFO_LEN];
	struct rtc_time build_date;
	size_t outlength, offset;
	const __le32 *dwords;
	const __le16 *words;
	u32 flags, build_id;
	const char *str;
	u64 tstamp;
	int rc;

	rc = efct_mcdi_rpc(efct, MC_CMD_GET_VERSION, inbuf, sizeof(inbuf),
			   outbuf, sizeof(outbuf), &outlength);
	if (rc || outlength < MC_CMD_GET_VERSION_V5_OUT_LEN)
		return;

	/* Handle previous output */
	words = (__le16 *)MCDI_PTR(outbuf,
				       GET_VERSION_V5_OUT_VERSION);
	offset = snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u.%u.%u.%u",
			  le16_to_cpu(words[0]),
			  le16_to_cpu(words[1]),
			  le16_to_cpu(words[2]),
			  le16_to_cpu(words[3]));

	devlink_info_version_running_put(req, DEVLINK_INFO_VERSION_GENERIC_FW_MGMT, buf);

	flags = MCDI_DWORD(outbuf, GET_VERSION_V5_OUT_FLAGS);
	if (flags & BIT(EFCT_VER_FLAG(BOARD_EXT_INFO))) {
		snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%s",
			 MCDI_PTR(outbuf, GET_VERSION_V5_OUT_BOARD_NAME));
		devlink_info_version_fixed_put(req, DEVLINK_INFO_VERSION_GENERIC_BOARD_ID, buf);
		/* Favour full board version if present (in V5 or later) */
		if (~flags & BIT(EFCT_VER_FLAG(BOARD_VERSION))) {
			snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u",
				 MCDI_DWORD(outbuf, GET_VERSION_V5_OUT_BOARD_REVISION));
			devlink_info_version_fixed_put(req, DEVLINK_INFO_VERSION_GENERIC_BOARD_REV,
						       buf);
		}

		str = MCDI_PTR(outbuf, GET_VERSION_V5_OUT_BOARD_SERIAL);
		if (str[0])
			devlink_info_board_serial_number_put(req, str);
	}

	if (flags & BIT(EFCT_VER_FLAG(FPGA_EXT_INFO))) {
		dwords = (__le32 *)MCDI_PTR(outbuf,
						GET_VERSION_V5_OUT_FPGA_VERSION);
		offset = snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u_%c%u",
				  le32_to_cpu(dwords[0]),
				  'A' + le32_to_cpu(dwords[1]),
				  le32_to_cpu(dwords[2]));

		str = MCDI_PTR(outbuf, GET_VERSION_V5_OUT_FPGA_EXTRA);
		if (str[0])
			snprintf(&buf[offset], EFCT_MAX_VERSION_INFO_LEN - offset,
				 " (%s)", str);

		devlink_info_version_running_put(req, EFCT_DEVLINK_INFO_VERSION_FPGA_REV, buf);
	}

	if (flags & BIT(EFCT_VER_FLAG(CMC_EXT_INFO))) {
		dwords = (__le32 *)MCDI_PTR(outbuf, GET_VERSION_V5_OUT_CMCFW_VERSION);
		offset = snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u.%u.%u.%u",
				  le32_to_cpu(dwords[0]),
				  le32_to_cpu(dwords[1]),
				  le32_to_cpu(dwords[2]),
				  le32_to_cpu(dwords[3]));

		tstamp = MCDI_QWORD(outbuf, GET_VERSION_V5_OUT_CMCFW_BUILD_DATE);
		if (tstamp) {
			rtc_time64_to_tm(tstamp, &build_date);
			snprintf(&buf[offset], EFCT_MAX_VERSION_INFO_LEN - offset,
				 " (%ptRd)", &build_date);
		}

		devlink_info_version_running_put(req, EFCT_DEVLINK_INFO_VERSION_FW_MGMT_CMC, buf);
	}

	words = (__le16 *)MCDI_PTR(outbuf, GET_VERSION_V5_OUT_VERSION);
	offset = snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u.%u.%u.%u",
			  le16_to_cpu(words[0]), le16_to_cpu(words[1]),
			  le16_to_cpu(words[2]), le16_to_cpu(words[3]));
	if (flags & BIT(EFCT_VER_FLAG(MCFW_EXT_INFO))) {
		build_id = MCDI_DWORD(outbuf, GET_VERSION_V5_OUT_MCFW_BUILD_ID);
		snprintf(&buf[offset], EFCT_MAX_VERSION_INFO_LEN - offset,
			 " (%x) %s", build_id,
			 MCDI_PTR(outbuf, GET_VERSION_V5_OUT_MCFW_BUILD_NAME));
	}
	devlink_info_version_running_put(req, DEVLINK_INFO_VERSION_GENERIC_FW_MGMT, buf);

	if (flags & BIT(EFCT_VER_FLAG(SUCFW_EXT_INFO))) {
		dwords = (__le32 *)MCDI_PTR(outbuf,	GET_VERSION_V5_OUT_SUCFW_VERSION);
		tstamp = MCDI_QWORD(outbuf, GET_VERSION_V5_OUT_SUCFW_BUILD_DATE);
		rtc_time64_to_tm(tstamp, &build_date);
		build_id = MCDI_DWORD(outbuf, GET_VERSION_V5_OUT_SUCFW_CHIP_ID);

		snprintf(buf, EFCT_MAX_VERSION_INFO_LEN,
			 "%u.%u.%u.%u type %x (%ptRd)",
			 le32_to_cpu(dwords[0]), le32_to_cpu(dwords[1]),
			 le32_to_cpu(dwords[2]), le32_to_cpu(dwords[3]),
			 build_id, &build_date);

		devlink_info_version_running_put(req, EFCT_DEVLINK_INFO_VERSION_FW_MGMT_SUC, buf);
	}

	if (flags & BIT(EFCT_VER_FLAG(DATAPATH_HW_VERSION))) {
		dwords = (__le32 *)MCDI_PTR(outbuf,	GET_VERSION_V5_OUT_DATAPATH_HW_VERSION);

		snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u.%u.%u",
			 le32_to_cpu(dwords[0]), le32_to_cpu(dwords[1]),
			 le32_to_cpu(dwords[2]));

		devlink_info_version_running_put(req, EFCT_DEVLINK_INFO_VERSION_DATAPATH_HW, buf);
	}

	if (flags & BIT(EFCT_VER_FLAG(DATAPATH_FW_VERSION))) {
		dwords = (__le32 *)MCDI_PTR(outbuf, GET_VERSION_V5_OUT_DATAPATH_FW_VERSION);

		snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u.%u.%u",
			 le32_to_cpu(dwords[0]), le32_to_cpu(dwords[1]),
			 le32_to_cpu(dwords[2]));

		devlink_info_version_running_put(req, EFCT_DEVLINK_INFO_VERSION_DATAPATH_FW,
						 buf);
	}

	if (flags & BIT(EFCT_VER_FLAG(SOC_BOOT_VERSION))) {
		dwords = (__le32 *)MCDI_PTR(outbuf, GET_VERSION_V5_OUT_SOC_BOOT_VERSION);

		snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u.%u.%u.%u",
			 le32_to_cpu(dwords[0]), le32_to_cpu(dwords[1]),
			 le32_to_cpu(dwords[2]),
			 le32_to_cpu(dwords[3]));

		devlink_info_version_running_put(req, EFCT_DEVLINK_INFO_VERSION_SOC_BOOT, buf);
	}

	if (flags & BIT(EFCT_VER_FLAG(SOC_UBOOT_VERSION))) {
		dwords = (__le32 *)MCDI_PTR(outbuf, GET_VERSION_V5_OUT_SOC_UBOOT_VERSION);

		snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u.%u.%u.%u",
			 le32_to_cpu(dwords[0]), le32_to_cpu(dwords[1]),
			 le32_to_cpu(dwords[2]),
			 le32_to_cpu(dwords[3]));

		devlink_info_version_running_put(req, EFCT_DEVLINK_INFO_VERSION_SOC_UBOOT, buf);
	}

	if (flags & BIT(EFCT_VER_FLAG(SOC_MAIN_ROOTFS_VERSION))) {
		dwords = (__le32 *)MCDI_PTR(outbuf, GET_VERSION_V5_OUT_SOC_MAIN_ROOTFS_VERSION);

		snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u.%u.%u.%u",
			 le32_to_cpu(dwords[0]), le32_to_cpu(dwords[1]),
			 le32_to_cpu(dwords[2]),
			 le32_to_cpu(dwords[3]));

		devlink_info_version_running_put(req, EFCT_DEVLINK_INFO_VERSION_SOC_MAIN, buf);
	}

	if (flags & BIT(EFCT_VER_FLAG(SOC_RECOVERY_BUILDROOT_VERSION))) {
		dwords = (__le32 *)MCDI_PTR(outbuf,
				GET_VERSION_V5_OUT_SOC_RECOVERY_BUILDROOT_VERSION);

		snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u.%u.%u.%u",
			 le32_to_cpu(dwords[0]), le32_to_cpu(dwords[1]),
			 le32_to_cpu(dwords[2]),
			 le32_to_cpu(dwords[3]));

		devlink_info_version_running_put(req, EFCT_DEVLINK_INFO_VERSION_SOC_RECOVERY, buf);
	}

	if (flags & BIT(EFCT_VER_FLAG(SUCFW_VERSION)) &&
	    ~flags & BIT(EFCT_VER_FLAG(SUCFW_EXT_INFO))) {
		dwords = (__le32 *)MCDI_PTR(outbuf, GET_VERSION_V5_OUT_SUCFW_VERSION);
		snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u.%u.%u.%u",
			 le32_to_cpu(dwords[0]), le32_to_cpu(dwords[1]),
			 le32_to_cpu(dwords[2]),
			 le32_to_cpu(dwords[3]));
		devlink_info_version_running_put(req, EFCT_DEVLINK_INFO_VERSION_FW_MGMT_SUC, buf);
	}

	if (flags & BIT(EFCT_VER_FLAG(BOARD_VERSION))) {
		dwords = (__le32 *)MCDI_PTR(outbuf, GET_VERSION_V5_OUT_BOARD_VERSION);

		snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u.%u.%u.%u",
			 le32_to_cpu(dwords[0]), le32_to_cpu(dwords[1]),
			 le32_to_cpu(dwords[2]),
			 le32_to_cpu(dwords[3]));

		devlink_info_version_running_put(req, DEVLINK_INFO_VERSION_GENERIC_BOARD_REV, buf);
	}

	if (flags & BIT(EFCT_VER_FLAG(BUNDLE_VERSION))) {
		dwords = (__le32 *)MCDI_PTR(outbuf, GET_VERSION_V5_OUT_BUNDLE_VERSION);

		snprintf(buf, EFCT_MAX_VERSION_INFO_LEN, "%u.%u.%u.%u",
			 le32_to_cpu(dwords[0]), le32_to_cpu(dwords[1]),
			 le32_to_cpu(dwords[2]),
			 le32_to_cpu(dwords[3]));

		devlink_info_version_running_put(req, DEVLINK_INFO_VERSION_GENERIC_FW_BUNDLE_ID,
						 buf);
	}
}

#undef EFCT_VER_FLAG

static void efct_devlink_info_query_all(struct efct_nic *efct,
					struct devlink_info_req *req)
{
#if defined(EFCT_USE_KCOMPAT) && defined(EFCT_HAVE_INFO_DRIVER_NAME_PUT)
	devlink_info_driver_name_put(req, efct->efct_dev->pci_dev->driver->name);
#endif
	efct_devlink_info_board_cfg(efct, req);
	efct_devlink_info_stored_versions(efct, req);
	efct_devlink_info_running_versions(efct, req);
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_USE_DEVLINK)

enum efct_devlink_param_id {
	EFCT_DEVLINK_PARAM_ID_BASE = DEVLINK_PARAM_GENERIC_ID_MAX,
	EFCT_DEVLINK_PARAM_ID_CT_THRESH,
	EFCT_DEVLINK_PARAM_ID_DIST_LAYOUT,
	EFCT_DEVLINK_PARAM_ID_SEPARATED_CPU,
	EFCT_DEVLINK_PARAM_ID_IRQ_ADAPT_LOW_THRESH,
	EFCT_DEVLINK_PARAM_ID_IRQ_ADAPT_HIGH_THRESH,
	EFCT_DEVLINK_PARAM_ID_IRQ_ADAPT_IRQS,
	EFCT_DEVLINK_PARAM_ID_RX_MERGE_TIMEOUT_NS,
	EFCT_DEVLINK_PARAM_ID_TX_MERGE_TIMEOUT_NS,
	EFCT_DEVLINK_PARAM_ID_NUM_PORTS_PF,
};

static int efct_irq_adapt_low_thresh_param_get(struct devlink *dl, u32 id,
					       struct devlink_param_gset_ctx *ctx)
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct = (*efct_dev)->efct[0];

	ctx->val.vu32 = efct->evq[0].irq_adapt_low_thresh;

	return 0;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_SET_EXTACK)
static int efct_irq_adapt_low_thresh_param_set(struct devlink *dl, u32 id,
					       struct devlink_param_gset_ctx *ctx,
						   struct netlink_ext_ack *extack)
#else
static int efct_irq_adapt_low_thresh_param_set(struct devlink *dl, u32 id,
					       struct devlink_param_gset_ctx *ctx)
#endif
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct;
	int i, j;

	for (i = 0; i < (*efct_dev)->num_ports; i++) {
		efct = (*efct_dev)->efct[i];
		for (j = 0; j < efct->max_evq_count; j++)
			efct->evq[j].irq_adapt_low_thresh = ctx->val.vu32;
	}
	return 0;
}

static int efct_irq_adapt_high_thresh_param_get(struct devlink *dl, u32 id,
						struct devlink_param_gset_ctx *ctx)
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct = (*efct_dev)->efct[0];

	ctx->val.vu32 = efct->evq[0].irq_adapt_high_thresh;

	return 0;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_SET_EXTACK)
static int efct_irq_adapt_high_thresh_param_set(struct devlink *dl, u32 id,
						struct devlink_param_gset_ctx *ctx,
						struct netlink_ext_ack *extack)
#else
static int efct_irq_adapt_high_thresh_param_set(struct devlink *dl, u32 id,
						struct devlink_param_gset_ctx *ctx)
#endif
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct;
	int i, j;

	for (i = 0; i < (*efct_dev)->num_ports; i++) {
		efct = (*efct_dev)->efct[i];
		for (j = 0; j < efct->max_evq_count; j++)
			efct->evq[j].irq_adapt_high_thresh = ctx->val.vu32;
	}
	return 0;
}

static int efct_irq_adapt_irqs_param_get(struct devlink *dl, u32 id,
					 struct devlink_param_gset_ctx *ctx)
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct = (*efct_dev)->efct[0];

	ctx->val.vu32 = efct->evq[0].irq_adapt_irqs;

	return 0;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_SET_EXTACK)
static int efct_irq_adapt_irqs_param_set(struct devlink *dl, u32 id,
					 struct devlink_param_gset_ctx *ctx,
					 struct netlink_ext_ack *extack)
#else
static int efct_irq_adapt_irqs_param_set(struct devlink *dl, u32 id,
					 struct devlink_param_gset_ctx *ctx)
#endif
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct;
	int i, j;

	for (i = 0; i < (*efct_dev)->num_ports; i++) {
		efct = (*efct_dev)->efct[i];
		for (j = 0; j < efct->max_evq_count; j++)
			efct->evq[j].irq_adapt_irqs = ctx->val.vu32;
	}
	return 0;
}

static int efct_rx_merge_timeout_get(struct devlink *dl, u32 id,
				     struct devlink_param_gset_ctx *ctx)
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct = (*efct_dev)->efct[0];

	ctx->val.vu32 = efct->evq[0].rx_merge_timeout_ns;

	return 0;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_SET_EXTACK)
static int efct_rx_merge_timeout_set(struct devlink *dl, u32 id,
				     struct devlink_param_gset_ctx *ctx,
					 struct netlink_ext_ack *extack)
#else
static int efct_rx_merge_timeout_set(struct devlink *dl, u32 id,
				     struct devlink_param_gset_ctx *ctx)
#endif
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct;
	int i, j;

	for (i = 0; i < (*efct_dev)->num_ports; i++) {
		efct = (*efct_dev)->efct[i];
		for (j = 0; j < efct->max_evq_count; j++)
			efct->evq[j].rx_merge_timeout_ns = ctx->val.vu32;
	}
	return 0;
}

static int efct_rx_merge_timeout_validate(struct devlink *dl, u32 id, union devlink_param_value val,
					  struct netlink_ext_ack *extack)
{
	if (val.vu32 > MAX_RX_MERGE_TIMEOUT_NS_VALUE) {
		pr_info("Receive event merge timeout too large for X3, It shall be less than %u\n",
			MAX_RX_MERGE_TIMEOUT_NS_VALUE);
		return -EINVAL;
	}

	if (val.vu32 > 0 && (val.vu32 % EV_TIMER_GRANULARITY_NS != 0)) {
		pr_info("Receive event merge timeout must be a multiple of %u\n",
			EV_TIMER_GRANULARITY_NS);
		return -EINVAL;
	}

	return 0;
}

static int efct_tx_merge_timeout_get(struct devlink *dl, u32 id, struct devlink_param_gset_ctx *ctx)
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct = (*efct_dev)->efct[0];

	ctx->val.vu32 = efct->evq[0].tx_merge_timeout_ns;

	return 0;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_SET_EXTACK)
static int efct_tx_merge_timeout_set(struct devlink *dl, u32 id, struct devlink_param_gset_ctx *ctx,
									 struct netlink_ext_ack *extack)
#else
static int efct_tx_merge_timeout_set(struct devlink *dl, u32 id, struct devlink_param_gset_ctx *ctx)
#endif
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct;
	int i, j;

	for (i = 0; i < (*efct_dev)->num_ports; i++) {
		efct = (*efct_dev)->efct[i];
		for (j = 0; j < efct->max_evq_count; j++)
			efct->evq[j].tx_merge_timeout_ns = ctx->val.vu32;
	}
	return 0;
}

static int efct_tx_merge_timeout_validate(struct devlink *dl, u32 id, union devlink_param_value val,
					  struct netlink_ext_ack *extack)
{
	if (val.vu32 > MAX_TX_MERGE_TIMEOUT_NS_VALUE) {
		pr_info("Transmit event merge timeout too large for X3, It shall be less than %u\n",
			MAX_TX_MERGE_TIMEOUT_NS_VALUE);
		return -EINVAL;
	}

	if (val.vu32 > 0 && (val.vu32 % EV_TIMER_GRANULARITY_NS != 0)) {
		pr_info("Transmit event merge timeout must be a multiple of %u\n",
			EV_TIMER_GRANULARITY_NS);
		return -EINVAL;
	}

	return 0;
}

static int efct_ct_thresh_param_get(struct devlink *dl, u32 id, struct devlink_param_gset_ctx *ctx)
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct = (*efct_dev)->efct[0];

	ctx->val.vu16 = efct->ct_thresh * 64;

	return 0;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_SET_EXTACK)
static int efct_ct_thresh_param_set(struct devlink *dl, u32 id, struct devlink_param_gset_ctx *ctx,
									struct netlink_ext_ack *extack)
#else
static int efct_ct_thresh_param_set(struct devlink *dl, u32 id, struct devlink_param_gset_ctx *ctx)
#endif
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct;
	int i, j;

	for (i = 0; i < (*efct_dev)->num_ports; i++) {
		efct = (*efct_dev)->efct[i];
		efct->ct_thresh = ctx->val.vu16 / 64;
		for (j = 0; j < EFCT_MAX_CORE_TX_QUEUES; j++)
			efct->txq[j].ct_thresh = efct->ct_thresh;
	}
	return 0;
}

static int efct_ct_thresh_param_validate(struct devlink *dl, u32 id, union devlink_param_value val,
					 struct netlink_ext_ack *extack)
{
	if (((val.vu16 % 64) != 0) || val.vu16 > MAX_CT_THRESHOLD_VALUE) {
		pr_info("Invalid CTPIO Threshold value passed. It shall be less than %d and a multiple of 64",
			MAX_CT_THRESHOLD_VALUE);
		return -EINVAL;
	}

	return 0;
}

static int efct_dist_layout_param_get(struct devlink *dl, u32 id,
				      struct devlink_param_gset_ctx *ctx)
{
	struct efct_device **efct_dev = devlink_priv(dl);

	ctx->val.vu8 = (*efct_dev)->dist_layout;

	return 0;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_SET_EXTACK)
static int efct_dist_layout_param_set(struct devlink *dl, u32 id,
				      struct devlink_param_gset_ctx *ctx,
					  struct netlink_ext_ack *extack)
#else
static int efct_dist_layout_param_set(struct devlink *dl, u32 id,
				      struct devlink_param_gset_ctx *ctx)
#endif
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct;
	int i;

	if ((*efct_dev)->dist_layout == ctx->val.vu8)
		return 0;

	(*efct_dev)->dist_layout = ctx->val.vu8;
	for (i = 0; i < (*efct_dev)->num_ports; i++) {
		efct = (*efct_dev)->efct[i];
		mutex_lock(&efct->state_lock);
		if (efct->state == STATE_NET_UP)
			efct_set_interrupt_affinity(efct);
		mutex_unlock(&efct->state_lock);
	}

	return 0;
}

static int efct_dist_layout_param_validate(struct devlink *dl, u32 id,
					   union devlink_param_value val,
					  struct netlink_ext_ack *extack)
{
	if (val.vu8 != RX_LAYOUT_DISTRIBUTED && val.vu8 != RX_LAYOUT_SEPARATED) {
		NL_SET_ERR_MSG_MOD(extack, "Invalid Layout value.Use 0-Distributed,1-Separated");
		return -EINVAL;
	}

	return 0;
}

static int efct_separated_cpu_param_get(struct devlink *dl, u32 id,
					struct devlink_param_gset_ctx *ctx)
{
	struct efct_device **efct_dev = devlink_priv(dl);

	ctx->val.vu8 = (*efct_dev)->separated_rx_cpu;

	return 0;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_SET_EXTACK)
static int efct_separated_cpu_param_set(struct devlink *dl, u32 id,
					struct devlink_param_gset_ctx *ctx,
					struct netlink_ext_ack *extack)
#else
static int efct_separated_cpu_param_set(struct devlink *dl, u32 id,
					struct devlink_param_gset_ctx *ctx)
#endif
{
	struct efct_device **efct_dev = devlink_priv(dl);
	struct efct_nic *efct;
	int i;

	if ((*efct_dev)->separated_rx_cpu == ctx->val.vu8)
		return 0;

	(*efct_dev)->separated_rx_cpu = ctx->val.vu8;
	for (i = 0; i < (*efct_dev)->num_ports; i++) {
		efct = (*efct_dev)->efct[i];
		mutex_lock(&efct->state_lock);
		if (efct->state == STATE_NET_UP)
			efct_set_interrupt_affinity(efct);
		mutex_unlock(&efct->state_lock);
	}

	return 0;
}

static int efct_separated_cpu_param_validate(struct devlink *dl, u32 id,
					     union devlink_param_value val,
					     struct netlink_ext_ack *extack)
{
	struct efct_device **efct_dev = devlink_priv(dl);

	if ((*efct_dev)->dist_layout == RX_LAYOUT_DISTRIBUTED) {
		NL_SET_ERR_MSG_MOD(extack, "Current layout is distributed. Use separated layout");
		return -EINVAL;
	}

	if (val.vu8 > num_online_cpus()) {
		NL_SET_ERR_MSG_MOD(extack,
				   "Invalid CPU value passed. Value exceeds current online CPUS");
		pr_info("Invalid CPU value. Exceeds current online CPUS [%u]", num_online_cpus());
		return -EINVAL;
	}

	return 0;
}

static int efct_num_ports_param_validate(struct devlink *dl, u32 id,
					 union devlink_param_value val,
					 struct netlink_ext_ack *extack)
{
	u32 value = val.vu32;

	if (value < 1 || value > MAX_PORTS) {
		NL_SET_ERR_MSG_MOD(extack, "Value out of range. Expecting either 1 or 2");
		return -ERANGE;
	}
	return 0;
}

static const struct devlink_param efct_devlink_params[] = {
	DEVLINK_PARAM_DRIVER(EFCT_DEVLINK_PARAM_ID_CT_THRESH,
			     "ct_thresh", DEVLINK_PARAM_TYPE_U16,
			     BIT(DEVLINK_PARAM_CMODE_RUNTIME),
			     efct_ct_thresh_param_get,
			     efct_ct_thresh_param_set,
			     efct_ct_thresh_param_validate),
	DEVLINK_PARAM_DRIVER(EFCT_DEVLINK_PARAM_ID_DIST_LAYOUT,
			     "dist_layout", DEVLINK_PARAM_TYPE_U8,
			     BIT(DEVLINK_PARAM_CMODE_RUNTIME),
			     efct_dist_layout_param_get,
			     efct_dist_layout_param_set,
			     efct_dist_layout_param_validate),
	DEVLINK_PARAM_DRIVER(EFCT_DEVLINK_PARAM_ID_SEPARATED_CPU,
			     "separated_cpu", DEVLINK_PARAM_TYPE_U8,
			     BIT(DEVLINK_PARAM_CMODE_RUNTIME),
			     efct_separated_cpu_param_get,
			     efct_separated_cpu_param_set,
			     efct_separated_cpu_param_validate),
	DEVLINK_PARAM_DRIVER(EFCT_DEVLINK_PARAM_ID_IRQ_ADAPT_LOW_THRESH,
			     "irq_adapt_low_thresh", DEVLINK_PARAM_TYPE_U32,
			     BIT(DEVLINK_PARAM_CMODE_RUNTIME),
			     efct_irq_adapt_low_thresh_param_get,
			     efct_irq_adapt_low_thresh_param_set,
			     NULL),
	DEVLINK_PARAM_DRIVER(EFCT_DEVLINK_PARAM_ID_IRQ_ADAPT_HIGH_THRESH,
			     "irq_adapt_high_thresh", DEVLINK_PARAM_TYPE_U32,
			     BIT(DEVLINK_PARAM_CMODE_RUNTIME),
			     efct_irq_adapt_high_thresh_param_get,
			     efct_irq_adapt_high_thresh_param_set,
			     NULL),
	DEVLINK_PARAM_DRIVER(EFCT_DEVLINK_PARAM_ID_IRQ_ADAPT_IRQS,
			     "irq_adapt_irqs", DEVLINK_PARAM_TYPE_U32,
			     BIT(DEVLINK_PARAM_CMODE_RUNTIME),
			     efct_irq_adapt_irqs_param_get,
			     efct_irq_adapt_irqs_param_set,
			     NULL),
	DEVLINK_PARAM_DRIVER(EFCT_DEVLINK_PARAM_ID_RX_MERGE_TIMEOUT_NS,
			     "rx_merge_timeout", DEVLINK_PARAM_TYPE_U32,
			     BIT(DEVLINK_PARAM_CMODE_RUNTIME),
			     efct_rx_merge_timeout_get,
			     efct_rx_merge_timeout_set,
			     efct_rx_merge_timeout_validate),
	DEVLINK_PARAM_DRIVER(EFCT_DEVLINK_PARAM_ID_TX_MERGE_TIMEOUT_NS,
			     "tx_merge_timeout", DEVLINK_PARAM_TYPE_U32,
			     BIT(DEVLINK_PARAM_CMODE_RUNTIME),
			     efct_tx_merge_timeout_get,
			     efct_tx_merge_timeout_set,
			     efct_tx_merge_timeout_validate),
	DEVLINK_PARAM_DRIVER(EFCT_DEVLINK_PARAM_ID_NUM_PORTS_PF,
			     "num_ports", DEVLINK_PARAM_TYPE_U32,
			     BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			     NULL, NULL, efct_num_ports_param_validate),
};

#if defined(EFCT_USE_KCOMPAT) && defined(EFCT_HAVE_NDO_GET_DEVLINK)
struct devlink_port *efct_get_devlink_port(struct net_device *dev)
{
	struct efct_nic *efct = efct_netdev_priv(dev);

	if (efct)
		return &efct->dl_port;
	else
		return NULL;
}
#endif

static int efct_devlink_info_get(struct devlink *devlink,
				 struct devlink_info_req *req,
				struct netlink_ext_ack *extack)
{
	struct efct_device **efct_dev = devlink_priv(devlink);
	struct efct_nic *efct = (*efct_dev)->efct[0];

	efct_devlink_info_query_all(efct, req);
	return 0;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_RELOAD_DOWN)
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_RELOAD_ACTION)
static int efct_devlink_reload_up(struct devlink *devlink,
				  enum devlink_reload_action action,
				  enum devlink_reload_limit limit,
				  u32 *actions_performed,
				  struct netlink_ext_ack *extack)
#elif defined(EFCT_HAVE_DEVLINK_RELOAD_UP_EXTACK)
static int efct_devlink_reload_up(struct devlink *devlink,
				  struct netlink_ext_ack *extack)
#endif
{
	struct efct_device **efct_device = devlink_priv(devlink);
	struct efct_device *efct_dev = *efct_device;
	union devlink_param_value saved_value;
	int rc = 0;

	rc = devl_param_driverinit_value_get(devlink,
					     EFCT_DEVLINK_PARAM_ID_NUM_PORTS_PF,
					     &saved_value);
	if (rc) {
		pr_err("Failed to get num ports value, uses default value\n");
		return rc;
	}

	efct_dev->num_ports = saved_value.vu32;
	rc = efct_load(efct_dev);
	if (rc) {
		pr_err("Devlink reload failed rc %d\n", rc);
		efct_dev->num_ports = 0;
		return rc;
	}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_RELOAD_ACTION)
	*actions_performed = BIT(DEVLINK_RELOAD_ACTION_DRIVER_REINIT);
#endif
	return 0;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_RELOAD_ACTION)
static int efct_devlink_reload_down(struct devlink *devlink,
				    bool netns_change,
				    enum devlink_reload_action action,
				    enum devlink_reload_limit limit,
				    struct netlink_ext_ack *extack)
#elif defined(EFCT_HAVE_DEVLINK_RELOAD_DOWN_NS)
static int efct_devlink_reload_down(struct devlink *devlink, bool netns_change,
				    struct netlink_ext_ack *extack)
#elif defined(EFCT_HAVE_DEVLINK_RELOAD_DOWN_EXTACK)
static int efct_devlink_reload_down(struct devlink *devlink,
				    struct netlink_ext_ack *extack)
#endif
{
	struct efct_device **efct_device = devlink_priv(devlink);
	struct efct_device *efct_dev = *efct_device;
	int rc = 0;

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_RELOAD_ACTION) || defined(EFCT_HAVE_DEVLINK_RELOAD_DOWN_NS)
	if (netns_change) {
		NL_SET_ERR_MSG_MOD(extack, "Namespace change is not supported");
		return -EOPNOTSUPP;
	}
#endif
	rc = efct_unload(efct_dev);
	if (rc) {
		pr_err("Devlink reload failed rc %d\n", rc);
		return rc;
	}
	return 0;
}
#endif

#if defined(EFCT_USE_KCOMPAT) && defined(EFCT_HAVE_DEVLINK_ONLY_RELOAD)
static int efct_devlink_reload(struct devlink *devlink,
			       struct netlink_ext_ack *extack)
{
	struct efct_device **efct_device = devlink_priv(devlink);
	struct efct_device *efct_dev = *efct_device;
	union devlink_param_value saved_value;
	int rc = 0;

	rc = efct_unload(efct_dev);
	if (rc) {
		pr_err("Devlink unload failed rc %d\n", rc);
		NL_SET_ERR_MSG_MOD(extack, "Devlink unload failed");
		return rc;
	}

	rc = devlink_param_driverinit_value_get(devlink,
						EFCT_DEVLINK_PARAM_ID_NUM_PORTS_PF,
						&saved_value);
	if (rc) {
		pr_err(" Failed to get num ports value, uses default value\n");
		return rc;
	}

	efct_dev->num_ports = saved_value.vu32;
	rc = efct_load(efct_dev);
	if (rc) {
		pr_err("Devlink reload failed rc %d\n", rc);
		NL_SET_ERR_MSG_MOD(extack, "Devlink reload failed");
		efct_dev->num_ports = 0;
		return rc;
	}
	return 0;
}
#endif

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_FLASH_UPDATE_PARAMS)
static int efct_devlink_flash_update(struct devlink *devlink,
				     struct devlink_flash_update_params *params,
				    struct netlink_ext_ack *extack)
#else
static int efct_devlink_flash_update(struct devlink *devlink,
				     const char *file_name,
				    const char *component,
				    struct netlink_ext_ack *extack)
#endif
{
	struct efct_device **efct_dev = devlink_priv(devlink);
	struct efct_nic *efct = (*efct_dev)->efct[0];
#if defined(EFCT_USE_KCOMPAT) && !defined(EFCT_HAVE_DEVLINK_FLASH_UPDATE_PARAMS_FW)
	const struct firmware *fw;
	int rc;
#endif

#if defined(EFCT_USE_KCOMPAT) && !defined(EFCT_HAVE_DEVLINK_FLASH_UPDATE_PARAMS)
	if (component) {
		netif_err(efct, drv, efct->net_dev,
			  "Updates to NVRAM component %s are not supported\n",
			  component);
		return -EINVAL;
	}
#endif

#if defined(EFCT_USE_KCOMPAT) && !defined(EFCT_HAVE_DEVLINK_FLASH_UPDATE_PARAMS_FW)
#ifdef EFCT_HAVE_DEVLINK_FLASH_UPDATE_PARAMS
	rc = request_firmware(&fw, params->file_name, &efct->efct_dev->pci_dev->dev);
#else
	rc = request_firmware(&fw, file_name, &efct->efct_dev->pci_dev->dev);
#endif
	if (rc)
		return rc;

	rc = efct_reflash_flash_firmware(efct, fw);

	release_firmware(fw);
	return rc;
#else
	return efct_reflash_flash_firmware(efct, params->fw);
#endif
}

static const struct devlink_ops xlnx_devlink_ops = {
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_OPS_SUPPORTED_FLASH_UPDATE_PARAMS)
	.supported_flash_update_params	= 0,
#endif
	.flash_update			= efct_devlink_flash_update,
	.info_get			= efct_devlink_info_get,
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_RELOAD_ACTION)
	.reload_actions = BIT(DEVLINK_RELOAD_ACTION_DRIVER_REINIT),
#endif
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_RELOAD_DOWN)
	.reload_down			= efct_devlink_reload_down,
	.reload_up			= efct_devlink_reload_up,
#elif defined(EFCT_HAVE_DEVLINK_ONLY_RELOAD)
	.reload				= efct_devlink_reload,
#endif
};

static void efct_devlink_params_unregister(struct efct_device *efct_dev)
{
	if (efct_dev->devlink) {
#if defined(EFCT_USE_KCOMPAT) && defined(EFCT_HAVE_DEVLINK_PARAMS_UNPUBLISH)
		devlink_params_unpublish(efct_dev->devlink);
#endif
		devlink_params_unregister(efct_dev->devlink, efct_devlink_params,
					  ARRAY_SIZE(efct_devlink_params));
	}
}

void efct_fini_devlink(struct efct_device *efct_dev)
{
	if (efct_dev->devlink)
		devlink_free(efct_dev->devlink);
	efct_dev->devlink = NULL;
}

int efct_probe_devlink(struct efct_device *efct_dev)
{
	struct efct_device **devlink_private;

#if defined(EFCT_USE_KCOMPAT) && !defined(EFCT_HAVE_DEVLINK_ALLOC_NS_DEV)
	struct devlink *dl;
#endif

	int rc = 0;

	efct_dev->devlink = devlink_alloc(&xlnx_devlink_ops,
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_ALLOC_DEV)
					  sizeof(struct efct_device **),
					  &efct_dev->pci_dev->dev);
#else
					  sizeof(struct efct_device **));
#endif
	if (!efct_dev->devlink)
		return -ENOMEM;

#if defined(EFCT_USE_KCOMPAT) && !defined(EFCT_HAVE_DEVLINK_ALLOC_NS_DEV)
	dl = (struct devlink *)efct_dev->devlink;
	dl->dev = &efct_dev->pci_dev->dev;
#endif
	devlink_private = devlink_priv(efct_dev->devlink);
	*devlink_private = efct_dev;

	return rc;
}

int efct_devlink_port_register(struct efct_nic *efct)
{
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_PORT_ATTR) || defined(EFCT_HAVE_DEVLINK_PORT_ATTR_CONST)
	struct devlink_port_attrs attrs = {};
#endif
	int rc;

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_PORT_ATTR) || defined(EFCT_HAVE_DEVLINK_PORT_ATTR_CONST)
	attrs.flavour = DEVLINK_PORT_FLAVOUR_PHYSICAL;
	attrs.phys.port_number = efct->port_num;
	devlink_port_attrs_set(&efct->dl_port, &attrs);
#else
	devlink_port_attrs_set(&efct->dl_port, DEVLINK_PORT_FLAVOUR_PHYSICAL,
			       efct->port_num, false, 0, NULL, 0);
#endif
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_SET_NETDEV_DEVLINK_PORT)
	SET_NETDEV_DEVLINK_PORT(efct->net_dev, &efct->dl_port);
#endif
	rc = devl_port_register(efct->efct_dev->devlink, &efct->dl_port,
				efct->port_num);
	if (rc)
		return rc;

	return 0;
}

static int efct_devlink_params_register(struct efct_device *efct_dev)
{
	union devlink_param_value value;
	int rc;

	rc = devlink_params_register(efct_dev->devlink, efct_devlink_params,
				     ARRAY_SIZE(efct_devlink_params));
	if (rc)
		return rc;

	value.vu32 = MAX_PORTS;
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_DEVL_DRIVERINIT_SET_GET)
	devl_lock(efct_dev->devlink);
#endif
	devl_param_driverinit_value_set(efct_dev->devlink,
					EFCT_DEVLINK_PARAM_ID_NUM_PORTS_PF,
					value);
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_DEVLINK_DEVL_DRIVERINIT_SET_GET)
	devl_unlock(efct_dev->devlink);
#endif

#if defined(EFCT_USE_KCOMPAT) && defined(EFCT_HAVE_DEVLINK_PARAMS_PUBLISH)
	devlink_params_publish(efct_dev->devlink);
#endif

	return 0;
}

int efct_devlink_register(struct efct_device *efct_dev)
{
	int rc;

	rc = efct_devlink_params_register(efct_dev);
	if (rc)
		return rc;

#if defined(EFCT_USE_KCOMPAT) && defined(EFCT_HAVE_DEVLINK_SET_FEATURES_RELOAD)
	devlink_set_features(efct_dev->devlink, DEVLINK_F_RELOAD);
#endif
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_VOID_DEVLINK_REGISTER)
	devlink_register(efct_dev->devlink);
#elif defined(EFCT_HAVE_DEVLINK_ALLOC_DEV)
	rc = devlink_register(efct_dev->devlink);
	if (rc) {
		efct_devlink_params_unregister(efct_dev);
		return rc;
	}
#else
	rc = devlink_register(efct_dev->devlink, &efct_dev->pci_dev->dev);
	if (rc) {
		efct_devlink_params_unregister(efct_dev);
		return rc;
	}
#endif
#if defined(EFCT_USE_KCOMPAT) &&  defined(EFCT_HAVE_DEVLINK_RELOAD_ENABLE_DISABLE)
	/* Enable reload support bit */
	devlink_reload_enable(efct_dev->devlink);
#endif

	return 0;
}

void efct_devlink_unregister(struct efct_device *efct_dev)
{
#if defined(EFCT_USE_KCOMPAT) &&  defined(EFCT_HAVE_DEVLINK_RELOAD_ENABLE_DISABLE)
	/* Disable reload support bit */
	devlink_reload_disable(efct_dev->devlink);
#endif
	devlink_unregister(efct_dev->devlink);
	efct_devlink_params_unregister(efct_dev);
}
#else
static ssize_t versions_show(struct device *dev, struct device_attribute *attr,
			     char *buf_out)
{
	struct efct_device *efct_dev = pci_get_drvdata(to_pci_dev(dev));
	struct devlink_info_req req = {
		.buf = buf_out,
		.bufsize = PAGE_SIZE
	};

	buf_out[0] = '\0';
	efct_devlink_info_query_all(efct_dev->efct[0], &req);
	return strlen(buf_out);
}

static DEVICE_ATTR_RO(versions);

int efct_probe_devlink(struct efct_device *efct_dev)
{
	return device_create_file(&efct_dev->pci_dev->dev, &dev_attr_versions);
}

void efct_fini_devlink(struct efct_device *efct_dev)
{
	device_remove_file(&efct_dev->pci_dev->dev, &dev_attr_versions);
}
#endif	/* EFCT_USE_DEVLINK */
