/* SPDX-License-Identifier: GPL-2.0
 ****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#ifndef EFCT_DRIVER_H
#define EFCT_DRIVER_H

/* Uncomment this to enable output from netif_vdbg
 * #define VERBOSE_DEBUG 1
 */

#include <linux/device.h>
#include <linux/ethtool.h>
#include <linux/if_vlan.h>
#include <linux/mii.h>
#include <linux/net_tstamp.h>
#include <linux/netdevice.h>
#include <linux/pci.h>

#ifdef EFCT_NOT_UPSTREAM
#ifndef CONFIG_EFCT_TEST
#include "config.h"
#endif
#endif
#ifdef CONFIG_XILINX_AUX_EFCT
#include <linux/auxiliary_bus.h>
#include "xlnx_efct.h"
#endif
#ifdef EFCT_USE_KCOMPAT
/* Must come before other headers */
#include "kernel_compat.h"
#endif
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_USE_DEVLINK)
#include <net/devlink.h>
#endif
#include "efct_enum.h"
#include "bitfield.h"

/**************************************************************************
 *
 * Build definitions
 *
 **************************************************************************/

#define EFCT_DRIVER_VERSION	"1.6.9.0"

/**************************************************************************
 *
 * Efx data structures
 *
 **************************************************************************/
#ifdef CONFIG_XILINX_PTP
struct efct_ptp_data;
#endif
#define EFCT_MAX_RX_QUEUES 16
#define EFCT_MAX_TX_QUEUES 32
#define EFCT_MAX_EV_QUEUES 48
#define EFCT_MAX_CORE_TX_QUEUES 1
#define EFCT_EV_UNSOL_MAX 64
#define EFCT_EV_CONSUMED_IN_OVERFLOW 1
#define EFCT_EV_UNSOL_GRANT_UPDATE 16
#define EFCT_EV_UNSOL_OVERFLOW_CLEAR 1
#define EFCT_DEFAULT_QUEUE 0
#define EFCT_MSIX_PER_PORT 32

#define EFCT_IRQ_ADAPT_LOW_THRESH_VAL 8000
#define EFCT_IRQ_ADAPT_HIGH_THRESH_VAL 16000
#define EFCT_IRQ_ADAPT_IRQS 1000
//TODO Setting the default moderation Rx/Tx to 0 for performance
#define EFCT_IRQ_RX_MOD_DEFAULT_VAL 0
#define EFCT_IRQ_TX_MOD_DEFAULT_VAL 0

/* Flush event wait timeout */
#define EFCT_MAX_FLUSH_TIME 5000
/*Reserve space for Eth Header, 4 vlan tags and FCS. refer XNET-606*/
#define EFCT_MTU_RESV (ETH_HLEN + (VLAN_HLEN * 4) + 4)
/* Maximum possible MTU the driver supports Refer X3-463*/
#define EFCT_MAX_MTU (1982 - EFCT_MTU_RESV)
/* Minimum MTU */
#define EFCT_MIN_MTU 64
/* Default CPU ID for separated receive queue layout */
#define DEFAULT_SEPARATED_RX_CPU 0

#define MAX_TX_BUFFERS	256
#define MAX_PORTS 2
#define EFCT_RX_BUF_SIZE_MULTIPLIER 4096
#define EFCT_PORT_LEN 0x62000
#define EFCT_PORT_OFFSET 0x80000
#define EFCT_MAX_VERSION_INFO_LEN 256
#define EFCT_FAILURE	-1
#define EFCT_SUCCESS	 0
#define EFCT_DEFAULT_FILTER_TBL_ROWS	256
#define EFCT_MAX_FILTER_TBL_ROWS	512

#define STRING_TABLE_LOOKUP(val, member) \
	((val) < member ## _max) && member ## _names[val] ? member ## _names[val] : "(invalid)"

/**@RESET_TYPE_ALL: Reset datapath, MAC and PHY
 * @RESET_TYPE_WORLD: Reset as much as possible
 * unsuccessful.
 * @RESET_TYPE_DATAPATH: Reset datapath only.
 * @RESET_TYPE_MC_BIST: MC entering BIST mode.
 * @RESET_TYPE_DISABLE: Reset datapath, MAC and PHY; leave NIC disabled
 * @RESET_TYPE_TX_WATCHDOG: reset due to TX watchdog
 * @RESET_TYPE_MC_FAILURE: MC reboot/assertion
 * @RESET_TYPE_MCDI_TIMEOUT: MCDI timeout.
 *
 * Reset methods are numbered in order of increasing scope.
 *
 * enum reset_type - reset types
 *
 * %RESET_TYPE_ALL, %RESET_TYPE_WORLD and
 * %RESET_TYPE_DISABLE specify the method/scope of the reset.  The
 * other values specify reasons, which efct_schedule_reset() will choose
 * a method for.
 *
 **/
enum reset_type {
	RESET_TYPE_ALL,
	RESET_TYPE_WORLD,
	RESET_TYPE_DATAPATH,
	RESET_TYPE_MC_BIST,
	RESET_TYPE_DISABLE,
	RESET_TYPE_MAX_METHOD,
	RESET_TYPE_TX_WATCHDOG,
	RESET_TYPE_MC_FAILURE,
	/* RESET_TYPE_MCDI_TIMEOUT is actually a method, not
	 * a reason, but it doesn't fit the scope hierarchy (it's not well-
	 * ordered by inclusion)
	 * We encode this by having its enum values be greater than
	 * RESET_TYPE_MAX_METHOD.  This also prevents issuing it with
	 * efct_ioctl_reset
	 */
	RESET_TYPE_MCDI_TIMEOUT,
	RESET_TYPE_MAX,
};

#ifdef CONFIG_XILINX_AUX_EFCT

/* Aux client wait timeout for rolled over super buffs */
#define EFCT_AUX_RO_CLEANUP_TIME_MS 5000

enum client_state {
	STATE_OPEN = 0,  /* Client is opened */
	STATE_CLOSED,   /* Client is closed */
};

struct efct_rxq_client_meta {
	atomic_t active_hp;
	u32 added_hp;
	u32 removed_hp;
	u32 ref_cnt;
};

struct xlnx_efct_client {
	struct efct_nic *efct;
	/* Event queue bit map*/
	DECLARE_BITMAP(active_evq, EFCT_MAX_EV_QUEUES);
	/* Tx queue bit map*/
	DECLARE_BITMAP(active_txq, EFCT_MAX_TX_QUEUES);
	/* Filter bitmap*/
	DECLARE_BITMAP(filter_map, EFCT_MAX_FILTER_TBL_ROWS);
	/* Per queue client information */
	struct efct_rxq_client_meta rxq_info[EFCT_MAX_RX_QUEUES];
	/* EFCT Driver opertations */
	const struct xlnx_efct_drvops *drvops;
	/*Driver private data*/
	void	*driver_priv;
	/*Client State */
	enum client_state state;
	/* Number of active hugepages */
	atomic_t active_hugepages;
	/* Client Index */
	u8 index;
#ifdef CONFIG_XILINX_DEBUGFS
	struct dentry *debug_dir;
#endif
};

#define MAX_AUX_CLIENTS 4
#define MIN_AUX_IRQ 1
#else
#define MIN_AUX_IRQ 0
#endif

extern const struct ethtool_ops efct_ethtool_ops;

/**
 * struct efct_buffer - A general-purpose DMA buffer
 * @addr: host base address of the buffer
 * @dma_addr: DMA base address of the buffer
 * @len: Buffer length, in bytes
 * @sentinel: sentinel value
 *
 * The NIC uses these buffers for its interrupt status registers and
 * MAC stats dumps.
 */
struct efct_buffer {
	void *addr;
	dma_addr_t dma_addr;
	u32 len;
	bool sentinel;
};

#define CT_DEFAULT_THRESHOLD    0xFF
#define MAX_CT_THRESHOLD_VALUE  (CT_DEFAULT_THRESHOLD * 64)

#define MAX_TX_MERGE_TIMEOUT_NS_VALUE 0x7e00
#define MAX_RX_MERGE_TIMEOUT_NS_VALUE 0x7e00
#define EV_TIMER_GRANULARITY_NS 0x200

struct efct_tx_buffer {
	struct sk_buff *skb;
};

struct efct_tx_queue {
	/* TXQ Identifier */
	u32 txq_index;
	/* EVQ Index */
	u32 evq_index;
	/* Queue reset */
	bool is_resetting;
	/*DP_TX_PACKET_FIFO_SIZE Design param*/
	u32 fifo_size;
	/*DP_TX_CTPIO_APERTURE_SIZE Design param*/
	u32 aperture_qword;
	u8 label;
	/*Number of entries*/
	u32 num_entries;
	/* efct_nic structure */
	struct efct_nic *efct;
	/* The networking core TX queue structure */
	struct netdev_queue *core_txq;
	/* Mapped address of CTPIO Windows base */
	u64 __iomem *piobuf;
	/* Current offset within CTPIO windows.
	 * When new packet comes, driver starts writing
	 * into piobuf + offset memory in 64 bits chunk.
	 * New offset is modulo of CTPIO window size
	 * If CTPIO is always going to be 4K stride then
	 * How about bitfield entry of 12 bits?
	 * This would automatically rollover?
	 */
	u32 piobuf_offset;
	/* When packet is added, this sequence is incremented */
	u8 added_sequence;
	/* When packet transfer is completed, this sequence is incremented from ISR */
	u8 completed_sequence;
	/* Below parameters help to identify free space within CTPIO window */
	/* Number of bytes in FIFO */
	atomic_t inuse_fifo_bytes;
	/* Number of packets in FIFO to track seq */
	atomic_t inflight_pkts;
	/* CT threshold */
	u8 ct_thresh;
	/* Transmit packet metadata */
	struct efct_tx_buffer tx_buffer[MAX_TX_BUFFERS];
	/* Number of packets completed to report to BQL */
	u32 pkts;
	/* Number of bytes completed to report to BQL */
	u32 bytes;
	/* Software stats */
	u64 tx_packets;
	u64 n_tx_stop_queue;
};

struct efct_driver_buffer {
	/* Buffer info */
	struct efct_buffer dma_buffer;
	/* Next pointer */
	struct efct_driver_buffer *next;
};

/* Default receivee buffers per queue */
#define RX_BUFFS_PER_QUEUE	4
/* Maximum driver buffers */
#define RX_MAX_DRIVER_BUFFS	 12
/* Lower water mark for NBL */
#define RX_MIN_DRIVER_BUFFS 2
/* Huge page size */
#define HUGE_PAGE_SZ BIT(21)      /* 2M Huge Page Size */
#define SUPER_BUFF_SZ BIT(20)     /* 1M Super Buff Size */
#define SUPER_BUFFS_PER_HUGE_PAGE DIV_ROUND_UP(HUGE_PAGE_SZ, \
					       SUPER_BUFF_SZ)

struct efct_driver_buffer_list {
	/* Array of driver buffers */
	struct efct_driver_buffer db[RX_MAX_DRIVER_BUFFS];
	/* free list */
	struct efct_driver_buffer *free_list;
};

struct efct_nic_buffer {
	/* DMA buffer Info */
	struct efct_buffer dma_buffer;
	/* Parent pool */
	unsigned char is_dbl;
	/* Buffer ID */
	int id;
};

/* NIC QUEUE LEN = RX_MAX_DRIVER_BUFFS + 1 space for rollover + 1 for queue ds */
#define NIC_BUFFS_PER_QUEUE	 (RX_MAX_DRIVER_BUFFS + 2)

struct efct_nic_buffer_list {
	/* Array of receive buffers */
	struct efct_nic_buffer nb[NIC_BUFFS_PER_QUEUE];
	/* In flight buffer count */
	u32 active_nic_buffs;
	/* HW Index or consumer index.
	 * This points to the current receive buffer where HW has posted packets
	 */
	unsigned char head_index;
	/* SW Index or producer index.
	 * Driver refills the queue from this index.
	 */
	unsigned char tail_index;
	/* Metadata offset within buffer */
	u32 meta_offset;
	/* Prev metadata offset within buffer */
	u32 prev_meta_offset;
	/* Frame location bit from packet start */
	u32 frame_offset_fixed;
	/* Monotonic buffer sequence number, used to refer to packets over a longer
	 * period than head_index
	 */
	u32 seq_no;
};

#ifdef CONFIG_XILINX_AUX_EFCT
struct efct_super_buffer {
	/* DMA buffer Info */
	struct efct_buffer dma_buffer;
	/* Client owner Id */
	struct xlnx_efct_client *client_id;
	/* Clearance flags */
	unsigned short flags;
	/* Reference count of shared clients */
	unsigned short ref_count;
	/* Next pointer */
	int next;
};

struct efct_super_buffer_list {
	/* For deallocation */
	struct rcu_head rcu;
	/* Free list */
	int free_list;
	/* Array of super buffer */
	struct efct_super_buffer sb[];
};

struct efct_huge_page_entry {
	/* Huge page entry */
	struct xlnx_efct_hugepage hp;
	/* dma address */
	dma_addr_t dma_addr;
	/* Clearance flags */
	unsigned short flags;
	/* Client owner Id */
	struct xlnx_efct_client *client_id;
	/* Active superbuf count */
	int active_sb_cnt;
	/* Next pointer */
	int next;
};

/* Default size of efct_huge_page_list::hp at queue creation time. Can
 * grow/realloc dynamically from here.
 */
#define INITIAL_HUGE_PAGE_LIST_SIZE  4

struct efct_huge_page_list {
	/* For deallocation */
	struct rcu_head rcu;
	/* free index */
	int free_list;
	/* Max allocated hp ID */
	int max_hp_id;
	/* Array of huge page entry */
	struct efct_huge_page_entry hp[];
};

struct poison_config {
	u64 value;
	size_t length;
};
#endif

struct efct_rx_queue {
	/* NUmber of filters added to this queue */
	u32 filter_count;
	/* Receive queue identifier */
	u32 index;
	/* EVQ Index */
	u32 evq_index;
	/* CPU mapping.
	 * Is this needed. We can keep one to one mapping using index variable
	 */
#ifdef CONFIG_XILINX_AUX_EFCT
	/*Bit map of aux clients*/
	DECLARE_BITMAP(active_aux_clients, MAX_AUX_CLIENTS);
	/*filter count for each client on this queue*/
	u16 aux_filter_count[MAX_AUX_CLIENTS];
	struct poison_config poison_cfg;
	s8 exclusive_client;
#endif
	/* Queue reset */
	bool is_resetting;
	/* Queue enable */
	bool enable;
	/* Number of receive buffs */
	u16 num_rx_buffs;
	/* Number of entries */
	u32 num_entries;
	u16 pkt_stride;
	u8 label;
	u32 cpu;
	/* efct_nic structure */
	struct efct_nic *efct;
	/* driver buffer size */
	u32 buffer_size;
	/* Receive queue base address within BAR */
	void __iomem *receive_base;
	/* NIC buffer list */
	struct efct_nic_buffer_list nbl;
	/* Driver buffer list */
	struct efct_driver_buffer_list dbl;
#ifdef CONFIG_XILINX_AUX_EFCT
	/* Huge page list */
	struct efct_huge_page_list *hpl;
	/* Super buffer list */
	struct efct_super_buffer_list *sbl;
	/* Number of entries in hpl->hp array. sbl->sb array is
	 * SUPER_BUFFS_PER_HUGE_PAGE times this size
	 */
	u32 hpl_alloced;
#endif
	/* Software stats */
	u64 n_rx_eth_crc_err;
	u64 n_rx_eth_len_err;
	u64 n_rx_ip_hdr_chksum_err;
	u64 n_rx_tcp_udp_chksum_err;
	u64 n_rx_sentinel_drop_count;
	u64 n_rx_mcast_mismatch;
	u64 n_rx_merge_events;
	u64 n_rx_merge_packets;
	u64 n_rx_alloc_skb_fail;
	u64 n_rx_broadcast_drop;
	u64 n_rx_other_host_drop;
	u64 n_rx_nbl_empty;
	u64 n_rx_buffers_posted;
	u64 n_rx_rollover_events;
	u64 n_rx_aux_pkts;
	u64 rx_packets;
#ifdef CONFIG_XILINX_DEBUGFS
	struct dentry *debug_dir;
#endif
};

enum nic_state {
	STATE_UNINIT = 0,	/* device being probed/removed */
	STATE_NET_DOWN,		/* netdev registered */
	STATE_NET_UP,		/* ready for traffic */
	STATE_DISABLED,		/* device disabled due to hardware errors */
};

static inline bool efct_net_active(enum nic_state state)
{
	return ((state == STATE_NET_UP) || (state == STATE_NET_DOWN));
}

enum driver_dist_layout {
	RX_LAYOUT_DISTRIBUTED = 0,
	RX_LAYOUT_SEPARATED,
};

/**
 * enum efct_phy_mode - PHY operating mode flags
 * @PHY_MODE_NORMAL: on and should pass traffic
 * @PHY_MODE_TX_DISABLED: on with TX disabled
 * @PHY_MODE_LOW_POWER: set to low power through MDIO
 * @PHY_MODE_OFF: switched off through external control
 * @PHY_MODE_SPECIAL: on but will not pass traffic
 */

enum efct_phy_mode {
	PHY_MODE_NORMAL		= 0, /*on and should pass traffic*/
	PHY_MODE_TX_DISABLED	= 1, /* on with TX disabled*/
	PHY_MODE_LOW_POWER	= 2, /* set to low power through MDIO */
	PHY_MODE_OFF		= 4, /* switched off through external control */
	PHY_MODE_SPECIAL	= 8, /* on but will not pass traffic*/
};

/* Forward declaration */
struct efct_nic;

/**
 * struct efct_link_state - Current state of the link
 * @up: Link is up
 * @fd: Link is full-duplex
 * @fc: Actual flow control flags
 * @speed: Link speed (Mbps)
 * @ld_caps: Local device capabilities
 * @lp_caps: Link partner capabilities
 */
struct efct_link_state {
	bool up;
	bool fd;
	u8 fc;
	u32 speed;
	u32 ld_caps;
	u32 lp_caps;
};

static inline bool efct_link_state_equal(const struct efct_link_state *left,
					 const struct efct_link_state *right)
{
	return left->up == right->up && left->fd == right->fd &&
		left->fc == right->fc && left->speed == right->speed;
}

enum efct_led_mode {
	EFCT_LED_OFF	= 0,
	EFCT_LED_ON	= 1,
	EFCT_LED_DEFAULT	= 2
};

/**
 * struct efct_hw_stat_desc - Description of a hardware statistic
 * @name: Name of the statistic as visible through ethtool, or %NULL if
 *	it should not be exposed
 * @dma_width: Width in bits (0 for non-DMA statistics)
 * @offset: Offset within stats (ignored for non-DMA statistics)
 */
struct efct_hw_stat_desc {
	const char *name;
	u16 dma_width;
	u16 offset;
};

/* Efx Error condition statistics */
struct efct_nic_errors {
	atomic_t missing_event;
	atomic_t rx_reset;
	atomic_t rx_desc_fetch;
	atomic_t tx_desc_fetch;
	atomic_t spurious_tx;
};

enum evq_type {
	EVQ_T_NONE,
	EVQ_T_RX,
	EVQ_T_TX,
	EVQ_T_AUX
};

struct msix_info {
		unsigned short irq;
		unsigned short idx;
		char name[IFNAMSIZ + 6];
};

struct efct_ev_queue {
	/* Event queue identifier */
	u32 index;
	/* Max entries */
	u32 entries;
	/* MSIx IRQ Number */
	struct msix_info msi;
	/* Event queue base address within BAR */
	void __iomem *evq_base;
	/* efct_nic structure */
	struct efct_nic *efct;
	/* Net device used with NAPI */
	struct net_device *napi_dev;
	/* NAPI control structure */
	struct napi_struct napi;
	/* phase bit */
	bool evq_phase;
#ifdef CONFIG_XILINX_AUX_EFCT
	/* subscribe for time sync events, used only for Aux TX queues*/
	bool subscribe_time_sync;
#endif
	/* DMA ring */
	struct efct_buffer buf;
	/* Event unsolicated consumer index */
	u32 unsol_consumer_index;
	/* Event queue consumer index */
	u32 consumer_index;
	u32 irq_moderation_ns;
	/* Eventy queue type Rx/Tx */
	enum evq_type type;
	/* Maximum number of queues associated for this evq */
	unsigned char queue_count;
	/* Associated queue */
	void *queue;
#ifdef CONFIG_XILINX_PTP
	u64 sync_timestamp_major;
	enum efct_sync_events_state sync_events_state;
#endif
	/* Number of IRQs since last adaptive moderation decision */
	u32 irq_count;
	/* IRQ moderation score */
	u32 irq_mod_score;
	/* Threshold score for reducing IRQ moderation */
	u32 irq_adapt_low_thresh;
	/* Threshold score for increasing IRQ moderation */
	u32 irq_adapt_high_thresh;
	/* Number of IRQs per IRQ moderation adaptation */
	u32 irq_adapt_irqs;
	/* Transmit event merge timeout to configure, in nanoseconds */
	u32 tx_merge_timeout_ns;
	/* Receive event merge timeout to configure, in nanoseconds */
	u32 rx_merge_timeout_ns;
	/* Last CPU to handle interrupt or test event */
	int event_test_cpu;
	int event_test_napi;
	/* Software stats */
	u64 n_evq_time_sync_events;
	u64 n_evq_error_events;
	u64 n_evq_flush_events;
	u64 n_evq_unsol_overflow;
	u64 n_evq_unhandled_events;
	/*unsolicited credit grant*/
	u16 unsol_credit;
};

#ifdef CONFIG_XILINX_AUX_EFCT
struct aux_resources {
	u16 evq_base;
	u16 evq_lim;
	u16 txq_base;
	u16 txq_lim;
};
#endif

struct efct_nic {
#ifdef CONFIG_XILINX_AUX_EFCT
	struct xlnx_efct_device aux_dev;
	/* List of registered AUX Clients */
	struct xlnx_efct_client aux_clients[MAX_AUX_CLIENTS];
	/*Bit map of aux clients*/
	DECLARE_BITMAP(aux_client_map, MAX_AUX_CLIENTS);
	/*Aux resource*/
	struct aux_resources aux_res;
	/* Aux client lock*/
	struct mutex client_list_lock;
	struct xlnx_efct_irq_resources *irq_resources;
	/* wait queue used to wait for huge pages cleanup completions */
	wait_queue_head_t hp_cleanup_wq;
#endif
	/* The following fields should be written very rarely */
	char name[IFNAMSIZ];
	const struct efct_nic_type *type;
	/* The Network device */
	struct net_device *net_dev;
	void *nic_data;
	void  *phy_data;
#ifdef CONFIG_XILINX_PTP
	struct efct_ptp_data *ptp_data;
	struct efct_ptp_data *phc_ptp_data;
#endif
	enum efct_phy_mode phy_mode;
	bool phy_power_force_off;
	enum efct_loopback_mode loopback_mode;
	u64 loopback_modes;
	struct notifier_block netdev_notifier;
	u32 datapath_caps;
	u32 datapath_caps2;
	/*used in MCDI code*/
	unsigned long mcdi_buf_use;
	bool mc_bist_for_other_fn;
	enum nic_state state;
	/* Nic State lock */
	struct mutex state_lock;
	struct rw_semaphore hpl_mutation_lock;
	struct efct_mcdi_data *mcdi;
	struct efct_buffer mcdi_buf;
	struct efct_nic_errors errors;
	u32 int_error_count;
	unsigned long int_error_expire;
	/* The efct physical function device data structure */
	struct efct_device *efct_dev;
	/* Port number or port identifier */
	u8 port_num;
	/* Port index */
	u8 port_idx;
	/*MTU size*/
	u32 mtu;
	/* link_advertising: Autonegotiation advertising flags */
	__ETHTOOL_DECLARE_LINK_MODE_MASK(link_advertising);
	/* Scheduled reset workitem */
	struct work_struct reset_work;
	/* Bitmask for pending resets */
	unsigned long reset_pending;
	unsigned long last_reset;
	u32 reset_count;
	/* Last seen MC warm boot count */
	int mc_warm_boot_count;

	/**
	 * mac_lock: MAC access lock. Protects @port_enabled, @link_up, @phy_mode,
	 * efct_monitor() and efct_mac_work()
	 */
	struct mutex mac_lock;

	/* Total active queues tx+rx */
	atomic_t active_queues;
	/* wait queue used to wait for flush completions */
	wait_queue_head_t flush_wq;
	/* Active event queue count */
	u8 max_evq_count;
	/* Active receive queue count */
	u8 rxq_count;
	/* Active transmit queue count */
	u8 max_txq_count;
	/* Tx ct_threshhold */
	u8 ct_thresh;
	u32 fc_disable;
	int last_irq_cpu;
	unsigned long evq_active_mask;
	unsigned long rxq_active_mask;
	unsigned long txq_active_mask;
	struct efct_link_state link_state;
	u32 n_link_state_changes;
	/* Interrupt timer maximum value, in nanoseconds */
	u32 timer_max_ns;
	u32 timer_quantum_ns;
	u32 msg_enable;
	/* Array of event queue structure */
	struct efct_ev_queue *evq;
	/* Array of receive queue structure */
	struct efct_rx_queue *rxq;
	/* Array of transmit queue structure */
	struct efct_tx_queue *txq;
	u32 port_base;
	void __iomem *membase;
	/* Base address of write-combining mapping of the memory BAR */
	void __iomem *wc_membase;
	netdev_features_t fixed_features;
	netdev_features_t hw_features;
#ifdef CONFIG_XILINX_DEBUGFS
	/* Port debugfs directory */
	struct dentry *debug_port_dir;
#ifdef CONFIG_XILINX_AUX_EFCT
	/* Aux debugfs dir */
	struct dentry *debug_aux_dir;
#endif
#endif
	/* MC MAC stats buffer */
	__le64 *mc_mac_stats;
	__le64 *mc_initial_stats;
	u16 num_mac_stats;
	bool stats_enabled;
	/* Software rx drop count */
	atomic64_t n_rx_sw_drops;
	/* Software tx drop count */
	atomic64_t n_tx_sw_drops;

	bool irq_rx_adaptive;
	/* wanted_fc: Wanted flow control flags */
	u8 wanted_fc;
	u32 fec_config;
	u32 stats_period_ms;
	/* Params required for IRQ Moderation */
	u32 irq_mod_step_ns;
	u32 irq_rx_moderation_ns;
	// ethtool stats lock
	spinlock_t stats_lock;
	bool stats_initialised;
	/* Mutex for serializing firmware reflash operations.*/
	struct mutex reflash_mutex;
	struct efct_mcdi_filter_table *filter_table;
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_USE_DEVLINK)
	struct devlink_port dl_port;
#endif
};

struct design_params {
	/* stride between entries in receive window */
	u32 rx_stride;
	/* stride between entries in event queue window */
	u32 evq_stride;
	/* stride between entries in CTPIO window */
	u32 ctpio_stride;
	/* Length of each receive buffer */
	u32 rx_buffer_len;
	/* Maximum Rx queues available */
	u32 rx_queues;
	/* Maximum Tx apertures available */
	u32 tx_apertures;
	/* Maximum number of receive buffers can be posted */
	u32 rx_buf_fifo_size;
	/* Fixed offset to the frame */
	u32 frame_offset_fixed;
	/* Receive metadata length */
	u32 rx_metadata_len;
	/* Largest window of reordered writes to the CTPIO */
	u32 tx_max_reorder;
	/* CTPIO aperture length */
	u32 tx_aperture_size;
	/* Size of packet FIFO per CTPIO aperture */
	u32 tx_fifo_size;
	/* partial time stamp in sub nano seconds */
	u32 ts_subnano_bit;
	/* Width of sequence number in EVQ_UNSOL_CREDIT_GRANT register */
	u32 unsol_credit_seq_mask;
	/* L4 csm feilds */
	u32 l4_csum_proto;
	/* MAx length of frame data when LEN_ERR indicates runt*/
	u32 max_runt;
	/* Event queue sizes */
	u32 evq_sizes;
	/* Number of event queues */
	u32 num_evq;
	/* Number of filter supported in H/W */
	u32 num_filter;
};

struct efct_device {
	struct design_params params;
	unsigned char num_ports;
	enum driver_dist_layout dist_layout;
	u8 separated_rx_cpu;
	struct pci_dev *pci_dev;
	const struct efct_nic_type *eth_type;
	/*spin lock for biu*/
	spinlock_t biu_lock;
	u64 max_dma_mask;
	u32 mem_bar;
	u32 reg_base;
	/* Memory BAR value as physical address */
	resource_size_t membase_phys;
	void __iomem *membase;
	/* BAR length */
	resource_size_t bar_len;
	bool mcdi_logging;
	struct efct_nic *efct[MAX_PORTS];
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_USE_DEVLINK)
	void *devlink;
#endif
#ifdef CONFIG_XILINX_DEBUGFS
	struct dentry *debug_dir;
#endif
	/*IRQ vectors per port*/
	u16 vec_per_port;
	struct msix_entry *xentries;
	struct delayed_work stats_work;
	struct workqueue_struct *stats_wq;
};

static inline int validate_evq_size(struct efct_nic *efct, int size)
{
	int evqsizes = 0;

	if (!is_power_of_2(size))
		return -EINVAL;
	evqsizes = efct->efct_dev->params.evq_sizes << 7;
	/*Since size is power of 2, Only 1 bit will be set in 'size'*/
	return (evqsizes & size) == size;
}

static inline struct efct_device *efct_nic_to_device(struct efct_nic *efct)
{
	return efct->efct_dev;
}

static inline struct efct_nic *efct_netdev_priv(struct net_device *dev)
{
	return netdev_priv(dev);
}

static inline int efct_dev_registered(struct efct_nic *efct)
{
	return efct->net_dev->reg_state == NETREG_REGISTERED;
}

static inline u32 efct_port_num(struct efct_nic *efct)
{
	return efct->port_num;
}

static inline bool efct_xmit_with_hwtstamp(struct sk_buff *skb)
{
	return skb_shinfo(skb)->tx_flags & SKBTX_HW_TSTAMP;
}

struct efct_nic_type {
	u32 (*mem_bar)(struct efct_nic *efct);
	u32 (*mem_map_size)(struct efct_nic *efct);
	int (*probe)(struct efct_nic *efct);
	void (*remove)(struct efct_nic *efct);
	int (*dimension_resources)(struct efct_nic *efct);
	void (*free_resources)(struct efct_nic *efct);
	int (*net_alloc)(struct efct_nic *efct);
	void (*net_dealloc)(struct efct_nic *efct);
	enum reset_type (*map_reset_reason)(enum reset_type reason);
	int (*map_reset_flags)(u32 *flags);
	int (*reset)(struct efct_nic *efct, enum reset_type method);
	int (*reconfigure_port)(struct efct_nic *efct);
	int (*reconfigure_mac)(struct efct_nic *efct);
	bool (*check_mac_fault)(struct efct_nic *efct);
	void (*mcdi_request)(struct efct_nic *efct, u8 bufid,
			     const union efct_dword *hdr, size_t hdr_len,
			     const union efct_dword *sdu, size_t sdu_len);
	bool (*mcdi_poll_response)(struct efct_nic *efct, u8 bufid);
	void (*mcdi_read_response)(struct efct_nic *efct,
				   u8 bufid,
				   union efct_dword *pdu,
				   size_t pdu_offset,
				   size_t pdu_len);
	int (*mcdi_poll_reboot)(struct efct_nic *efct);
	void (*mcdi_record_bist_event)(struct efct_nic *efct);
	int (*mcdi_poll_bist_end)(struct efct_nic *efct);
	void (*mcdi_reboot_detected)(struct efct_nic *efct);
	bool (*mcdi_get_buf)(struct efct_nic *efct, u8 *bufid);
	void (*mcdi_put_buf)(struct efct_nic *efct, u8 bufid);
	bool (*ev_mcdi_pending)(struct efct_nic *efct, u16 index);
	int (*ev_probe)(struct efct_nic *efct, u32 index, u32 size);
	int (*ev_init)(struct efct_ev_queue *eventq);
	void (*ev_remove)(struct efct_ev_queue *eventq);
	void (*ev_fini)(struct efct_ev_queue *eventq);
	void (*ev_purge)(struct efct_ev_queue *eventq);
	int (*ev_process)(struct efct_ev_queue *evq, int quota);
	int (*tx_probe)(struct efct_nic *efct, u32 evq_index, int txq_index);
	int (*tx_init)(struct efct_tx_queue *tx_queue);
	void (*tx_remove)(struct efct_tx_queue *tx_queue);
	void (*tx_fini)(struct efct_tx_queue *tx_queue);
	void (*tx_purge)(struct efct_tx_queue *tx_queue);
	int (*rx_probe)(struct efct_nic *efct, u16 index, u16 evq_index);
	int (*rx_init)(struct efct_rx_queue *rx_queue);
	void (*rx_remove)(struct efct_rx_queue *rx_queue);
	void (*rx_fini)(struct efct_rx_queue *rx_queue);
	void (*rx_purge)(struct efct_rx_queue *rx_queue);
	u32 (*mcdi_rpc_timeout)(struct efct_nic *efct, u32 cmd);
	size_t (*describe_stats)(struct efct_nic *efct, u8 *names);
	void (*update_stats)(struct efct_nic *efct, bool force);
	void (*pull_stats)(struct efct_nic *efct);
	irqreturn_t (*irq_handle_msix)(int irq, void *dev_id);
	u32 (*check_caps)(const struct efct_nic *efct, u8 flag, u32 offset);
	bool (*has_dynamic_sensors)(struct efct_nic *efct);
#ifdef CONFIG_XILINX_PTP
	int (*ptp_set_ts_config)(struct efct_nic *efct, struct hwtstamp_config *init);
	u32 hwtstamp_filters;
#endif
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_NDO_GET_PHYS_PORT_ID)
	int (*get_phys_port_id)(struct efct_nic *efct, struct netdev_phys_item_id *ppid);
#endif
	int (*irq_test_generate)(struct efct_nic *efct);
	void (*ev_test_generate)(struct efct_ev_queue *evq);
	int mcdi_max_ver;
	u64 max_dma_mask;
};

int efct_unload(struct efct_device *efct_dev);
int efct_load(struct efct_device *efct_dev);

/* Pseudo bit-mask flow control field */
#define EFCT_FC_RX   FLOW_CTRL_RX
#define EFCT_FC_TX   FLOW_CTRL_TX
#define EFCT_FC_AUTO 4
#define EFCT_FC_OFF 0

/**
 * EFCT_MAX_FRAME_LEN - calculate maximum frame length
 * @mtu:		Maximum Transmission Unit
 *
 * This calculates the maximum frame length that will be used for a
 * given MTU.  The frame length will be equal to the MTU plus a
 * constant amount of header space. This is the quantity
 * that the net driver will program into the MAC as the maximum frame
 * length.
 */
#define EFCT_MAX_FRAME_LEN(mtu) \
	((mtu) + EFCT_MTU_RESV)

#endif /* EFCT_DRIVER_H */
