/* SPDX-License-Identifier: GPL-2.0
 ****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#ifndef EFCT_IO_H
#define EFCT_IO_H

#include <linux/io.h>
#include <linux/spinlock.h>
#ifdef CONFIG_EFCT_TEST
#include "efct-test-mock.h"
#endif

/**************************************************************************
 *
 * NIC register I/O
 *
 **************************************************************************
 *
 * Notes on locking strategy:
 *
 * Many CSRs are very wide and cannot be read or written atomically.
 * Writes from the host are buffered by the Bus Interface Unit (BIU)
 * up to 128 bits.  Whenever the host writes part of such a register,
 * the BIU collects the written value and does not write to the
 * underlying register until all 4 dwords have been written.  A
 * similar buffering scheme applies to host access to the NIC's 64-bit
 * SRAM.
 *
 * Writes to different CSRs and 64-bit SRAM words must be serialised,
 * since interleaved access can result in lost writes.  We use
 * efct_nic::biu_lock for this.
 *
 * We also serialise reads from 128-bit CSRs and SRAM with the same
 * spinlock.  This may not be necessary, but it doesn't really matter
 * as there are no such reads on the fast path.
 */
static inline u32 efct_reg(struct efct_device *efct_dev, u32 reg)
{
	return efct_dev->reg_base + reg;
}

static inline void _efct_writed(__le32 value, void __iomem *reg)
{
	__raw_writel((__force u32)value, reg);
}

#ifndef CONFIG_EFCT_TEST

static inline __le32 _efct_readd(void __iomem *reg)
{
	return (__force __le32)__raw_readl(reg);
}
#endif

/* Write a 32-bit CSR or the last dword of a special 128-bit CSR */
static inline void efct_writed(const union efct_dword *value, void __iomem *reg)
{
	_efct_writed(value->word32, reg);
}

/* Read a 32-bit CSR or SRAM */
static inline void efct_readd(union efct_dword *value, void __iomem *reg)
{
	value->word32 = _efct_readd(reg);
}

/* Read a 128-bit CSR, locking as appropriate. */
static inline void efct_reado(struct efct_device *efct_dev,
			      union efct_oword *value, void __iomem *reg)
{
	unsigned long flags __maybe_unused;

	spin_lock_irqsave(&efct_dev->biu_lock, flags);
	value->u32[0] = (__force __le32)__raw_readl((u8 __iomem *)reg + 0);
	value->u32[1] = (__force __le32)__raw_readl((u8 __iomem *)reg + 4);
	value->u32[2] = (__force __le32)__raw_readl((u8 __iomem *)reg + 8);
	value->u32[3] = (__force __le32)__raw_readl((u8 __iomem *)reg + 12);
	spin_unlock_irqrestore(&efct_dev->biu_lock, flags);
}

static inline void _efct_writeq(__le64 value, void __iomem *reg)
{
	__raw_writeq((__force u64)value, reg);
}

static inline __le64 _efct_readq(void __iomem *reg)
{
	return (__force __le64)__raw_readq(reg);
}
#endif /* EFCT_IO_H */
