// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */
#include "efct_netdev.h"
#include "efct_common.h"
#include "efct_evq.h"
#include "efct_tx.h"
#include "efct_nic.h"
#include "mcdi.h"
#include "mcdi_port_common.h"
#ifdef EFCT_NOT_UPSTREAM
#include "efct_ioctl.h"
#include "ioctl_common.h"
#endif
#include "efct_devlink.h"
#ifdef CONFIG_XILINX_PTP
#include "efct_ptp.h"
#endif
#ifdef CONFIG_XILINX_AUX_EFCT
#include "efct_auxbus.h"
#endif
#ifdef CONFIG_XILINX_DEBUGFS
#include "debugfs.h"
#endif

static int efct_netdev_event(struct notifier_block *this,
			     unsigned long event, void *ptr)
{
	struct efct_nic *efct = container_of(this, struct efct_nic, netdev_notifier);
	struct net_device *net_dev = netdev_notifier_info_to_dev(ptr);

	if (efct->net_dev == net_dev &&
	    (event == NETDEV_CHANGENAME || event == NETDEV_REGISTER)) {
		netif_dbg(efct, probe, efct->net_dev,
			  "Event %lu Interface name %s\n", event, efct->net_dev->name);
		strcpy(efct->name, efct->net_dev->name);
	}

	return NOTIFY_DONE;
}

static int efct_start_all_queues(struct efct_nic *efct)
{
	int i, j, k, l, rc;

	i = 0;
	j = 0;
	k = 0;
	atomic_set(&efct->active_queues, 0);
	netif_dbg(efct, probe, efct->net_dev, "Starting queues\n");

	for_each_set_bit(i, &efct->evq_active_mask, efct->max_evq_count) {
		if (efct->evq[i].type == EVQ_T_AUX)
			continue;

		rc = efct->type->ev_init(&efct->evq[i]);
		if (rc) {
			netif_err(efct, ifup, efct->net_dev, "EV queue init failed, index %d\n", i);
			goto fail1;
		}
	}

	for (j = 0; j < efct->rxq_count; j++) {
		rc = efct->type->rx_init(&efct->rxq[j]);
		if (rc) {
			netif_err(efct, ifup, efct->net_dev, "RX queue init failed, index %d\n", j);
			goto fail2;
		}
		atomic_inc(&efct->active_queues);
	}

	for (k = 0; k < EFCT_MAX_CORE_TX_QUEUES; k++) {
		rc = efct->type->tx_init(&efct->txq[k]);
		if (rc) {
			netif_err(efct, ifup, efct->net_dev, "TX queue init failed, index %d\n", k);
			goto fail3;
		}
	}

	atomic_add(EFCT_MAX_CORE_TX_QUEUES, &efct->active_queues);

	return 0;

fail3:
	for (l = 0; l < k; l++) {
		efct->type->tx_fini(&efct->txq[l]);
		efct->type->tx_purge(&efct->txq[l]);
	}
fail2:
	for (l = 0; l < j ; l++) {
		efct->type->rx_fini(&efct->rxq[l]);
		efct->type->rx_purge(&efct->rxq[l]);
	}
fail1:
	for (l = 0; l < i; ++l) {
		if (!test_bit(l, &efct->evq_active_mask) || efct->evq[l].type == EVQ_T_AUX)
			continue;

		efct->type->ev_fini(&efct->evq[l]);
		efct->type->ev_purge(&efct->evq[l]);
	}

	return rc;
}

static void efct_stop_all_queues(struct efct_nic *efct)
{
	int i, pending = 0;

	netif_dbg(efct, probe, efct->net_dev, "Stopping queues\n");
	if ((efct->reset_pending & (1 << RESET_TYPE_DATAPATH))) {
		for (i = 0; i < efct->rxq_count; i++) {
			efct->rxq[i].enable = false;
			efct->type->rx_purge(&efct->rxq[i]);
		}

		for (i = 0; i < EFCT_MAX_CORE_TX_QUEUES; i++)
			efct->type->tx_purge(&efct->txq[i]);

		for_each_set_bit(i, &efct->evq_active_mask, efct->max_evq_count) {
			if (efct->evq[i].type == EVQ_T_AUX)
				continue;
			efct->type->ev_purge(&efct->evq[i]);
		}

#ifdef CONFIG_XILINX_PTP
		for_each_set_bit(i, &efct->evq_active_mask, efct->max_evq_count)
			efct->evq[i].sync_events_state = SYNC_EVENTS_DISABLED;
#endif
		return;
	}

	for (i = 0; i < efct->rxq_count; i++)
		efct->type->rx_fini(&efct->rxq[i]);

	for (i = 0; i < EFCT_MAX_CORE_TX_QUEUES; i++)
		efct->type->tx_fini(&efct->txq[i]);

	wait_event_timeout(efct->flush_wq, atomic_read(&efct->active_queues) == 0,
			   msecs_to_jiffies(EFCT_MAX_FLUSH_TIME));

	pending = atomic_read(&efct->active_queues);
	if (pending) {
		netif_err(efct, hw, efct->net_dev, "failed to flush %d queues\n",
			  pending);
	}

	for (i = 0; i < efct->rxq_count; i++)
		efct->type->rx_purge(&efct->rxq[i]);

	for (i = 0; i < EFCT_MAX_CORE_TX_QUEUES; i++)
		efct->type->tx_purge(&efct->txq[i]);

	for_each_set_bit(i, &efct->evq_active_mask, efct->max_evq_count) {
		if (efct->evq[i].type == EVQ_T_AUX)
			continue;
		efct->type->ev_fini(&efct->evq[i]);
		efct->type->ev_purge(&efct->evq[i]);
	}
}

/* Context: process, rtnl_lock() held. */
static int efct_net_open(struct net_device *net_dev)
{
	struct efct_nic *efct = efct_netdev_priv(net_dev);
	int rc;

	mutex_lock(&efct->state_lock);

	if (efct->state == STATE_DISABLED || efct->state == STATE_UNINIT) {
		rc = -EINVAL;
		goto fail;
	}

	/* Always come out of low power unless we're forced off */
	if (!efct->phy_power_force_off)
		efct->phy_mode &= ~PHY_MODE_LOW_POWER;

	mutex_lock(&efct->mac_lock);
	rc = efct_mac_reconfigure(efct);
	mutex_unlock(&efct->mac_lock);
	if (rc)
		goto fail;

#ifdef CONFIG_XILINX_PTP
	efct_ptp_evt_data_init(efct);
#endif
	rc = efct_start_all_queues(efct);
	if (rc) {
		netif_err(efct, ifup, efct->net_dev, "efct_start_all_queues failed, index %d\n", rc);
		goto fail;
	}

	rc = efct_init_napi(efct);
	if (rc)
		goto fail1;
	/* Link state detection is event-driven; we have to poll the state before enabling
	 * interrupt and after queue setup. It will avoid race, In case cable plugged after
	 * phy_poll and before status change
	 */
	mutex_lock(&efct->mac_lock);
	if (efct_mcdi_phy_poll(efct))
		efct_link_status_changed(efct);
	mutex_unlock(&efct->mac_lock);
	rc = efct_start_evqs(efct);
	if (rc)
		goto fail2;
#if !defined(EFCT_NOT_UPSTREAM) || !defined(CONFIG_X3_BUSYPOLL)
	efct_set_interrupt_affinity(efct);
#endif
	netif_start_queue(net_dev);
#ifdef CONFIG_XILINX_PTP
	if (efct->ptp_data->txtstamp)
		efct_ptp_tx_ts_event(efct, true);
#endif
	efct->state = STATE_NET_UP;
	mutex_unlock(&efct->state_lock);

	return 0;

fail2:
	efct_finish_napi(efct);
fail1:
	efct_stop_all_queues(efct);
fail:
	mutex_unlock(&efct->state_lock);

	return rc;
}

static int efct_net_stop(struct net_device *net_dev)
{
	struct efct_nic *efct = efct_netdev_priv(net_dev);

	mutex_lock(&efct->state_lock);
	efct->state = STATE_NET_DOWN;
#ifdef CONFIG_XILINX_PTP
	if (efct->ptp_data->txtstamp &&  !((efct->reset_pending & (1 << RESET_TYPE_DATAPATH))))
		efct_ptp_tx_ts_event(efct, false);
#endif
	netif_stop_queue(net_dev);
#if !defined(EFCT_NOT_UPSTREAM) || !defined(CONFIG_X3_BUSYPOLL)
	efct_clear_interrupt_affinity(efct);
#endif
	efct_stop_all_queues(efct);

	efct_stop_evqs(efct);
	efct_finish_napi(efct);
	mutex_unlock(&efct->state_lock);

	/* Update stats to avoid last update delta */
	efct->type->update_stats(efct, true);

	return 0;
}

static netdev_tx_t efct_start_xmit(struct sk_buff *skb, struct net_device *net_dev)
{
	struct efct_nic *efct = efct_netdev_priv(net_dev);
	struct efct_tx_queue *tx_queue;
	int qid;
	int rc;

	qid = skb_get_queue_mapping(skb);
	tx_queue = &efct->txq[qid];
	rc = efct_enqueue_skb(tx_queue, skb, net_dev);
	if (rc)
		atomic64_inc(&tx_queue->efct->n_tx_sw_drops);

	return NETDEV_TX_OK;
}

/* Context: netif_tx_lock held, BHs disabled. */
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_NDO_TX_TIMEOUT_TXQUEUE)
static void efct_watchdog(struct net_device *net_dev, unsigned int txqueue)
#else
static void efct_watchdog(struct net_device *net_dev)
#endif
{
	struct efct_nic *efct = efct_netdev_priv(net_dev);

	netif_err(efct, tx_err, efct->net_dev, "TX watchdog. Resetting the device\n");

	efct_schedule_reset(efct, RESET_TYPE_TX_WATCHDOG);
}

static int efct_phy_probe(struct efct_nic *efct)
{
	struct efct_mcdi_phy_data *phy_data;
	int rc;

	/* Probe for the PHY */
	efct->phy_data = kzalloc(sizeof(*phy_data), GFP_KERNEL);
	if (!efct->phy_data)
		return -ENOMEM;

	rc = efct_mcdi_get_phy_cfg(efct, efct->phy_data);
	if (rc)
		goto fail;

	/* Populate driver and ethtool settings */
	phy_data = efct->phy_data;

	mcdi_to_ethtool_linkset(efct, phy_data->media, phy_data->supported_cap,
				efct->link_advertising);

	/* Default to Autonegotiated flow control if the PHY supports it */
	efct->wanted_fc = EFCT_FC_RX | EFCT_FC_TX;
	if (phy_data->supported_cap & (1 << MC_CMD_PHY_CAP_AN_LBN))
		efct->wanted_fc |= EFCT_FC_AUTO;

	efct->fec_config = mcdi_fec_caps_to_ethtool(phy_data->supported_cap);
	efct_link_set_wanted_fc(efct, efct->wanted_fc);
	/* Push settings to the PHY. Failure is not fatal, the user can try to
	 * fix it using ethtool.
	 */
	rc = efct_mcdi_port_reconfigure(efct);
	if (rc)
		netif_warn(efct, probe, efct->net_dev,
			   "could not initialise PHY settings\n");

	rc = efct_mcdi_loopback_modes(efct, &efct->loopback_modes);
	if (rc != 0)
		goto fail;
	/* The MC indicates that LOOPBACK_NONE is a valid loopback mode,
	 * but by convention we don't
	 */
	efct->loopback_modes &= ~(1 << LOOPBACK_NONE);
	return 0;
fail:
	kfree(efct->phy_data);
	efct->phy_data = NULL;
	return rc;
}

#ifdef EFCT_NOT_UPSTREAM
static int efct_do_sioc(struct net_device *net_dev, struct ifreq *ifr,
			struct efct_sock_ioctl __user *user_data)
{
	struct efct_nic *efct = efct_netdev_priv(net_dev);
	u16 efct_cmd;
	int rc;

	if (copy_from_user(&efct_cmd, &user_data->cmd, sizeof(efct_cmd)))
		return -EFAULT;
	rc = efct_private_ioctl_common(efct, efct_cmd, &user_data->u);
	if (rc == -EOPNOTSUPP)
		netif_err(efct, probe, efct->net_dev,
			  "Private ioctl cmd 0x%x is not supported\n", efct_cmd);
	return rc;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_NDO_SIOCDEVPRIVATE)
static int efct_siocdevprivate(struct net_device *net_dev, struct ifreq *ifr,
			       void __user *data, int cmd)
{
	struct efct_nic *efct = efct_netdev_priv(net_dev);

	if (cmd == SIOCEFCT)
		return efct_do_sioc(net_dev, ifr, data);

	netif_err(efct, probe, efct->net_dev,
		  "Unknown private ioctl cmd 0x%x\n", cmd);
	return -EOPNOTSUPP;
}
#endif
#endif

static int efct_eth_ioctl(struct net_device *net_dev, struct ifreq *ifr,
			  int cmd)
{
	switch (cmd) {
#ifdef CONFIG_XILINX_PTP
	case SIOCGHWTSTAMP:
		return efct_ptp_get_ts_config(net_dev, ifr);
	case SIOCSHWTSTAMP:
		return efct_ptp_set_ts_config(net_dev, ifr);
#endif
	default:
		return -EOPNOTSUPP;
	}
}

#if defined(EFCT_USE_KCOMPAT) && !defined(EFCT_HAVE_NDO_SIOCDEVPRIVATE) && !defined(EFCT_HAVE_NDO_ETH_IOCTL)
static int efct_ioctl(struct net_device *net_dev, struct ifreq *ifr, int cmd)
{
#ifdef EFCT_NOT_UPSTREAM
#if defined(EFCT_USE_KCOMPAT) && !defined(EFCT_HAVE_NDO_SIOCDEVPRIVATE)
	if (cmd == SIOCEFCT)
		return efct_do_sioc(net_dev, ifr, ifr->ifr_data);

#endif
#endif
	return efct_eth_ioctl(net_dev, ifr, cmd);
}
#endif

static void efct_set_rx_mode(struct net_device *dev)
{
}

static void efct_net_stats(struct net_device *net_dev, struct rtnl_link_stats64 *stats)
{
	struct efct_nic *efct = efct_netdev_priv(net_dev);

	spin_lock_bh(&efct->stats_lock);
	efct_update_stats_common(efct, NULL, stats);
	spin_unlock_bh(&efct->stats_lock);
}

static const struct net_device_ops efct_netdev_ops = {
	.ndo_open               = efct_net_open,
	.ndo_stop               = efct_net_stop,
	.ndo_start_xmit         = efct_start_xmit,
	.ndo_tx_timeout         = efct_watchdog,
	.ndo_get_stats64        = efct_net_stats,
#if defined(EFCT_NOT_UPSTREAM)
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_NDO_SIOCDEVPRIVATE)
	.ndo_siocdevprivate	= efct_siocdevprivate,
#endif
#endif
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_NDO_ETH_IOCTL)
	.ndo_eth_ioctl		= efct_eth_ioctl,
#endif
#if defined(EFCT_USE_KCOMPAT) && !defined(EFCT_HAVE_NDO_SIOCDEVPRIVATE) && !defined(EFCT_HAVE_NDO_ETH_IOCTL)
	.ndo_do_ioctl		= efct_ioctl,
#endif
	.ndo_set_mac_address    = efct_set_mac_address,
	.ndo_set_rx_mode        = efct_set_rx_mode,
#if defined(EFCT_USE_KCOMPAT) && defined(EFCT_HAVE_NDO_EXT_CHANGE_MTU)
	.extended.ndo_change_mtu = efct_change_mtu,
#else
	.ndo_change_mtu         = efct_change_mtu,
#endif
#if defined(EFCT_USE_KCOMPAT) && defined(EFCT_USE_DEVLINK) && defined(EFCT_HAVE_NDO_GET_DEVLINK)
	.ndo_get_devlink_port	= efct_get_devlink_port,
#endif
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_NDO_GET_PHYS_PORT_ID)
	.ndo_get_phys_port_id   = efct_get_phys_port_id,
#endif
#if defined(EFCT_USE_KCOMPAT) && defined(EFCT_HAVE_NDO_GET_PHYS_PORT_NAME) && !defined(CONFIG_NET_DEVLINK)
	.ndo_get_phys_port_name = efct_get_phys_port_name,
#endif
};

int efct_close_netdev(struct efct_nic *efct)
{
	rtnl_lock();
	dev_close(efct->net_dev);
	rtnl_unlock();

	return 0;
}

int efct_remove_netdev(struct efct_nic *efct)
{
	efct_mcdi_mac_fini_stats(efct);
	kfree(efct->phy_data);
#if defined(EFCT_NOT_UPSTREAM)
	efct_mcdi_mon_remove(efct);
#endif
	efct->phy_data = NULL;

	return 0;
}

int efct_get_mac_address(struct efct_nic *efct, u8 *mac_address)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_MAC_ADDRESSES_OUT_LEN);
	size_t outlen;
	int rc;

	BUILD_BUG_ON(MC_CMD_GET_MAC_ADDRESSES_IN_LEN != 0);

	rc = efct_mcdi_rpc(efct, MC_CMD_GET_MAC_ADDRESSES, NULL,
			   0, outbuf, sizeof(outbuf), &outlen);
	if (rc)
		return rc;

	if (outlen < MC_CMD_GET_MAC_ADDRESSES_OUT_LEN)
		return -EIO;

	memcpy(mac_address, MCDI_PTR(outbuf, GET_MAC_ADDRESSES_OUT_MAC_ADDR_BASE), ETH_ALEN);

	return 0;
}

int efct_probe_netdev(struct efct_nic *efct)
{
	struct net_device *net_dev = efct->net_dev;
	u8 addr[ETH_ALEN];
	int rc = 0;

	/* Netdev Modifiable features */
	net_dev->hw_features |= NETIF_F_RXCSUM | NETIF_F_RXALL;
	/* Netdev features */
	net_dev->features |= (net_dev->hw_features | NETIF_F_HIGHDMA | NETIF_F_SG);

	/* Disable receiving frames with bad FCS, by default. */
	net_dev->features &= ~NETIF_F_RXALL;

#ifndef CONFIG_EFCT_TEST
	SET_NETDEV_DEV(net_dev, &efct->efct_dev->pci_dev->dev);
#else
	SET_NETDEV_DEV(net_dev, NULL);
#endif
	rc = efct_get_mac_address(efct, addr);
	if (rc) {
		pci_err(efct->efct_dev->pci_dev, "Failed to get MAC address, rc=%d", rc);
		goto out;
	}
	/* Assign MAC address */
	eth_hw_addr_set(net_dev, addr);

	rc = efct_init_datapath_caps(efct);
	if (rc < 0) {
		pci_err(efct->efct_dev->pci_dev,
			"Failed to get datapath capabilities from MC, rc=%d", rc);
		goto out;
	}
#if defined(EFCT_NOT_UPSTREAM)
	rc = efct_mcdi_mon_probe(efct);
	if (rc)
		pr_warn("Failed to initialize sensors\n");
#endif
	rc = efct_phy_probe(efct);
	if (rc)
		goto out;

	rc = efct_mcdi_mac_init_stats(efct);
	if (rc)
		goto out;
out:
	return rc;
}

int efct_register_netdev(struct efct_nic *efct)
{
	struct net_device *net_dev = efct->net_dev;
	int rc = 0;

	/*registering netdev*/
	net_dev->watchdog_timeo = 5 * HZ;
	net_dev->netdev_ops = &efct_netdev_ops;
#if !defined(EFCT_USE_KCOMPAT) || !defined(EFCT_HAVE_NDO_SET_MULTICAST_LIST)
	net_dev->priv_flags |= IFF_UNICAST_FLT | IFF_LIVE_ADDR_CHANGE;
#endif

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_NETDEV_MTU_LIMITS)
	net_dev->min_mtu = EFCT_MIN_MTU;
	net_dev->max_mtu = EFCT_MAX_MTU;
#elif defined(EFCT_HAVE_NETDEV_EXT_MTU_LIMITS)
	net_dev->extended->min_mtu = EFCT_MIN_MTU;
	net_dev->extended->max_mtu = EFCT_MAX_MTU;
#endif
	net_dev->ethtool_ops = &efct_ethtool_ops;
	efct->netdev_notifier.notifier_call = efct_netdev_event;
	rc = register_netdevice_notifier(&efct->netdev_notifier);
	if (rc) {
		pci_err(efct->efct_dev->pci_dev, "Error: netdev notifier register fail\n");
		goto fail;
	}

	rtnl_lock();

	rc = dev_alloc_name(net_dev, net_dev->name);
	if (rc < 0) {
		pci_err(efct->efct_dev->pci_dev, "dev_alloc_name failed, rc=%d", rc);
		goto fail_locked;
	}
	strscpy(efct->name, efct->net_dev->name, sizeof(efct->name));

	rc = register_netdevice(net_dev);
	if (rc) {
		pci_err(efct->efct_dev->pci_dev, "register_netdevice failed, rc=%d", rc);
		goto fail_locked;
	}

	/* Always start with carrier off; PHY events will detect the link */
	netif_carrier_off(net_dev);

	rtnl_unlock();

	return 0;

fail_locked:
	rtnl_unlock();
	unregister_netdevice_notifier(&efct->netdev_notifier);
fail:
	netif_err(efct, probe, efct->net_dev, "could not register net dev\n");
	return rc;
}

void efct_unregister_netdev(struct efct_nic *efct)
{
	if (WARN_ON(efct_netdev_priv(efct->net_dev) != efct))
		return;

	if (efct_dev_registered(efct)) {
		unregister_netdevice_notifier(&efct->netdev_notifier);
		rtnl_lock();
		unregister_netdevice(efct->net_dev);
		rtnl_unlock();
	}
}

void efct_shutdown(struct efct_nic *efct)
{
#ifdef CONFIG_XILINX_PTP
	struct efct_ptp_data *ptp;

	ptp = efct->ptp_data;
#endif
	rtnl_lock();
	netif_device_detach(efct->net_dev);
	rtnl_unlock();
	mutex_lock(&efct->state_lock);
	efct->state = STATE_UNINIT;
#ifdef CONFIG_XILINX_PTP
	if (efct_phc_exposed(efct)) {
		efct_ptp_hw_pps_enable(ptp->efct, false);
		efct_ptp_stop(efct);
	}
#endif
	if (netif_running(efct->net_dev)) {
		efct_stop_all_queues(efct);
		efct_clear_interrupt_affinity(efct);
		efct_stop_evqs(efct);
	}
	mutex_unlock(&efct->state_lock);
}
