// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include <linux/module.h>
#include "efct_reg.h"
#include "efct_common.h"
#include "efct_tx.h"
#include "efct_rx.h"
#include "mcdi.h"
#include "mcdi_pcol.h"
#include "efct_io.h"
#include "mcdi_functions.h"
#include "efct_nic.h"
#ifdef CONFIG_XILINX_AUX_EFCT
#include "efct_auxbus.h"
#endif
#include "mcdi_port_common.h"
#include "efct_ptp.h"
#include "efct_evq.h"
#ifdef CONFIG_XILINX_DEBUGFS
#include "debugfs.h"
#endif

#define EFCT_NUM_MCDI_BUFFERS	1

static bool efct_has_dynamic_sensors(struct efct_nic *efct)
{
	return efct_has_cap(efct, DYNAMIC_SENSORS);
}

static int txq_get_free_index(struct efct_nic *efct)
{
	unsigned long mask = (1UL << efct->max_txq_count) - 1;
	unsigned long txq_active;
	int index;

	do {
		txq_active = efct->txq_active_mask;
		/*No free index present, all event queues are in use. ffz()
		 *behavior is undefined if at least one zero is not present.
		 * This check will make sure at least 1 zero bit is there in evq_active
		 */
		if ((txq_active & mask) == mask)
			return -EAGAIN;
		index = ffz(txq_active);
		/* In case queue is already allocated because of contention. It will
		 * retry for next available index
		 */
		if (test_and_set_bit(index, &efct->txq_active_mask)) {
			netif_dbg(efct, drv, efct->net_dev,
				  "Tx queue index %d is already in use\n", index);
			continue;
		}

		if (index < efct->max_txq_count)
			return index;
		else
			return -ENOSPC;
	} while (1);

	return -ENOSPC;
}

int efct_start_evqs(struct efct_nic *efct)
{
#if !defined(EFCT_NOT_UPSTREAM) || !defined(CONFIG_X3_BUSYPOLL)
	int rc;
#endif
	int i;

	efct_set_evq_names(efct);
	for (i = 0; i < efct->max_evq_count; ++i) {
		if (!efct->evq[i].napi_dev)
			continue;
		napi_enable(&efct->evq[i].napi);
#if defined(EFCT_NOT_UPSTREAM) && defined(CONFIG_X3_BUSYPOLL)
		napi_schedule(&efct->evq[i].napi);
#else
		rc = request_irq(efct->evq[i].msi.irq, efct->type->irq_handle_msix, 0,
				 efct->evq[i].msi.name, &efct->evq[i]);
		if (rc) {
			netif_err(efct, ifup, efct->net_dev, "failed to hook IRQ %d\n",
				  efct->evq[i].msi.irq);
			goto fail;
		}
		// Host needs to prime the interrupt initially
		efct_write_evt_prime(efct, efct->evq[i].index, efct->evq[i].consumer_index);
#endif
	}

	return 0;
#if !defined(EFCT_NOT_UPSTREAM) || !defined(CONFIG_X3_BUSYPOLL)
fail:
	napi_disable(&efct->evq[i].napi);
	for (i = i - 1; i >= 0; i--) {
		free_irq(efct->evq[i].msi.irq, &efct->evq[i]);
		napi_disable(&efct->evq[i].napi);
	}
	return rc;
#endif
}

void efct_stop_evqs(struct efct_nic *efct)
{
	u32 i;

	for (i = 0; i < efct->max_evq_count; ++i) {
		if (!efct->evq[i].napi_dev)
			continue;
		napi_disable(&efct->evq[i].napi);
#if !defined(EFCT_NOT_UPSTREAM) || !defined(CONFIG_X3_BUSYPOLL)
		free_irq(efct->evq[i].msi.irq, &efct->evq[i]);
#endif
	}
}

static void efct_process_timer_config(struct efct_nic *efct,
				      const union efct_dword *data)
{
	efct->timer_max_ns = MCDI_DWORD(data, GET_EVQ_TMR_PROPERTIES_OUT_MCDI_TMR_MAX_NS);
	efct->irq_mod_step_ns = MCDI_DWORD(data, GET_EVQ_TMR_PROPERTIES_OUT_MCDI_TMR_STEP_NS);
}

static int efct_get_timer_config(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_EVQ_TMR_PROPERTIES_OUT_LEN);
	int rc;

	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_GET_EVQ_TMR_PROPERTIES, NULL, 0,
				 outbuf, sizeof(outbuf), NULL);

	if (rc == 0) {
		efct_process_timer_config(efct, outbuf);
	} else {
		efct_mcdi_display_error(efct, MC_CMD_GET_EVQ_TMR_PROPERTIES,
					MC_CMD_GET_EVQ_TMR_PROPERTIES_OUT_LEN,
				       NULL, 0, rc);
	}

	return rc;
}

static int efct_get_warm_boot_count(struct efct_device *efct_dev)
{
	union efct_dword reg;

	efct_readd(&reg, efct_dev->membase + efct_reg(efct_dev,
						      ER_HZ_FUN_WIN_REG_HOST_MC_SFT_STATUS));
	if (EFCT_DWORD_FIELD(reg, EFCT_DWORD_0) == 0xffffffff) {
		pr_info("X3 Hardware unavailable\n");
		return -ENETDOWN;
	} else {
		return EFCT_DWORD_FIELD(reg, EFCT_WORD_1) == 0xb007 ?
			EFCT_DWORD_FIELD(reg, EFCT_WORD_0) : -EIO;
	}
}

#ifdef CONFIG_XILINX_MCDI_LOGGING
static ssize_t mcdi_logging_show(struct device *dev, struct device_attribute *attr,
				 char *buf)
{
	struct efct_device *efct_dev = pci_get_drvdata(to_pci_dev(dev));

	return scnprintf(buf, PAGE_SIZE, "%d\n", efct_dev->mcdi_logging);
}

static ssize_t mcdi_logging_store(struct device *dev, struct device_attribute *attr,
				  const char *buf, size_t count)
{
	struct efct_device *efct_dev = pci_get_drvdata(to_pci_dev(dev));
	bool enable = count > 0 && *buf != '0';
	int i = 0;

	efct_dev->mcdi_logging = enable;
	for (i = 0; i < efct_dev->num_ports; i++) {
		struct efct_nic *efct = efct_dev->efct[i];
		struct efct_mcdi_iface *mcdi = efct_mcdi(efct);

		if (mcdi)
			mcdi->logging_enabled = enable;
	}

	return count;
}

static DEVICE_ATTR_RW(mcdi_logging);

void efct_init_mcdi_logging(struct efct_device *efct_dev)
{
	int rc = device_create_file(&efct_dev->pci_dev->dev,
				    &dev_attr_mcdi_logging);
	if (rc)
		pr_warn("failed to init net dev attributes\n");
}

void efct_fini_mcdi_logging(struct efct_device *efct_dev)
{
	device_remove_file(&efct_dev->pci_dev->dev, &dev_attr_mcdi_logging);
}
#endif

int efct_init_datapath_caps(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_CAPABILITIES_V10_OUT_LEN);
	size_t outlen = 0;
	int rc;

	BUILD_BUG_ON(MC_CMD_GET_CAPABILITIES_IN_LEN != 0);

	rc = efct_mcdi_rpc(efct, MC_CMD_GET_CAPABILITIES, NULL, 0,
			   outbuf, sizeof(outbuf), &outlen);
	if (rc) {
		netif_err(efct, ifup, efct->net_dev,
			  "MCDI command to read datapath firmware capabilities failed\n");
		return rc;
	}

	if (outlen < MC_CMD_GET_CAPABILITIES_V10_OUT_LEN) {
		netif_err(efct, ifup, efct->net_dev,
			  "unable to read datapath firmware capabilities\n");
		return -EIO;
	}

	efct->datapath_caps = MCDI_DWORD(outbuf, GET_CAPABILITIES_V10_OUT_FLAGS1);
	efct->datapath_caps2 = MCDI_DWORD(outbuf, GET_CAPABILITIES_V10_OUT_FLAGS2);
	efct->num_mac_stats = MCDI_WORD(outbuf, GET_CAPABILITIES_V10_OUT_MAC_STATS_NUM_STATS);

	return 0;
}

static int try_get_warm_boot_count(struct efct_nic *efct)
{
	int i, rc = -1;

	for (i = 0; i < 5; ++i) {
		rc = efct_get_warm_boot_count(efct->efct_dev);
		if (rc >= 0)
			break;
		ssleep(1);
	}

	return rc;
}

static int efct_probe_main(struct efct_nic *efct)
{
	struct efct_nic_data *nic_data;
	int rc;

	rc = try_get_warm_boot_count(efct);
	if (efct->mc_warm_boot_count < 0) {
		pci_err(efct->efct_dev->pci_dev,
			"Failed to get MC warm boot count, rc=%d\n", rc);
		return rc;
	}

	efct->mc_warm_boot_count = rc;

	nic_data = kzalloc(sizeof(*nic_data), GFP_KERNEL);
	if (!nic_data)
		return -ENOMEM;
	efct->nic_data = nic_data;
	nic_data->efct = efct;

	/* MCDI buffers must be 256 byte aligned. */
	rc = efct_nic_alloc_buffer(efct, &efct->mcdi_buf, MCDI_BUF_LEN, GFP_KERNEL);
	if (rc) {
		pci_err(efct->efct_dev->pci_dev,
			"Failed to allocate memory for MCDI buffers, rc=%d\n", rc);
		goto fail1;
	}
	rc = efct_probe_common(efct);
	if (rc) {
		pci_err(efct->efct_dev->pci_dev, "efct_probe_common function failed\n");
		goto fail2;
	}

	rc = efct_mcdi_port_get_number(efct);
	if (rc < 0) {
		pci_err(efct->efct_dev->pci_dev,
			"MCDI command to get port number failed, rc=%d\n", rc);
		goto fail3;
	}

	efct->port_num = rc;
	rc = efct_get_timer_config(efct);
	if (rc) {
		pci_err(efct->efct_dev->pci_dev, "failed to get timer details\n");
		goto fail3;
	}
#ifdef CONFIG_XILINX_DEBUGFS
	efct_init_debugfs_nic(efct);
#endif
#ifdef CONFIG_XILINX_PTP
	rc = efct_ptp_probe_setup(efct);
	if (rc) {
		pci_err(efct->efct_dev->pci_dev, "failed to init PTP\n");
		goto fail4;
	}
#endif
	return 0;

#ifdef CONFIG_XILINX_PTP
fail4:
#ifdef CONFIG_XILINX_DEBUGFS
	efct_fini_debugfs_nic(efct);
#endif
#endif
fail3:
	efct_remove_common(efct);
fail2:
	efct_nic_free_buffer(efct, &efct->mcdi_buf);
fail1:
	kfree(nic_data);
	efct->nic_data = NULL;
	return rc;
}

static void efct_remove_main(struct efct_nic *efct)
{
#ifdef CONFIG_XILINX_PTP
	efct_ptp_remove_setup(efct);
#endif
#ifdef CONFIG_XILINX_DEBUGFS
	efct_fini_debugfs_nic(efct);
#endif
	efct_remove_common(efct);
	efct_nic_free_buffer(efct, &efct->mcdi_buf);
	kfree(efct->nic_data);
	efct->nic_data = NULL;
}

static u32 efct_mcdi_rpc_timeout
					(struct efct_nic *efct,
					 u32 cmd)
{
	switch (cmd) {
	case MC_CMD_NVRAM_ERASE:
	case MC_CMD_NVRAM_UPDATE_FINISH:
		return MCDI_RPC_LONG_TIMEOUT;
	default:
		return MCDI_RPC_TIMEOUT;
	}
}

/*	MCDI
 */
static u8 *efct_mcdi_buf
			(struct efct_nic *efct, u8 bufid,
			 dma_addr_t *dma_addr)
{
	if (dma_addr)
		*dma_addr = efct->mcdi_buf.dma_addr +
			    bufid * ALIGN(MCDI_BUF_LEN, 256);
	return (u8 *)efct->mcdi_buf.addr + bufid * ALIGN(MCDI_BUF_LEN, 256);
}

static void write_to_pio_region(struct efct_nic *efct, size_t offset,
				const union efct_dword *req_data, size_t len)
{
	u32 pio_reg = offset + ER_HZ_PORT0_REG_HOST_MC_PIO;
	int dwords = (len + 3) / 4;

	while (dwords > 0) {
		u64 val = (uint64_t)(le32_to_cpu(req_data[0].word32)) |
				((dwords > 1) ?
				((uint64_t)(le32_to_cpu(req_data[1].word32)) << 32) : 0);

		_efct_writeq(cpu_to_le64((u64)val), efct->membase + pio_reg);
		pio_reg += 8;
		dwords -= 2;
		req_data += 2;
	}
	/* Memory Barrier */
	wmb();
}

static void efct_mcdi_request(struct efct_nic *efct, u8 bufid,
			      const union efct_dword *hdr, size_t hdr_len,
			      const union efct_dword *sdu, size_t sdu_len)
{
	u32 doorbell_reg_lwrd, doorbell_reg_hwrd;
	dma_addr_t dma_addr;
	u8 *pdu;

	pdu = efct_mcdi_buf(efct, bufid, &dma_addr);
	doorbell_reg_lwrd = ER_HZ_PORT0_REG_HOST_MC_DOORBELL;
	doorbell_reg_hwrd = doorbell_reg_lwrd + 4;

	if (!pdu)
		return;

	/* Set content of the host memory to 0 while sending the request */
	memset(pdu, 0x0, MCDI_BUF_LEN);

	/* Write out the hdr */
	write_to_pio_region(efct, 0, hdr, hdr_len);

	/* Write out the payload */
	write_to_pio_region(efct, hdr_len, sdu, sdu_len);

	/* The hardware provides 'low' and 'high' (doorbell) registers
	 * for passing the 64-bit address of an MCDI request to
	 * firmware.  However the dwords are swapped by firmware.  The
	 * least significant bits of the doorbell are then 0 for all
	 * MCDI requests due to alignment.
	 */
	_efct_writed(cpu_to_le32((u32)dma_addr),
		     (efct->membase + doorbell_reg_lwrd));
	_efct_writed(cpu_to_le32((u64)dma_addr >> 32),
		     (efct->membase + doorbell_reg_hwrd));
}

static bool efct_mcdi_poll_response(struct efct_nic *efct, u8 bufid)
{
	const union efct_dword hdr =
		*(const union efct_dword *)(efct_mcdi_buf(efct, bufid, NULL));

	//memory barrier
	rmb();
	return EFCT_DWORD_FIELD(hdr, MCDI_HEADER_RESPONSE);
}

static void efct_mcdi_read_response(struct efct_nic *efct, u8 bufid,
				    union efct_dword *outbuf, size_t offset,
				    size_t outlen)
{
	const u8 *pdu = efct_mcdi_buf(efct, bufid, NULL);

	memcpy(outbuf, pdu + offset, outlen);
}

static int mcdi_poll_reboot(struct efct_nic *efct)
{
	int rc;

	rc = efct_get_warm_boot_count(efct->efct_dev);
	if (rc < 0) {
		/* The firmware is presumably in the process of
		 * rebooting. However, we are supposed to report each
		 * reboot just once, so we must only do that once we
		 * can read and store the updated warm boot count.
		 */
		return 0;
	}

	if (rc == efct->mc_warm_boot_count)
		return 0;

	return -EIO;
}

/* Get an MCDI buffer
 *
 * The caller is responsible for preventing racing by holding the
 * MCDI iface_lock.
 */
static bool efct_mcdi_get_buf(struct efct_nic *efct, u8 *bufid)
{
	if (!bufid)
		return false;

	*bufid = ffz(efct->mcdi_buf_use);

	if (*bufid < EFCT_NUM_MCDI_BUFFERS) {
		set_bit(*bufid, &efct->mcdi_buf_use);
		return true;
	}

	return false;
}

/* Return an MCDI buffer */
static void efct_mcdi_put_buf(struct efct_nic *efct, u8 bufid)
{
	if (bufid >= EFCT_NUM_MCDI_BUFFERS) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid mcdi buffer index %d\n", bufid);
		return;
	}
	if (!test_and_clear_bit(bufid, &efct->mcdi_buf_use))
		netif_warn(efct, probe, efct->net_dev, "Buffer %u already freed\n", bufid);
}

/*	Event queue initialization
 */
static int efct_ev_probe(struct efct_nic *efct, u32 index, u32 size)
{
	int rc;

	if (!efct)
		return -EINVAL;
	size = roundup_pow_of_two(size + EFCT_EV_UNSOL_MAX + EFCT_EV_CONSUMED_IN_OVERFLOW);
	rc = validate_evq_size(efct, size);
	if (rc < 1) {
		netif_err(efct, probe, efct->net_dev,
			  "Invalid event queue size: %d\n",
			  size);
		return -EINVAL;
	}

	efct->evq[index].evq_base =
			efct->membase + ER_HZ_PORT0_REG_HOST_EVQ_UNSOL_CREDIT_GRANT
			+ (index * efct->efct_dev->params.evq_stride);
	efct->evq[index].entries = size;
	return efct_nic_alloc_buffer(efct, &efct->evq[index].buf,
				(efct->evq[index].entries) *
				sizeof(union efct_qword), GFP_KERNEL);
}

static void efct_ev_unsol_credit_grant(struct efct_ev_queue *eventq,
				       bool overflow)
{
	union efct_qword qword;
	u32 credit_mask;
	u64 value;

	if (!eventq->unsol_credit)
		return;
	credit_mask = eventq->efct->efct_dev->params.unsol_credit_seq_mask;
	value = (eventq->unsol_consumer_index + eventq->unsol_credit) & credit_mask;
	EFCT_POPULATE_QWORD_2(qword,
			      ERF_HZ_GRANT_SEQ, value,
			     ERF_HZ_CLEAR_OVERFLOW, overflow);
	_efct_writeq(qword.u64[0], eventq->evq_base);
}

static int efct_ev_init(struct efct_ev_queue *eventq)
{
	int rc;

	if (!eventq)
		return -EINVAL;

	/* Set initial phase to 0 */
	eventq->evq_phase = false;
	eventq->consumer_index = 0;
	eventq->unsol_consumer_index = 0;
#ifdef CONFIG_XILINX_PTP
	eventq->sync_events_state = SYNC_EVENTS_DISABLED;
#endif
	rc = efct_mcdi_ev_init(eventq);
	if (rc) {
		netif_err(eventq->efct, ifup, eventq->efct->net_dev,
			  "MCDI init failed for event queue index = %d\n", eventq->index);
		return rc;
	}
#ifdef CONFIG_XILINX_DEBUGFS
	efct_init_debugfs_ev_queue(eventq);
#endif
	if (eventq->type != EVQ_T_AUX) {
		efct_mcdi_ev_set_timer(eventq, eventq->irq_moderation_ns,
				       MC_CMD_SET_EVQ_TMR_IN_TIMER_MODE_INT_HLDOFF, false);
	}
	efct_ev_unsol_credit_grant(eventq, EFCT_EV_UNSOL_OVERFLOW_CLEAR);
	return 0;
}

//TX queue initialization
static int efct_tx_probe(struct efct_nic *efct, u32 evq_index, int txq_index)
{
	if (!efct)
		return -EINVAL;
	if (txq_index == -1)
		txq_index = txq_get_free_index(efct);
	else
		if (test_and_set_bit(txq_index, &efct->txq_active_mask))
			txq_index = -EINVAL;

	if (txq_index < 0) {
		netif_err(efct, probe, efct->net_dev,
			  "Failed to find the free Tx index\n");
		return txq_index;
	}
	efct->txq[txq_index].piobuf = efct->wc_membase +
				      (txq_index * efct->efct_dev->params.ctpio_stride);
	efct->txq[txq_index].evq_index = evq_index;
	efct->txq[txq_index].label = txq_index;
	if (efct->evq[evq_index].type != EVQ_T_AUX)
		efct->txq[txq_index].core_txq = netdev_get_tx_queue(efct->net_dev, 0);
	else
		efct->txq[txq_index].core_txq = NULL;
	efct->txq[txq_index].aperture_qword = efct->efct_dev->params.tx_aperture_size / 8;
	efct->txq[txq_index].fifo_size = efct->efct_dev->params.tx_fifo_size;

	return txq_index;
}

static int efct_tx_init(struct efct_tx_queue *tx_queue)
{
	int rc;

	if (!tx_queue)
		return -EINVAL;

	tx_queue->completed_sequence = 255;
	tx_queue->added_sequence = 0;
	tx_queue->piobuf_offset = 0;
	tx_queue->ct_thresh = tx_queue->efct->ct_thresh;
	tx_queue->pkts = 0;
	tx_queue->bytes = 0;
	atomic_set(&tx_queue->inuse_fifo_bytes, 0);
	atomic_set(&tx_queue->inflight_pkts, 0);

	rc = efct_mcdi_tx_init(tx_queue);
	if (rc) {
		netif_err(tx_queue->efct, ifup, tx_queue->efct->net_dev,
			  "MCDI init failed for tx queue index = %d\n", tx_queue->txq_index);
		return rc;
	}
#ifdef CONFIG_XILINX_DEBUGFS
	efct_init_debugfs_tx_queue(tx_queue);
#endif
	return 0;
}

//RX queue initialization
static int efct_rx_probe(struct efct_nic *efct, u16 index, u16 evq_index)
{
	struct efct_rx_queue *rxq;
	int rc;

	if (!efct)
		return -EINVAL;

	if (test_and_set_bit(index, &efct->rxq_active_mask)) {
		netif_err(efct, probe, efct->net_dev,
			  "Rx queue index %d is already in use\n", index);
		return -EINVAL;
	}

	rxq = &efct->rxq[index];

	rxq->evq_index = evq_index;
	rxq->label = index;
	rxq->cpu = index;
#ifdef CONFIG_XILINX_AUX_EFCT
	rxq->poison_cfg.length = 0;
#endif
	rxq->n_rx_eth_crc_err = 0;
	rxq->n_rx_eth_len_err = 0;
	rxq->n_rx_ip_hdr_chksum_err = 0;
	rxq->n_rx_tcp_udp_chksum_err = 0;
	rxq->receive_base = efct->membase + ER_HZ_PORT0_REG_HOST_RX_BUFFER_POST
			+ (index * efct->efct_dev->params.rx_stride);

	efct->evq[evq_index].type = EVQ_T_RX;

	rc = dbl_init(rxq);
	if (rc)
		goto fail1;

	nbl_init(rxq);
#ifdef CONFIG_XILINX_AUX_EFCT
	rc = hpl_init(rxq);
	if (rc < 0)
		goto fail2;
	rc = sbl_init(rxq);
	if (rc < 0)
		goto fail3;
#endif
	return 0;

#ifdef CONFIG_XILINX_AUX_EFCT
fail3:
	hpl_fini(rxq);
fail2:
	dbl_fini(rxq);
#endif
fail1:
	clear_bit(index, &efct->rxq_active_mask);
	return rc;
}

static int efct_rx_init(struct efct_rx_queue *rx_queue)
{
	int rc;

	if (!rx_queue)
		return -EINVAL;
	rc = efct_mcdi_rx_init(rx_queue);
	if (rc) {
		netif_err(rx_queue->efct, ifup, rx_queue->efct->net_dev,
			  "MCDI init failed for rx queue index = %d\n", rx_queue->index);
		return rc;
	}
	rx_queue->enable = true;

#ifdef CONFIG_XILINX_DEBUGFS
	efct_init_debugfs_rx_queue(rx_queue);
#endif

#if defined(CONFIG_XILINX_PTP) && defined(CONFIG_XILINX_AUX_EFCT)
	efct_ptp_subscribe_timesync(&rx_queue->efct->evq[rx_queue->evq_index]);
#endif
	/* Post buffers in rx queue */
	rc = efct_fill_rx_buffs(rx_queue);
	if (rc) {
		netif_err(rx_queue->efct, ifup, rx_queue->efct->net_dev,
			  "Failed to fill rx queue index = %d\n", rx_queue->index);
#if defined(CONFIG_XILINX_PTP) && defined(CONFIG_XILINX_AUX_EFCT)
		efct_ptp_unsubscribe_timesync(&rx_queue->efct->evq[rx_queue->evq_index]);
#endif
#ifdef CONFIG_XILINX_DEBUGFS
		efct_fini_debugfs_rx_queue(rx_queue);
#endif
		rx_queue->enable = false;
		efct_mcdi_rx_fini(rx_queue);
	}

	return rc;
}

static void efct_ev_fini(struct efct_ev_queue *ev_queue)
{
	if (!ev_queue || !ev_queue->efct) {
		pr_err("Error:event queue or nic is NULL\n");
		return;
	}

	if (!test_bit(ev_queue->index, &ev_queue->efct->evq_active_mask)) {
		netif_err(ev_queue->efct, ifdown, ev_queue->efct->net_dev,
			  "Event queue is already removed\n");
		return;
	}

	efct_mcdi_ev_fini(ev_queue);
}

static void efct_ev_purge(struct efct_ev_queue *evq)
{
#ifdef CONFIG_XILINX_DEBUGFS
	efct_fini_debugfs_ev_queue(evq);
#endif
}

static void efct_ev_remove(struct efct_ev_queue *ev_queue)
{
	struct efct_nic *efct = ev_queue->efct;

	efct_nic_free_buffer(efct, &ev_queue->buf);
	if (!test_and_clear_bit(ev_queue->index, &efct->evq_active_mask)) {
		netif_err(efct, ifdown, efct->net_dev,
			  "Event queue is already removed\n");
	}
}

/* Read the current event from the event queue */
static union efct_qword *efct_event(struct efct_ev_queue *evq, u32 index)
{
	return ((union efct_qword *)(evq->buf.addr)) + index;
}

bool efct_nic_event_present(struct efct_ev_queue *evq)
{
	union efct_qword *p_event;
	u32 consumer_index;
	bool read_phase;
	bool evq_phase;

	consumer_index = evq->consumer_index;
	evq_phase = evq->evq_phase;
	p_event = efct_event(evq, consumer_index);

	read_phase = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_PHASE);
	if (read_phase != evq_phase)
		return true;
	return false;
}

static void efct_handle_flush_ev(struct efct_nic *efct, bool is_txq, int qid)
{
	if (is_txq) {
		struct efct_tx_queue *txq = &efct->txq[qid];

		if (txq->is_resetting) {
			txq->is_resetting = false;
			/* Reinit the Tx queue */
			efct_schedule_queue_reset(efct, is_txq, qid);
			return;
		}
	} else {
		struct efct_rx_queue *rxq = &efct->rxq[qid];

		if (rxq->is_resetting) {
			rxq->is_resetting = false;
			/* Reinit the Rx queue */
			efct_schedule_queue_reset(efct, is_txq, qid);
			return;
		}
	}

	if (atomic_dec_and_test(&efct->active_queues))
		wake_up(&efct->flush_wq);

	WARN_ON(atomic_read(&efct->active_queues) < 0);
}

#ifdef CONFIG_XILINX_PTP
static int efct_time_sync_event(struct efct_ev_queue *evq, union efct_qword *p_event, int quota)
{
	int spent = 1;

	evq->sync_timestamp_major =  EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_TSYNC_TIME_HIGH_48);
	/* if sync events have been disabled then we want to silently ignore
	 * this event, so throw away result.
	 */
	(void)cmpxchg(&evq->sync_events_state, SYNC_EVENTS_REQUESTED, SYNC_EVENTS_VALID);

#ifdef CONFIG_XILINX_AUX_EFCT
	if (evq->type == EVQ_T_RX) {
		struct efct_rx_queue *rxq = (struct efct_rx_queue *)evq->queue;
		int i;

		for (i = 0; i < evq->queue_count; ++i, ++rxq) {
			spent += efct_aux_handle_event(evq->efct, XLNX_EVENT_TIME_SYNC,
					rxq->index, le64_to_cpu(p_event->u64[0]), quota - spent);
		}
		if (spent > quota)
			spent = quota;
	}
#endif
	return spent;
}
#endif

/* TODO : Remove this Macro and use from efct_reg.h after the hw yml file is
 * updated with Error events change
 */
#define	ESE_HZ_X3_CTL_EVENT_SUBTYPE_ERROR 0x3
static int
efct_ev_control(struct efct_ev_queue *evq, union efct_qword *p_event, int quota)
{
	int subtype, spent = 1;
	struct efct_nic *efct;

	efct = evq->efct;
	subtype = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_CTL_SUBTYPE);

	evq->unsol_consumer_index++;

	switch (subtype) {
#ifdef CONFIG_XILINX_PTP
	case ESE_HZ_X3_CTL_EVENT_SUBTYPE_TIME_SYNC:
		evq->n_evq_time_sync_events++;
		spent = efct_time_sync_event(evq, p_event, quota);
		break;
#endif
	case ESE_HZ_X3_CTL_EVENT_SUBTYPE_ERROR:
		{
			u8 qlabel, ftype, reason;

			ftype = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_FLSH_FLUSH_TYPE);
			qlabel = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_FLSH_LABEL);
			reason = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_FLSH_REASON);

			netif_err(efct, probe, efct->net_dev,
				  "Error event received for %s %d reason code: %u\n",
				  ftype ? "Rxq" : "Txq", qlabel, reason);

			if (ftype)
				efct->rxq[qlabel].is_resetting = true;
			else
				efct->txq[qlabel].is_resetting = true;
		}
		evq->n_evq_error_events++;
		break;
	case ESE_HZ_X3_CTL_EVENT_SUBTYPE_FLUSH:
		{
			u8 qlabel, ftype;

			ftype = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_FLSH_FLUSH_TYPE);
			qlabel = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_FLSH_LABEL);

			netif_dbg(efct, drv, efct->net_dev,
				  "Flush event received for %s %d\n",
				  ftype ? "Rxq" : "Txq", qlabel);

#ifdef CONFIG_XILINX_AUX_EFCT
			if (ftype) /* Rx flush type */
				efct_aux_handle_event(efct, XLNX_EVENT_RX_FLUSH, -1, 0ul, INT_MAX);
#endif
			efct_handle_flush_ev(efct, !ftype, qlabel);
			evq->n_evq_flush_events++;
		}
		break;
	case ESE_HZ_X3_CTL_EVENT_SUBTYPE_UNSOL_OVERFLOW:
		/*TODO: Take action on overflow event, Reset unsol_consumer_index*/
		evq->n_evq_unsol_overflow++;
		efct_ev_unsol_credit_grant(evq, EFCT_EV_UNSOL_OVERFLOW_CLEAR);
		break;
	default:
		evq->n_evq_unhandled_events++;
		netif_info(efct, probe, efct->net_dev, "Unhandled control event %08x:%08x\n",
			   le32_to_cpu(p_event->u32[1]), le32_to_cpu(p_event->u32[0]));
		break;
	}

	if ((evq->unsol_consumer_index % EFCT_EV_UNSOL_GRANT_UPDATE == 0)) {
		efct_ev_unsol_credit_grant(evq, 0);
		evq->unsol_consumer_index &= evq->efct->efct_dev->params.unsol_credit_seq_mask;
	}

	return spent;
}

static void efct_ev_mcdi(struct efct_ev_queue *evq, union efct_qword *p_event)
{
	if (!efct_mcdi_process_event(evq->efct, p_event) &&
	    !efct_mcdi_port_process_event_common(evq, p_event)) {
		int code = EFCT_QWORD_FIELD(*p_event, MCDI_EVENT_CODE);
		struct efct_nic *efct = evq->efct;

		netif_info(efct, probe, efct->net_dev,
			   "Unhandled MCDI event %08x:%08x code %d\n",
			   le32_to_cpu(p_event->u32[1]), le32_to_cpu(p_event->u32[0]), code);
	}

	evq->unsol_consumer_index++;
	if ((evq->unsol_consumer_index % EFCT_EV_UNSOL_GRANT_UPDATE == 0)) {
		efct_ev_unsol_credit_grant(evq, 0);
		evq->unsol_consumer_index &= evq->efct->efct_dev->params.unsol_credit_seq_mask;
	}
}

static void efct_handle_driver_generated_event(struct efct_ev_queue *evq, union efct_qword *event)
{
	struct efct_nic *efct;
	u32 subcode;

	subcode = EFCT_QWORD_FIELD(*event, EFCT_DWORD_0);
	efct = evq->efct;
	switch (EFCT_DRVGEN_CODE(subcode)) {
	case EFCT_TEST:
		evq->event_test_napi = raw_smp_processor_id();
		netif_dbg(efct, probe, efct->net_dev,
			  "Driver initiated event %08x:%08x\n",
			  le32_to_cpu(event->u32[1]), le32_to_cpu(event->u32[0]));
		break;
	default:
		netif_err(efct, hw, efct->net_dev,
			  "Event queue %d unknown driver event type %u data %08x:%08x\n",
			  evq->index, (u32)subcode,
			  le32_to_cpu(event->u32[1]),
			  le32_to_cpu(event->u32[0]));
	}
}

static int efct_ev_process(struct efct_ev_queue *evq, int quota)
{
	bool evq_phase, read_phase;
	union efct_qword *p_event;
	int i, spent, ev_type;
	struct efct_nic *efct;
	u32 consumer_index;
	u8 qlabel = 0;

	evq_phase = evq->evq_phase;
	efct = evq->efct;

	spent = 0;
	consumer_index = evq->consumer_index;
	p_event = efct_event(evq, consumer_index);
	read_phase = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_PHASE);

	if (read_phase != evq_phase)
		goto end;

#ifdef CONFIG_XILINX_AUX_EFCT
	if (evq->type == EVQ_T_RX) {
		struct efct_rx_queue *rxq = (struct efct_rx_queue *)evq->queue;

		for (i = 0; i < evq->queue_count; ++i, ++rxq)
			efct_aux_poll(efct, rxq->index, quota);
	}
#endif

	while (spent < quota) {
		netif_vdbg(efct, drv, efct->net_dev, "processing event on %d %08x:%08x\n",
			   evq->index, le32_to_cpu(p_event->u32[1]),
			   le32_to_cpu(p_event->u32[0]));

		ev_type = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_TYPE);

		switch (ev_type) {
		case ESE_HZ_XN_EVENT_TYPE_RX_PKTS: {
			qlabel = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_RXPKTS_LABEL);
			netif_vdbg(efct, drv, efct->net_dev,
				   "RX completion event received for queue %d\n", qlabel);
			efct_ev_rx(&efct->rxq[qlabel], p_event);
			++spent;
			break;
		}
		case ESE_HZ_XN_EVENT_TYPE_TX_COMPLETION: {
			qlabel = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_TXCMPL_LABEL);
			netif_vdbg(efct, drv, efct->net_dev,
				   "TX completion event received for queue %d\n", qlabel);
			efct_ev_tx(&efct->txq[qlabel], p_event, false);
			++spent;
			break;
		}
		case ESE_HZ_XN_EVENT_TYPE_CONTROL:
			spent += efct_ev_control(evq, p_event, quota - spent);
			break;
		case ESE_HZ_XN_EVENT_TYPE_MCDI:
			efct_ev_mcdi(evq, p_event);
			++spent;
			break;
		case ESE_HZ_XN_EVENT_TYPE_DRIVER:
			efct_handle_driver_generated_event(evq, p_event);
			break;
		default:
			netif_warn(efct, probe, efct->net_dev, "Unhandled event %08x:%08x\n",
				   le32_to_cpu(p_event->u32[1]), le32_to_cpu(p_event->u32[0]));
		}

		++consumer_index;
		/*Entries */
		if (consumer_index == evq->entries) {
			evq_phase = !read_phase;
			consumer_index = 0;
		}
		p_event = efct_event(evq, consumer_index);
		read_phase = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_PHASE);
		if (read_phase != evq_phase)
			break;
	}

	if (evq->type == EVQ_T_TX) {
		struct efct_tx_queue *txq = (struct efct_tx_queue *)evq->queue;

		for (i = 0; i < evq->queue_count; ++i, ++txq) {
			if (txq->bytes) {
				netdev_tx_completed_queue(txq->core_txq, txq->pkts, txq->bytes);
				txq->pkts = 0;
				txq->bytes = 0;
			}
		}
	}

	evq->consumer_index = consumer_index;
	evq->evq_phase = evq_phase;

#ifdef CONFIG_XILINX_AUX_EFCT
	if (evq->type == EVQ_T_RX) {
		struct efct_rx_queue *rxq = (struct efct_rx_queue *)evq->queue;

		for (i = 0; i < evq->queue_count; ++i, ++rxq) {
			u32 sbseq = rxq->nbl.seq_no;

			if (rxq->nbl.meta_offset == 0)
				++sbseq;  /* nbl.seq_no tracks the payload's sb, so we need */
					  /* to increment it to track the meta sb */
			spent += efct_aux_handle_event(efct, XLNX_EVENT_WAKEUP,
						       rxq->index, ((u64)sbseq << 32) |
						       (rxq->nbl.meta_offset / rxq->pkt_stride),
						       quota - spent);
		}

		/* Aux client can take extra bite when remaining quota is zero.
		 * To avoid quota overflow, keeping spent to quota.
		 */
		if (spent > quota)
			spent = quota;
	}
#endif
end:
#if defined(EFCT_NOT_UPSTREAM) && defined(CONFIG_X3_BUSYPOLL)
	spent = quota;
#endif
	return spent;
}

static void efct_rx_fini(struct efct_rx_queue *rx_queue)
{
#if defined(CONFIG_XILINX_PTP) && defined(CONFIG_XILINX_AUX_EFCT)
	efct_ptp_unsubscribe_timesync(&rx_queue->efct->evq[rx_queue->evq_index]);
#endif
	rx_queue->enable = false;

	efct_mcdi_rx_fini(rx_queue);
}

static void efct_purge_rxq(struct efct_rx_queue *rxq)
{
	nbl_reset(rxq);
#ifdef CONFIG_XILINX_DEBUGFS
	efct_fini_debugfs_rx_queue(rxq);
#endif
}

static void efct_rx_remove(struct efct_rx_queue *rx_queue)
{
	struct efct_nic *efct = rx_queue->efct;

	/* Free driver buffers */
#ifdef CONFIG_XILINX_AUX_EFCT
	sbl_fini(rx_queue);
	hpl_fini(rx_queue);
#endif
	dbl_fini(rx_queue);

	if (!test_and_clear_bit(rx_queue->index, &efct->rxq_active_mask)) {
		netif_err(efct, probe, efct->net_dev,
			  "Rx queue is already removed, index = %d\n", rx_queue->index);
	}
}

static void efct_purge_txq(struct efct_tx_queue *txq)
{
	if (txq->core_txq) {
		_efct_ev_tx(txq, txq->added_sequence - 1, 0, 0, true);
		txq->pkts = 0;
		txq->bytes = 0;

		netdev_tx_reset_queue(txq->core_txq);
	}
#ifdef CONFIG_XILINX_DEBUGFS
	efct_fini_debugfs_tx_queue(txq);
#endif
}

static void efct_tx_fini(struct efct_tx_queue *tx_queue)
{
	efct_mcdi_tx_fini(tx_queue);
}

static void efct_tx_remove(struct efct_tx_queue *tx_queue)
{
	struct efct_nic *efct = tx_queue->efct;

	if (efct->evq[tx_queue->evq_index].type == EVQ_T_TX)
		efct->evq[tx_queue->evq_index].type = EVQ_T_NONE;

	if (!test_and_clear_bit(tx_queue->txq_index, &efct->txq_active_mask)) {
		netif_err(efct, ifdown, efct->net_dev,
			  "Tx queue is already removed\n");
	}
}

static irqreturn_t efct_msix_handler(int irq, void *dev_id)
{
	struct efct_ev_queue *evq = (struct efct_ev_queue *)dev_id;

	netif_vdbg(evq->efct, intr, evq->efct->net_dev, "Evq %d scheduling NAPI poll on CPU%d\n",
		   evq->index, raw_smp_processor_id());
	evq->event_test_cpu = raw_smp_processor_id();
	napi_schedule(&evq->napi);
	if (evq->index == 0)
		evq->efct->last_irq_cpu = raw_smp_processor_id();
	return IRQ_HANDLED;
}

static void efct_common_stat_mask(unsigned long *mask)
{
	__set_bit(EFCT_STAT_port_rx_packets, mask);
	__set_bit(EFCT_STAT_port_tx_packets, mask);
	__set_bit(EFCT_STAT_port_rx_bytes, mask);
	__set_bit(EFCT_STAT_port_tx_bytes, mask);
	__set_bit(EFCT_STAT_port_rx_multicast, mask);
	__set_bit(EFCT_STAT_port_rx_bad, mask);
	__set_bit(EFCT_STAT_port_rx_align_error, mask);
	__set_bit(EFCT_STAT_port_rx_overflow, mask);
}

static void efct_ethtool_stat_mask(unsigned long *mask)
{
	__set_bit(EFCT_STAT_port_tx_pause, mask);
	__set_bit(EFCT_STAT_port_tx_unicast, mask);
	__set_bit(EFCT_STAT_port_tx_multicast, mask);
	__set_bit(EFCT_STAT_port_tx_broadcast, mask);
	__set_bit(EFCT_STAT_port_tx_lt64, mask);
	__set_bit(EFCT_STAT_port_tx_64, mask);
	__set_bit(EFCT_STAT_port_tx_65_to_127, mask);
	__set_bit(EFCT_STAT_port_tx_128_to_255, mask);
	__set_bit(EFCT_STAT_port_tx_256_to_511, mask);
	__set_bit(EFCT_STAT_port_tx_512_to_1023, mask);
	__set_bit(EFCT_STAT_port_tx_1024_to_15xx, mask);
	__set_bit(EFCT_STAT_port_tx_15xx_to_jumbo, mask);
	__set_bit(EFCT_STAT_port_rx_good, mask);
	__set_bit(EFCT_STAT_port_rx_bad_bytes, mask);
	__set_bit(EFCT_STAT_port_rx_pause, mask);
	__set_bit(EFCT_STAT_port_rx_unicast, mask);
	__set_bit(EFCT_STAT_port_rx_broadcast, mask);
	__set_bit(EFCT_STAT_port_rx_lt64, mask);
	__set_bit(EFCT_STAT_port_rx_64, mask);
	__set_bit(EFCT_STAT_port_rx_65_to_127, mask);
	__set_bit(EFCT_STAT_port_rx_128_to_255, mask);
	__set_bit(EFCT_STAT_port_rx_256_to_511, mask);
	__set_bit(EFCT_STAT_port_rx_512_to_1023, mask);
	__set_bit(EFCT_STAT_port_rx_1024_to_15xx, mask);
	__set_bit(EFCT_STAT_port_rx_15xx_to_jumbo, mask);
	__set_bit(EFCT_STAT_port_rx_gtjumbo, mask);
	__set_bit(EFCT_STAT_port_rx_bad, mask);
	__set_bit(EFCT_STAT_port_rx_align_error, mask);
	__set_bit(EFCT_STAT_port_rx_bad_gtjumbo, mask);
	__set_bit(EFCT_STAT_port_rx_length_error, mask);
	__set_bit(EFCT_STAT_port_rx_nodesc_drops, mask);
	__set_bit(EFCT_STAT_port_pm_discard_vfifo_full, mask);
	__set_bit(EFCT_STAT_port_rxdp_q_disabled_pkts, mask);
	__set_bit(EFCT_STAT_port_rxdp_di_dropped_pkts, mask);
	__set_bit(EFCT_STAT_port_ctpio_underflow_fail, mask);
	__set_bit(EFCT_STAT_port_ctpio_success, mask);
	__set_bit(GENERIC_STAT_rx_drops, mask);
	__set_bit(GENERIC_STAT_tx_drops, mask);
}

#define EFCT_DMA_STAT(ext_name, mcdi_name)			\
	[EFCT_STAT_ ## ext_name] =				\
	{ #ext_name, 64, 8 * MC_CMD_MAC_ ## mcdi_name }

static const struct efct_hw_stat_desc efct_stat_desc[EFCT_STAT_COUNT] = {
	EFCT_DMA_STAT(port_tx_bytes, TX_BYTES),
	EFCT_DMA_STAT(port_tx_packets, TX_PKTS),
	EFCT_DMA_STAT(port_tx_pause, TX_PAUSE_PKTS),
	EFCT_DMA_STAT(port_tx_unicast, TX_UNICAST_PKTS),
	EFCT_DMA_STAT(port_tx_multicast, TX_MULTICAST_PKTS),
	EFCT_DMA_STAT(port_tx_broadcast, TX_BROADCAST_PKTS),
	EFCT_DMA_STAT(port_tx_lt64, TX_LT64_PKTS),
	EFCT_DMA_STAT(port_tx_64, TX_64_PKTS),
	EFCT_DMA_STAT(port_tx_65_to_127, TX_65_TO_127_PKTS),
	EFCT_DMA_STAT(port_tx_128_to_255, TX_128_TO_255_PKTS),
	EFCT_DMA_STAT(port_tx_256_to_511, TX_256_TO_511_PKTS),
	EFCT_DMA_STAT(port_tx_512_to_1023, TX_512_TO_1023_PKTS),
	EFCT_DMA_STAT(port_tx_1024_to_15xx, TX_1024_TO_15XX_PKTS),
	EFCT_DMA_STAT(port_tx_15xx_to_jumbo, TX_15XX_TO_JUMBO_PKTS),
	EFCT_DMA_STAT(port_rx_bytes, RX_BYTES),
	EFCT_DMA_STAT(port_rx_packets, RX_PKTS),
	EFCT_DMA_STAT(port_rx_good, RX_GOOD_PKTS),
	EFCT_DMA_STAT(port_rx_bad, RX_BAD_FCS_PKTS),
	EFCT_DMA_STAT(port_rx_bad_bytes, RX_BAD_BYTES),
	EFCT_DMA_STAT(port_rx_pause, RX_PAUSE_PKTS),
	EFCT_DMA_STAT(port_rx_unicast, RX_UNICAST_PKTS),
	EFCT_DMA_STAT(port_rx_multicast, RX_MULTICAST_PKTS),
	EFCT_DMA_STAT(port_rx_broadcast, RX_BROADCAST_PKTS),
	EFCT_DMA_STAT(port_rx_lt64, RX_UNDERSIZE_PKTS),
	EFCT_DMA_STAT(port_rx_64, RX_64_PKTS),
	EFCT_DMA_STAT(port_rx_65_to_127, RX_65_TO_127_PKTS),
	EFCT_DMA_STAT(port_rx_128_to_255, RX_128_TO_255_PKTS),
	EFCT_DMA_STAT(port_rx_256_to_511, RX_256_TO_511_PKTS),
	EFCT_DMA_STAT(port_rx_512_to_1023, RX_512_TO_1023_PKTS),
	EFCT_DMA_STAT(port_rx_1024_to_15xx, RX_1024_TO_15XX_PKTS),
	EFCT_DMA_STAT(port_rx_15xx_to_jumbo, RX_15XX_TO_JUMBO_PKTS),
	EFCT_DMA_STAT(port_rx_gtjumbo, RX_GTJUMBO_PKTS),
	EFCT_DMA_STAT(port_rx_bad_gtjumbo, RX_JABBER_PKTS),
	EFCT_DMA_STAT(port_rx_align_error, RX_ALIGN_ERROR_PKTS),
	EFCT_DMA_STAT(port_rx_length_error, RX_LENGTH_ERROR_PKTS),
	EFCT_DMA_STAT(port_rx_overflow, RX_OVERFLOW_PKTS),
	EFCT_DMA_STAT(port_rx_nodesc_drops, RX_NODESC_DROPS),
	EFCT_DMA_STAT(port_pm_discard_vfifo_full, PM_DISCARD_VFIFO_FULL),
	EFCT_DMA_STAT(port_rxdp_q_disabled_pkts, RXDP_Q_DISABLED_PKTS),
	EFCT_DMA_STAT(port_rxdp_di_dropped_pkts, RXDP_DI_DROPPED_PKTS),
	EFCT_DMA_STAT(port_ctpio_underflow_fail, CTPIO_UNDERFLOW_FAIL),
	EFCT_DMA_STAT(port_ctpio_success, CTPIO_SUCCESS),
	EFCT_GENERIC_SW_STAT(rx_drops),
	EFCT_GENERIC_SW_STAT(tx_drops),
};

static size_t efct_describe_stats(struct efct_nic *efct, u8 *names)
{
	DECLARE_BITMAP(mask, EFCT_STAT_COUNT) = {};

	efct_ethtool_stat_mask(mask);
	return efct_nic_describe_stats(efct_stat_desc, EFCT_STAT_COUNT, mask, names);
}

static void efct_update_sw_stats(struct efct_nic *efct, u64 *stats)
{
	stats[GENERIC_STAT_rx_drops] = atomic64_read(&efct->n_rx_sw_drops);
	stats[GENERIC_STAT_tx_drops] = atomic64_read(&efct->n_tx_sw_drops);
}

size_t efct_update_stats_common(struct efct_nic *efct, u64 *full_stats,
				struct rtnl_link_stats64 *core_stats)
{
	struct efct_nic_data *nic_data = efct->nic_data;
	DECLARE_BITMAP(mask, EFCT_STAT_COUNT) = {};
	u64 *stats = nic_data->stats;
	int stats_count = 0, index;

	efct_ethtool_stat_mask(mask);

	if (full_stats) {
		for_each_set_bit(index, mask, EFCT_STAT_COUNT) {
			if (efct_stat_desc[index].name) {
				*full_stats++ = stats[index];
				++stats_count;
			}
		}
	}
	if (core_stats) {
		core_stats->rx_packets = stats[EFCT_STAT_port_rx_packets];
		core_stats->tx_packets = stats[EFCT_STAT_port_tx_packets];
		core_stats->rx_bytes = stats[EFCT_STAT_port_rx_bytes];
		core_stats->tx_bytes = stats[EFCT_STAT_port_tx_bytes];
		core_stats->rx_dropped = stats[EFCT_STAT_port_rx_nodesc_drops] +
				stats[GENERIC_STAT_rx_drops] +
				stats[EFCT_STAT_port_pm_discard_vfifo_full] +
				stats[EFCT_STAT_port_rxdp_q_disabled_pkts];
		core_stats->tx_dropped = stats[GENERIC_STAT_tx_drops];
		core_stats->multicast = stats[EFCT_STAT_port_rx_multicast];
		core_stats->rx_length_errors =
				stats[EFCT_STAT_port_rx_gtjumbo] +
				stats[EFCT_STAT_port_rx_length_error];
		core_stats->rx_crc_errors = stats[EFCT_STAT_port_rx_bad];
		core_stats->rx_frame_errors =
				stats[EFCT_STAT_port_rx_align_error];
		core_stats->rx_fifo_errors = stats[EFCT_STAT_port_rx_overflow];
		core_stats->rx_errors = (core_stats->rx_length_errors +
				core_stats->rx_crc_errors +
				core_stats->rx_frame_errors);
	}

	return stats_count;
}

static void efct_update_stats(struct efct_nic *efct, bool force)
{
	struct efct_nic_data *nic_data = efct->nic_data;
	DECLARE_BITMAP(mask, EFCT_STAT_COUNT) = {};
	u64 *stats = nic_data->stats;
	int rc = -1;

	efct_common_stat_mask(mask);
	efct_ethtool_stat_mask(mask);

	mutex_lock(&efct->state_lock);
	if (efct->state == STATE_NET_UP || force) {
		rc = efct_mcdi_mac_stats(efct, EFCT_STATS_NODMA, 0,
					 efct->mc_mac_stats, efct->num_mac_stats * sizeof(__le64));
	}
	mutex_unlock(&efct->state_lock);

	if (rc)
		return;

	spin_lock_bh(&efct->stats_lock);
	efct_nic_update_stats(efct_stat_desc, EFCT_STAT_COUNT, mask,
			      stats, efct->mc_initial_stats, efct->mc_mac_stats);
	efct_update_sw_stats(efct, stats);
	spin_unlock_bh(&efct->stats_lock);
}

static int efct_nic_reset_stats(struct efct_nic *efct)
{
	return efct_mcdi_mac_stats(efct, EFCT_STATS_NODMA, 0, efct->mc_initial_stats,
				  efct->num_mac_stats * sizeof(__le64));
}

static void efct_pull_stats(struct efct_nic *efct)
{
	if (!efct->stats_initialised) {
		efct_reset_sw_stats(efct);
		efct_nic_reset_stats(efct);
#ifdef CONFIG_XILINX_PTP
		efct_ptp_reset_stats(efct);
#endif
		efct->stats_initialised = true;
	}
}

/**
 * efct_nic_describe_stats - Describe supported statistics for ethtool
 * @desc: Array of &struct efct_hw_stat_desc describing the statistics
 * @count: Length of the @desc array
 * @mask: Bitmask of which elements of @desc are enabled
 * @names: Buffer to copy names to, or %NULL.  The names are copied
 *	starting at intervals of %ETH_GSTRING_LEN bytes.
 *
 * return: Number of visible statistics, i.e. the number of set
 * bits in the first @count bits of @mask for which a name is defined.
 */
int efct_nic_describe_stats(const struct efct_hw_stat_desc *desc, size_t count,
			    const unsigned long *mask, u8 *names)
{
	int visible = 0;
	int index;

	for_each_set_bit(index, mask, count) {
		if (desc[index].name) {
			if (names) {
				strscpy(names, desc[index].name, ETH_GSTRING_LEN);
				names += ETH_GSTRING_LEN;
			}
			++visible;
		}
	}

	return visible;
}

/**
 * efct_nic_update_stats - Convert statistics DMA buffer to array of u64
 * @desc: Array of &struct efct_hw_stat_desc describing the DMA buffer
 *	layout.  DMA widths of 0, 16, 32 and 64 are supported; where
 *	the width is specified as 0 the corresponding element of
 *	@stats is not updated.
 * @count: Length of the @desc array
 * @mask: Bitmask of which elements of @desc are enabled
 * @stats: Buffer to update with the converted statistics.  The length
 *	of this array must be at least @count.
 * @mc_initial_stats: Copy of DMA buffer containing initial stats. Subtracted
 *	from the stats in mc_stats.
 * @mc_stats: DMA buffer containing hardware statistics
 */
void efct_nic_update_stats(const struct efct_hw_stat_desc *desc, size_t count,
			   const unsigned long *mask, u64 *stats,
			  const __le64 *mc_initial_stats, const __le64 *mc_stats)
{
	u8 zero[8] = {0};
	size_t index;

	for_each_set_bit(index, mask, count) {
		if (desc[index].dma_width) {
			const void *addr = mc_stats ? (u8 *)mc_stats + desc[index].offset : zero;
			const void *init = mc_initial_stats && mc_stats ? (u8 *)mc_initial_stats +
					   desc[index].offset : zero;

			switch (desc[index].dma_width) {
			case 16:
				stats[index] = le16_to_cpup((__le16 *)addr) -
						le16_to_cpup((__le16 *)init);
				break;
			case 32:
				stats[index] = le32_to_cpup((__le32 *)addr) -
						le32_to_cpup((__le32 *)init);
				break;
			case 64:
				stats[index] = le64_to_cpup((__le64 *)addr) -
						le64_to_cpup((__le64 *)init);
				break;
			default:
				WARN_ON_ONCE(1);
				stats[index] = 0;
				break;
			}
		}
	}
}

static u32 efct_check_caps(const struct efct_nic *efct, u8 flag, u32 offset)
{
	switch (offset) {
	case(MC_CMD_GET_CAPABILITIES_V10_OUT_FLAGS1_OFST):
		return efct->datapath_caps & BIT_ULL(flag);
	case(MC_CMD_GET_CAPABILITIES_V10_OUT_FLAGS2_OFST):
		return efct->datapath_caps2 & BIT_ULL(flag);
	default:
		return 0;
	}
	return 0;
}

static int efct_reconfigure_mac(struct efct_nic *efct)
{
	int rc;

	WARN_ON(!mutex_is_locked(&efct->mac_lock));

	rc = efct_mcdi_set_mac(efct);
	if (rc)
		netif_err(efct, hw, efct->net_dev,
			  "efct_mcdi_set_mac failed\n");

	return rc;
}

static int efct_map_reset_flags(u32 *flags)
{
	enum {
		EFCT_RESET_PORT = ((ETH_RESET_MAC | ETH_RESET_PHY) <<
				ETH_RESET_SHARED_SHIFT),
		EFCT_RESET_MC = ((ETH_RESET_DMA | ETH_RESET_FILTER |
					ETH_RESET_MAC | ETH_RESET_PHY | ETH_RESET_MGMT) <<
					ETH_RESET_SHARED_SHIFT)
	};

	if ((*flags & EFCT_RESET_MC) == EFCT_RESET_MC) {
		*flags &= ~EFCT_RESET_MC;
		return RESET_TYPE_WORLD;
	}

	if ((*flags & EFCT_RESET_PORT) == EFCT_RESET_PORT) {
		*flags &= ~EFCT_RESET_PORT;
		return RESET_TYPE_ALL;
	}

	/* no invisible reset implemented */
	return -EINVAL;
}

static enum reset_type efct_map_reset_reason(enum reset_type reason)
{
	if (reason == RESET_TYPE_MC_FAILURE)
		return RESET_TYPE_DATAPATH;
	else if (reason == RESET_TYPE_MC_BIST)
		return reason;
	else if (reason == RESET_TYPE_MCDI_TIMEOUT)
		return reason;
	else
		return RESET_TYPE_ALL;
}

static int efct_type_reset(struct efct_nic *efct, enum reset_type reset_type)
{
	u32 attach_flags;
	bool was_up;
	int rc = 0;

	if (efct->net_dev)
		was_up = netif_running(efct->net_dev);
	else
		return -EINVAL;

	if (efct->reset_pending & (1 << RESET_TYPE_MCDI_TIMEOUT))
		return efct_flr(efct);

	if (was_up) {
		netif_device_detach(efct->net_dev);
		dev_close(efct->net_dev);
	}

	/* Bring down filter table */
	if (reset_type != RESET_TYPE_DATAPATH) {
		rc = efct_filter_table_down(efct);
		if (rc)
			goto err;
	}
	rc = efct_mcdi_reset(efct, reset_type);
	if (rc)
		goto err;

	efct->last_reset = jiffies;

	rc = efct_mcdi_drv_attach(efct, MC_CMD_FW_FULL_FEATURED, &attach_flags, true);
	if (rc) {
		/* This is not fatal as there are no fw variants */
		netif_warn(efct, probe, efct->net_dev,
			   "failed to re-attach driver after reset rc=%d\n", rc);
	}
#ifdef CONFIG_XILINX_AUX_EFCT
	/* Clear AUX resources after reset */
	efct_aux_on_dev_reset(efct);
#endif
	/* Bring up filter table */
	rc = efct_filter_table_up(efct);
	if (rc)
		goto err;

	/* Resetting statistics */
	netif_info(efct, drv, efct->net_dev, "Resetting statistics.\n");
	efct->stats_initialised = false;
	efct->type->pull_stats(efct);
#ifdef CONFIG_XILINX_PTP
	if (efct_phc_exposed(efct)) {
		rc = efct_ptp_start(efct);
		if (rc < 0) {
			netif_err(efct, probe, efct->net_dev, "PTP enable failed in reset\n");
			goto err;
		}
		rc = efct_ptp_hw_pps_enable(efct, true);
		if (rc < 0) {
			netif_err(efct, probe, efct->net_dev, "PPS enable failed in reset.\n");
			goto err;
		}
	}
#endif
	if (was_up) {
		netif_device_attach(efct->net_dev);
		return dev_open(efct->net_dev, NULL);
	}

	/* Update stats forcefully so stats are reset when interface was already down */
	efct->type->update_stats(efct, true);
err:
	return rc;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_NDO_GET_PHYS_PORT_ID)
static int efct_type_get_phys_port_id(struct efct_nic *efct, struct netdev_phys_item_id *ppid)
{
	if (!is_valid_ether_addr(efct->net_dev->dev_addr))
		return -EOPNOTSUPP;

	ppid->id_len = ETH_ALEN;
	memcpy(ppid->id, efct->net_dev->dev_addr, ppid->id_len);

	return 0;
}
#endif

static void efct_mcdi_reboot_detected(struct efct_nic *efct)
{
	/*dummy function*/
}

static int efct_type_irq_test_generate(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_TRIGGER_INTERRUPT_IN_LEN);
	int index;

	// 2nd port of PF
	index = (efct->port_num == 2) || (efct->port_num == 3);
	BUILD_BUG_ON(MC_CMD_TRIGGER_INTERRUPT_OUT_LEN != 0);

	MCDI_SET_DWORD(inbuf, TRIGGER_INTERRUPT_IN_INTR_LEVEL, index * EFCT_MSIX_PER_PORT);

	return efct_mcdi_rpc_quiet(efct, MC_CMD_TRIGGER_INTERRUPT, inbuf, sizeof(inbuf),
				  NULL, 0, NULL);
}

static void efct_type_ev_test_generate(struct efct_ev_queue *evq)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_DRIVER_EVENT_IN_LEN);
	struct efct_nic *efct = evq->efct;
	union efct_qword event;
	int rc;

	EFCT_POPULATE_QWORD_2(event,
			      ESF_HZ_EV_TYPE, ESE_HZ_XN_EVENT_TYPE_DRIVER,
			      ESF_HZ_DRIVER_DATA, EFCT_TEST);

	MCDI_SET_DWORD(inbuf, DRIVER_EVENT_IN_EVQ, evq->index);

	/* MCDI_SET_QWORD is not appropriate here since EFCT_POPULATE_* has
	 * already swapped the data to little-endian order.
	 */
	memcpy(MCDI_PTR(inbuf, DRIVER_EVENT_IN_DATA), &event.u64[0],
	       sizeof(union efct_qword));

	rc = efct_mcdi_rpc(efct, MC_CMD_DRIVER_EVENT, inbuf, sizeof(inbuf),
			   NULL, 0, NULL);
	if (rc && (rc != -ENETDOWN))
		goto fail;

	return;

fail:
	netif_err(efct, hw, efct->net_dev, "%s: failed rc=%d\n", __func__, rc);
}

const struct efct_nic_type efct_nic_type = {
	.probe = efct_probe_main,
	.remove = efct_remove_main,
	.mcdi_max_ver = 2,
	.mcdi_rpc_timeout = efct_mcdi_rpc_timeout,
	.mcdi_request = efct_mcdi_request,
	.mcdi_poll_response = efct_mcdi_poll_response,
	.mcdi_read_response = efct_mcdi_read_response,
	.mcdi_poll_reboot = mcdi_poll_reboot,
	.mcdi_get_buf = efct_mcdi_get_buf,
	.mcdi_put_buf = efct_mcdi_put_buf,
	.mem_bar = NULL,
	.ev_probe = efct_ev_probe,
	.ev_init = efct_ev_init,
	.ev_remove = efct_ev_remove,
	.ev_fini = efct_ev_fini,
	.ev_purge = efct_ev_purge,
	.ev_process = efct_ev_process,
	.tx_probe = efct_tx_probe,
	.tx_init = efct_tx_init,
	.tx_remove = efct_tx_remove,
	.tx_fini = efct_tx_fini,
	.tx_purge = efct_purge_txq,
	.rx_probe = efct_rx_probe,
	.rx_init = efct_rx_init,
	.rx_remove = efct_rx_remove,
	.rx_fini = efct_rx_fini,
	.rx_purge = efct_purge_rxq,
	.irq_handle_msix = efct_msix_handler,
	.describe_stats = efct_describe_stats,
	.update_stats = efct_update_stats,
	.pull_stats = efct_pull_stats,
	.check_caps = efct_check_caps,
	.reconfigure_mac = efct_reconfigure_mac,
	.map_reset_flags = efct_map_reset_flags,
	.map_reset_reason = efct_map_reset_reason,
	.reset = efct_type_reset,
	.has_dynamic_sensors = efct_has_dynamic_sensors,
#ifdef CONFIG_XILINX_PTP
	.ptp_set_ts_config = efct_ptp_enable_ts,
	.hwtstamp_filters = 1 << HWTSTAMP_FILTER_NONE | 1 << HWTSTAMP_FILTER_ALL,
#endif
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_NDO_GET_PHYS_PORT_ID)
	.get_phys_port_id = efct_type_get_phys_port_id,
#endif
	.mcdi_reboot_detected = efct_mcdi_reboot_detected,
	.irq_test_generate = efct_type_irq_test_generate,
	.ev_test_generate = efct_type_ev_test_generate,
};
