// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include "mcdi_pcol.h"
#include "mcdi.h"
#include "efct_netdev.h"
#include "mcdi_functions.h"
#include "efct_io.h"
#include "efct_reg.h"
#include "efct_ptp.h"
#include "efct_nic.h"

static LIST_HEAD(efct_phcs_list);
static DEFINE_SPINLOCK(efct_phcs_list_lock);

/* Maximum parts-per-billion adjustment that is acceptable */
#define MAX_PPB			100000000

#define PTP_SYNC_SAMPLE	10

/*  */
#define	SYNCHRONISATION_GRANULARITY_NS	1400

#define PTP_TIME_READ_SAMPLE 6

/* Number of bits of sub-nanosecond in partial timestamps */
#define PARTIAL_TS_SUB_BITS_2 2

#define PARTIAL_TS_NANO_MASK	0xffffffff
#define PARTIAL_TS_SEC_MASK	0xff
#define TIME_TO_SEC_SHIFT	32
#define SYNC_TS_SEC_SHIFT	16
#define ONE_BILLION		1000000000
/* 64 bit second not supported, Use zero high 32 bit of second*/
#define SEC_HIGH32		0

static void efct_ptp_ns_to_s_qns(s64 ns, u32 *nic_major, u32 *nic_minor, u32 *nic_hi)
{
	struct timespec64 ts = ns_to_timespec64(ns);

	*nic_hi = (u32)(ts.tv_sec >> TIME_TO_SEC_SHIFT);
	*nic_major = (u32)ts.tv_sec;
	*nic_minor = ts.tv_nsec << PARTIAL_TS_SUB_BITS_2;
}

static u64 efct_ptp_time(struct efct_nic *efct)
{
	return le64_to_cpu(_efct_readq(efct->membase + ER_HZ_PORT0_REG_HOST_THE_TIME));
}

void efct_ptp_evt_data_init(struct efct_nic *efct)
{
	struct efct_ptp_data *ptp;

	ptp = efct->ptp_data;

	ptp->evt_frag_idx = 0;
	ptp->evt_code = 0;
}

static ktime_t efct_ptp_s_qns_to_ktime_correction(u32 nic_major, u32 nic_minor,
						  s32 correction)
{
	ktime_t kt;

	nic_minor = DIV_ROUND_CLOSEST(nic_minor, 4);
	correction = DIV_ROUND_CLOSEST(correction, 4);

	kt = ktime_set(nic_major, nic_minor);

	if (correction >= 0)
		kt = ktime_add_ns(kt, (u64)correction);
	else
		kt = ktime_sub_ns(kt, (u64)-correction);
	return kt;
}

static ktime_t efct_ptp_s64_qns_to_ktime_correction(u32 nich, u64 timereg,
						    s64 correction)

{
	u64 nic_major;
	u32 nic_minor;
	ktime_t kt;

	nic_minor = (timereg & 0xFFFFFFFF);
	nic_minor = DIV_ROUND_CLOSEST(nic_minor, 4);
	correction = DIV_ROUND_CLOSEST(correction, 4);
	nic_major = (timereg >> TIME_TO_SEC_SHIFT) | ((u64)nich << 32);
	kt = ktime_set(nic_major, nic_minor);

	if (correction >= 0)
		kt = ktime_add_ns(kt, (u64)correction);
	else
		kt = ktime_sub_ns(kt, (u64)-correction);
	return kt;
}

bool efct_phc_exposed(struct efct_nic *efct)
{
	return efct->phc_ptp_data == efct->ptp_data && efct->ptp_data;
}

static void efct_ptp_delete_data(struct kref *kref)
{
	struct efct_ptp_data *ptp = container_of(kref, struct efct_ptp_data,
						kref);

	ptp->efct = NULL;
	kfree(ptp);
}

static s64 efct_get_the_time_read_delay(struct efct_nic *efct)
{
	u64 time[PTP_TIME_READ_SAMPLE];
	struct efct_ptp_data *ptp;
	s64 mindiff = LONG_MAX;
	ktime_t t1, t0;
	int i;

	ptp = efct->ptp_data;

	/* Assuming half of time taken to read PCI register
	 * as correction
	 */
	for (i = 0; i < ARRAY_SIZE(time); i++)
		time[i] = efct_ptp_time(efct);
	for (i = 0; i < ARRAY_SIZE(time) - 1; i++) {
		t1 = ptp->nic64_to_kernel_time(0, time[i + 1], 0);
		t0 = ptp->nic64_to_kernel_time(0, time[i], 0);
		if (ktime_to_ns(ktime_sub(t1, t0)) < mindiff)
			mindiff = ktime_to_ns(ktime_sub(t1, t0));
	}

	return -(mindiff  >> 1);
}

int efct_ptp_ts_set_sync_status(struct efct_nic *efct, u32 in_sync, u32 timeout)
{
	MCDI_DECLARE_BUF(mcdi_req, MC_CMD_PTP_IN_SET_SYNC_STATUS_LEN);
	u32 flag;
	int rc;

	if (!efct->ptp_data)
		return -ENOTTY;

	if (!(efct->ptp_data->capabilities &
		(1 << MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_REPORT_SYNC_STATUS_LBN)))
		return -EOPNOTSUPP;

	if (in_sync != 0)
		flag = MC_CMD_PTP_IN_SET_SYNC_STATUS_IN_SYNC;
	else
		flag = MC_CMD_PTP_IN_SET_SYNC_STATUS_NOT_IN_SYNC;

	MCDI_SET_DWORD(mcdi_req, PTP_IN_OP, MC_CMD_PTP_OP_SET_SYNC_STATUS);
	MCDI_SET_DWORD(mcdi_req, PTP_IN_PERIPH_ID, 0);
	MCDI_SET_DWORD(mcdi_req, PTP_IN_SET_SYNC_STATUS_STATUS, flag);
	MCDI_SET_DWORD(mcdi_req, PTP_IN_SET_SYNC_STATUS_TIMEOUT, timeout);

	rc = efct_mcdi_rpc(efct, MC_CMD_PTP, mcdi_req, sizeof(mcdi_req),
			   NULL, 0, NULL);
	return rc;
}

static int efct_ptp_synchronize(struct efct_nic *efct, u32 num_readings)
{
	struct pps_event_time last_time;
	u32 ngood = 0, last_good = 0;
	struct efct_ptp_timeset *ts;
	struct timespec64 mc_time;
	struct efct_ptp_data *ptp;
	struct timespec64 delta;
	s64 diff_min = LONG_MAX;
	struct timespec64 diff;
	s64 diff_avg = 0;
	s64 correction;
	int rc = 0;
	u32 i;

	ts = kmalloc_array(num_readings, sizeof(*ts), GFP_KERNEL);
	if (!ts)
		return -ENOMEM;
	ptp = efct->ptp_data;

	for (i = 0; i <  num_readings; i++) {
		ktime_get_real_ts64(&ts[ngood].prets);
		ts[ngood].nictime = efct_ptp_time(efct);
		ktime_get_real_ts64(&ts[ngood].posts);
		diff = timespec64_sub(ts[ngood].posts, ts[ngood].prets);
		ts[ngood].window = timespec64_to_ns(&diff);
		if (ts[ngood].window > SYNCHRONISATION_GRANULARITY_NS) {
			//TODO addstat. Adjust SYNCHRONISATION_GRANULARITY_NS
			// macro value based on experiments such this it can guess there is
			// context switch between pre and post reading
			++ptp->sw_stats.invalid_sync_windows;
		} else {
			mc_time = ktime_to_timespec64(ptp->nic64_to_kernel_time
						      (SEC_HIGH32, ts[ngood].nictime, 0));
			diff = timespec64_sub(mc_time, ts[ngood].prets);
			ts[ngood].mc_host_diff = timespec64_to_ns(&diff);

			diff_avg += div_s64((ts[ngood].mc_host_diff - diff_avg), (ngood + 1));
			ngood++;
		}
	}
	pps_get_ts(&last_time);
	if (ngood == 0) {
		++ptp->sw_stats.skipped_sync;
		rc = -EAGAIN;
		goto out;
	}

	if (ngood > 2) { /* No point doing this if only 1-2 valid samples, Use last_good 0*/
		/* Find the sample which is closest to the average */
		for (i = 0; i < ngood; i++) {
			s64 d = abs(ts[i].mc_host_diff - diff_avg);

			if (d < diff_min) {
				diff_min = d;
				last_good = i;
			}
		}
	}
	correction = efct_get_the_time_read_delay(efct);
	// Pass correction in quarter nano second format
	mc_time = ktime_to_timespec64(ptp->nic64_to_kernel_time
				(SEC_HIGH32, ts[last_good].nictime, correction <<
				 efct->efct_dev->params.ts_subnano_bit));
	ptp->last_delta = timespec64_sub(mc_time, ts[last_good].prets);
	delta = timespec64_sub(last_time.ts_real, ts[last_good].prets);
	timespec64_add_ns(&delta, mc_time.tv_nsec);
	pps_sub_ts(&last_time, delta);
	ptp->host_time_pps = last_time;
out:
	kfree(ts);
	return rc;
}

static void *ptp_data_alloc(struct efct_nic *efct, gfp_t flags)
{
	struct efct_ptp_data *ptp;

	ptp = kzalloc(sizeof(*ptp), flags);
	if (!ptp)
		return ERR_PTR(-ENOMEM);
	kref_init(&ptp->kref);
	ptp->efct = efct;

	return ptp;
}

static unsigned int ptp_data_kref_read(struct efct_ptp_data *ptp)
{
	return kref_read(&ptp->kref);
}

static void ptp_data_get(struct efct_ptp_data *ptp)
{
	kref_get(&ptp->kref);
}

static void ptp_data_put(struct efct_ptp_data *ptp)
{
	kref_put(&ptp->kref, efct_ptp_delete_data);
}

static void ptp_data_del(struct efct_ptp_data *ptp)
{
	ptp->efct = NULL;
	kref_put(&ptp->kref, efct_ptp_delete_data);
}

/* Convert ppb value, clamped to +/- 10^9, to FP16 ppm. */
static s64 ppb_to_ppm_fp16(s64 ppb)
{
	ppb = clamp_val(ppb, -ONE_BILLION, ONE_BILLION);

	/* Multiply by floor(2^41 / 1000) and shift right by 41 - 16). */
	return (ppb * 2199023255LL) >> (41 - 16);
}

static int efct_ptp_get_attributes(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_PTP_IN_GET_ATTRIBUTES_LEN);
	struct efct_ptp_data *ptp;
	s64 freq_adj_min;
	s64 freq_adj_max;
	size_t out_len;
	u32 fmt;
	int rc;

	ptp = efct->ptp_data;

	MCDI_SET_DWORD(inbuf, PTP_IN_OP, MC_CMD_PTP_OP_GET_ATTRIBUTES);
	MCDI_SET_DWORD(inbuf, PTP_IN_PERIPH_ID, 0);
	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_PTP, inbuf, sizeof(inbuf),
				 outbuf, sizeof(outbuf), &out_len);

	if (rc != 0) {
		pci_err(efct->efct_dev->pci_dev, "No PTP support\n");
		efct_mcdi_display_error(efct, MC_CMD_PTP, sizeof(inbuf),
					outbuf, sizeof(outbuf), rc);
		goto out;
	}
	fmt = MCDI_DWORD(outbuf, PTP_OUT_GET_ATTRIBUTES_V2_TIME_FORMAT);
	switch (fmt) {
	case MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_SECONDS_QTR_NANOSECONDS:
		if (efct->efct_dev->params.ts_subnano_bit != PARTIAL_TS_SUB_BITS_2) {
			rc = -EINVAL;
			pci_err(efct->efct_dev->pci_dev,
				"Error: Time format %d and design param %d not in sync\n",
				fmt, efct->efct_dev->params.ts_subnano_bit);
			goto out;
		}
		ptp->ns_to_nic_time = efct_ptp_ns_to_s_qns;
		ptp->nic_to_kernel_time = efct_ptp_s_qns_to_ktime_correction;
		ptp->nic64_to_kernel_time = efct_ptp_s64_qns_to_ktime_correction;
		ptp->nic_time.minor_max = 4000000000UL;
		ptp->nic_time.sync_event_minor_shift = 24;
		break;
	default:
		pci_err(efct->efct_dev->pci_dev, "Time format not supported 0x%x\n", fmt);
		rc = -ERANGE;
		goto out;
	}

	ptp->capabilities = MCDI_DWORD(outbuf, PTP_OUT_GET_ATTRIBUTES_V2_CAPABILITIES);
	/* Set up the shift for conversion between frequency
	 * adjustments in parts-per-billion and the fixed-point
	 * fractional ns format that the adapter uses.
	 */
	if (ptp->capabilities & (1 << MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_FP44_FREQ_ADJ_LBN)) {
		ptp->adjfreq_prec = 44;
	} else {
		pci_err(efct->efct_dev->pci_dev, "Unsupported fixed-point representation of frequency adjustments\n");
		return -EINVAL;
	}
	if (out_len >= MC_CMD_PTP_OUT_GET_ATTRIBUTES_V2_LEN) {
		freq_adj_min = MCDI_QWORD(outbuf, PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MIN);
		freq_adj_max = MCDI_QWORD(outbuf, PTP_OUT_GET_ATTRIBUTES_V2_FREQ_ADJ_MAX);

		/* The Linux PTP Hardware Clock interface expects frequency adjustments(adj_ppb) in
		 * parts per billion(e.g. +10000000 means go 1% faster; -50000000 means go 5%
		 * slower). The MCDI interface between the driver and the NMC firmware uses a
		 * nanosecond based adjustment
		 * (e.g. +0.01 means go 1% faster, -0.05 means go 5% slower).
		 *	adj_ns = adj_ppb / 10 ^ 9
		 * adj_ns is represented in the MCDI messages as a signed fixed-point 64-bit value
		 * with either 40 or 44 bits for the fractional part.
		 * For the 44 bit format:
		 * adj_ns_fp44 = adj_ns * 2^44
		 * adj_ppb = adj_ns_fp44 * 10^9 / 2^44
		 * The highest common factor of those is 2^9 so that is equivalent to:
		 * adj_ppb = adj_ns_fp44 * 1953125 / 2^35\
		 * adj_ppb = 10000000 equivalent to adj_ns = 0.01, corresponding representation in
		 * fixed point 44 bit format  is 028F5C28F5C.
		 */
		freq_adj_min = (freq_adj_min * 1953125LL) >> 35;
		freq_adj_max = (freq_adj_max * 1953125LL) >> 35;
		ptp->max_adjfreq = min_t(s64, abs(freq_adj_min), abs(freq_adj_max));
	} else {
		ptp->max_adjfreq = MAX_PPB;
	}
	ptp->max_adjfine = ppb_to_ppm_fp16(ptp->max_adjfreq);
out:
	return rc;
}

static int efct_mcdi_ptp_op_enable(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_PTP_IN_ENABLE_LEN);
	MCDI_DECLARE_BUF_ERR(outbuf);
	int rc;

	MCDI_SET_DWORD(inbuf, PTP_IN_OP, MC_CMD_PTP_OP_ENABLE);
	MCDI_SET_DWORD(inbuf, PTP_IN_PERIPH_ID, 0);

	MCDI_SET_DWORD(inbuf, PTP_IN_ENABLE_MODE, 0);

	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_PTP, inbuf, sizeof(inbuf),
				 outbuf, sizeof(outbuf), NULL);
	rc = (rc == -EALREADY) ? 0 : rc;
	if (rc)
		efct_mcdi_display_error(efct, MC_CMD_PTP,
					MC_CMD_PTP_IN_ENABLE_LEN,
				       outbuf, sizeof(outbuf), rc);
	return rc;
}

static int efct_mcdi_ptp_op_disable(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_PTP_IN_DISABLE_LEN);
	MCDI_DECLARE_BUF_ERR(outbuf);
	int rc;

	MCDI_SET_DWORD(inbuf, PTP_IN_OP, MC_CMD_PTP_OP_DISABLE);
	MCDI_SET_DWORD(inbuf, PTP_IN_PERIPH_ID, 0);
	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_PTP, inbuf, sizeof(inbuf),
				 outbuf, sizeof(outbuf), NULL);
	rc = (rc == -EALREADY) ? 0 : rc;
	if (rc)
		efct_mcdi_display_error(efct, MC_CMD_PTP,
					MC_CMD_PTP_IN_DISABLE_LEN,
				       outbuf, sizeof(outbuf), rc);
	return rc;
}

int efct_ptp_start(struct efct_nic *efct)
{
	struct efct_ptp_data *ptp = efct->ptp_data;
	int rc;

	rc = efct_mcdi_ptp_op_enable(efct);
	if (rc != 0)
		goto fail;

	ptp->evt_frag_idx = 0;
	ptp->current_adjfreq = 0;
	ptp->enabled = true;
	return 0;

fail:
	return rc;
}

int efct_ptp_stop(struct efct_nic *efct)
{
	struct efct_ptp_data *ptp = efct->ptp_data;
	int rc;

	rc = efct_mcdi_ptp_op_disable(efct);
	ptp->enabled = false;

	return rc;
}

static ktime_t efct_ptp_nic_to_kernel_time(struct efct_tx_queue *txq, u32 sec, u32 nano)
{
	struct efct_ev_queue *evq;
	struct efct_ptp_data *ptp;
	struct efct_nic *efct;
	ktime_t kt = { 0 };
	u32 sync_major;
	s8 delta;

	efct = txq->efct;
	ptp = efct->ptp_data;
	evq = &txq->efct->evq[txq->evq_index];
	//TODO How to handle ovelapping nanoseconds
	sync_major = evq->sync_timestamp_major >> SYNC_TS_SEC_SHIFT;
	delta = sec - sync_major;
	sec = sync_major + delta;
	kt = ptp->nic_to_kernel_time(sec, nano, ptp->ts_corrections.general_tx);
	return kt;
}

void efct_include_ts_in_skb(struct efct_tx_queue *txq, u64 partial_ts, struct sk_buff *skb)
{
	struct skb_shared_hwtstamps timestamps;
	struct efct_ev_queue *evq;
	u32 nano;
	u32 sec;

	evq = &txq->efct->evq[txq->evq_index];
	if (evq->sync_events_state != SYNC_EVENTS_VALID)
		return;
	memset(&timestamps, 0, sizeof(timestamps));
	nano = (partial_ts & PARTIAL_TS_NANO_MASK);
	sec = (partial_ts >> TIME_TO_SEC_SHIFT) & PARTIAL_TS_SEC_MASK;
	timestamps.hwtstamp = efct_ptp_nic_to_kernel_time(txq, sec, nano);
	//TODO: Enable PTP stat to track dropped timestamp once validation of timestamp is added
	// ++ptp->sw_stats.invalid_sync_windows;
	skb_tstamp_tx(skb, &timestamps);
}

#define PTP_SW_STAT(ext_name, field_name)				\
	{ #ext_name, 0, offsetof(struct efct_ptp_data, field_name) }
#define PTP_MC_STAT(ext_name, mcdi_name)				\
	{ #ext_name, 32, MC_CMD_PTP_OUT_STATUS_STATS_ ## mcdi_name ## _OFST }
static const struct efct_hw_stat_desc efct_ptp_stat_desc[] = {
	PTP_SW_STAT(ptp_invalid_sync_windows, sw_stats.invalid_sync_windows),
	PTP_SW_STAT(ptp_skipped_sync, sw_stats.skipped_sync),
	PTP_SW_STAT(pps_fw, sw_stats.pps_fw),
	PTP_SW_STAT(pps_in_count, sw_stats.pps_hw),
	PTP_MC_STAT(pps_in_offset_mean, PPS_OFF_MEAN),
	PTP_MC_STAT(pps_in_offset_last, PPS_OFF_LAST),
	PTP_MC_STAT(pps_in_offset_max, PPS_OFF_MAX),
	PTP_MC_STAT(pps_in_offset_min, PPS_OFF_MIN),
	PTP_MC_STAT(pps_in_period_mean, PPS_PER_MEAN),
	PTP_MC_STAT(pps_in_period_last, PPS_PER_LAST),
	PTP_MC_STAT(pps_in_period_max, PPS_PER_MAX),
	PTP_MC_STAT(pps_in_period_min, PPS_PER_MIN),
	PTP_MC_STAT(pps_in_bad, PPS_BAD),
	PTP_MC_STAT(pps_in_oflow, PPS_OFLOW),
};

#define PTP_STAT_COUNT ARRAY_SIZE(efct_ptp_stat_desc)

static const unsigned long efct_ptp_stat_mask[] = {
	[0 ... BITS_TO_LONGS(PTP_STAT_COUNT) - 1] = ~0UL,
};

void efct_ptp_reset_stats(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(in_rst_stats, MC_CMD_PTP_IN_RESET_STATS_LEN);
	struct efct_ptp_data *ptp = efct->ptp_data;
	int rc;

	if (!ptp)
		return;
	memset(&ptp->sw_stats, 0, sizeof(ptp->sw_stats));

	MCDI_SET_DWORD(in_rst_stats, PTP_IN_OP, MC_CMD_PTP_OP_RESET_STATS);
	MCDI_SET_DWORD(in_rst_stats, PTP_IN_PERIPH_ID, 0);

	rc = efct_mcdi_rpc(efct, MC_CMD_PTP, in_rst_stats, sizeof(in_rst_stats),
			   NULL, 0, NULL);
	if (rc < 0)
		netif_dbg(efct, probe, efct->net_dev, "PPS stats MCDI fail\n");
}

size_t efct_ptp_describe_stats(struct efct_nic *efct, u8 *strings)
{
	if (!efct->ptp_data)
		return 0;

	return efct_nic_describe_stats(efct_ptp_stat_desc, PTP_STAT_COUNT,
				      efct_ptp_stat_mask, strings);
}

size_t efct_ptp_update_stats(struct efct_nic *efct, u64 *stats)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_PTP_OUT_STATUS_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_PTP_IN_STATUS_LEN);
	struct efct_ptp_data *ptp = efct->ptp_data;
	size_t i;
	int rc;

	if (!ptp)
		return 0;

	/* Copy software statistics */
	for (i = 0; i < PTP_STAT_COUNT; i++) {
		if (efct_ptp_stat_desc[i].dma_width)
			continue;
		stats[i] = *(u32 *)((char *)efct->ptp_data +
					     efct_ptp_stat_desc[i].offset);
	}
	/* Fetch MC statistics.  We *must* fill in all statistics or
	 * risk leaking kernel memory to userland, so if the MCDI
	 * request fails we pretend we got zeroes.
	 */
	MCDI_SET_DWORD(inbuf, PTP_IN_OP, MC_CMD_PTP_OP_STATUS);
	MCDI_SET_DWORD(inbuf, PTP_IN_PERIPH_ID, 0);
	rc = efct_mcdi_rpc(efct, MC_CMD_PTP, inbuf, sizeof(inbuf),
			   outbuf, sizeof(outbuf), NULL);
	if (rc)
		memset(outbuf, 0, sizeof(outbuf));
	efct_nic_update_stats(efct_ptp_stat_desc, PTP_STAT_COUNT,
			      efct_ptp_stat_mask,
			     stats,
			     NULL,
			     (__le64 *)_MCDI_PTR(outbuf, 0));

	return PTP_STAT_COUNT;
}

/* Get PTP timestamp corrections */
static int efct_ptp_get_timestamp_corrections(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_PTP_IN_GET_TIMESTAMP_CORRECTIONS_LEN);
	size_t out_len;
	int rc;

	MCDI_SET_DWORD(inbuf, PTP_IN_OP,
		       MC_CMD_PTP_OP_GET_TIMESTAMP_CORRECTIONS);
	MCDI_SET_DWORD(inbuf, PTP_IN_PERIPH_ID, 0);

	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_PTP, inbuf, sizeof(inbuf),
				 outbuf, sizeof(outbuf), &out_len);
	if (rc == 0) {
		efct->ptp_data->ts_corrections.ptp_tx =
			MCDI_DWORD(outbuf,
				   PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_PTP_TX);
		efct->ptp_data->ts_corrections.ptp_rx =
			MCDI_DWORD(outbuf, PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_PTP_RX);
		efct->ptp_data->ts_corrections.pps_out =
			MCDI_DWORD(outbuf, PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_PPS_OUT);
		efct->ptp_data->ts_corrections.pps_in =
			MCDI_DWORD(outbuf, PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_PPS_IN);

		if (out_len >= MC_CMD_PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_LEN) {
			efct->ptp_data->ts_corrections.general_tx =
				MCDI_DWORD(outbuf, PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_GENERAL_TX);
			efct->ptp_data->ts_corrections.general_rx =
				MCDI_DWORD(outbuf, PTP_OUT_GET_TIMESTAMP_CORRECTIONS_V2_GENERAL_RX);
		} else {
			efct->ptp_data->ts_corrections.general_tx =
				efct->ptp_data->ts_corrections.ptp_tx;
			efct->ptp_data->ts_corrections.general_rx =
				efct->ptp_data->ts_corrections.ptp_rx;
		}
	} else {
		efct_mcdi_display_error(efct, MC_CMD_PTP, sizeof(inbuf), outbuf,
					sizeof(outbuf), rc);
		return rc;
	}

	return 0;
}

int efct_ptp_hw_pps_enable(struct efct_nic *efct, bool enable)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_PTP_IN_PPS_ENABLE_LEN);
	int rc;

	if (!efct->ptp_data)
		return -ENOTTY;

	MCDI_SET_DWORD(inbuf, PTP_IN_OP, MC_CMD_PTP_OP_PPS_ENABLE);
	MCDI_SET_DWORD(inbuf, PTP_IN_PERIPH_ID, 0);
	MCDI_SET_DWORD(inbuf, PTP_IN_PPS_ENABLE_OP,
		       enable ? MC_CMD_PTP_ENABLE_PPS :
				MC_CMD_PTP_DISABLE_PPS);
	rc = efct_mcdi_rpc(efct, MC_CMD_PTP, inbuf, sizeof(inbuf),
			   NULL, 0, NULL);

	if (rc && rc != -MC_CMD_ERR_EALREADY)
		return rc;

	return 0;
}

static void efct_ptp_pps_worker(struct work_struct *work)
{
	struct ptp_clock_event ptp_evt;
	struct efct_ptp_data *ptp;
	struct efct_nic *efct;

	ptp = container_of(work, struct efct_ptp_data, pps_work);
	efct = (ptp ? ptp->efct : NULL);
	if (!ptp || !efct || !ptp->pps_workwq)
		return;
	ptp_data_get(ptp);

	if (efct_ptp_synchronize(efct, PTP_SYNC_SAMPLE))
		goto out;

	if (ptp->usr_evt_enabled & (1 << PTP_CLK_REQ_PPS)) {
		ptp_evt.type = PTP_CLOCK_PPSUSR;
		ptp_evt.pps_times = ptp->host_time_pps;
		ptp_clock_event(ptp->phc_clock, &ptp_evt);
	}
out:
	ptp_data_put(ptp);
}

static void efct_remove_pps_workqueue(struct efct_ptp_data *ptp_data)
{
	struct workqueue_struct *pps_workwq = ptp_data->pps_workwq;

	ptp_data->pps_workwq = NULL; /* tells worker to do nothing */
	if (!pps_workwq)
		return;
	cancel_work_sync(&ptp_data->pps_work);
	destroy_workqueue(pps_workwq);
}

static int efct_create_pps_workqueue(struct efct_ptp_data *ptp)
{
	struct efct_device *efct;
	char busdevice[11];

	efct = ptp->efct->efct_dev;

	snprintf(busdevice, sizeof(busdevice), "%04x:%02x:%02x",
		 pci_domain_nr(efct->pci_dev->bus),
		 efct->pci_dev->bus->number,
		 efct->pci_dev->devfn);

	INIT_WORK(&ptp->pps_work, efct_ptp_pps_worker);
	ptp->pps_workwq = alloc_workqueue("xlnx_pps_%s", WQ_UNBOUND |
					  WQ_MEM_RECLAIM | WQ_SYSFS, 1,
					  busdevice);
	if (!ptp->pps_workwq)
		return -ENOMEM;
	return 0;
}

/* Convert FP16 ppm value to frequency adjustment in specified fixed point form.
 *
 * Input should be within the range +/- 0.1. If MAX_PPB is increased beyond
 * 10^8 then the formula here must be revised to avoid overflow.
 *
 * Precalculate multipliers to scale FP16 PPM values to a fixed point frequency
 * adjustment factor.
 * - The required scaling factor is: 1 / (10^6 * 2^16).
 * - The resultant factor will be represented in FP40 or FP44 as specified.
 * - Required adjustment range is: +/- 0.1.
 * - This requires 33 bits to represent FP16 PPM precision plus sign bit.
 * - The multiplier width cannot exceed 31 bits, to avoid overflow.
 * - 2^50 / 10^6 fits into 31 bits so we precalculate this as the scale word.
 * - Cancelling out the 2^16 in the denominator this gives a result in FP66.
 * - Shifting this down by 22 gets us the FP44 value required for some hardware.
 * - Shifting this down by 26 gets us the FP40 value required by other hardware.
 * - A further scale word corrects error in low order bits of large adjustments.
 * - This additional scale word arbitrarily use 30 bits of precision.
 */

static s64 ppm_fp16_to_factor(s64 ppm_fp16, unsigned int prec)
{
	/* FP66 multipliers to achieve division of multiplicand */
	static const s64 scale_word_frac = (((1LL << 50) % 1000000LL) << 30) / 1000000LL;
	static const s64 scale_word_whole = (1LL << 50) / 1000000LL;
	/* 1. Apply whole and fractional scale words to effect division.
	 * 2. Round to nearest at target precision.
	 * 3. Shift down to target precision.
	 */
	return (ppm_fp16 * scale_word_whole +
		(ppm_fp16 * scale_word_frac >> 30) +
		(1 << (66 - prec - 1))) >> (66 - prec);
}

static int efct_phc_adjfine(struct ptp_clock_info *ptp, long ppm_fp16)
{
	MCDI_DECLARE_BUF(inadj, MC_CMD_PTP_IN_ADJUST_V2_LEN);
	struct efct_ptp_data *ptp_data;
	struct efct_nic *efct;
	s64 adjustment;
	int rc;

	ptp_data = container_of(ptp, struct efct_ptp_data, phc_clock_info);
	efct = ptp_data->efct;
	ppm_fp16 = clamp_val(ppm_fp16, -ptp_data->max_adjfine, ptp_data->max_adjfine);
	adjustment = ppm_fp16_to_factor(ppm_fp16, ptp_data->adjfreq_prec);
	MCDI_SET_DWORD(inadj, PTP_IN_OP, MC_CMD_PTP_OP_ADJUST);
	MCDI_SET_DWORD(inadj, PTP_IN_PERIPH_ID, 0);
	MCDI_SET_QWORD(inadj, PTP_IN_ADJUST_V2_FREQ, adjustment);
	MCDI_SET_DWORD(inadj, PTP_IN_ADJUST_V2_MAJOR_HI, 0);
	MCDI_SET_DWORD(inadj, PTP_IN_ADJUST_V2_MAJOR, 0);
	MCDI_SET_DWORD(inadj, PTP_IN_ADJUST_V2_MINOR, 0);
	rc = efct_mcdi_rpc(efct, MC_CMD_PTP, inadj, sizeof(inadj),
			   NULL, 0, NULL);
	if (rc != 0)
		return rc;

	ptp_data->current_adjfreq = adjustment;
	return 0;
}

static int efct_adjtime(struct ptp_clock_info *ptp, u32 nic_major_hi, u32 nic_major, u32 nic_minor)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_PTP_IN_ADJUST_V2_LEN);
	struct efct_ptp_data *ptp_data;
	struct efct_nic *efct;
	int rc;

	ptp_data = container_of(ptp, struct efct_ptp_data, phc_clock_info);
	efct = ptp_data->efct;
	MCDI_SET_DWORD(inbuf, PTP_IN_OP, MC_CMD_PTP_OP_ADJUST);
	MCDI_SET_DWORD(inbuf, PTP_IN_PERIPH_ID, 0);
	MCDI_SET_QWORD(inbuf, PTP_IN_ADJUST_V2_FREQ, ptp_data->current_adjfreq);
	MCDI_SET_DWORD(inbuf, PTP_IN_ADJUST_V2_MAJOR_HI, nic_major_hi);
	MCDI_SET_DWORD(inbuf, PTP_IN_ADJUST_V2_MAJOR, nic_major);
	MCDI_SET_DWORD(inbuf, PTP_IN_ADJUST_V2_MINOR, nic_minor);
	rc = efct_mcdi_rpc(efct, MC_CMD_PTP, inbuf, sizeof(inbuf), NULL, 0, NULL);

	return rc;
}

static int efct_phc_adjtime(struct ptp_clock_info *ptp, s64 delta)
{
	u32 nic_major, nic_minor, nic_major_hi;
	struct efct_ptp_data *ptp_data;
	struct efct_nic *efct;

	ptp_data = container_of(ptp, struct efct_ptp_data, phc_clock_info);
	efct = ptp_data->efct;
	efct->ptp_data->ns_to_nic_time(delta, &nic_major, &nic_minor, &nic_major_hi);
	return efct_adjtime(ptp, nic_major_hi, nic_major, nic_minor);
}

static int efct_phc_gettime64(struct ptp_clock_info *ptp, struct timespec64 *ts)
{
	struct efct_ptp_data *ptp_data;
	struct efct_nic *efct;
	u64 timer_l;
	ktime_t kt;

	ptp_data = container_of(ptp, struct efct_ptp_data, phc_clock_info);
	efct = ptp_data->efct;
	timer_l = efct_ptp_time(efct);
	kt = ptp_data->nic64_to_kernel_time(SEC_HIGH32, timer_l, 0);
	*ts = ktime_to_timespec64(kt);

	return 0;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_PTP_GETTIMEX64)
static int efct_phc_gettimex64(struct ptp_clock_info *ptp, struct timespec64 *ts,
			       struct ptp_system_timestamp *sts)
{
	struct efct_ptp_data *ptp_data;
	struct efct_nic *efct;
	u64 timer;
	ktime_t kt;

	ptp_data = container_of(ptp, struct efct_ptp_data, phc_clock_info);
	efct = ptp_data->efct;
	ptp_read_system_prets(sts);
	timer = efct_ptp_time(efct);
	ptp_read_system_postts(sts);
	kt = ptp_data->nic64_to_kernel_time(SEC_HIGH32, timer, 0);
	*ts = ktime_to_timespec64(kt);

	return 0;
}
#endif

static int efct_phc_getcrosststamp(struct ptp_clock_info *ptp,
				   struct system_device_crosststamp *cts)
{
	struct system_time_snapshot snap;
	struct efct_ptp_data *ptp_data;
	struct efct_nic *efct;
	int rc;

	ptp_data = container_of(ptp, struct efct_ptp_data, phc_clock_info);
	efct = ptp_data->efct;
	rc = efct_ptp_synchronize(efct, PTP_SYNC_SAMPLE);
	if (rc)
		goto out;
	ktime_get_snapshot(&snap);
	cts->device = ktime_add(snap.real, timespec64_to_ktime(ptp_data->last_delta));
	cts->sys_realtime = snap.real;
	cts->sys_monoraw = snap.raw;
out:
	return rc;
}

static int efct_phc_settime64(struct ptp_clock_info *p, const struct timespec64 *ts)
{
	u32 nic_major, nic_minor, nic_major_hi;
	struct efct_ptp_data *ptp_data;
	struct timespec64 time_now;
	struct timespec64 delta;
	struct efct_nic *efct;
	int rc;

	ptp_data = container_of(p, struct efct_ptp_data, phc_clock_info);
	efct = ptp_data->efct;
	rc = efct_phc_gettime64(p, &time_now);
	if (rc != 0)
		return rc;
	delta = timespec64_sub(*ts, time_now);
	nic_major_hi = delta.tv_sec >> TIME_TO_SEC_SHIFT;
	nic_major = (u32)delta.tv_sec;
	nic_minor = (u32)delta.tv_nsec << efct->efct_dev->params.ts_subnano_bit;
	rc = efct_adjtime(p, nic_major_hi, nic_major, nic_minor);
	if (rc != 0)
		return rc;

	return 0;
}

static int efct_setup_pps_worker(struct efct_ptp_data *ptp, int enable)
{
	int rc = 0;

	if (enable && !ptp->pps_workwq) {
		rc = efct_create_pps_workqueue(ptp);
		if (rc < 0)
			goto err;
	} else if (!enable && ptp->pps_workwq) {
		efct_remove_pps_workqueue(ptp);
	}
err:
	return rc;
}

static int efct_phc_enable(struct ptp_clock_info *ptp,
			   struct ptp_clock_request *request,
			   int enable)
{
	struct efct_ptp_data *ptp_data = container_of(ptp,
						     struct efct_ptp_data,
						     phc_clock_info);
	int rc = 0;

	switch (request->type) {
	case PTP_CLK_REQ_EXTTS:
		if (ptp->pin_config[0].func != PTP_PF_EXTTS)
			enable = false;
		if (enable)
			ptp_data->usr_evt_enabled |= (1 << request->type);
		else
			ptp_data->usr_evt_enabled &= ~(1 << request->type);
		break;

	case PTP_CLK_REQ_PPS:
		rc = efct_setup_pps_worker(ptp_data, enable);
		if (rc < 0)
			goto err;
		if (enable)
			ptp_data->usr_evt_enabled |= (1 << request->type);
		else
			ptp_data->usr_evt_enabled &= ~(1 << request->type);
		break;
	default:
		rc = -EOPNOTSUPP;
		goto err;
	}
	return 0;
err:
	return rc;
}

static int efct_phc_verify(struct ptp_clock_info *ptp, unsigned int pin,
			   enum ptp_pin_function func, unsigned int chan)
{
	switch (func) {
	case PTP_PF_NONE:
	case PTP_PF_EXTTS:
		break;
	default:
		return -EOPNOTSUPP;
	}

	return 0;
}

static const struct ptp_clock_info efct_phc_clock_info = {
	.owner		= THIS_MODULE,
	.name		= "efct",
	.max_adj	= MAX_PPB, /* unused, ptp_data->max_adjfreq used instead */
	.n_alarm	= 0,
	.n_ext_ts	= 1,
	.n_pins		= 1,
	.n_per_out	= 0,
	.pps		= 1,
	.adjfine	= efct_phc_adjfine,
	.adjtime	= efct_phc_adjtime,
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_PTP_GETTIMEX64)
	.gettimex64	= efct_phc_gettimex64,
#else
	.gettime64	= efct_phc_gettime64,
#endif
	.settime64	= efct_phc_settime64,
	.getcrosststamp = efct_phc_getcrosststamp,
	.enable		= efct_phc_enable,
	.verify		= efct_phc_verify,
};

static int efct_associate_phc(struct efct_nic *efct, unsigned char *serial)
{
	struct efct_ptp_data *other, *next;
	struct efct_ptp_data *ptp;

	if (efct->phc_ptp_data || efct->ptp_data) {
		netif_err(efct, probe, efct->net_dev,
			  "PHC/PTP already associated. It can be a bug in driver\n");
		return -EALREADY;
	}
	ptp = ptp_data_alloc(efct, GFP_KERNEL);
	if (IS_ERR(ptp))
		return PTR_ERR(ptp);
	efct->ptp_data = ptp;
	spin_lock(&efct_phcs_list_lock);

	list_for_each_entry_safe(other, next, &efct_phcs_list,
				 phcs_node) {
		if (!strncmp(other->serial,  serial, EFCT_MAX_VERSION_INFO_LEN)) {
			if (!other->efct) {
				ptp_data_del(ptp);
				efct->ptp_data = other;
				other->efct = efct;
			}
			efct->phc_ptp_data = other;
			ptp_data_get(other);
			goto out;
		}
	}
	efct->phc_ptp_data = efct->ptp_data;
	list_add(&efct->phc_ptp_data->phcs_node, &efct_phcs_list);
	spin_unlock(&efct_phcs_list_lock);
	efct_ptp_start(efct);

	return 0;
out:
	spin_unlock(&efct_phcs_list_lock);
	return 0;
}

static void efct_dissociate_phc(struct efct_nic *efct)
{
	if (!efct->phc_ptp_data)
		return;
	if (ptp_data_kref_read(efct->phc_ptp_data) == 1) {
		efct_ptp_stop(efct);
		spin_lock(&efct_phcs_list_lock);
		list_del(&efct->phc_ptp_data->phcs_node);
		spin_unlock(&efct_phcs_list_lock);
	}
	if (efct->ptp_data == efct->phc_ptp_data) {
		efct->phc_ptp_data->efct = NULL;
	} else {
		ptp_data_put(efct->phc_ptp_data);
		efct->phc_ptp_data = NULL;
	}
	ptp_data_del(efct->ptp_data);
	efct->ptp_data = NULL;
}

static int efct_get_board_serial(struct efct_nic *efct, u8 *serial)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_GET_VERSION_EXT_IN_LEN);
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_VERSION_V5_OUT_LEN);
	size_t outlength;
	const char *str;
	u32 flags;
	int rc;

	rc = efct_mcdi_rpc(efct, MC_CMD_GET_VERSION, inbuf, sizeof(inbuf),
			   outbuf, sizeof(outbuf), &outlength);
	if (rc || outlength < MC_CMD_GET_VERSION_V5_OUT_LEN)
		return -EINVAL;
		/* Handle V2 additions */
	flags = MCDI_DWORD(outbuf, GET_VERSION_V5_OUT_FLAGS);
	if (!(flags & BIT(MC_CMD_GET_VERSION_V5_OUT_BOARD_EXT_INFO_PRESENT_LBN)))
		return -EINVAL;

	str = MCDI_PTR(outbuf, GET_VERSION_V5_OUT_BOARD_SERIAL);
	strscpy(serial, str, EFCT_MAX_VERSION_INFO_LEN);

	return rc;
}

int efct_ptp_probe_setup(struct efct_nic *efct)
{
	unsigned char serial[EFCT_MAX_VERSION_INFO_LEN];
	struct efct_ptp_data *ptp;
	struct ptp_pin_desc *ppd;
	int rc;

	rc = efct_get_board_serial(efct, serial);
	if (rc) {
		pr_err("Failed to get PTP UID, rc=%d", rc);
		goto fail1;
	}
	rc = efct_associate_phc(efct, serial);
	if (rc)
		goto fail1;
	ptp = efct->ptp_data;
	strscpy(ptp->serial, serial, EFCT_MAX_VERSION_INFO_LEN);
	ptp->config.flags = 0;
	ptp->config.tx_type = HWTSTAMP_TX_OFF;
	ptp->config.rx_filter = HWTSTAMP_FILTER_NONE;
	rc = efct_ptp_get_attributes(efct);
	if (rc < 0)
		goto fail2;

	/* Get the timestamp corrections */
	rc = efct_ptp_get_timestamp_corrections(efct);
	if (rc < 0)
		goto fail2;
	if (efct_phc_exposed(efct)) {
		ptp->phc_clock_info = efct_phc_clock_info;
		ptp->phc_clock_info.max_adj = ptp->max_adjfreq;
		ppd = &ptp->pin_config[0];
		snprintf(ppd->name, sizeof(ppd->name), "pps0");
		ppd->index = 0;
		ppd->func = PTP_PF_EXTTS;
		ptp->phc_clock_info.pin_config = ptp->pin_config;
		ptp->phc_clock = ptp_clock_register(&ptp->phc_clock_info,
						    &efct->efct_dev->pci_dev->dev);
		if (IS_ERR(ptp->phc_clock)) {
			rc = PTR_ERR(ptp->phc_clock);
			goto fail2;
		}
		if (efct_ptp_hw_pps_enable(ptp->efct, true))
			pci_err(efct->efct_dev->pci_dev, "PPS not enabled\n");
	}
	return 0;
fail2:
	efct_dissociate_phc(efct);
fail1:
	return rc;
}

void efct_ptp_remove_setup(struct efct_nic *efct)
{
	struct efct_ptp_data *ptp;

	ptp = efct->ptp_data;
	if (efct_phc_exposed(efct)) {
		efct_ptp_hw_pps_enable(ptp->efct, false);
		efct_remove_pps_workqueue(ptp);
	}

	if (ptp->phc_clock)
		ptp_clock_unregister(ptp->phc_clock);
	ptp->phc_clock = NULL;
	efct_dissociate_phc(efct);
}

int efct_ptp_get_ts_config(struct net_device *net_dev, struct ifreq *ifr)
{
	struct efct_nic *efct;

	efct = efct_netdev_priv(net_dev);
	if (!efct->ptp_data)
		return -EOPNOTSUPP;

	return copy_to_user(ifr->ifr_data, &efct->ptp_data->config,
			    sizeof(efct->ptp_data->config)) ? -EFAULT : 0;
}

void efct_ptp_get_ts_info(struct efct_nic *efct, struct kernel_ethtool_ts_info *ts_info)
{
	struct efct_ptp_data *phc_ptp = efct->phc_ptp_data;

	ASSERT_RTNL();

	if (!phc_ptp)
		return;

	ts_info->so_timestamping |= (SOF_TIMESTAMPING_TX_HARDWARE |
				     SOF_TIMESTAMPING_RX_HARDWARE |
				     SOF_TIMESTAMPING_RAW_HARDWARE);

	if (phc_ptp->phc_clock)
		ts_info->phc_index = ptp_clock_index(phc_ptp->phc_clock);
	ts_info->tx_types = 1 << HWTSTAMP_TX_OFF | 1 << HWTSTAMP_TX_ON;
	ts_info->rx_filters = efct->type->hwtstamp_filters;
}

int efct_ptp_subscribe_timesync(struct efct_ev_queue *eventq)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_PTP_IN_TIME_EVENT_SUBSCRIBE_LEN);
	int rc;

	if (eventq->sync_events_state == SYNC_EVENTS_REQUESTED ||
	    eventq->sync_events_state == SYNC_EVENTS_VALID)
		return 0;
	eventq->sync_events_state = SYNC_EVENTS_REQUESTED;
	MCDI_SET_DWORD(inbuf, PTP_IN_OP, MC_CMD_PTP_OP_TIME_EVENT_SUBSCRIBE);
	MCDI_SET_DWORD(inbuf, PTP_IN_PERIPH_ID, 0);
	MCDI_POPULATE_DWORD_2(inbuf, PTP_IN_TIME_EVENT_SUBSCRIBE_QUEUE,
			      PTP_IN_TIME_EVENT_SUBSCRIBE_QUEUE_ID, eventq->index,
			      PTP_IN_TIME_EVENT_SUBSCRIBE_REPORT_SYNC_STATUS, 1);
	rc = efct_mcdi_rpc(eventq->efct, MC_CMD_PTP, inbuf, sizeof(inbuf), NULL, 0, NULL);
	if (rc != 0) {
		netif_err(eventq->efct, probe, eventq->efct->net_dev,
			  "Time sync event subscribe failed\n");
		return rc;
	}

	return rc;
}

int efct_ptp_unsubscribe_timesync(struct efct_ev_queue *eventq)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_PTP_IN_TIME_EVENT_UNSUBSCRIBE_LEN);
	int rc;

	if (eventq->sync_events_state == SYNC_EVENTS_DISABLED)
		return 0;
	eventq->sync_events_state =  SYNC_EVENTS_DISABLED;

	MCDI_SET_DWORD(inbuf, PTP_IN_OP, MC_CMD_PTP_OP_TIME_EVENT_UNSUBSCRIBE);
	MCDI_SET_DWORD(inbuf, PTP_IN_PERIPH_ID, 0);
	MCDI_SET_DWORD(inbuf, PTP_IN_TIME_EVENT_UNSUBSCRIBE_CONTROL,
		       MC_CMD_PTP_IN_TIME_EVENT_UNSUBSCRIBE_SINGLE);
	MCDI_SET_DWORD(inbuf, PTP_IN_TIME_EVENT_UNSUBSCRIBE_QUEUE,
		       eventq->index);

	rc = efct_mcdi_rpc(eventq->efct, MC_CMD_PTP, inbuf, sizeof(inbuf), NULL, 0, NULL);

	return rc;
}

int efct_ptp_tx_ts_event(struct efct_nic *efct, bool flag)
{
	struct efct_ev_queue *eventq;
	int eidx, rc;
	int k;

	for_each_set_bit(k, &efct->txq_active_mask, efct->max_txq_count) {
		eidx = efct->txq[k].evq_index;
		eventq = &efct->evq[eidx];
		if (eventq->type != EVQ_T_TX)
			continue;
		if (flag) {
			rc = efct_ptp_subscribe_timesync(eventq);
			if (rc)
				goto fail;
		} else {
			rc = efct_ptp_unsubscribe_timesync(eventq);
		}
	}
	return 0;
fail:
	for_each_set_bit(k, &efct->txq_active_mask, efct->max_txq_count) {
		eidx = efct->txq[k].evq_index;
		eventq = &efct->evq[k];
		if (eventq->type != EVQ_T_TX)
			continue;

		efct_ptp_unsubscribe_timesync(&efct->evq[k]);
	}
	return rc;
}

int efct_ptp_enable_ts(struct efct_nic *efct, struct hwtstamp_config *init)
{
	struct efct_ptp_data *ptp;
	int rc = 0;

	ptp = efct->ptp_data;
	switch (init->tx_type) {
	case HWTSTAMP_TX_OFF:
		if (ptp->txtstamp) {
			ptp->txtstamp = false;
			mutex_lock(&efct->state_lock);
			if (efct->state == STATE_NET_UP)
				efct_ptp_tx_ts_event(efct, false);
			mutex_unlock(&efct->state_lock);
		}
		break;
	case HWTSTAMP_TX_ON:
		if (!ptp->txtstamp) {
			ptp->txtstamp = true;
			mutex_lock(&efct->state_lock);
			if (efct->state == STATE_NET_UP)
				efct_ptp_tx_ts_event(efct, true);
			mutex_unlock(&efct->state_lock);
		}
		break;
	default:
		return -ERANGE;
	}

	switch (init->rx_filter) {
	case HWTSTAMP_FILTER_NONE:
		if (ptp->rxtstamp) {
			ptp->rxtstamp = false;

			init->rx_filter = HWTSTAMP_FILTER_NONE;
		}
		break;
	case HWTSTAMP_FILTER_ALL:
	case HWTSTAMP_FILTER_PTP_V1_L4_EVENT:
	case HWTSTAMP_FILTER_PTP_V1_L4_SYNC:
	case HWTSTAMP_FILTER_PTP_V1_L4_DELAY_REQ:
	case HWTSTAMP_FILTER_PTP_V2_L4_EVENT:
	case HWTSTAMP_FILTER_PTP_V2_L4_SYNC:
	case HWTSTAMP_FILTER_PTP_V2_L4_DELAY_REQ:
	case HWTSTAMP_FILTER_PTP_V2_L2_EVENT:
	case HWTSTAMP_FILTER_PTP_V2_L2_SYNC:
	case HWTSTAMP_FILTER_PTP_V2_L2_DELAY_REQ:
	case HWTSTAMP_FILTER_PTP_V2_EVENT:
	case HWTSTAMP_FILTER_PTP_V2_SYNC:
	case HWTSTAMP_FILTER_PTP_V2_DELAY_REQ:
		if (!ptp->rxtstamp) {
			init->rx_filter = HWTSTAMP_FILTER_ALL;
			ptp->rxtstamp = true;
		}
		break;
	default:
		return -ERANGE;
	}

	efct_ptp_synchronize(efct, PTP_SYNC_SAMPLE);

	return rc;
}

int efct_ptp_set_ts_config(struct net_device *net_dev, struct ifreq *ifr)
{
	struct hwtstamp_config config;
	struct efct_nic *efct;
	int rc;

	efct = efct_netdev_priv(net_dev);
	/* Not a PTP enabled port */
	if (!efct->ptp_data)
		return -EOPNOTSUPP;

	if (copy_from_user(&config, ifr->ifr_data, sizeof(config)))
		return -EFAULT;

	if (config.flags)
		return -EINVAL;

	rc = efct->type->ptp_set_ts_config(efct, &config);
	if (rc != 0)
		return rc;
	efct->ptp_data->config = config;
	return copy_to_user(ifr->ifr_data, &config, sizeof(config))
		? -EFAULT : 0;
}

static void hw_pps_event_pps(struct efct_ptp_data *ptp)
{
	struct ptp_clock_event ptp_evt;
	ktime_t n_assert;

	n_assert = ptp->nic_to_kernel_time(EFCT_QWORD_FIELD(ptp->evt_frags[0],
						MCDI_EVENT_DATA),
						EFCT_QWORD_FIELD(ptp->evt_frags[1],
								 MCDI_EVENT_DATA),
						ptp->ts_corrections.pps_in);
	if (ptp->usr_evt_enabled & (1 << PTP_CLK_REQ_EXTTS)) {
		ptp_evt.type = PTP_CLOCK_EXTTS;
		ptp_evt.index = 0;
		ptp_evt.timestamp = ktime_to_ns(n_assert);
		ptp_clock_event(ptp->phc_clock, &ptp_evt);
	}
	ptp->sw_stats.pps_hw++;
}

static void ptp_event_pps(struct efct_ptp_data *ptp)
{
	if (ptp->pps_workwq)
		queue_work(ptp->pps_workwq, &ptp->pps_work);
	ptp->sw_stats.pps_fw++;
}

void efct_ptp_event(struct efct_nic *efct, union efct_qword *ev)
{
	struct efct_ptp_data *ptp;
	int code;

	code = EFCT_QWORD_FIELD(*ev, MCDI_EVENT_CODE);
	ptp = efct->phc_ptp_data;
	if (ptp->evt_frag_idx == 0) {
		ptp->evt_code = code;
	} else if (ptp->evt_code != code) {
		netif_err(efct, hw, efct->net_dev,
			  "PTP out of sequence event %d\n", code);
		ptp->evt_frag_idx = 0;
	}
	ptp->evt_frags[ptp->evt_frag_idx++] = *ev;
	if (!MCDI_EVENT_FIELD(*ev, CONT)) {
		/* Process resulting event */
		switch (code) {
		case MCDI_EVENT_CODE_PTP_PPS:
			ptp_event_pps(ptp);
			break;
		case MCDI_EVENT_CODE_HW_PPS:
			hw_pps_event_pps(ptp);
			break;
		default:
			netif_err(efct, hw, efct->net_dev,
				  "PTP unknown event %d\n", code);
			break;
		}
		ptp->evt_frag_idx = 0;
	} else if (ptp->evt_frag_idx == MAX_EVENT_FRAGS) {
		netif_err(efct, hw, efct->net_dev,
			  "PTP too many event fragments\n");
		ptp->evt_frag_idx = 0;
	}
}

