// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include <linux/highmem.h>
#include <linux/module.h>
#include <net/busy_poll.h>

#include "efct_rx.h"
#include "efct_common.h"
#include "efct_reg.h"
#include "efct_io.h"
#include "efct_ptp.h"
#ifdef CONFIG_XILINX_EFCT_TRACING
#define CREATE_TRACE_POINTS
#include <trace/events/xilinx_efct.h>
#endif

#ifdef CONFIG_XILINX_AUX_EFCT
#include "efct_auxbus.h"
static void hpl_free_sb(struct efct_rx_queue *, int);

/* Huge Page Buffer Flags */
#define HP_LAZY_FREE 1
#define HP_ORPHAN 2
/* Superbuf Flags */
#define SB_RETURN_TO_HPL 1
#define SB_MOVE_OUT 2

static void efct_poison_buffer(struct efct_rx_queue *rxq, struct efct_buffer *buffer)
{
	size_t length = rxq->poison_cfg.length;
	bool header_poison_needed = false;
	union efct_qword *last_meta;
	u32 len = buffer->len;
	u64 header_poison = 0;
	u32 offset, i;
	u64 *packet;

	/* check the sentinel value of last packet, if equal
	 * to new sentinel, we need header poisoning.
	 */
	last_meta = (union efct_qword *)((u8 *)buffer->addr + len - rxq->pkt_stride);
	if (buffer->sentinel == EFCT_QWORD_FIELD(*last_meta, ESF_HZ_RX_PREFIX_SENTINEL)) {
		header_poison_needed = true;
		header_poison = (buffer->sentinel) ? 0 : ~0ull;
	}

	if (!length && !header_poison_needed)
		return;

	offset = rxq->efct->efct_dev->params.frame_offset_fixed / 8;

	if (!offset) {
		/*In case frame offset is not fixed we need to */
		if (length)
			length += 48;
		offset = 2; // 2 octets = 16B rx metadata
	}

	packet = (u64 *)buffer->addr;
	while (len) {
		if (header_poison_needed)
			*packet = header_poison;
		if (length) {
			for (i = 0; i < (length / 8); i++)
				*(packet + offset + i) = rxq->poison_cfg.value;
		}
		packet += (rxq->pkt_stride / 8);
		len -= (rxq->pkt_stride);
	}

	/* Ensure sentinels is written before hw starts using it */
	wmb();
}
#endif

/* Post buffer to NIC */
static void efct_rx_buff_post(struct efct_rx_queue *rxq, struct efct_buffer *buffer, bool rollover)
{
	union efct_qword qword;
	u64 value;

#ifdef CONFIG_XILINX_AUX_EFCT
	efct_poison_buffer(rxq, buffer);
#endif
	rxq->n_rx_buffers_posted++;
	value = buffer->dma_addr >> 12;
	EFCT_POPULATE_QWORD_3(qword,
			      ERF_HZ_PAGE_ADDRESS, value,
			     ERF_HZ_SENTINEL_VALUE, buffer->sentinel,
			     ERF_HZ_ROLLOVER, rollover);
	_efct_writeq(qword.u64[0], rxq->receive_base);
}

/* Initialize driver buffer list.
 * Allocate driver DMA buffers.
 *
 * Context : Driver
 *
 * Return :  0 - Success
 *          -1 - Failure
 */
int dbl_init(struct efct_rx_queue *rxq)
{
	struct efct_driver_buffer_list *dbl = &rxq->dbl;
	struct efct_nic *efct = rxq->efct;
	int rc, i, j;

	/* Allocate driver DMA buffers */
	for (i = 0; i < rxq->num_rx_buffs; ++i) {
		rc = efct_nic_alloc_buffer(efct, &dbl->db[i].dma_buffer,
					   rxq->buffer_size, GFP_KERNEL);
		if (rc)
			goto fail;
		/* Expect alloc call to memset to zero. So set original
		 * sentinel to zero.
		 */
		dbl->db[i].dma_buffer.sentinel = 0;
	}

	/* Add all buffers to free list */
	for (i = 0; i < rxq->num_rx_buffs - 1; ++i)
		dbl->db[i].next = &dbl->db[i + 1];

	dbl->db[i].next = NULL;
	dbl->free_list = &dbl->db[0];

	return 0;
fail:
	for (j = 0; j < i; ++j)
		efct_nic_free_buffer(efct, &dbl->db[j].dma_buffer);

	return rc;
}

/* Free driver DMA buffers
 *
 * Context : Driver
 *
 * Return : None
 */
void dbl_fini(struct efct_rx_queue *rxq)
{
	int i;

	for (i = 0; i < rxq->num_rx_buffs; ++i)
		efct_nic_free_buffer(rxq->efct, &rxq->dbl.db[i].dma_buffer);
}

/* Allocate buffer from free list */
static int dbl_alloc_buff(struct efct_rx_queue *rxq, struct efct_buffer *buffer)
{
	struct efct_driver_buffer_list *dbl = &rxq->dbl;
	int id;

	if (dbl->free_list) {
		memcpy(buffer, &dbl->free_list->dma_buffer, sizeof(*buffer));
		id = dbl->free_list - &dbl->db[0];
		dbl->free_list = dbl->free_list->next;
		return id;
	}

	return -ENOSPC;
}

/* Move buffer to free list */
static void dbl_free_buff(struct efct_rx_queue *rxq, int id)
{
	struct efct_driver_buffer_list *dbl = &rxq->dbl;

	/* Flip the original sentinel value */
	dbl->db[id].dma_buffer.sentinel = !dbl->db[id].dma_buffer.sentinel;
	dbl->db[id].next = dbl->free_list;
	dbl->free_list = &dbl->db[id];
}

/* Initialize NBL
 *
 * Context : Driver
 *
 * Return : 0 - Success
 */
int nbl_init(struct efct_rx_queue *rxq)
{
	struct efct_nic_buffer_list *nbl = &rxq->nbl;

	memset(&nbl->nb[0], 0, ARRAY_SIZE(nbl->nb) * sizeof(nbl->nb[0]));
	nbl->head_index = 0u;
	nbl->tail_index = 0u;
	nbl->seq_no = 0u;
	/* Packet meta starts at next packet location */
	nbl->meta_offset = rxq->pkt_stride;
	nbl->prev_meta_offset = 0u;
	nbl->active_nic_buffs = 0u;
	nbl->frame_offset_fixed = rxq->efct->efct_dev->params.frame_offset_fixed;
	return 0;
}

#ifdef CONFIG_XILINX_AUX_EFCT
static u32 nbl_tail_seq_no(struct efct_rx_queue *rxq)
{
	struct efct_nic_buffer_list *nbl = &rxq->nbl;

	return nbl->seq_no + nbl->active_nic_buffs - 1;
}
#endif

/* Add buffer to NBL */
static int nbl_buff_push(struct efct_rx_queue *rxq, struct efct_buffer *buffer,
			 int id, bool is_dbl, bool rollover)
{
	struct efct_nic_buffer_list *nbl = &rxq->nbl;

	WARN_ON_ONCE(id < 0);
	if (nbl->head_index == ((nbl->tail_index + 1) % NIC_BUFFS_PER_QUEUE)) {
		netif_err(rxq->efct, probe, rxq->efct->net_dev,
			  "Rx queue %d nbl full\n", rxq->index);
		return EFCT_FAILURE;
	}

	memcpy(&nbl->nb[nbl->tail_index].dma_buffer, buffer, sizeof(*buffer));
	nbl->nb[nbl->tail_index].id = id;
	nbl->nb[nbl->tail_index].is_dbl = is_dbl;

	efct_rx_buff_post(rxq, buffer, rollover);

	nbl->tail_index = (nbl->tail_index + 1) % NIC_BUFFS_PER_QUEUE;
	++nbl->active_nic_buffs;
#ifdef CONFIG_XILINX_EFCT_TRACING
	trace_xilinx_efct_buff_push(rxq, nbl, id, is_dbl, rollover);
#endif
	return 0;
}

/* Remove buffer from NBL */
static int nbl_buff_pop(struct efct_rx_queue *rxq, bool is_rollover, int *id, bool *is_dbl)
{
	struct efct_nic_buffer_list *nbl = &rxq->nbl;

	if (nbl->head_index == nbl->tail_index) {
		netif_err(rxq->efct, rx_err, rxq->efct->net_dev,
			  "Rx queue %d nbl empty\n", rxq->index);
		return -1;
	}

	*id = nbl->nb[nbl->head_index].id;
	*is_dbl = nbl->nb[nbl->head_index].is_dbl;

	if (is_rollover) {
		nbl->meta_offset = rxq->pkt_stride;
		nbl->prev_meta_offset = 0u;
	}

	/* Rollover to next buffer */
	nbl->head_index = (nbl->head_index + 1) % NIC_BUFFS_PER_QUEUE;
	++nbl->seq_no;
	--nbl->active_nic_buffs;
#ifdef CONFIG_XILINX_EFCT_TRACING
	trace_xilinx_efct_buff_pop(rxq, nbl, *id, *is_dbl, is_rollover);
#endif
	return 0;
}

/* Return current packet meta header and packet start location */
static int nbl_buff_pkt_peek(struct efct_rx_queue *rxq, u8 **pkt_start)
{
	struct efct_nic_buffer_list *nbl = &rxq->nbl;
	u32 frame_loc;

	/* TODO: Make below check with likely or unlikely
	 * based on default design param.
	 */
	if (nbl->frame_offset_fixed) {
		frame_loc = nbl->frame_offset_fixed;
	} else {
		frame_loc = EFCT_QWORD_FIELD(*((union efct_qword *)*pkt_start),
					     ESF_HZ_RX_PREFIX_NEXT_FRAME_LOC);

		if (frame_loc == ESE_HZ_XN_NEXT_FRAME_LOC_SEPARATELY) {
			frame_loc = 64;
		} else if (frame_loc == ESE_HZ_XN_NEXT_FRAME_LOC_TOGETHER) {
			frame_loc = 16;
		} else {
			netif_err(rxq->efct, rx_err, rxq->efct->net_dev,
				  "Rx queue %d Unknown frame location %0X\n",
				  rxq->index, frame_loc);
			frame_loc = 16;
		}
	}

	*pkt_start = (*pkt_start + frame_loc + 2);

	return 0;
}

static int nbl_buff_pkt_extract_meta(struct efct_rx_queue *rxq, union efct_qword **p_meta,
				     u8 **pkt_start, bool *is_sentinel_mismatch)
{
	struct efct_nic_buffer_list *nbl = &rxq->nbl;
	unsigned char head_index;

	if (nbl->meta_offset < nbl->prev_meta_offset) {
		/* We are in split */
		if (nbl->head_index == nbl->tail_index) {
			/* Oh no */
			netif_err(rxq->efct, rx_err, rxq->efct->net_dev,
				  "Rx queue %d nbl empty for packet metadata\n", rxq->index);
			return EFCT_FAILURE;
		}

		head_index = (nbl->head_index + 1) % NIC_BUFFS_PER_QUEUE;
	} else {
		head_index = nbl->head_index;
	}

	/* Pick next buffer for metadata */
	*p_meta = (union efct_qword *)((u8 *)nbl->nb[head_index].dma_buffer.addr +
	nbl->meta_offset);
	*pkt_start = (void *)(((u8 *)nbl->nb[nbl->head_index].dma_buffer.addr) +
	nbl->prev_meta_offset);

	if (nbl->nb[head_index].dma_buffer.sentinel
			!= EFCT_QWORD_FIELD(*(*p_meta), ESF_HZ_RX_PREFIX_SENTINEL))
		*is_sentinel_mismatch = true;
	return 0;
}

/* Consume packet from nic buffer.
 * This function returns true when nic buffer needs rollover
 * else false
 */
static bool nbl_buff_pkt_consume(struct efct_rx_queue *rxq)
{
	struct efct_nic_buffer_list *nbl = &rxq->nbl;

	nbl->meta_offset += rxq->pkt_stride;
	if (rxq->buffer_size <= nbl->meta_offset)
		nbl->meta_offset = 0u;

	nbl->prev_meta_offset += rxq->pkt_stride;
	if (rxq->buffer_size <= nbl->prev_meta_offset) {
		nbl->prev_meta_offset = 0u;
		/* Once prev meta offset moves to next buffer, we
		 * can rollover to next buffer
		 */
		return true;
	}

	return false;
}

/* Free the buffer to appropriate pool */
static void
rx_free_buffer(struct efct_rx_queue *rxq, int id, bool is_dbl, bool is_internal_ref)
{
#ifdef CONFIG_XILINX_AUX_EFCT
	int i, ref_cnt = 1;
#endif
	WARN_ON_ONCE(id < 0);
	if (is_dbl) {
		dbl_free_buff(rxq, id);
		return;
	}
#ifdef CONFIG_XILINX_AUX_EFCT
	if (is_internal_ref) {
		ref_cnt = efct_aux_buffer_end(rxq->efct, rxq->index, id, false);
		/* Update kernel and client ref count for superbuf */
		if (ref_cnt > 0)
			ref_cnt += 1;
		else
			ref_cnt = 1;
	}
	for (i = 0; i < ref_cnt; ++i)
		sbl_release_buff(rxq, id);
#endif
}

/* Remove buffer from NBL */
void nbl_reset(struct efct_rx_queue *rxq)
{
	struct efct_nic_buffer_list *nbl = &rxq->nbl;
#ifdef CONFIG_XILINX_AUX_EFCT
	union efct_qword *p_meta, rolled_over;
#endif
	bool is_dbl;
#ifdef CONFIG_XILINX_AUX_EFCT
	u64 *meta;
#endif
	int id;
#ifdef CONFIG_XILINX_AUX_EFCT

	do {
		/* Find the last written valid packet by HW */
		if (nbl->meta_offset == 0u &&
		    ((nbl->head_index + 1) % NIC_BUFFS_PER_QUEUE) == nbl->tail_index) {
			/* at last of Nbl */
			nbl_buff_pop(rxq, true, &id, &is_dbl);
			rx_free_buffer(rxq, id, is_dbl, true);

			return;
		}

		/* Check sentinel of meta for last written packet */
		p_meta = (union efct_qword *)((u8 *)nbl->nb[nbl->head_index].dma_buffer.addr +
					  nbl->meta_offset);

		if (nbl->nb[nbl->head_index].dma_buffer.sentinel !=
			EFCT_QWORD_FIELD(*p_meta, ESF_HZ_RX_PREFIX_SENTINEL))
			break;

		if (nbl_buff_pkt_consume(rxq)) {
			if (nbl_buff_pop(rxq, false, &id, &is_dbl))
				break;

			rx_free_buffer(rxq, id, is_dbl, true);
		}
	} while (1);
#endif

	while (nbl->head_index != nbl->tail_index) {
#ifdef CONFIG_XILINX_AUX_EFCT
		meta = (u64 *)((u8 *)nbl->nb[nbl->head_index].dma_buffer.addr + nbl->meta_offset);

		EFCT_POPULATE_QWORD_2(rolled_over, ESF_HZ_RX_PREFIX_ROLLOVER, 1,
				      ESF_HZ_RX_PREFIX_SENTINEL,
				     nbl->nb[nbl->head_index].dma_buffer.sentinel);
		WRITE_ONCE(*meta, (__force u64)(rolled_over.u64[0]));
#endif
		nbl_buff_pop(rxq, true, &id, &is_dbl);
		rx_free_buffer(rxq, id, is_dbl, true);
	}
}

#ifdef CONFIG_XILINX_AUX_EFCT
/* Check water mark level for refill */
static bool is_below_higher_watermark(struct efct_rx_queue *rxq)
{
	struct efct_nic_buffer_list *nbl = &rxq->nbl;

	if (nbl->active_nic_buffs < rxq->num_rx_buffs)
		return true;

	return false;
}

/* Check water mark level for refill */
static bool is_below_lower_watermark(struct efct_rx_queue *rxq)
{
	struct efct_nic_buffer_list *nbl = &rxq->nbl;

	if (bitmap_empty(rxq->active_aux_clients, MAX_AUX_CLIENTS)) {
		if (nbl->active_nic_buffs < rxq->num_rx_buffs)
			return true;
	} else {
		if (nbl->active_nic_buffs < RX_MIN_DRIVER_BUFFS)
			return true;
	}

	return false;
}

static void sbl_init_one(struct efct_super_buffer *sb)
{
	sb->ref_count = 0xFF;
	sb->next = -1;
	sb->client_id = NULL;
	sb->flags = 0;
}

/* Initialize superbuf list
 *
 * Context : Driver
 *
 * Return : 0 - Success
 */
int sbl_init(struct efct_rx_queue *rxq)
{
	struct efct_super_buffer_list *sbl = rxq->sbl;
	int i;

	for (i = 0; i < rxq->hpl_alloced * SUPER_BUFFS_PER_HUGE_PAGE; ++i)
		sbl_init_one(&sbl->sb[i]);

	sbl->free_list = -1;
	return 0;
}

void sbl_fini(struct efct_rx_queue *rxq)
{
	/* Deallocation is in hpl_fini() (because the alloc was in hpl_init()) */
}

/* Allocate superbuf from free list */
static int sbl_alloc_buff(struct efct_rx_queue *rxq, struct efct_buffer *buffer)
{
	struct efct_super_buffer_list *sbl = rxq->sbl;
	int id;

	id = sbl->free_list;
	if (id >= 0) {
		struct efct_super_buffer *sb = &sbl->sb[id];

		memcpy(buffer, &sb->dma_buffer, sizeof(*buffer));
		sbl->free_list = sb->next;
		return id;
	}

	return -ENOSPC;
}

/* Update superbuf reference count by count */
static void sbl_update_buff_ref(struct efct_rx_queue *rxq, int id, int count)
{
	rxq->sbl->sb[id].ref_count = count;
}

/* Release superbuf to free list
 *
 * Context : NAPI
 *
 * Return : None
 */
void sbl_release_buff(struct efct_rx_queue *rxq, int id)
{
	struct efct_super_buffer_list *sbl = rxq->sbl;

	--sbl->sb[id].ref_count;

	if (sbl->sb[id].ref_count == 0) {
		if (unlikely(sbl->sb[id].flags & (1 << SB_RETURN_TO_HPL))) {
			/* Free the superbuf and return to hpl */
			hpl_free_sb(rxq, id);
			sbl->sb[id].client_id = NULL;
			sbl->sb[id].flags = 0u;
			sbl->sb[id].ref_count = ~sbl->sb[id].ref_count;
		} else {
			/* Flip the original sentinel value and add to free list */
			sbl->sb[id].dma_buffer.sentinel = !sbl->sb[id].dma_buffer.sentinel;
			sbl->sb[id].next = sbl->free_list;
			sbl->free_list = id;
		}
	}
}

/* Add superbuf to a provided location */
static void sbl_add(struct efct_rx_queue *rxq, struct efct_buffer *buffer, int loc,
		    struct xlnx_efct_client *client_id)
{
	struct efct_super_buffer_list *sbl = rxq->sbl;

	memcpy(&sbl->sb[loc].dma_buffer, buffer, sizeof(*buffer));
	sbl->sb[loc].client_id = client_id;
	sbl->sb[loc].ref_count = 0;
	sbl->sb[loc].next = sbl->free_list;
	sbl->free_list = loc;
}

/* Mark sbl for given flag */
static void sbl_add_flag(struct efct_rx_queue *rxq, int loc, unsigned short flag)
{
	rxq->sbl->sb[loc].flags |= (1 << flag);
}

/* Mask sbl for given flag */
static void sbl_clear_flag(struct efct_rx_queue *rxq, int loc, unsigned short flag)
{
	rxq->sbl->sb[loc].flags &= ~(1 << flag);
}

/* Initialize huge page list
 *
 * Context : driver
 *
 * return :  0 - Success
 *          <0 - Failure
 */
int hpl_init(struct efct_rx_queue *rxq)
{
	struct efct_huge_page_list *hpl;
	int i;

	if (!rxq->hpl) {
		rxq->hpl_alloced = INITIAL_HUGE_PAGE_LIST_SIZE;
		rxq->hpl = kvzalloc(struct_size(hpl, hp, rxq->hpl_alloced), GFP_KERNEL);
		if (!rxq->hpl) {
			netif_err(rxq->efct, rx_err, rxq->efct->net_dev,
				  "Rx queue %d no memory for initial huge page list\n", rxq->index);
			return -ENOMEM;
		}
		rxq->sbl = kvzalloc(struct_size(rxq->sbl, sb,
						rxq->hpl_alloced * SUPER_BUFFS_PER_HUGE_PAGE),
						GFP_KERNEL);
		if (!rxq->sbl) {
			netif_err(rxq->efct, rx_err, rxq->efct->net_dev,
				  "Rx queue %d no memory for initial superbuf list\n", rxq->index);
			kvfree(rxq->hpl);
			rxq->hpl = NULL;
			return -ENOMEM;
		}
	}
	hpl = rxq->hpl;
	memset(&hpl->hp[0], 0, rxq->hpl_alloced * sizeof(hpl->hp[0]));

	/* Add all buffers to free list */
	for (i = 0; i < rxq->hpl_alloced - 1; ++i)
		hpl->hp[i].next = i + 1;

	hpl->hp[i].next = -1;
	hpl->free_list = 0;
	hpl->max_hp_id = 0;

	return 0;
}

void hpl_fini(struct efct_rx_queue *rxq)
{
	kvfree(rxq->hpl);
	kvfree(rxq->sbl);
	rxq->hpl = NULL;
	rxq->sbl = NULL;
}

/* Create superbufs from huge page and add created buffers to superbuf list from base */
static int hpl_hp_to_sb(struct efct_rx_queue *rxq, struct xlnx_efct_hugepage *hp, int base,
			struct xlnx_efct_client *client_id)
{
	struct efct_nic *efct = rxq->efct;
	struct efct_buffer buffer;
	int i, nbufs, end;

	/* Map the page */
	buffer.dma_addr = dma_map_page(&efct->efct_dev->pci_dev->dev,
				       hp->page, 0, HUGE_PAGE_SZ, DMA_FROM_DEVICE);
	if (unlikely(dma_mapping_error(&efct->efct_dev->pci_dev->dev, buffer.dma_addr))) {
		netif_err(rxq->efct, probe, rxq->efct->net_dev,
			  "Rx queue %d hp dma map failed\n", rxq->index);
		return -ENOMEM;
	}

	/* store dma mapping for post cleanup */
	rxq->hpl->hp[base].dma_addr = buffer.dma_addr;

	buffer.addr = kmap(hp->page);
	buffer.len = rxq->buffer_size;
	buffer.sentinel = 0;
	nbufs = HUGE_PAGE_SZ / rxq->buffer_size;
	end = (base + 1) * nbufs;

	/* Do mapping of huge page to superbuffs in sbl */
	for (i = base * nbufs; i < end; ++i) {
		sbl_add(rxq, &buffer, i, client_id);

		buffer.addr = (void *)((u8 *)buffer.addr + rxq->buffer_size);
		buffer.dma_addr = buffer.dma_addr + rxq->buffer_size;
	}

	/* Finally mark in use superbufs */
	rxq->hpl->hp[base].active_sb_cnt = nbufs;

	return 0;
}

static void efct_kvfree_rcu(struct rcu_head *rcu)
{
	kvfree(rcu);
}

static int grow_hp_alloc(struct efct_rx_queue *rxq)
{
	struct efct_super_buffer_list *sbl;
	struct efct_huge_page_list *hpl;
	u32 new_alloced;
	int i;

	new_alloced = rxq->hpl_alloced + (rxq->hpl_alloced >> 1);
	hpl = kvzalloc(struct_size(hpl, hp, new_alloced), GFP_KERNEL);
	if (!hpl)
		return -ENOMEM;
	sbl = kvzalloc(struct_size(sbl, sb,
				   new_alloced * SUPER_BUFFS_PER_HUGE_PAGE),
				   GFP_KERNEL);
	if (!sbl) {
		kvfree(hpl);
		return -ENOMEM;
	}
	memcpy(hpl, rxq->hpl, struct_size(hpl, hp, rxq->hpl_alloced));
	memcpy(sbl, rxq->sbl,
	       struct_size(sbl, sb, rxq->hpl_alloced * SUPER_BUFFS_PER_HUGE_PAGE));
	for (i = rxq->hpl_alloced * SUPER_BUFFS_PER_HUGE_PAGE;
	     i < new_alloced * SUPER_BUFFS_PER_HUGE_PAGE; ++i)
		sbl_init_one(&sbl->sb[i]);
	for (i = rxq->hpl_alloced; i < new_alloced - 1; ++i)
		hpl->hp[i].next = i + 1;
	hpl->hp[i].next = -1;
	BUG_ON(hpl->free_list != -1);
	hpl->free_list = rxq->hpl_alloced;

	/* It's mostly safe just to resize the lists without locking since this
	 * function can only be called with NAPI disabled, however we do still need
	 * hpl_get_hugepages() and various debugfs ops to continue to work, hence
	 * doing RCU.
	 */
	/* kvfree_rcu() isn't available on RHEL7 kernels */
	call_rcu(&rxq->sbl->rcu, efct_kvfree_rcu);
	call_rcu(&rxq->hpl->rcu, efct_kvfree_rcu);
	rxq->sbl = sbl;
	rxq->hpl = hpl;
	rxq->hpl_alloced = new_alloced;
	return 0;
}

/* Add huge page for a client */
static int hpl_add_hp(struct efct_rx_queue *rxq, struct xlnx_efct_hugepage *hp,
		      struct xlnx_efct_client *client_id)
{
	struct efct_huge_page_list *hpl = rxq->hpl;
	int rc;

	if (hpl->free_list < 0) {
		rc = grow_hp_alloc(rxq);
		if (rc < 0)
			return EFCT_FAILURE;
		hpl = rxq->hpl;
	}

	/* Create superbufs and replicate to superbuf list */
	rc = hpl_hp_to_sb(rxq, hp, hpl->free_list, client_id);
	if (unlikely(rc))
		return rc;

	if (hpl->max_hp_id < hpl->free_list)
		hpl->max_hp_id = hpl->free_list;

	hpl->hp[hpl->free_list].hp = *hp;
	hpl->hp[hpl->free_list].client_id = client_id;
	hpl->free_list = hpl->hp[hpl->free_list].next;
	return 0;
}

/* Add huge pages for this client
 *
 * Context : Process
 *           Disable NAPI before calling this function.
 *
 * Return : 0 - Success
 *          Negative error number - Failure
 *
 * Additionally, on failure, set hp_count to the number of added huge pages.
 */
int hpl_add_hp_for_client(struct efct_rx_queue *rxq,
			  struct xlnx_efct_client *client_id, int *hp_count)
{
	struct xlnx_efct_hugepage hp;
	int i, rc;

	for (i = 0; i < *hp_count; ++i) {
		rc = client_id->drvops->alloc_hugepage(client_id->driver_priv, &hp);
		if (rc) {
			netif_err(rxq->efct, probe, rxq->efct->net_dev,
				  "Rx queue %d failed to allocate client hp\n", rxq->index);
			goto fail;
		}

		if (!atomic_read(&client_id->active_hugepages)) {
			if (!try_module_get(THIS_MODULE)) {
				client_id->drvops->free_hugepage(client_id->driver_priv, &hp);
				netif_err(rxq->efct, probe, rxq->efct->net_dev,
					  "Cannot get module for aux client:%d\n",
					  client_id->index);
				rc = -EBUSY;
				goto fail;
			}
		}

		rc = hpl_add_hp(rxq, &hp, client_id);
		if (rc) {
			netif_err(rxq->efct, probe, rxq->efct->net_dev,
				  "Rx queue %d failed to add hp\n", rxq->index);
			client_id->drvops->free_hugepage(client_id->driver_priv, &hp);
			if (!atomic_read(&client_id->active_hugepages))
				module_put(THIS_MODULE);
			rc = -ENOMEM;
			goto fail;
		}

		atomic_inc(&client_id->active_hugepages);

		/* Increment active HP count for this client */
		atomic_inc(&client_id->rxq_info[rxq->index].active_hp);
	}

	/* HP count argument stays unchanged */
	return 0;

fail:
	*hp_count = i;
	return rc;
}

struct deferred_hp_free {
	struct rcu_head rcu;
	u32 rxq_index;
	struct xlnx_efct_client *client_id;
	struct xlnx_efct_hugepage hp;
};

static void free_aux_hp(struct rcu_head *rcu)
{
	struct deferred_hp_free *defer =
				container_of(rcu, struct deferred_hp_free, rcu);
	struct xlnx_efct_client *client_id = defer->client_id;

	/* Free huge page with client op */
	client_id->drvops->free_hugepage(client_id->driver_priv, &defer->hp);

	if (atomic_dec_and_test(&client_id->active_hugepages))
		module_put(THIS_MODULE);

	/* Decrement active HP count for this client */
	if (atomic_dec_and_test(&client_id->rxq_info[defer->rxq_index].active_hp))
		wake_up(&client_id->efct->hp_cleanup_wq);
	kfree(defer);
}

static void hpl_free_hp(struct efct_rx_queue *rxq, struct efct_huge_page_entry *hp)
{
	struct xlnx_efct_client *client_id = hp->client_id;
	struct efct_huge_page_list *hpl = rxq->hpl;
	struct deferred_hp_free *defer;

	/* Unmap hp */
	kunmap(hp->hp.page);
	dma_unmap_page(&rxq->efct->efct_dev->pci_dev->dev,
		       hp->dma_addr, HUGE_PAGE_SZ,
		       DMA_FROM_DEVICE);

	if (hp->flags & (1 << HP_ORPHAN))
		return;

	defer = kmalloc(sizeof(*defer), GFP_ATOMIC | __GFP_NOFAIL);
	/* call_rcu will skip Hugepage free if kmalloc failed. */
	if (defer) {
		defer->rxq_index = rxq->index;
		defer->client_id = client_id;
		defer->hp = hp->hp;
		call_rcu(&defer->rcu, free_aux_hp);
	}

	hp->client_id = NULL;
	hp->flags = 0u;
	hp->active_sb_cnt = 0;

	/* Add huge page entry to free list */
	hp->next = hpl->free_list;
	hpl->free_list = hp - &hpl->hp[0];
}

/* Free superbuf from this huge page */
static void hpl_free_sb(struct efct_rx_queue *rxq, int sb_id)
{
	struct efct_huge_page_list *hpl = rxq->hpl;
	int id;

	id = sb_id / (HUGE_PAGE_SZ / rxq->buffer_size);

	if (hpl->hp[id].active_sb_cnt) {
		--hpl->hp[id].active_sb_cnt;
		if (!hpl->hp[id].active_sb_cnt)
			hpl_free_hp(rxq, &hpl->hp[id]);
	} else {
		netif_err(rxq->efct, probe, rxq->efct->net_dev,
			  "Rx queue %d hp %d already freed\n", rxq->index, id);
	}
}

/* Remove N huge pages for this client.
 * When less than n_hp are removed, subsequently rollover
 * is triggered.
 *
 * Context : Process
 *           Disable NAPI before calling this function.
 *
 * Return : Number of freed huge pages.
 */
int hpl_remove_hp_for_client(struct efct_rx_queue *rxq, struct xlnx_efct_client *client_id,
			     int n_hp)
{
	struct efct_super_buffer_list *sbl = rxq->sbl;
	struct efct_huge_page_list *hpl = rxq->hpl;
	int hp_count = n_hp;
	int i, j, nbufs;

	nbufs = (HUGE_PAGE_SZ / rxq->buffer_size);
	/* First, find free huge pages and mark them to be moved
	 * out of sbl free list
	 */
	for (i = 0; i < rxq->hpl_alloced && hp_count; ++i) {
		if (hpl->hp[i].client_id == client_id &&
		    !(hpl->hp[i].flags & (1 << HP_LAZY_FREE))) {
			for (j = i * nbufs; j < (i + 1) * nbufs; ++j) {
				if (sbl->sb[j].ref_count)
					break;
			}

			if (j == ((i + 1) * nbufs)) {
				hpl_free_hp(rxq, &hpl->hp[i]);

				/* Mark sbs to be moved out from free list */
				for (j = i * nbufs; j < (i + 1) * nbufs; ++j) {
					/* Poison SB */
					sbl->sb[j].client_id = NULL;
					sbl->sb[j].ref_count = ~sbl->sb[j].ref_count;
					sbl_add_flag(rxq, j, SB_MOVE_OUT);
				}

				--hp_count;
			}
		}
	}

	if (hp_count != 0) {
		n_hp -= hp_count;

		/* Second, mark any remaining huge pages for lazy free */
		for (i = 0; i < rxq->hpl_alloced && hp_count; ++i) {
			if (hpl->hp[i].client_id == client_id &&
			    !(hpl->hp[i].flags & (1 << HP_LAZY_FREE))) {
				for (j = i * nbufs; j < (i + 1) * nbufs; ++j) {
					if (sbl->sb[j].ref_count == 0) {
						hpl_free_sb(rxq, j);
						/* Poisen SB */
						sbl->sb[j].client_id = NULL;
						sbl->sb[j].ref_count = ~sbl->sb[j].ref_count;
						sbl_add_flag(rxq, j, SB_MOVE_OUT);
					} else {
						sbl_add_flag(rxq, j, SB_RETURN_TO_HPL);
					}
				}

				hpl->hp[i].flags |= (1 << HP_LAZY_FREE);
				--hp_count;
			}
		}
	}

	/* Third, remove freed sbs from the list */
	sbl->free_list = -1;
	for (i = 0; i < rxq->hpl_alloced * SUPER_BUFFS_PER_HUGE_PAGE; ++i) {
		if (sbl->sb[i].flags & (1 << SB_MOVE_OUT)) {
			/* Clear the flag */
			sbl_clear_flag(rxq, i, SB_MOVE_OUT);
		} else if (sbl->sb[i].ref_count == 0) {
			sbl->sb[i].next = sbl->free_list;
			sbl->free_list = i;
		}
	}

	return n_hp;
}

/* Return huge page list for this rxq
 *
 * Context : Process, rcu read locked
 */
int hpl_get_hugepages(struct efct_rx_queue *rxq, struct xlnx_efct_hugepage *pages,
		      size_t n_pages)
{
	struct efct_huge_page_list *hpl = rxq->hpl;
	int i;

	if (n_pages < hpl->max_hp_id + 1)
		return -ENOSPC;

	for (i = 0; i < hpl->max_hp_id + 1; ++i) {
		if (READ_ONCE(hpl->hp[i].active_sb_cnt) == 0)
			memset(&pages[i], 0, sizeof(hpl->hp[i].hp));
		else
			memcpy(&pages[i], &hpl->hp[i].hp, sizeof(hpl->hp[i].hp));
	}

	for (; i < n_pages; ++i)
		memset(&pages[i], 0, sizeof(hpl->hp[i].hp));

	return 0;
}

/* Rollover existing NIC buffers. This function draws a new buffer
 * either from sbl or dbl and posts it to NIC with rollover enabled.
 *
 * Context: Process.
 *          Disable NAPI before calling this function.
 */
int efct_rx_rollover(struct efct_rx_queue *rxq)
{
	struct efct_buffer buffer;
	bool is_dbl = false;
#ifdef CONFIG_XILINX_AUX_EFCT
	int ref_cnt;
#endif
	int rc, id;

	if (!rxq->enable)
		return -EINVAL;

	/* Add new buffer to nbl */
	id = sbl_alloc_buff(rxq, &buffer);
	if (id < 0) {
		id = dbl_alloc_buff(rxq, &buffer);
		is_dbl = true;
	}
	if (id < 0)
		return EFCT_FAILURE;
	rc = nbl_buff_push(rxq, &buffer, id, is_dbl, true);
	if (rc) {
		rx_free_buffer(rxq, id, is_dbl, false);
		return rc;
	}

#ifdef CONFIG_XILINX_AUX_EFCT
	ref_cnt = efct_aux_buffer_start(rxq->efct, rxq->index, nbl_tail_seq_no(rxq),
					is_dbl ? -1 : id, buffer.sentinel);
	if (is_dbl)
		return EFCT_SUCCESS;

	/* Update kernel and client ref count for superbuf */
	if (ref_cnt > 0)
		ref_cnt += 1;
	else
		ref_cnt = 1;
	sbl_update_buff_ref(rxq, id, ref_cnt);
#endif
	return EFCT_SUCCESS;
}

#endif

/* Allocate rx buffer and push to nbl */
static void rx_push_buffer_to_nbl(struct efct_rx_queue *rxq, int id,
				  struct efct_buffer *buffer, bool is_dbl)
{
#ifdef CONFIG_XILINX_AUX_EFCT
	int ref_cnt;
#endif
	int rc;

	rc = nbl_buff_push(rxq, buffer, id, is_dbl, false);
	if (unlikely(rc)) {
		rx_free_buffer(rxq, id, is_dbl, false);
		return;
	}
#ifdef CONFIG_XILINX_AUX_EFCT
	ref_cnt = efct_aux_buffer_start(rxq->efct, rxq->index, nbl_tail_seq_no(rxq),
					is_dbl ? -1 : id, buffer->sentinel);
	if (is_dbl)
		return;

	/* Update kernel and client ref count for superbuf */
	if (ref_cnt > 0)
		ref_cnt += 1;
	else
		ref_cnt = 1;
	sbl_update_buff_ref(rxq, id, ref_cnt);
#endif
}

static void rx_add_buffer(struct efct_rx_queue *rxq)
{
	struct efct_buffer buffer;
	int id;

	if (!rxq->enable)
		return;
#ifdef CONFIG_XILINX_AUX_EFCT
	while (is_below_higher_watermark(rxq)) {
		id = sbl_alloc_buff(rxq, &buffer);
		if (id < 0)
			break;
		rx_push_buffer_to_nbl(rxq, id, &buffer, false);
	}

	while (is_below_lower_watermark(rxq)) {
		id = dbl_alloc_buff(rxq, &buffer);
		if (unlikely(id < 0))
			return;
		rx_push_buffer_to_nbl(rxq, id, &buffer, true);
	}
#else
	id = dbl_alloc_buff(rxq, &buffer);
	if (unlikely(id < 0))
		return;
	rx_push_buffer_to_nbl(rxq, id, &buffer, true);
#endif
}

/* Fill receive queue with receive buffers */
int efct_fill_rx_buffs(struct efct_rx_queue *rxq)
{
	struct efct_buffer buffer;
	int id, filled_buffs;

	filled_buffs = 0;
#ifdef CONFIG_XILINX_AUX_EFCT
	while (is_below_higher_watermark(rxq)) {
		id = sbl_alloc_buff(rxq, &buffer);
		if (id < 0)
			break;
		rx_push_buffer_to_nbl(rxq, id, &buffer, false);
		++filled_buffs;
	}

	while (is_below_lower_watermark(rxq)) {
		id = dbl_alloc_buff(rxq, &buffer);
		if (unlikely(id < 0))
			break;
		rx_push_buffer_to_nbl(rxq, id, &buffer, true);
		++filled_buffs;
	}
#else
	for (filled_buffs = 0; filled_buffs < rxq->num_rx_buffs; ++filled_buffs) {
		id = dbl_alloc_buff(rxq, &buffer);
		if (id < 0)
			break;
		rx_push_buffer_to_nbl(rxq, id, &buffer, true);
	}
#endif

	netif_dbg(rxq->efct, drv, rxq->efct->net_dev, "Rxq %d filled with %d buffers\n",
		  rxq->index, filled_buffs);

	if (filled_buffs == 0)
		return -ENOMEM;

	return 0;
}

static __wsum get_csum(union efct_qword *p_meta)
{
	__u16 hw1;

	hw1 = be16_to_cpu((__force __be16)EFCT_QWORD_FIELD(*p_meta, ESF_HZ_RX_PREFIX_CSUM_FRAME));
	WARN_ON_ONCE(!hw1);

	return (__force __wsum)hw1;
}

static bool is_l2_status_ok(struct efct_rx_queue *rx_queue, union efct_qword *p_meta)
{
	u16 l2_status;

	l2_status = EFCT_QWORD_FIELD(*p_meta, ESF_HZ_RX_PREFIX_L2_STATUS);
	if (l2_status == ESE_HZ_X3_L2_STATUS_FCS_ERR) {
		rx_queue->n_rx_eth_crc_err++;
		return false;
	}
	if (l2_status == ESE_HZ_X3_L2_STATUS_LEN_ERR) {
		rx_queue->n_rx_eth_len_err++;
		return false;
	}

	return true;
}

#ifdef CONFIG_XILINX_PTP
#define NSEC_BITS_MASK 0xffffffff

static void efct_include_ts_in_rxskb(struct efct_rx_queue *rxq, union efct_qword *p_meta,
				     struct sk_buff *skb)
{
	struct skb_shared_hwtstamps *timestamps;
	struct efct_ptp_data *ptp;
	struct efct_nic *efct;
	u64 pkt_ts_major;
	u32 pkt_ts_minor;

	efct = rxq->efct;
	ptp = efct->ptp_data;
	timestamps = skb_hwtstamps(skb);
	pkt_ts_major = EFCT_OWORD_FIELD(*((union efct_oword *)p_meta), ESF_HZ_RX_PREFIX_TIMESTAMP);
	pkt_ts_minor = (pkt_ts_major & NSEC_BITS_MASK);
	pkt_ts_major = pkt_ts_major >> 32;
	timestamps->hwtstamp = ptp->nic_to_kernel_time(pkt_ts_major, pkt_ts_minor,
				ptp->ts_corrections.general_rx);
}
#endif
/* Deliver packet to stack */
static void efct_rx_deliver(struct efct_rx_queue *rxq, u8 *pkt_start, union efct_qword *p_meta)
{
	struct sk_buff *skb = NULL;
#ifdef CONFIG_XILINX_PTP
	struct efct_ptp_data *ptp;
#endif
	struct efct_nic *efct;
	struct ethhdr *eth;
	__wsum csum = 0;
	u16 len;

	efct = rxq->efct;

	len = EFCT_QWORD_FIELD(*p_meta, ESF_HZ_RX_PREFIX_LENGTH);
#ifdef CONFIG_XILINX_PTP
	ptp = efct->ptp_data;
#endif
	if (unlikely(!is_l2_status_ok(rxq, p_meta))) {
		if (!(efct->net_dev->features & NETIF_F_RXALL))
			goto drop;
	}

	skb = napi_alloc_skb(&efct->evq[rxq->evq_index].napi, len);
	if (unlikely(!skb)) {
		rxq->n_rx_alloc_skb_fail++;
		goto drop;
	}
#ifdef CONFIG_XILINX_PTP
	if (ptp->rxtstamp && EFCT_QWORD_FIELD(*p_meta, ESF_HZ_RX_PREFIX_TIMESTAMP_STATUS))
		efct_include_ts_in_rxskb(rxq, p_meta, skb);
#endif
	/* Copy packet from rx buffer to skb */
	memcpy(skb_put(skb, len), pkt_start, len);
	skb_mark_napi_id(skb, &efct->evq[rxq->evq_index].napi);
	skb->protocol = eth_type_trans(skb, efct->net_dev);

	eth = eth_hdr(skb);
	do {
		/* Accept all packets */
		if (efct->net_dev->flags & IFF_PROMISC)
			break;
		if (skb->pkt_type == PACKET_BROADCAST) {
			/* Accept all broadcast packets */
			if (efct->net_dev->flags & IFF_BROADCAST)
				break;
			rxq->n_rx_broadcast_drop++;
			goto drop;
		} else if (skb->pkt_type == PACKET_MULTICAST) {
			if ((efct->net_dev->flags & IFF_MULTICAST) &&
			    (netdev_mc_count(efct->net_dev))) {
				struct netdev_hw_addr *hw_addr;
				int found_mc_addr = 0;

				netdev_for_each_mc_addr(hw_addr, efct->net_dev) {
					if (ether_addr_equal(eth->h_dest, hw_addr->addr)) {
						found_mc_addr = 1;
						break;
					}
				}

				if (!found_mc_addr) {
					/* Unwanted multicast traffic. Drop packet and
					 * count as mismatch but don't report as drop
					 */
					rxq->n_rx_mcast_mismatch++;
					goto free_buf;
				}
			}
		} else if (skb->pkt_type == PACKET_HOST) {
			break;
		} else if (skb->pkt_type == PACKET_OTHERHOST) {
			bool found_uc_addr = false;

			if ((efct->net_dev->priv_flags & IFF_UNICAST_FLT) &&
			    netdev_uc_count(efct->net_dev)) {
				struct netdev_hw_addr *uc;

				netdev_for_each_uc_addr(uc, efct->net_dev) {
					if (ether_addr_equal(eth->h_dest, uc->addr)) {
						found_uc_addr = true;
						break;
					}
				}
			}

			if (!found_uc_addr) {
				/* Unwanted unicast traffic. Drop packet and
				 * count as other_host but don't report as drop.
				 */
				rxq->n_rx_other_host_drop++;
				goto free_buf;
			}
		}
	} while (0);

	/* RX checksum offload */
	if (EFCT_QWORD_FIELD(*p_meta, ESF_HZ_RX_PREFIX_L2_CLASS) ==
				ESE_HZ_X3_L2_CLASS_ETH_01VLAN) {
		if (EFCT_QWORD_FIELD(*p_meta, ESF_HZ_RX_PREFIX_L3_CLASS) ==
				ESE_HZ_X3_L3_CLASS_IP4) {
			if (EFCT_QWORD_FIELD(*p_meta, ESF_HZ_RX_PREFIX_L3_STATUS) ==
					ESE_HZ_X3_L3_STATUS_BAD_OR_UNKNOWN) {
				rxq->n_rx_ip_hdr_chksum_err++;
			} else {
				if (likely(efct->net_dev->features & NETIF_F_RXCSUM))
					csum = get_csum(p_meta);
				switch (EFCT_QWORD_FIELD(*p_meta, ESF_HZ_RX_PREFIX_L4_CLASS)) {
				case ESE_HZ_X3_L4_CLASS_TCP:
				case ESE_HZ_X3_L4_CLASS_UDP:
					if (EFCT_QWORD_FIELD(*p_meta, ESF_HZ_RX_PREFIX_L4_STATUS) ==
						ESE_HZ_X3_L4_STATUS_BAD_OR_UNKNOWN)
						rxq->n_rx_tcp_udp_chksum_err++;
					break;
				}
			}
		}
	}

	if (csum) {
		skb->csum = csum;
		skb->ip_summed = CHECKSUM_COMPLETE;
	} else {
		skb->ip_summed = CHECKSUM_NONE;
	}
#ifdef CONFIG_XILINX_EFCT_TRACING
	trace_xilinx_efct_receive(skb, pkt_start, p_meta, rxq);
#endif

	napi_gro_receive(&efct->evq[rxq->evq_index].napi, skb);
	++rxq->rx_packets;
	return;

drop:
	atomic64_inc(&efct->n_rx_sw_drops);
free_buf:
	if (skb)
		dev_kfree_skb_any(skb);
}

/* Process rx event
 * This function return number of packets processed
 */
void efct_ev_rx(struct efct_rx_queue *rxq, const union efct_qword *p_event)
{
	bool is_dbl, rollover, pkt_handled = false, is_sentinel_mismatch;
#ifdef CONFIG_XILINX_AUX_EFCT
	bool flow_lookup;
#endif
	u16 i, n_pkts;
	int rc, id;

	rollover = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_RXPKTS_ROLLOVER);
	if (rollover) {
		rc = nbl_buff_pop(rxq, true, &id, &is_dbl);
		if (!rc) {
			rx_free_buffer(rxq, id, is_dbl, true);
			rx_add_buffer(rxq);
		}
		rxq->n_rx_rollover_events++;
		return;
	}

	n_pkts = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_RXPKTS_NUM_PACKETS);
#ifdef CONFIG_XILINX_AUX_EFCT
	flow_lookup = EFCT_QWORD_FIELD(*p_event, ESF_HZ_EV_RXPKTS_FLOW_LOOKUP);
#endif
	rxq->efct->evq[rxq->evq_index].irq_mod_score += 2 * n_pkts;

	if (n_pkts > 1) {
		++rxq->n_rx_merge_events;
		rxq->n_rx_merge_packets += n_pkts;
	}

	for (i = 0; i < n_pkts; ++i) {
		/* Packet belong to stack */
		union efct_qword *p_meta;
		u8 *pkt_start;

		is_sentinel_mismatch = false;

		rc = nbl_buff_pkt_extract_meta(rxq, &p_meta, &pkt_start, &is_sentinel_mismatch);
		if (unlikely(rc)) {
			rxq->n_rx_nbl_empty++;
			/* Drop the event */
			return;
		}
#ifdef CONFIG_XILINX_AUX_EFCT
		pkt_handled = efct_aux_packet_handled(rxq->efct, rxq->index, flow_lookup,
						      p_meta, pkt_start);
#endif
		if (pkt_handled) {
			rxq->n_rx_aux_pkts++;
		} else if (is_sentinel_mismatch) {
			pr_err_once("Interface %s, Rx queue %d: Packet dropped due to sentinel mismatch\n",
				    rxq->efct->net_dev->name, rxq->index);
			rxq->n_rx_sentinel_drop_count++;
			atomic64_inc(&rxq->efct->n_rx_sw_drops);
		} else {
			rc = nbl_buff_pkt_peek(rxq, &pkt_start);
			if (unlikely(rc)) {
				/* Drop the event */
				return;
			}

			efct_rx_deliver(rxq, pkt_start, p_meta);
		}

		if (nbl_buff_pkt_consume(rxq)) {
			/* Pop out buffer from nbl and free to approp list */

			rc = nbl_buff_pop(rxq, false, &id, &is_dbl);
			if (unlikely(rc))
				return;

			rx_free_buffer(rxq, id, is_dbl, true);
			rx_add_buffer(rxq);
		}
	}
}
