// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include "efct_driver.h"
#include "efct_ioctl.h"
#include "efct_common.h"
#include "ioctl_common.h"
#include "mcdi.h"
#include "mcdi_pcol.h"
#ifdef CONFIG_XILINX_PTP
#include "efct_ptp.h"
#endif

static void efct_ioctl_mcdi_complete_reset(struct efct_nic *efct, u32 cmd, int rc)
{
	/* efct_mcdi_rpc() will not schedule a reset if MC_CMD_REBOOT causes
	 * a reboot. But from the user's POV, they're triggering a reboot
	 * 'externally', and want both ports to recover. So schedule the
	 * reset here.
	 */
	if (cmd == MC_CMD_REBOOT && rc == -EIO) {
		netif_warn(efct, hw, efct->net_dev, "Expected MC reboot.\n");
		efct_schedule_reset(efct, RESET_TYPE_MC_FAILURE);
	}
}

static int efct_ioctl_do_mcdi(struct efct_nic *efct,
			      struct efct_mcdi_request2 __user *user_req)
{
	size_t inbuf_len, req_outlen, outlen_actual;
	union efct_dword *outbuf = NULL;
	struct efct_mcdi_request2 *req;
	union efct_dword *inbuf = NULL;
	int rc;

	req = memdup_user(user_req, sizeof(*req));
	if (!req)
		return -ENOMEM;

	/* No input flags are defined yet */
	if (req->flags != 0) {
		rc = -EINVAL;
		goto out_free;
	}

	/* efct_mcdi_rpc() will check the length anyway, but this avoids
	 * trying to allocate an extreme amount of memory.
	 */
	if (req->inlen > MCDI_CTL_SDU_LEN_MAX_V2 ||
	    req->outlen > MCDI_CTL_SDU_LEN_MAX_V2) {
		rc = -EINVAL;
		goto out_free;
	}

	inbuf_len = ALIGN(req->inlen, 4);
	inbuf = kmalloc(inbuf_len, GFP_USER);
	if (!inbuf) {
		rc = -ENOMEM;
		goto out_free;
	}
	/* Ensure zero-padding if req.inlen not a multiple of 4 */
	if (req->inlen % 4)
		inbuf[req->inlen / 4].word32 = 0;

	outbuf = kmalloc(ALIGN(req->outlen, 4), GFP_USER);
	if (!outbuf) {
		rc = -ENOMEM;
		goto out_free;
	}

	if (copy_from_user(inbuf, &user_req->payload, req->inlen)) {
		rc = -EFAULT;
		goto out_free;
	}

	/* We use inbuf_len as an inlen not divisible by 4 annoys mcdi-logging.
	 * It doesn't care about outlen however.
	 */
	rc = efct_mcdi_rpc_quiet(efct, req->cmd, inbuf, inbuf_len,
				 outbuf, req->outlen, &outlen_actual);

	efct_ioctl_mcdi_complete_reset(efct, req->cmd, rc);

	if (rc) {
		if (outlen_actual) {
			/* Error was reported by the MC */
			req->flags |= EFCT_MCDI_REQUEST_ERROR;
			req->host_errno = -rc;
			rc = 0;
		} else {
			/* Communication failure */
			goto out_free;
		}
	}
	req_outlen = req->outlen;
	req->outlen = outlen_actual;

	if (copy_to_user(user_req, req, sizeof(*req)) ||
	    copy_to_user(&user_req->payload, outbuf,
			 min(outlen_actual, req_outlen)))
		rc = -EFAULT;

out_free:
	kfree(outbuf);
	kfree(inbuf);
	kfree(req);
	return rc;
}

static int efct_ioctl_get_device_ids(struct efct_nic *efct,
				     union efct_ioctl_data *data)
{
	struct efct_device_ids *ids = &data->device_ids;

	ids->vendor_id = efct->efct_dev->pci_dev->vendor;
	ids->device_id = efct->efct_dev->pci_dev->device;
	ids->subsys_vendor_id = efct->efct_dev->pci_dev->subsystem_vendor;
	ids->subsys_device_id = efct->efct_dev->pci_dev->subsystem_device;
	ids->port_num = efct_port_num(efct);
	/* ids->perm_addr isn't __aligned(2), so we can't use ether_addr_copy
	 * (and we can't change it because it's an ioctl argument)
	 */
	ether_addr_copy(ids->perm_addr, efct->net_dev->perm_addr);

	return 0;
}

static int efct_ioctl_set_carrier(struct efct_nic *efct,
				  union efct_ioctl_data *data)
{
	struct efct_set_carrier_ioctl *eci = &data->set_carrier;

	if (eci->on)
		netif_carrier_on(efct->net_dev);
	else
		netif_carrier_off(efct->net_dev);

	return 0;
}

static int efct_ioctl_set_loopback(struct efct_nic *efct,
				   union efct_ioctl_data *data)
{
	enum efct_loopback_mode mode = data->set_loopback.mode;
	enum efct_loopback_mode old_mode;
	int rc;

	/* Check that any mode is supported before we search for a set bit */
	if (efct->loopback_modes == 0)
		return -EOPNOTSUPP;
	if (!efct_net_active(efct->state))
		return -EOPNOTSUPP;

	if (mode == LOOPBACK_NEAR)
		mode = ffs(efct->loopback_modes) - 1;
	if (mode == LOOPBACK_FAR)
		/* The furthest internal facing loopback, so exclude network
		 * loopback
		 */
		mode = fls(efct->loopback_modes & ~LOOPBACKS_WS) - 1;

	/* Check mode is supported by port */
	if (mode && (!(efct->loopback_modes & (1 << mode))))
		return -EOPNOTSUPP;

	mutex_lock(&efct->mac_lock);
	old_mode = efct->loopback_mode;
	efct->loopback_mode = mode;
	rc = __efct_reconfigure_port(efct);
	if (rc)
		efct->loopback_mode = old_mode;
	mutex_unlock(&efct->mac_lock);

	return rc;
}

static int efct_ioctl_set_phy_power(struct efct_nic *efct,
				    union efct_ioctl_data *data)
{
	enum efct_phy_mode old_mode;
	int rc;

	mutex_lock(&efct->mac_lock);

	old_mode = efct->phy_mode;
	if (data->set_phy_power.on)
		efct->phy_mode &= ~PHY_MODE_LOW_POWER;
	else
		efct->phy_mode |= PHY_MODE_LOW_POWER;

	rc = __efct_reconfigure_port(efct);
	if (rc != 0)
		efct->phy_mode = old_mode;
	else
		efct->phy_power_force_off = !data->set_phy_power.on;

	mutex_unlock(&efct->mac_lock);

	return rc;
}

#ifdef CONFIG_XILINX_PTP
static int efct_ioctl_ts_set_sync_status(struct efct_nic *efct,
					 union efct_ioctl_data *data)
{
	return efct_ptp_ts_set_sync_status(efct, data->ts_set_sync_status.in_sync,
					data->ts_set_sync_status.timeout);
}
#endif

int efct_private_ioctl_common(struct efct_nic *efct, u16 cmd,
			      union efct_ioctl_data __user *user_data)
{
	int (*op)(struct efct_nic *efct, union efct_ioctl_data *data);
	union efct_ioctl_data *data = NULL;
	size_t size;
	int rc;

	if (!capable(CAP_NET_ADMIN))
		return -EPERM;

	switch (cmd) {
	case EFCT_MCDI_REQUEST2:
		/* This command has variable length */
		return efct_ioctl_do_mcdi(efct, &user_data->mcdi_request2);
	case EFCT_GET_DEVICE_IDS:
		size = sizeof(data->device_ids);
		op = efct_ioctl_get_device_ids;
		break;
	case EFCT_SET_LOOPBACK:
		size = sizeof(data->set_loopback);
		op = efct_ioctl_set_loopback;
		break;
	case EFCT_SET_CARRIER:
		size = sizeof(data->set_carrier);
		op = efct_ioctl_set_carrier;
		break;
	case EFCT_SET_PHY_POWER:
		size = sizeof(data->set_phy_power);
		op = efct_ioctl_set_phy_power;
		break;
#ifdef CONFIG_XILINX_PTP
	case EFCT_TS_SET_SYNC_STATUS:
		size = sizeof(data->ts_set_sync_status);
		op = efct_ioctl_ts_set_sync_status;
		break;
#endif
	default:
		return -EOPNOTSUPP;
	}

	data = kmalloc(sizeof(*data), GFP_KERNEL);
	if (!data)
		return -ENOMEM;

	if (copy_from_user(data, user_data, size)) {
		kfree(data);
		return -EFAULT;
	}

	rc = op(efct, data);
	if (!rc) {
		if (copy_to_user(user_data, data, size))
			rc = -EFAULT;
	}

	kfree(data);
	return rc;
}
