/* SPDX-License-Identifier: GPL-2.0
 ****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#ifndef EFCT_MCDI_H
#define EFCT_MCDI_H

#include "efct_driver.h"
#include <linux/hwmon.h>
#include <linux/rhashtable.h>
#include "mcdi_pcol.h"
/**
 * enum efct_mcdi_mode - MCDI transaction mode
 * @MCDI_MODE_POLL: poll for MCDI completion, until timeout
 * @MCDI_MODE_EVENTS: wait for an mcdi_event.  On timeout, poll once
 * @MCDI_MODE_FAIL: we think MCDI is dead, so fail-fast all calls
 */
enum efct_mcdi_mode {
	MCDI_MODE_POLL,
	MCDI_MODE_EVENTS,
	MCDI_MODE_FAIL,
};

/* On older firmwares there is only a single thread on the MC, so even
 * the shortest operation can be blocked for some time by an operation
 * requested by a different function.
 * See bug61269 for further discussion.
 *
 * On newer firmwares that support multithreaded MCDI commands we extend
 * the timeout for commands we know can run longer.
 */
#define MCDI_RPC_TIMEOUT       (10 * HZ)
#define MCDI_RPC_LONG_TIMEOUT  (60 * HZ)
#define MCDI_RPC_POST_RST_TIME (10 * HZ)

#define MC_OWNERSHIP_STATUS_DELAY_MS      1
#define MC_OWNERSHIP_STATUS_DELAY_COUNT   1000

#define MCDI_BUF_LEN (8 + MCDI_CTL_SDU_LEN_MAX)

/**
 * enum efct_mcdi_cmd_state - State for an individual MCDI command
 * @MCDI_STATE_QUEUED: Command not started
 * @MCDI_STATE_RETRY: Command was submitted and MC rejected with no resources.
 *                    Command will be retried once another command returns.
 * @MCDI_STATE_PROXY: Command needs authenticating with proxy auth. Will be sent
 *                    again after a PROXY_COMPLETE event.
 * @MCDI_STATE_RUNNING: Command was accepted and is running.
 *		      race between completion in another threads and the worker.
 * @MCDI_STATE_PROXY_CANCELLED:
 * @MCDI_STATE_RUNNING_CANCELLED:
 * @MCDI_STATE_FINISHED:
 */

enum efct_mcdi_cmd_state {
	/* waiting to run */
	MCDI_STATE_QUEUED,
	/* we tried to run, but the MC said we have too many outstanding
	 * commands
	 */
	MCDI_STATE_RETRY,
	/* we sent the command and the MC is waiting for proxy auth */
	MCDI_STATE_PROXY,
	/* the command is running */
	MCDI_STATE_RUNNING,
	/* state was PROXY but the issuer cancelled the command */
	MCDI_STATE_PROXY_CANCELLED,
	/* the command is running but the issuer cancelled the command */
	MCDI_STATE_RUNNING_CANCELLED,
	/* processing of this command has completed.
	 * used to break races between contexts.
	 */
	MCDI_STATE_FINISHED,
};

typedef void efct_mcdi_async_completer(struct efct_nic *efct,
				      unsigned long cookie, int rc,
				      union efct_dword *outbuf,
				      size_t outlen_actual);

/**
 * struct efct_mcdi_cmd - An outstanding MCDI command
 * @ref: Reference count. There will be one reference if the command is
 *	in the mcdi_iface cmd_list, another if it's on a cleanup list,
 *	and a third if it's queued in the work queue.
 * @list: The data for this entry in mcdi->cmd_list
 * @cleanup_list: The data for this entry in a cleanup list
 * @work: The work item for this command, queued in mcdi->workqueue
 * @mcdi: The mcdi_iface for this command
 * @state: The state of this command
 * @inlen: inbuf length
 * @inbuf: Input buffer
 * @quiet: Whether to silence errors
 * @polled: Whether this command is polled or evented
 * @reboot_seen: Whether a reboot has been seen during this command,
 *	to prevent duplicates
 * @seq: Sequence number
 * @bufid: Buffer ID from the NIC implementation
 * @started: Jiffies this command was started at
 * @cookie: Context for completion function
 * @completer: Completion function
 * @cmd: Command number
 * @proxy_handle: Handle if this command was proxied
 * @atomic_completer: Completion function
 * @handle: Handle
 * @rc: Return code
 * @outlen: Length of response buffer
 * @outbuf: Response buffer
 *
 */
struct efct_mcdi_cmd {
	struct kref ref;
	struct list_head list;
	struct list_head cleanup_list;
	struct delayed_work work;
	struct efct_mcdi_iface *mcdi;
	enum efct_mcdi_cmd_state state;
	size_t inlen;
	const union efct_dword *inbuf;
	bool quiet;
	bool polled;
	bool reboot_seen;
	u8 seq;
	u8 bufid;
	unsigned long started;
	unsigned long cookie;
	efct_mcdi_async_completer *atomic_completer;
	efct_mcdi_async_completer *completer;
	u32 handle;
	u32 cmd;
	int rc;
	size_t outlen;
	union efct_dword *outbuf;
	u32 proxy_handle;
	/* followed by inbuf data if necessary */
};

/**
 * struct efct_mcdi_iface - MCDI protocol context
 * @efct: The associated NIC
 * @iface_lock: Serialise access to this structure
 * @cmd_list: List of outstanding and running commands
 * @workqueue: Workqueue used for delayed processing
 * @outstanding_cleanups: Count of cleanups
 * @cmd_complete_wq: Waitqueue for command completion
 * @db_held_by: Command the MC doorbell is in use by
 * @seq_held_by: Command each sequence number is in use by
 * @prev_seq: The last used sequence number
 * @prev_handle: The last used command handle
 * @mode: Poll for mcdi completion, or wait for an mcdi_event
 * @new_epoch: Indicates start of day or start of MC reboot recovery
 * @logging_buffer: Buffer that may be used to build MCDI tracing messages
 * @logging_enabled: Whether to trace MCDI
 */
struct efct_mcdi_iface {
	struct efct_nic *efct;
	/* Serialise access*/
	spinlock_t iface_lock;
	u32 outstanding_cleanups;
	struct list_head cmd_list;
	struct workqueue_struct *workqueue;
	wait_queue_head_t cmd_complete_wq;
	struct efct_mcdi_cmd *db_held_by;
	struct efct_mcdi_cmd *seq_held_by[16];
	u32 prev_handle;
	enum efct_mcdi_mode mode;
	u8 prev_seq;
	bool new_epoch;
#ifdef CONFIG_XILINX_MCDI_LOGGING
	bool logging_enabled;
	char *logging_buffer;
#endif
};

#ifdef CONFIG_XILINX_MCDI_MON
#define EFCT_HWMON_TYPES_COUNT	hwmon_pwm

struct efct_sensor_reading {
	u32 handle;
	u32 value;
	u32 state;
};

struct efct_mcdi_mon {
	struct efct_nic *efct;
	struct efct_sensor_reading *sensor_buf;
	/*update lock*/
	struct mutex update_lock;
	unsigned long last_update;
	struct device *device;
	struct efct_mcdi_mon_attribute *attrs;
	u32 n_attrs;
	void *sensor_list;
	struct rhashtable sensor_table;
	u32 generation_count;
	u32 n_dynamic_sensors;
	u32 pend_sensor_handle;
	u32 pend_sensor_state;
	struct hwmon_chip_info *chip_info;
	struct workqueue_struct *monwq;
	int type_idx[EFCT_HWMON_TYPES_COUNT];
#if defined(EFCT_USE_KCOMPAT) && !defined(EFCT_HAVE_HWMON_DEVICE_REGISTER_WITH_INFO)
	struct kobject kobj;
	struct attribute_group sysfs_sensor_attr_group;
	struct kobj_type sensor_ktype;
#endif
};

struct mon_state_work {
	struct work_struct work;
	struct efct_nic *efct;
	u32 handle;
	int state;
	int value;
};
#endif

/**
 * struct efct_mcdi_data - extra state for NICs that implement MCDI
 * @iface: Interface/protocol state
 * @hwmon: Hardware monitor state
 * @fn_flags: Flags for this function, as returned by %MC_CMD_DRV_ATTACH.
 */
struct efct_mcdi_data {
	struct efct_mcdi_iface iface;
#ifdef CONFIG_XILINX_MCDI_MON
	struct efct_mcdi_mon hwmon;
#endif
	u32 fn_flags;
};

static inline struct efct_mcdi_iface *efct_mcdi(struct efct_nic *efct)
{
	return efct->mcdi ? &efct->mcdi->iface : NULL;
}

#ifdef CONFIG_XILINX_MCDI_MON
static inline struct efct_mcdi_mon *efct_mcdi_mon(struct efct_nic *efct)
{
	return efct->mcdi ? &efct->mcdi->hwmon : NULL;
}
#endif

int efct_mcdi_init(struct efct_nic *efct);
void efct_mcdi_detach(struct efct_nic *efct);
void efct_mcdi_fini(struct efct_nic *efct);

int efct_mcdi_rpc(struct efct_nic *efct, u32 cmd,
		  const union efct_dword *inbuf, size_t inlen,
		  union efct_dword *outbuf, size_t outlen, size_t *outlen_actual);
int efct_mcdi_rpc_quiet(struct efct_nic *efct, u32 cmd,
			const union efct_dword *inbuf, size_t inlen,
			union efct_dword *outbuf, size_t outlen,
			size_t *outlen_actual);

int efct_mcdi_rpc_async(struct efct_nic *efct, u32 cmd,
			const union efct_dword *inbuf, size_t inlen,
			efct_mcdi_async_completer *complete,
			unsigned long cookie);
int efct_mcdi_rpc_async_ext(struct efct_nic *efct, u32 cmd,
			    const union efct_dword *inbuf, size_t inlen,
			    efct_mcdi_async_completer *atomic_completer,
			    efct_mcdi_async_completer *completer,
			    unsigned long cookie, bool quiet,
			    bool immediate_only, u32 *handle);
/* Attempt to cancel an outstanding command.
 * This function guarantees that the completion function will never be called
 * after it returns. The command may or may not actually be cancelled.
 */
void efct_mcdi_cancel_cmd(struct efct_nic *efct, u32 handle);

void efct_mcdi_display_error(struct efct_nic *efct, u32 cmd,
			     size_t inlen, union efct_dword *outbuf,
			     size_t outlen, int rc);

int efct_mcdi_poll_reboot(struct efct_nic *efct);
/* Wait for all commands and all cleanup for them to be complete */
void efct_mcdi_wait_for_cleanup(struct efct_nic *efct);
bool efct_mcdi_process_event(struct efct_nic *efct, union efct_qword *event);
#ifdef CONFIG_XILINX_MCDI_MON
void efct_mcdi_dynamic_sensor_event(struct efct_nic *efct, union efct_qword *ev);
#else
static inline void efct_mcdi_dynamic_sensor_event(struct efct_nic *efct, union efct_qword *ev)
{
}
#endif

/* We expect that 16- and 32-bit fields in MCDI requests and responses
 * are appropriately aligned, but 64-bit fields are only
 * 32-bit-aligned.  Also, on Siena we must copy to the MC shared
 * memory strictly 32 bits at a time, so add any necessary padding.
 */
#define MCDI_DECLARE_BUF(_name, _len) union efct_dword _name[DIV_ROUND_UP(_len, 4)] = {{0}}
#define MCDI_DECLARE_BUF_ERR(_name)					\
	MCDI_DECLARE_BUF(_name, 8)
#define _MCDI_PTR(_buf, _offset)					\
	((u8 *)(_buf) + (_offset))
#define MCDI_PTR(_buf, _field)						\
	_MCDI_PTR(_buf, MC_CMD_ ## _field ## _OFST)
#define _MCDI_CHECK_ALIGN(_ofst, _align)				\
	((void)BUILD_BUG_ON_ZERO((_ofst) & ((_align) - 1)), (_ofst))
#define _MCDI_DWORD(_buf, _field)					\
	((_buf) + (_MCDI_CHECK_ALIGN(MC_CMD_ ## _field ## _OFST, 4) >> 2))
#define MCDI_WORD(_buf, _field)						\
	((void)BUILD_BUG_ON_ZERO(MC_CMD_ ## _field ## _LEN != 2),	\
	 le16_to_cpu(*(__force const __le16 *)MCDI_PTR(_buf, _field)))
#define MCDI_SET_DWORD(_buf, _field, _value)				\
	EFCT_POPULATE_DWORD_1(*_MCDI_DWORD(_buf, _field), EFCT_DWORD_0, _value)
#define MCDI_DWORD(_buf, _field)					\
	EFCT_DWORD_FIELD(*_MCDI_DWORD(_buf, _field), EFCT_DWORD_0)
#define MCDI_POPULATE_DWORD_1(_buf, _field, _name1, _value1)		\
	EFCT_POPULATE_DWORD_1(*_MCDI_DWORD(_buf, _field),		\
			     MC_CMD_ ## _name1, _value1)
#define MCDI_POPULATE_DWORD_2(_buf, _field, _name1, _value1,		\
			      _name2, _value2)				\
	EFCT_POPULATE_DWORD_2(*_MCDI_DWORD(_buf, _field),		\
			     MC_CMD_ ## _name1, _value1,		\
			     MC_CMD_ ## _name2, _value2)
#define MCDI_POPULATE_DWORD_3(_buf, _field, _name1, _value1,		\
			      _name2, _value2, _name3, _value3)		\
	EFCT_POPULATE_DWORD_3(*_MCDI_DWORD(_buf, _field),		\
			     MC_CMD_ ## _name1, _value1,		\
			     MC_CMD_ ## _name2, _value2,		\
			     MC_CMD_ ## _name3, _value3)
#define MCDI_POPULATE_DWORD_4(_buf, _field, _name1, _value1,		\
			      _name2, _value2, _name3, _value3,		\
			      _name4, _value4)				\
	EFCT_POPULATE_DWORD_4(*_MCDI_DWORD(_buf, _field),		\
			     MC_CMD_ ## _name1, _value1,		\
			     MC_CMD_ ## _name2, _value2,		\
			     MC_CMD_ ## _name3, _value3,		\
			     MC_CMD_ ## _name4, _value4)
#define MCDI_POPULATE_DWORD_5(_buf, _field, _name1, _value1,		\
			      _name2, _value2, _name3, _value3,		\
			      _name4, _value4, _name5, _value5)		\
	EFCT_POPULATE_DWORD_5(*_MCDI_DWORD(_buf, _field),		\
			     MC_CMD_ ## _name1, _value1,		\
			     MC_CMD_ ## _name2, _value2,		\
			     MC_CMD_ ## _name3, _value3,		\
			     MC_CMD_ ## _name4, _value4,		\
			     MC_CMD_ ## _name5, _value5)
#define MCDI_POPULATE_DWORD_6(_buf, _field, _name1, _value1,		\
			      _name2, _value2, _name3, _value3,		\
			      _name4, _value4, _name5, _value5,		\
			      _name6, _value6)				\
	EFCT_POPULATE_DWORD_6(*_MCDI_DWORD(_buf, _field),		\
			     MC_CMD_ ## _name1, _value1,		\
			     MC_CMD_ ## _name2, _value2,		\
			     MC_CMD_ ## _name3, _value3,		\
			     MC_CMD_ ## _name4, _value4,		\
			     MC_CMD_ ## _name5, _value5,		\
			     MC_CMD_ ## _name6, _value6)
#define MCDI_POPULATE_DWORD_7(_buf, _field, _name1, _value1,		\
			      _name2, _value2, _name3, _value3,		\
			      _name4, _value4, _name5, _value5,		\
			      _name6, _value6, _name7, _value7)		\
	EFCT_POPULATE_DWORD_7(*_MCDI_DWORD(_buf, _field),		\
			     MC_CMD_ ## _name1, _value1,		\
			     MC_CMD_ ## _name2, _value2,		\
			     MC_CMD_ ## _name3, _value3,		\
			     MC_CMD_ ## _name4, _value4,		\
			     MC_CMD_ ## _name5, _value5,		\
			     MC_CMD_ ## _name6, _value6,		\
			     MC_CMD_ ## _name7, _value7)
#define MCDI_POPULATE_DWORD_8(_buf, _field, _name1, _value1,		\
			      _name2, _value2, _name3, _value3,		\
			      _name4, _value4, _name5, _value5,		\
			      _name6, _value6, _name7, _value7,		\
			      _name8, _value8)		\
	EFCT_POPULATE_DWORD_8(*_MCDI_DWORD(_buf, _field),		\
			     MC_CMD_ ## _name1, _value1,		\
			     MC_CMD_ ## _name2, _value2,		\
			     MC_CMD_ ## _name3, _value3,		\
			     MC_CMD_ ## _name4, _value4,		\
			     MC_CMD_ ## _name5, _value5,		\
			     MC_CMD_ ## _name6, _value6,		\
			     MC_CMD_ ## _name7, _value7,		\
			     MC_CMD_ ## _name8, _value8)
#define MCDI_SET_QWORD(_buf, _field, _value)				\
	do {								\
		EFCT_POPULATE_DWORD_1(_MCDI_DWORD(_buf, _field)[0],	\
				     EFCT_DWORD_0, (u32)(_value));	\
		EFCT_POPULATE_DWORD_1(_MCDI_DWORD(_buf, _field)[1],	\
				     EFCT_DWORD_0, (u64)(_value) >> 32);	\
	} while (0)
#define MCDI_QWORD(_buf, _field)					\
	(EFCT_DWORD_FIELD(_MCDI_DWORD(_buf, _field)[0], EFCT_DWORD_0) |	\
	(u64)EFCT_DWORD_FIELD(_MCDI_DWORD(_buf, _field)[1], EFCT_DWORD_0) << 32)
#define _MCDI_ARRAY_PTR(_buf, _field, _index, _align)			\
	(_MCDI_PTR(_buf, _MCDI_CHECK_ALIGN(MC_CMD_ ## _field ## _OFST, _align))\
	 + (_index) * _MCDI_CHECK_ALIGN(MC_CMD_ ## _field ## _LEN, _align))
#define MCDI_ARRAY_STRUCT_PTR(_buf, _field, _index)			\
	((union efct_dword *)_MCDI_ARRAY_PTR(_buf, _field, _index, 4))
#define _MCDI_ARRAY_DWORD(_buf, _field, _index)				\
	(BUILD_BUG_ON_ZERO(MC_CMD_ ## _field ## _LEN != 4) +		\
	 (union efct_dword *)_MCDI_ARRAY_PTR(_buf, _field, _index, 4))
#define MCDI_VAR_ARRAY_LEN(_len, _field)				\
		min_t(size_t, MC_CMD_ ## _field ## _MAXNUM,			\
		      ((_len) - MC_CMD_ ## _field ## _OFST) / MC_CMD_ ## _field ## _LEN)

#define MCDI_SET_ARRAY_DWORD(_buf, _field, _index, _value)		\
	EFCT_SET_DWORD_FIELD(*_MCDI_ARRAY_DWORD(_buf, _field, _index),	\
			    EFCT_DWORD_0, _value)
#define MCDI_ARRAY_DWORD(_buf, _field, _index)				\
	EFCT_DWORD_FIELD(*_MCDI_ARRAY_DWORD(_buf, _field, _index), EFCT_DWORD_0)
#define MCDI_EVENT_FIELD(_ev, _field)			\
	EFCT_QWORD_FIELD(_ev, MCDI_EVENT_ ## _field)
#define MCDI_CAPABILITY(field)						\
	MC_CMD_GET_CAPABILITIES_V10_OUT_ ## field ## _LBN
#define MCDI_CAPABILITY_OFST(field) \
	MC_CMD_GET_CAPABILITIES_V10_OUT_ ## field ## _OFST
#define efct_has_cap(efct, field) \
	((efct)->type->check_caps((efct), \
			      MCDI_CAPABILITY(field), \
			      MCDI_CAPABILITY_OFST(field)))

int efct_flr(struct efct_nic *efct);
int efct_mcdi_drv_attach(struct efct_nic *efct, u32 fw_variant, u32 *out_flags,
			 bool reattach);
int efct_mcdi_drv_detach(struct efct_nic *efct);
int efct_mcdi_log_ctrl(struct efct_nic *efct, bool evq, bool uart, u32 dest_evq);
int efct_mcdi_nvram_info(struct efct_nic *efct, u32 type,
			 size_t *size_out, size_t *erase_size_out,
			 size_t *write_size_out, bool *protected_out);
int efct_mcdi_handle_assertion(struct efct_nic *efct);
int efct_mcdi_port_reconfigure(struct efct_nic *efct);
int efct_mcdi_reset(struct efct_nic *efct, enum reset_type method);
#if defined(EFCT_NOT_UPSTREAM)
#ifdef CONFIG_XILINX_MCDI_MON
int efct_mcdi_mon_probe(struct efct_nic *efct);
void efct_mcdi_mon_remove(struct efct_nic *efct);
#else
static inline int efct_mcdi_mon_probe(struct efct_nic *efct) { return 0; }
static inline void efct_mcdi_mon_remove(struct efct_nic *efct) {}
#endif
#endif
enum efct_update_finish_mode {
	EFCT_UPDATE_FINISH_WAIT,
	EFCT_UPDATE_FINISH_BACKGROUND,
	EFCT_UPDATE_FINISH_POLL,
	EFCT_UPDATE_FINISH_ABORT,
};

#endif /* EFCT_MCDI_H */
