// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include <linux/vmalloc.h> // needed for vzalloc/vfree

#include "efct_driver.h"
#include "mcdi.h"
#include "mcdi_functions.h"
#include "efct_nic.h"

int efct_mcdi_ev_init(struct efct_ev_queue *eventq)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_INIT_EVQ_V3_OUT_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_INIT_EVQ_V3_IN_LEN);
	struct efct_nic *efct = eventq->efct;
	dma_addr_t dma_addr;
	size_t outlen;
	int rc;

	/* Fill event queue with all ones (i.e. empty events) */
	memset(eventq->buf.addr, 0xff, eventq->buf.len);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_SIZE, eventq->entries);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_INSTANCE, eventq->index);
	/* INIT_EVQ expects index in vector table, not absolute */
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_IRQ_NUM, eventq->msi.idx);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_TMR_MODE, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_TMR_LOAD, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_TMR_RELOAD, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_COUNT_MODE, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_COUNT_THRSHLD, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_FLAG_INT_ARMD, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_FLAG_RPTR_DOS, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_RX_MERGE_TIMEOUT_NS, eventq->rx_merge_timeout_ns);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_TX_MERGE_TIMEOUT_NS, eventq->tx_merge_timeout_ns);

	MCDI_POPULATE_DWORD_6(inbuf, INIT_EVQ_V3_IN_FLAGS,
			      INIT_EVQ_V3_IN_FLAG_INTERRUPTING, 1,
			      INIT_EVQ_V3_IN_FLAG_RX_MERGE, 1,
			      INIT_EVQ_V3_IN_FLAG_TX_MERGE, 1,
			      INIT_EVQ_V3_IN_FLAG_TYPE, 0,
			      INIT_EVQ_V3_IN_FLAG_USE_TIMER, 1,
			      INIT_EVQ_V3_IN_FLAG_CUT_THRU, 0);

	dma_addr = eventq->buf.dma_addr;
	MCDI_SET_QWORD(inbuf, INIT_EVQ_V3_IN_DMA_ADDR, dma_addr);
	rc = efct_mcdi_rpc(efct, MC_CMD_INIT_EVQ, inbuf, MC_CMD_INIT_EVQ_V3_IN_LEN,
			   outbuf, sizeof(outbuf), &outlen);

	if (outlen >= MC_CMD_INIT_EVQ_V3_OUT_LEN)
		netif_dbg(efct, drv, efct->net_dev,
			  "Index %d using event queue flags %08x\n",
			  eventq->index,
			  MCDI_DWORD(outbuf, INIT_EVQ_V3_OUT_FLAGS));

	return rc;
}

int efct_mcdi_ev_set_timer(struct efct_ev_queue *eventq, u32 ns, u32 mode, bool async)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_SET_EVQ_TMR_OUT_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_SET_EVQ_TMR_IN_LEN);
	struct efct_nic *efct = eventq->efct;
	size_t outlen;
	int rc;

	MCDI_SET_DWORD(inbuf, SET_EVQ_TMR_IN_INSTANCE, eventq->index);
	MCDI_SET_DWORD(inbuf, SET_EVQ_TMR_IN_TMR_LOAD_REQ_NS, ns);
	MCDI_SET_DWORD(inbuf, SET_EVQ_TMR_IN_TMR_RELOAD_REQ_NS, ns);
	MCDI_SET_DWORD(inbuf, SET_EVQ_TMR_IN_TMR_MODE, mode);

	if (async) {
		rc = efct_mcdi_rpc_async(efct, MC_CMD_SET_EVQ_TMR, inbuf, sizeof(inbuf), NULL, 0);
	} else {
		rc = efct_mcdi_rpc(efct, MC_CMD_SET_EVQ_TMR, inbuf, sizeof(inbuf),
				   outbuf, sizeof(outbuf), &outlen);
		if (rc)
			efct_mcdi_display_error(efct, MC_CMD_SET_EVQ_TMR,
						MC_CMD_SET_EVQ_TMR_IN_LEN, NULL, 0, rc);
		else
			/* Saving the actual value set */
			eventq->irq_moderation_ns = MCDI_DWORD(outbuf,
							       SET_EVQ_TMR_OUT_TMR_RELOAD_ACT_NS);
	}

	return rc;
}

int efct_mcdi_ev_fini(struct efct_ev_queue *eventq)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_FINI_EVQ_IN_LEN);
	MCDI_DECLARE_BUF_ERR(outbuf);
	struct efct_nic *efct;
	size_t outlen;
	int rc;

	if (!eventq || !eventq->efct)
		return -EINVAL;

	efct = eventq->efct;
	MCDI_SET_DWORD(inbuf, FINI_EVQ_IN_INSTANCE, eventq->index);

	rc = efct_mcdi_rpc(efct, MC_CMD_FINI_EVQ, inbuf, sizeof(inbuf),
			   outbuf, sizeof(outbuf), &outlen);
	if (rc && rc != -EALREADY)
		efct_mcdi_display_error(efct, MC_CMD_FINI_EVQ, MC_CMD_FINI_EVQ_IN_LEN, NULL, 0, rc);

	return rc;
}

int efct_mcdi_rx_init(struct efct_rx_queue *rx_queue)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_INIT_RXQ_V5_IN_LEN);
	struct efct_nic *efct = rx_queue->efct;
	int rc;

	BUILD_BUG_ON(MC_CMD_INIT_RXQ_V5_OUT_LEN != 0);
	/*set the inbuf memory to zero*/
	memset(inbuf, 0, MC_CMD_INIT_RXQ_V5_IN_LEN);

	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_SIZE, rx_queue->num_entries);
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_TARGET_EVQ, rx_queue->evq_index);
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_LABEL, rx_queue->label);
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_INSTANCE, rx_queue->index);
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_PORT_ID, EVB_PORT_ID_ASSIGNED);
	MCDI_POPULATE_DWORD_4(inbuf, INIT_RXQ_V5_IN_FLAGS,
			      INIT_RXQ_V5_IN_DMA_MODE,
			      MC_CMD_INIT_RXQ_V5_IN_EQUAL_STRIDE_SUPER_BUFFER,
			      INIT_RXQ_V5_IN_FLAG_TIMESTAMP, 1,
			      INIT_RXQ_V5_IN_FLAG_PREFIX, 1,
			      INIT_RXQ_V5_IN_FLAG_DISABLE_SCATTER, 1);
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_ES_PACKET_STRIDE, roundup_pow_of_two(efct->mtu));
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_ES_MAX_DMA_LEN, efct->mtu);
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_ES_PACKET_BUFFERS_PER_BUCKET,
		       DIV_ROUND_UP(rx_queue->buffer_size, rx_queue->pkt_stride));

	rc = efct_mcdi_rpc(efct, MC_CMD_INIT_RXQ, inbuf,
			   MC_CMD_INIT_RXQ_V5_IN_LEN, NULL, 0, NULL);
	if (rc && rc != -ENETDOWN && rc != -EAGAIN)
		netif_err(efct, ifup, efct->net_dev,
			  "failed to initialise RXQ %d, error %d\n", rx_queue->index, rc);

	return rc;
}

int efct_mcdi_rx_fini(struct efct_rx_queue *rx_queue)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_FINI_RXQ_IN_LEN);
	MCDI_DECLARE_BUF_ERR(outbuf);
	struct efct_nic *efct = rx_queue->efct;
	size_t outlen;
	int rc;

	MCDI_SET_DWORD(inbuf, FINI_RXQ_IN_INSTANCE, rx_queue->index);

	rc = efct_mcdi_rpc(efct, MC_CMD_FINI_RXQ, inbuf, sizeof(inbuf),
			   outbuf, sizeof(outbuf), &outlen);
	if (rc && rc != -EALREADY)
		efct_mcdi_display_error(efct, MC_CMD_FINI_RXQ, MC_CMD_FINI_RXQ_IN_LEN,
					outbuf, outlen, rc);

	return rc;
}

int efct_mcdi_tx_init(struct efct_tx_queue *tx_queue)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_INIT_TXQ_EXT_IN_LEN);
	struct efct_nic *efct = tx_queue->efct;
	int rc;

	BUILD_BUG_ON(MC_CMD_INIT_TXQ_OUT_LEN != 0);
	/*set the inbuf memory to zero*/
	memset(inbuf, 0, MC_CMD_INIT_TXQ_EXT_IN_LEN);

	MCDI_SET_DWORD(inbuf, INIT_TXQ_EXT_IN_TARGET_EVQ, tx_queue->evq_index);
	MCDI_SET_DWORD(inbuf, INIT_TXQ_EXT_IN_LABEL, tx_queue->label);
	MCDI_SET_DWORD(inbuf, INIT_TXQ_EXT_IN_INSTANCE, tx_queue->txq_index);
	MCDI_SET_DWORD(inbuf, INIT_TXQ_EXT_IN_PORT_ID, EVB_PORT_ID_ASSIGNED);
	//TBD crc mode
	MCDI_POPULATE_DWORD_4(inbuf, INIT_TXQ_EXT_IN_FLAGS,
			      INIT_TXQ_EXT_IN_FLAG_IP_CSUM_DIS, 1,
			      INIT_TXQ_EXT_IN_FLAG_TCP_CSUM_DIS, 1,
			      INIT_TXQ_EXT_IN_FLAG_CTPIO, 1,
			      INIT_TXQ_EXT_IN_FLAG_CTPIO_UTHRESH, 1);

	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_INIT_TXQ,
				 inbuf, sizeof(inbuf),
				NULL, 0, NULL);
	if (rc) {
		efct_mcdi_display_error(efct, MC_CMD_INIT_TXQ,
					MC_CMD_INIT_TXQ_EXT_IN_LEN,
				       NULL, 0, rc);
		return rc;
	}

	return 0;
}

int efct_mcdi_tx_fini(struct efct_tx_queue *tx_queue)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_FINI_TXQ_IN_LEN);
	MCDI_DECLARE_BUF_ERR(outbuf);
	struct efct_nic *efct;
	size_t outlen;
	int rc;

	efct = tx_queue->efct;
	MCDI_SET_DWORD(inbuf, FINI_TXQ_IN_INSTANCE, tx_queue->txq_index);

	rc = efct_mcdi_rpc(efct, MC_CMD_FINI_TXQ, inbuf, sizeof(inbuf),
			   outbuf, sizeof(outbuf), &outlen);
	if (rc && rc != -EALREADY)
		efct_mcdi_display_error(efct, MC_CMD_FINI_TXQ, MC_CMD_FINI_TXQ_IN_LEN,
					outbuf, outlen, rc);

	return rc;
}

int efct_mcdi_filter_insert(struct efct_nic *efct, struct efct_filter_spec *rule, u64 *handle)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_FILTER_OP_EXT_OUT_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_FILTER_OP_V3_IN_LEN);
	size_t outlen;
	int rc;

	memset(inbuf, 0, MC_CMD_FILTER_OP_V3_IN_LEN);
	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_OP, MC_CMD_FILTER_OP_IN_OP_INSERT);

	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_RX_MODE, MC_CMD_FILTER_OP_V3_IN_RX_MODE_SIMPLE);
	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_RX_DEST,
		       (rule->queue_id == RX_CLS_FLOW_DISC ?
		       MC_CMD_FILTER_OP_V3_IN_RX_DEST_DROP :
		       MC_CMD_FILTER_OP_V3_IN_RX_DEST_HOST));
	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_TX_DEST, MC_CMD_FILTER_OP_V3_IN_TX_DEST_DEFAULT);
	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_PORT_ID, EVB_PORT_ID_ASSIGNED);

	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_MATCH_FIELDS, rule->match_fields);
	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_RX_QUEUE,
		       (rule->queue_id == RX_CLS_FLOW_DISC ?
		       0 : rule->queue_id));
	memcpy(MCDI_PTR(inbuf, FILTER_OP_V3_IN_DST_MAC), &rule->loc_mac, sizeof(rule->loc_mac));
	memcpy(MCDI_PTR(inbuf, FILTER_OP_V3_IN_OUTER_VLAN), &rule->outer_vid,
	       sizeof(rule->outer_vid));
	memcpy(MCDI_PTR(inbuf, FILTER_OP_V3_IN_DST_IP), &rule->dst_ip, sizeof(rule->dst_ip));
	memcpy(MCDI_PTR(inbuf, FILTER_OP_V3_IN_DST_PORT), &rule->dst_port, sizeof(rule->dst_port));
	memcpy(MCDI_PTR(inbuf, FILTER_OP_V3_IN_IP_PROTO), &rule->ip_proto, sizeof(rule->ip_proto));
	memcpy(MCDI_PTR(inbuf, FILTER_OP_V3_IN_ETHER_TYPE), &rule->ether_type,
	       sizeof(rule->ether_type));

	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_FILTER_OP, inbuf, sizeof(inbuf),
				 outbuf, sizeof(outbuf), &outlen);
	if (rc == 0)
		*handle = MCDI_QWORD(outbuf, FILTER_OP_EXT_OUT_HANDLE);

	return rc;
}

int efct_mcdi_filter_remove(struct efct_nic *efct, u64 handle)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_FILTER_OP_V3_IN_LEN);
	int rc;

	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_OP, MC_CMD_FILTER_OP_IN_OP_REMOVE);
	MCDI_SET_QWORD(inbuf, FILTER_OP_V3_IN_HANDLE, handle);
	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_FILTER_OP, inbuf,
				 sizeof(inbuf), NULL, 0, NULL);
	return rc;
}

static int efct_mcdi_filter_parser_info(struct efct_nic *efct, struct efct_mcdi_filter_table *table)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_PARSER_DISP_INFO_OUT_LENMAX);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_GET_PARSER_DISP_INFO_IN_LEN);
	unsigned int pd_match_pri, pd_match_count;
	size_t outlen;
	int rc;

	/* Find out which RX filter types are supported, and their priorities */
	MCDI_SET_DWORD(inbuf, GET_PARSER_DISP_INFO_IN_OP,
		       MC_CMD_GET_PARSER_DISP_INFO_IN_OP_GET_SUPPORTED_RX_MATCHES);
	rc = efct_mcdi_rpc(efct, MC_CMD_GET_PARSER_DISP_INFO,
			   inbuf, sizeof(inbuf), outbuf, sizeof(outbuf),
			   &outlen);
	if (rc)
		return rc;

	pd_match_count = MCDI_VAR_ARRAY_LEN(outlen, GET_PARSER_DISP_INFO_OUT_SUPPORTED_MATCHES);

	if (pd_match_count == 0)
		netif_warn(efct, probe, efct->net_dev, "%s: pd_match_count = 0\n", __func__);
	for (pd_match_pri = 0; pd_match_pri < pd_match_count; pd_match_pri++) {
		u32 mcdi_flags =
			MCDI_ARRAY_DWORD(outbuf, GET_PARSER_DISP_INFO_OUT_SUPPORTED_MATCHES,
					 pd_match_pri);
		table->rx_match_fields[table->rx_match_count++] = mcdi_flags;
	}

	return 0;
}

int efct_mcdi_filter_table_probe(struct efct_nic *efct)
{
	struct efct_mcdi_filter_table *table;
	int rc = 0, i;

	if (efct->filter_table) /* already probed */
		return rc;

	table = kzalloc(sizeof(*table), GFP_KERNEL);
	if (!table)
		return -ENOMEM;

	efct->filter_table = table;

	init_rwsem(&table->lock);
	table->entry = vzalloc(efct->efct_dev->params.num_filter *
			       sizeof(*table->entry));
	if (!table->entry) {
		rc = -ENOMEM;
		goto freetable;
	}

	for (i = 0; i < efct->efct_dev->params.num_filter; i++) {
		table->entry[i].handle = EFCT_HANDLE_INVALID;
		table->entry[i].ref_cnt = 0;
	}
	rc = efct_mcdi_filter_parser_info(efct, table);
	if (rc)
		goto freeentry;

	return rc;

freeentry:
	vfree(table->entry);
freetable:
	kfree(table);
	return rc;
}

void efct_mcdi_filter_table_remove(struct efct_nic *efct)
{
	struct efct_mcdi_filter_table *table = efct->filter_table;
	int i;

	if (!table)
		return;
	for (i = 0; i < efct->efct_dev->params.num_filter; i++) {
		if (table->entry[i].spec)
			kfree((struct efct_filter_spec *)table->entry[i].spec);
	}
	vfree(table->entry);
	table->entry = NULL;
	efct->filter_table = NULL;
	kfree(table);
}

#define EFCT_MCDI_NVRAM_LEN_MAX 128

int efct_mcdi_nvram_update_start(struct efct_nic *efct, u32 type)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_NVRAM_UPDATE_START_V2_IN_LEN);
	int rc;

	MCDI_SET_DWORD(inbuf, NVRAM_UPDATE_START_IN_TYPE, type);
	MCDI_POPULATE_DWORD_1(inbuf, NVRAM_UPDATE_START_V2_IN_FLAGS,
			      NVRAM_UPDATE_START_V2_IN_FLAG_REPORT_VERIFY_RESULT, 1);

	BUILD_BUG_ON(MC_CMD_NVRAM_UPDATE_START_OUT_LEN != 0);

	rc = efct_mcdi_rpc(efct, MC_CMD_NVRAM_UPDATE_START, inbuf, sizeof(inbuf),
			   NULL, 0, NULL);

	return rc;
}

int efct_mcdi_nvram_write(struct efct_nic *efct, u32 type,
			  loff_t offset, const u8 *buffer, size_t length)
{
	union efct_dword *inbuf;
	size_t inlen;
	int rc;

	inlen = ALIGN(MC_CMD_NVRAM_WRITE_IN_LEN(length), 4);
	inbuf = kzalloc(inlen, GFP_KERNEL);
	if (!inbuf)
		return -ENOMEM;

	MCDI_SET_DWORD(inbuf, NVRAM_WRITE_IN_TYPE, type);
	MCDI_SET_DWORD(inbuf, NVRAM_WRITE_IN_OFFSET, offset);
	MCDI_SET_DWORD(inbuf, NVRAM_WRITE_IN_LENGTH, length);
	memcpy(MCDI_PTR(inbuf, NVRAM_WRITE_IN_WRITE_BUFFER), buffer, length);

	BUILD_BUG_ON(MC_CMD_NVRAM_WRITE_OUT_LEN != 0);

	rc = efct_mcdi_rpc(efct, MC_CMD_NVRAM_WRITE, inbuf, inlen, NULL, 0, NULL);
	kfree(inbuf);

	return rc;
}

int efct_mcdi_nvram_erase(struct efct_nic *efct, u32 type,
			  loff_t offset, size_t length)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_NVRAM_ERASE_IN_LEN);
	int rc;

	MCDI_SET_DWORD(inbuf, NVRAM_ERASE_IN_TYPE, type);
	MCDI_SET_DWORD(inbuf, NVRAM_ERASE_IN_OFFSET, offset);
	MCDI_SET_DWORD(inbuf, NVRAM_ERASE_IN_LENGTH, length);

	BUILD_BUG_ON(MC_CMD_NVRAM_ERASE_OUT_LEN != 0);

	rc = efct_mcdi_rpc(efct, MC_CMD_NVRAM_ERASE, inbuf, sizeof(inbuf),
			   NULL, 0, NULL);
	return rc;
}

int efct_mcdi_nvram_update_finish(struct efct_nic *efct, u32 type,
				  enum efct_update_finish_mode mode)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_NVRAM_UPDATE_FINISH_V2_OUT_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_LEN);
	size_t outlen;
	int rc, rc2;
	u32 reboot;

	/* Reboot PHY's into the new firmware. mcfw reboot is handled
	 * explicitly via ethtool.
	 */
	reboot = (type == MC_CMD_NVRAM_TYPE_PHY_PORT0 ||
		  type == MC_CMD_NVRAM_TYPE_PHY_PORT1 ||
		  type == MC_CMD_NVRAM_TYPE_DISABLED_CALLISTO);
	MCDI_SET_DWORD(inbuf, NVRAM_UPDATE_FINISH_IN_TYPE, type);
	MCDI_SET_DWORD(inbuf, NVRAM_UPDATE_FINISH_IN_REBOOT, reboot);

	/* Old firmware doesn't support background update finish and abort
	 * operations. Fallback to waiting if the requested mode is not
	 * supported.
	 */
	if (!efct_has_cap(efct, NVRAM_UPDATE_POLL_VERIFY_RESULT) ||
	    (!efct_has_cap(efct, NVRAM_UPDATE_ABORT_SUPPORTED) &&
	     mode == EFCT_UPDATE_FINISH_ABORT))
		mode = EFCT_UPDATE_FINISH_WAIT;

	MCDI_POPULATE_DWORD_4(inbuf, NVRAM_UPDATE_FINISH_V2_IN_FLAGS,
			      NVRAM_UPDATE_FINISH_V2_IN_FLAG_REPORT_VERIFY_RESULT,
			      (mode != EFCT_UPDATE_FINISH_ABORT),
			      NVRAM_UPDATE_FINISH_V2_IN_FLAG_RUN_IN_BACKGROUND,
			      (mode == EFCT_UPDATE_FINISH_BACKGROUND),
			      NVRAM_UPDATE_FINISH_V2_IN_FLAG_POLL_VERIFY_RESULT,
			      (mode == EFCT_UPDATE_FINISH_POLL),
			      NVRAM_UPDATE_FINISH_V2_IN_FLAG_ABORT,
			      (mode == EFCT_UPDATE_FINISH_ABORT));

	rc = efct_mcdi_rpc(efct, MC_CMD_NVRAM_UPDATE_FINISH, inbuf, sizeof(inbuf),
			   outbuf, sizeof(outbuf), &outlen);
	if (!rc && outlen >= MC_CMD_NVRAM_UPDATE_FINISH_V2_OUT_LEN) {
		rc2 = MCDI_DWORD(outbuf, NVRAM_UPDATE_FINISH_V2_OUT_RESULT_CODE);
		if (rc2 != MC_CMD_NVRAM_VERIFY_RC_SUCCESS &&
		    rc2 != MC_CMD_NVRAM_VERIFY_RC_PENDING)
			netif_err(efct, probe, efct->net_dev,
				  "NVRAM update failed verification with code 0x%x\n",
				  rc2);
		switch (rc2) {
		case MC_CMD_NVRAM_VERIFY_RC_SUCCESS:
			break;
		case MC_CMD_NVRAM_VERIFY_RC_PENDING:
			rc = -EAGAIN;
			break;
		case MC_CMD_NVRAM_VERIFY_RC_CMS_CHECK_FAILED:
		case MC_CMD_NVRAM_VERIFY_RC_MESSAGE_DIGEST_CHECK_FAILED:
		case MC_CMD_NVRAM_VERIFY_RC_SIGNATURE_CHECK_FAILED:
		case MC_CMD_NVRAM_VERIFY_RC_TRUSTED_APPROVERS_CHECK_FAILED:
		case MC_CMD_NVRAM_VERIFY_RC_SIGNATURE_CHAIN_CHECK_FAILED:
			rc = -EIO;
			break;
		case MC_CMD_NVRAM_VERIFY_RC_INVALID_CMS_FORMAT:
		case MC_CMD_NVRAM_VERIFY_RC_BAD_MESSAGE_DIGEST:
			rc = -EINVAL;
			break;
		case MC_CMD_NVRAM_VERIFY_RC_NO_VALID_SIGNATURES:
		case MC_CMD_NVRAM_VERIFY_RC_NO_TRUSTED_APPROVERS:
		case MC_CMD_NVRAM_VERIFY_RC_NO_SIGNATURE_MATCH:
		case MC_CMD_NVRAM_VERIFY_RC_REJECT_TEST_SIGNED:
		case MC_CMD_NVRAM_VERIFY_RC_SECURITY_LEVEL_DOWNGRADE:
			rc = -EPERM;
			break;
		default:
			netif_err(efct, probe, efct->net_dev,
				  "Unknown response to NVRAM_UPDATE_FINISH\n");
			rc = -EIO;
		}
	}
	return rc;
}

#define	EFCT_MCDI_NVRAM_UPDATE_FINISH_INITIAL_POLL_DELAY_MS 5
#define	EFCT_MCDI_NVRAM_UPDATE_FINISH_MAX_POLL_DELAY_MS 5000
#define	EFCT_MCDI_NVRAM_UPDATE_FINISH_RETRIES 185

int efct_mcdi_nvram_update_finish_polled(struct efct_nic *efct, u32 type)
{
	u32 delay = EFCT_MCDI_NVRAM_UPDATE_FINISH_INITIAL_POLL_DELAY_MS;
	u32 retry = 0;
	int rc;

	/* NVRAM updates can take a long time (e.g. up to 1 minute for bundle
	 * images). Polling for NVRAM update completion ensures that other MCDI
	 * commands can be issued before the background NVRAM update completes.
	 *
	 * The initial call either completes the update synchronously, or
	 * returns -EAGAIN to indicate processing is continuing. In the latter
	 * case, we poll for at least 900 seconds, at increasing intervals
	 * (5ms, 50ms, 500ms, 5s).
	 */
	rc = efct_mcdi_nvram_update_finish(efct, type, EFCT_UPDATE_FINISH_BACKGROUND);
	while (rc == -EAGAIN) {
		if (retry > EFCT_MCDI_NVRAM_UPDATE_FINISH_RETRIES)
			return -ETIMEDOUT;
		retry++;

		msleep(delay);
		if (delay < EFCT_MCDI_NVRAM_UPDATE_FINISH_MAX_POLL_DELAY_MS)
			delay *= 10;

		rc = efct_mcdi_nvram_update_finish(efct, type, EFCT_UPDATE_FINISH_POLL);
	}
	return rc;
}

int efct_mcdi_nvram_metadata(struct efct_nic *efct, u32 type,
			     u32 *subtype, u16 version[4], char *desc,
			    size_t descsize)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_NVRAM_METADATA_IN_LEN);
	union efct_dword *outbuf;
	size_t outlen;
	u32 flags;
	int rc;

	outbuf = kzalloc(MC_CMD_NVRAM_METADATA_OUT_LENMAX_MCDI2, GFP_KERNEL);
	if (!outbuf)
		return -ENOMEM;

	MCDI_SET_DWORD(inbuf, NVRAM_METADATA_IN_TYPE, type);

	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_NVRAM_METADATA, inbuf,
				 sizeof(inbuf), outbuf,
				MC_CMD_NVRAM_METADATA_OUT_LENMAX_MCDI2,
				&outlen);
	if (rc)
		goto out_free;
	if (outlen < MC_CMD_NVRAM_METADATA_OUT_LENMIN) {
		rc = -EIO;
		goto out_free;
	}

	flags = MCDI_DWORD(outbuf, NVRAM_METADATA_OUT_FLAGS);

	if (desc && descsize > 0) {
		if (flags & BIT(MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_VALID_LBN)) {
			if (descsize <=
			    MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_NUM(outlen)) {
				rc = -E2BIG;
				goto out_free;
			}

			strncpy(desc,
				MCDI_PTR(outbuf, NVRAM_METADATA_OUT_DESCRIPTION),
				MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_NUM(outlen));
			desc[MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_NUM(outlen)] = '\0';
		} else {
			desc[0] = '\0';
		}
	}

	if (subtype) {
		if (flags & BIT(MC_CMD_NVRAM_METADATA_OUT_SUBTYPE_VALID_LBN))
			*subtype = MCDI_DWORD(outbuf, NVRAM_METADATA_OUT_SUBTYPE);
		else
			*subtype = 0;
	}

	if (version) {
		if (flags & BIT(MC_CMD_NVRAM_METADATA_OUT_VERSION_VALID_LBN)) {
			version[0] = MCDI_WORD(outbuf, NVRAM_METADATA_OUT_VERSION_W);
			version[1] = MCDI_WORD(outbuf, NVRAM_METADATA_OUT_VERSION_X);
			version[2] = MCDI_WORD(outbuf, NVRAM_METADATA_OUT_VERSION_Y);
			version[3] = MCDI_WORD(outbuf, NVRAM_METADATA_OUT_VERSION_Z);
		} else {
			version[0] = 0;
			version[1] = 0;
			version[2] = 0;
			version[3] = 0;
		}
	}

out_free:
	kfree(outbuf);
	return rc;
}

#define EFCT_MCDI_NVRAM_DEFAULT_WRITE_LEN 128

int efct_mcdi_nvram_info(struct efct_nic *efct, u32 type,
			 size_t *size_out, size_t *erase_size_out,
			size_t *write_size_out, bool *protected_out)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_NVRAM_INFO_V2_OUT_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_NVRAM_INFO_IN_LEN);
	size_t write_size = 0;
	size_t outlen;
	int rc;

	MCDI_SET_DWORD(inbuf, NVRAM_INFO_IN_TYPE, type);

	rc = efct_mcdi_rpc(efct, MC_CMD_NVRAM_INFO, inbuf, sizeof(inbuf),
			   outbuf, sizeof(outbuf), &outlen);
	if (rc)
		goto fail;
	if (outlen < MC_CMD_NVRAM_INFO_OUT_LEN) {
		rc = -EIO;
		goto fail;
	}

	if (outlen >= MC_CMD_NVRAM_INFO_V2_OUT_LEN)
		write_size = MCDI_DWORD(outbuf, NVRAM_INFO_V2_OUT_WRITESIZE);
	else
		write_size = EFCT_MCDI_NVRAM_DEFAULT_WRITE_LEN;

	*write_size_out = write_size;
	*size_out = MCDI_DWORD(outbuf, NVRAM_INFO_V2_OUT_SIZE);
	*erase_size_out = MCDI_DWORD(outbuf, NVRAM_INFO_V2_OUT_ERASESIZE);
	*protected_out = !!(MCDI_DWORD(outbuf, NVRAM_INFO_V2_OUT_FLAGS) &
				(1 << MC_CMD_NVRAM_INFO_V2_OUT_PROTECTED_LBN));
	return 0;

fail:
	netif_err(efct, hw, efct->net_dev, "%s: failed rc=%d\n", __func__, rc);
	return rc;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_NEED_ETHTOOL_EROM_VERSION)
void efct_mcdi_erom_ver(struct efct_nic *efct,
			char *buf,
			size_t len)
{
	u16 version[4];
	int rc;

	rc = efct_mcdi_nvram_metadata(efct, NVRAM_PARTITION_TYPE_EXPANSION_ROM,
				      NULL, version, NULL, 0);
	if (rc)
		return;
	len = min_t(size_t, EFCT_MAX_VERSION_INFO_LEN, len);
	snprintf(buf, len, "%u.%u.%u.%u", version[0],
		 version[1], version[2], version[3]);
}
#endif

void efct_mcdi_print_fwver(struct efct_nic *efct, char *buf, size_t len)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_VERSION_V5_OUT_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_GET_VERSION_EXT_IN_LEN);
	const __le16 *ver_words;
	size_t outlength;
	int rc;

	BUILD_BUG_ON(MC_CMD_GET_VERSION_IN_LEN != 0);
	rc = efct_mcdi_rpc(efct, MC_CMD_GET_VERSION, inbuf, sizeof(inbuf), outbuf,
			   sizeof(outbuf), &outlength);
	if (rc)
		goto fail;
	if (outlength < MC_CMD_GET_VERSION_V5_OUT_LEN)
		goto fail;

	ver_words = (__le16 *)MCDI_PTR(outbuf, GET_VERSION_V5_OUT_VERSION);
	snprintf(buf, len, "%u.%u.%u.%u",
		 le16_to_cpu(ver_words[0]), le16_to_cpu(ver_words[1]),
		 le16_to_cpu(ver_words[2]), le16_to_cpu(ver_words[3]));

	return;

fail:
	buf[0] = 0;
}
