// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include "efct_common.h"
#include "mcdi.h"
#include "mcdi_port_common.h"
#include "bitfield.h"

#define SFF_READ_BYTES_3	3

/*	MAC statistics
 */

int efct_mcdi_mac_stats(struct efct_nic *efct,
			enum efct_stats_action action, int clear,
		       void *outbuf, size_t buflen)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_MAC_STATS_IN_LEN);
	int dma, change;
	size_t outlen;
	u32 dma_len;
	int rc = 0;

	change = 0;
	if (action == EFCT_STATS_ENABLE)
		efct->stats_enabled = true;
	else if (action == EFCT_STATS_DISABLE)
		efct->stats_enabled = false;

	dma = (action == EFCT_STATS_PULL) || efct->stats_enabled;
	if (action == EFCT_STATS_NODMA)
		change = 0;
	else if (action != EFCT_STATS_PULL)
		change = 1;

	dma_len = dma ? efct->num_mac_stats * sizeof(u64) : 0;

	BUILD_BUG_ON(MC_CMD_MAC_STATS_OUT_DMA_LEN != 0);

	MCDI_SET_QWORD(inbuf, MAC_STATS_IN_DMA_ADDR, 0ul);
	MCDI_POPULATE_DWORD_7(inbuf, MAC_STATS_IN_CMD,
			      MAC_STATS_IN_DMA, dma,
			      MAC_STATS_IN_CLEAR, clear,
			      MAC_STATS_IN_PERIODIC_CHANGE, change,
			      MAC_STATS_IN_PERIODIC_ENABLE, efct->stats_enabled,
			      MAC_STATS_IN_PERIODIC_CLEAR, 0,
			      MAC_STATS_IN_PERIODIC_NOEVENT, 1,
			      MAC_STATS_IN_PERIOD_MS, efct->stats_period_ms);
	MCDI_SET_DWORD(inbuf, MAC_STATS_IN_DMA_LEN, dma_len);
	MCDI_SET_DWORD(inbuf, MAC_STATS_IN_PORT_ID, EVB_PORT_ID_ASSIGNED);
	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_MAC_STATS, inbuf, sizeof(inbuf),
				 outbuf, buflen, &outlen);
	if (rc)
		efct_mcdi_display_error(efct, MC_CMD_MAC_STATS, sizeof(inbuf),
					NULL, 0, rc);

	return rc;
}

void efct_mcdi_mac_fini_stats(struct efct_nic *efct)
{
	kfree(efct->mc_initial_stats);
	efct->mc_initial_stats = NULL;

	kfree(efct->mc_mac_stats);
	efct->mc_mac_stats = NULL;
}

int efct_mcdi_mac_init_stats(struct efct_nic *efct)
{
	if (!efct->num_mac_stats)
		return 0;

	efct->mc_initial_stats =
		kcalloc(efct->num_mac_stats, sizeof(u64), GFP_KERNEL);
	if (!efct->mc_initial_stats) {
		netif_warn(efct, probe, efct->net_dev,
			   "failed to allocate initial MC stats buffer\n");
		return -ENOMEM;
	}

	efct->mc_mac_stats =
		kcalloc(efct->num_mac_stats, sizeof(u64), GFP_KERNEL);
	if (!efct->mc_mac_stats) {
		netif_warn(efct, probe, efct->net_dev,
			   "failed to allocate MC stats buffer\n");
		efct_mcdi_mac_fini_stats(efct);
		return -ENOMEM;
	}

	return 0;
}

int efct_mcdi_port_get_number(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_PORT_ASSIGNMENT_OUT_LEN);
	size_t outlen = 0;
	int rc;

	rc = efct_mcdi_rpc(efct, MC_CMD_GET_PORT_ASSIGNMENT, NULL, 0,
			   outbuf, sizeof(outbuf), &outlen);
	if (rc)
		return rc;

	if (outlen < MC_CMD_GET_PORT_ASSIGNMENT_OUT_LEN)
		return -EIO;

	return MCDI_DWORD(outbuf, GET_PORT_ASSIGNMENT_OUT_PORT);
}

/*	Event processing
 */
static u32 efct_mcdi_event_link_speed[] = {
	[MCDI_EVENT_LINKCHANGE_SPEED_100M] = 100,
	[MCDI_EVENT_LINKCHANGE_SPEED_1G] = 1000,
	[MCDI_EVENT_LINKCHANGE_SPEED_10G] = 10000,
	[MCDI_EVENT_LINKCHANGE_SPEED_40G] = 40000,
	[MCDI_EVENT_LINKCHANGE_SPEED_25G] = 25000,
	[MCDI_EVENT_LINKCHANGE_SPEED_50G] = 50000,
	[MCDI_EVENT_LINKCHANGE_SPEED_100G] = 100000,
};

static u8 efct_mcdi_link_state_flags(struct efct_link_state *link_state)
{
	return (link_state->up ? (1 << MC_CMD_GET_LINK_OUT_V2_LINK_UP_LBN) : 0) |
		(link_state->fd ? (1 << MC_CMD_GET_LINK_OUT_V2_FULL_DUPLEX_LBN) : 0);
}

static u8 efct_mcdi_link_state_fcntl(struct efct_link_state *link_state)
{
	switch (link_state->fc) {
	case EFCT_FC_AUTO | EFCT_FC_TX | EFCT_FC_RX:
		return MC_CMD_FCNTL_AUTO;
	case EFCT_FC_TX | EFCT_FC_RX:
		return MC_CMD_FCNTL_BIDIR;
	case EFCT_FC_RX:
		return MC_CMD_FCNTL_RESPOND;
	case EFCT_FC_OFF:
		return MC_CMD_FCNTL_OFF;
	default:
		WARN_ON_ONCE(1);
		return MC_CMD_FCNTL_OFF;
	}
}

bool efct_mcdi_phy_poll(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_LINK_OUT_V2_LEN);
	struct efct_link_state old_state = efct->link_state;
	int rc;

	WARN_ON(!mutex_is_locked(&efct->mac_lock));

	BUILD_BUG_ON(MC_CMD_GET_LINK_IN_LEN != 0);

	rc = efct_mcdi_rpc(efct, MC_CMD_GET_LINK, NULL, 0,
			   outbuf, sizeof(outbuf), NULL);
	if (rc) {
		efct->link_state.up = false;
	} else {
		u32 lp_caps = MCDI_DWORD(outbuf, GET_LINK_OUT_V2_LP_CAP);
		u32 ld_caps = MCDI_DWORD(outbuf, GET_LINK_OUT_V2_CAP);

		efct_mcdi_phy_decode_link(efct, &efct->link_state,
					  MCDI_DWORD(outbuf, GET_LINK_OUT_V2_LINK_SPEED),
					 MCDI_DWORD(outbuf, GET_LINK_OUT_V2_FLAGS),
					 MCDI_DWORD(outbuf, GET_LINK_OUT_V2_FCNTL),
					 ld_caps, lp_caps);
	}

	return !efct_link_state_equal(&efct->link_state, &old_state);
}

void efct_mcdi_phy_decode_link(struct efct_nic *efct,
			       struct efct_link_state *link_state,
			      u32 speed, u32 flags, u32 fcntl,
			      u32 ld_caps, u32 lp_caps)
{
	switch (fcntl) {
	case MC_CMD_FCNTL_AUTO:
		WARN_ON(1); /* This is not a link mode */
		link_state->fc = EFCT_FC_AUTO | EFCT_FC_TX | EFCT_FC_RX;
		break;
	case MC_CMD_FCNTL_BIDIR:
		link_state->fc = EFCT_FC_TX | EFCT_FC_RX;
		break;
	case MC_CMD_FCNTL_RESPOND:
		link_state->fc = EFCT_FC_RX;
		break;
	case MC_CMD_FCNTL_OFF:
		link_state->fc = 0;
		break;
	default:
		WARN_ON(1);
		link_state->fc = 0;
	}

	link_state->up = !!(flags & (1 << MC_CMD_GET_LINK_OUT_V2_LINK_UP_LBN));
	link_state->fd = !!(flags & (1 << MC_CMD_GET_LINK_OUT_V2_FULL_DUPLEX_LBN));
	link_state->speed = speed;
	link_state->ld_caps = ld_caps;
	link_state->lp_caps = lp_caps;
}

void efct_mcdi_process_link_change_v2(struct efct_nic *efct, union efct_qword *ev)
{
	u32 link_up, flags, fcntl, speed, lpa;

	speed = EFCT_QWORD_FIELD(*ev, MCDI_EVENT_LINKCHANGE_V2_SPEED);
	if (speed >= ARRAY_SIZE(efct_mcdi_event_link_speed)) {
		netif_err(efct, ifup, efct->net_dev,
			  "Link speed %u received in FW response is not supported\n", speed);
		return;
	}
	speed = efct_mcdi_event_link_speed[speed];

	link_up = EFCT_QWORD_FIELD(*ev, MCDI_EVENT_LINKCHANGE_V2_FLAGS_LINK_UP);
	fcntl = EFCT_QWORD_FIELD(*ev, MCDI_EVENT_LINKCHANGE_V2_FCNTL);
	lpa = EFCT_QWORD_FIELD(*ev, MCDI_EVENT_LINKCHANGE_V2_LP_CAP);
	flags = efct_mcdi_link_state_flags(&efct->link_state) &
		~(1 << MC_CMD_GET_LINK_OUT_V2_LINK_UP_LBN);
	flags |= !!link_up << MC_CMD_GET_LINK_OUT_V2_LINK_UP_LBN;
	flags |= 1 << MC_CMD_GET_LINK_OUT_V2_FULL_DUPLEX_LBN;

	efct_mcdi_phy_decode_link(efct, &efct->link_state, speed, flags, fcntl,
				  efct->link_state.ld_caps, lpa);

	efct_link_status_changed(efct);

	// efct_mcdi_phy_check_fcntl(efct, lpa); //TODO
}

void efct_mcdi_process_module_change(struct efct_nic *efct, union efct_qword *ev)
{
	u8 flags = efct_mcdi_link_state_flags(&efct->link_state);
	u32 ld_caps;

	ld_caps = EFCT_QWORD_FIELD(*ev, MCDI_EVENT_MODULECHANGE_LD_CAP);

	flags |= 1 << MC_CMD_GET_LINK_OUT_V2_FULL_DUPLEX_LBN;

	/* efct->link_state is only modified by efct_mcdi_phy_get_link().
	 * Therefore, it is safe to modify the link state outside of the mac_lock here.
	 */
	efct_mcdi_phy_decode_link(efct, &efct->link_state, efct->link_state.speed,
				  flags,
				 efct_mcdi_link_state_fcntl(&efct->link_state),
				 ld_caps, 0);

	efct_link_status_changed(efct);
}

bool efct_mcdi_port_process_event_common(struct efct_ev_queue *evq, union efct_qword *event)
{
	int code = EFCT_QWORD_FIELD(*event, MCDI_EVENT_CODE);
	struct efct_nic *efct = evq->efct;

	switch (code) {
	case MCDI_EVENT_CODE_LINKCHANGE_V2:
		efct_mcdi_process_link_change_v2(efct, event);
		return true;
	case MCDI_EVENT_CODE_MODULECHANGE:
		efct_mcdi_process_module_change(efct, event);
		return true;
	case MCDI_EVENT_CODE_DYNAMIC_SENSORS_STATE_CHANGE:
	case MCDI_EVENT_CODE_DYNAMIC_SENSORS_CHANGE:
		efct_mcdi_dynamic_sensor_event(efct, event);
		return true;
	}

	return false;
}

u32 ethtool_linkset_to_mcdi_cap(const unsigned long *linkset)
{
	u32 result = 0;

	if (test_bit(ETHTOOL_LINK_MODE_10000baseCR_Full_BIT, linkset) ||
	    test_bit(ETHTOOL_LINK_MODE_10000baseSR_Full_BIT, linkset))
		result |= (1 << MC_CMD_PHY_CAP_10000FDX_LBN);
	if (test_bit(ETHTOOL_LINK_MODE_25000baseCR_Full_BIT, linkset) ||
	    test_bit(ETHTOOL_LINK_MODE_25000baseSR_Full_BIT, linkset))
		result |= (1 << MC_CMD_PHY_CAP_25000FDX_LBN);
	if (test_bit(ETHTOOL_LINK_MODE_Pause_BIT, linkset))
		result |= (1 << MC_CMD_PHY_CAP_PAUSE_LBN);
	if (test_bit(ETHTOOL_LINK_MODE_Asym_Pause_BIT, linkset))
		result |= (1 << MC_CMD_PHY_CAP_ASYM_LBN);
	if (test_bit(ETHTOOL_LINK_MODE_Autoneg_BIT, linkset))
		result |= (1 << MC_CMD_PHY_CAP_AN_LBN);

	return result;
}

#define FEC_BIT(x) (1 << MC_CMD_PHY_CAP_##x##_LBN)
/* Invert ethtool_fec_caps_to_mcdi. */
u32 mcdi_fec_caps_to_ethtool(u32 caps)
{
	bool rs, rs_req;

	rs = caps & FEC_BIT(RS_FEC);
	rs_req = caps & FEC_BIT(RS_FEC_REQUESTED);

	if (!rs)
		return ETHTOOL_FEC_OFF;

	return (rs_req ? ETHTOOL_FEC_RS : 0) | (rs == rs_req ? 0 : ETHTOOL_FEC_AUTO);
}

u32 ethtool_fec_caps_to_mcdi(u32 supported_cap, u32 ethtool_cap)
{
	u32 ret = 0;

	if (ethtool_cap & ETHTOOL_FEC_AUTO)
		ret |= FEC_BIT(RS_FEC) & supported_cap;
	if ((ethtool_cap & ETHTOOL_FEC_RS) && (supported_cap & FEC_BIT(RS_FEC)))
		ret |= FEC_BIT(RS_FEC) | FEC_BIT(RS_FEC_REQUESTED);

	return ret;
}

#undef FEC_BIT

u32 efct_get_mcdi_caps(struct efct_nic *efct)
{
	struct efct_mcdi_phy_data *phy_data = efct->phy_data;

	return ethtool_linkset_to_mcdi_cap(efct->link_advertising) |
		ethtool_fec_caps_to_mcdi(phy_data->supported_cap,
					 efct->fec_config);
}

void efct_link_set_wanted_fc(struct efct_nic *efct, u8 wanted_fc)
{
	efct->wanted_fc = wanted_fc;
	if (efct->link_advertising[0] & ADVERTISED_Autoneg) {
		if (wanted_fc & EFCT_FC_RX)
			efct->link_advertising[0] |= (ADVERTISED_Pause |
						     ADVERTISED_Asym_Pause);
		else
			efct->link_advertising[0] &= ~(ADVERTISED_Pause |
						      ADVERTISED_Asym_Pause);
		if (wanted_fc & EFCT_FC_TX)
			efct->link_advertising[0] ^= ADVERTISED_Asym_Pause;
	}
}

u32 efct_get_mcdi_phy_flags(struct efct_nic *efct)
{
	struct efct_mcdi_phy_data *phy_cfg = efct->phy_data;
	enum efct_phy_mode mode, supported;
	u32 flags;

	/* TODO: Advertise the capabilities supported by this PHY */
	supported = 0;
	if (phy_cfg->flags & (1 << MC_CMD_GET_PHY_CFG_OUT_TXDIS_LBN))
		supported |= PHY_MODE_TX_DISABLED;
	if (phy_cfg->flags & (1 << MC_CMD_GET_PHY_CFG_OUT_LOWPOWER_LBN))
		supported |= PHY_MODE_LOW_POWER;
	if (phy_cfg->flags & (1 << MC_CMD_GET_PHY_CFG_OUT_POWEROFF_LBN))
		supported |= PHY_MODE_OFF;

	mode = efct->phy_mode & supported;
	flags = 0;
	if (mode & PHY_MODE_TX_DISABLED)
		flags |= (1 << MC_CMD_SET_LINK_IN_V2_TXDIS_LBN);
	if (mode & PHY_MODE_LOW_POWER)
		flags |= (1 << MC_CMD_SET_LINK_IN_V2_LOWPOWER_LBN);
	if (mode & PHY_MODE_OFF)
		flags |= (1 << MC_CMD_SET_LINK_IN_V2_POWEROFF_LBN);

	if (efct->state != STATE_UNINIT && !netif_running(efct->net_dev))
		flags |= (1 << MC_CMD_SET_LINK_IN_V2_LINKDOWN_LBN);

	return flags;
}

int efct_mcdi_port_reconfigure(struct efct_nic *efct)
{
	return efct_mcdi_set_link(efct, efct_get_mcdi_caps(efct),
				 efct_get_mcdi_phy_flags(efct),
				 efct->loopback_mode, SET_LINK_SEQ_IGNORE);
}

int efct_mcdi_get_phy_cfg(struct efct_nic *efct, struct efct_mcdi_phy_data *cfg)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_PHY_CFG_OUT_LEN);
	size_t outlen;
	int rc;

	BUILD_BUG_ON(MC_CMD_GET_PHY_CFG_IN_LEN != 0);
	BUILD_BUG_ON(sizeof(cfg->name) != MC_CMD_GET_PHY_CFG_OUT_NAME_LEN);

	rc = efct_mcdi_rpc(efct, MC_CMD_GET_PHY_CFG, NULL, 0,
			   outbuf, sizeof(outbuf), &outlen);
	if (rc)
		goto fail;

	if (outlen < MC_CMD_GET_PHY_CFG_OUT_LEN) {
		rc = -EIO;
		goto fail;
	}

	cfg->flags = MCDI_DWORD(outbuf, GET_PHY_CFG_OUT_FLAGS);
	cfg->type = MCDI_DWORD(outbuf, GET_PHY_CFG_OUT_TYPE);
	cfg->supported_cap =
		MCDI_DWORD(outbuf, GET_PHY_CFG_OUT_SUPPORTED_CAP);
	cfg->channel = MCDI_DWORD(outbuf, GET_PHY_CFG_OUT_CHANNEL);
	cfg->port = MCDI_DWORD(outbuf, GET_PHY_CFG_OUT_PRT);
	cfg->stats_mask = MCDI_DWORD(outbuf, GET_PHY_CFG_OUT_STATS_MASK);
	memcpy(cfg->name, MCDI_PTR(outbuf, GET_PHY_CFG_OUT_NAME),
	       sizeof(cfg->name));
	cfg->media = MCDI_DWORD(outbuf, GET_PHY_CFG_OUT_MEDIA_TYPE);
	cfg->mmd_mask = MCDI_DWORD(outbuf, GET_PHY_CFG_OUT_MMD_MASK);
	memcpy(cfg->revision, MCDI_PTR(outbuf, GET_PHY_CFG_OUT_REVISION),
	       sizeof(cfg->revision));

	return 0;

fail:
	netif_err(efct, hw, efct->net_dev, "%s: failed rc=%d\n", __func__, rc);
	return rc;
}

static u32 efct_calc_mac_mtu(struct efct_nic *efct)
{
	return EFCT_MAX_FRAME_LEN(efct->net_dev->mtu);
}

int efct_mcdi_set_mac(struct efct_nic *efct)
{
	bool forward_fcs = !!(efct->net_dev->features & NETIF_F_RXFCS);
	MCDI_DECLARE_BUF(cmdbytes, MC_CMD_SET_MAC_V3_IN_LEN);
	u32 fcntl;

	ether_addr_copy(MCDI_PTR(cmdbytes, SET_MAC_V3_IN_ADDR),
			efct->net_dev->dev_addr);

	MCDI_POPULATE_DWORD_3(cmdbytes, SET_MAC_V3_IN_CONTROL,
			      SET_MAC_V3_IN_CFG_MTU, 1,
			      SET_MAC_V3_IN_CFG_FCNTL, 1,
			      SET_MAC_V3_IN_CFG_FCS, 1);
	MCDI_SET_DWORD(cmdbytes, SET_MAC_V3_IN_MTU, efct_calc_mac_mtu(efct));
	MCDI_SET_DWORD(cmdbytes, SET_MAC_V3_IN_DRAIN, 0);

	switch (efct->wanted_fc) {
	case EFCT_FC_RX | EFCT_FC_TX:
		fcntl = MC_CMD_FCNTL_BIDIR;
		break;
	case EFCT_FC_RX:
		fcntl = MC_CMD_FCNTL_RESPOND;
		break;
	case EFCT_FC_AUTO:
		fcntl = MC_CMD_FCNTL_AUTO;
		break;
	default:
		fcntl = MC_CMD_FCNTL_OFF;
		break;
	}

	if (efct->wanted_fc & EFCT_FC_AUTO)
		fcntl = MC_CMD_FCNTL_AUTO;
	if (efct->fc_disable)
		fcntl = MC_CMD_FCNTL_OFF;

	MCDI_SET_DWORD(cmdbytes, SET_MAC_V3_IN_FCNTL, fcntl);
	MCDI_POPULATE_DWORD_1(cmdbytes, SET_MAC_V3_IN_FLAGS,
			      SET_MAC_V3_IN_FLAG_INCLUDE_FCS, forward_fcs);

	return efct_mcdi_rpc(efct, MC_CMD_SET_MAC, cmdbytes, sizeof(cmdbytes),
			    NULL, 0, NULL);
}

int efct_mcdi_set_link(struct efct_nic *efct, u32 capabilities,
		       u32 flags, u32 loopback_mode,  u8 seq)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_SET_LINK_IN_V2_LEN);
	int rc;

	BUILD_BUG_ON(MC_CMD_SET_LINK_OUT_LEN != 0);
	MCDI_SET_DWORD(inbuf, SET_LINK_IN_V2_CAP, capabilities);
	MCDI_SET_DWORD(inbuf, SET_LINK_IN_V2_FLAGS, flags);
	MCDI_SET_DWORD(inbuf, SET_LINK_IN_V2_LOOPBACK_MODE, loopback_mode);
	MCDI_SET_DWORD(inbuf, SET_LINK_IN_V2_LOOPBACK_SPEED, 0 /* Auto */);
	MCDI_SET_DWORD(inbuf, SET_LINK_IN_V2_MODULE_SEQ, seq);
	rc = efct_mcdi_rpc(efct, MC_CMD_SET_LINK, inbuf, sizeof(inbuf),
			   NULL, 0, NULL);
	return rc;
}

int efct_mcdi_loopback_modes(struct efct_nic *efct, u64 *loopback_modes)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_LOOPBACK_MODES_OUT_LEN);
	size_t outlen;
	int rc;

	rc = efct_mcdi_rpc(efct, MC_CMD_GET_LOOPBACK_MODES, NULL, 0,
			   outbuf, sizeof(outbuf), &outlen);
	if (rc)
		goto fail;

	if (outlen < (MC_CMD_GET_LOOPBACK_MODES_OUT_SUGGESTED_OFST +
		MC_CMD_GET_LOOPBACK_MODES_OUT_SUGGESTED_LEN)) {
		rc = -EIO;
		goto fail;
	}

	*loopback_modes = MCDI_QWORD(outbuf, GET_LOOPBACK_MODES_OUT_SUGGESTED);
	return 0;

fail:
	netif_err(efct, hw, efct->net_dev, "%s: failed rc=%d\n", __func__, rc);
	return rc;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_ETHTOOL_LINKSETTINGS)
#define MAP_CAP(mcdi_cap, ethtool_cap) do { \
	if (cap & (1 << MC_CMD_PHY_CAP_##mcdi_cap##_LBN)) \
		SET_CAP(ethtool_cap); \
	cap &= ~(1 << MC_CMD_PHY_CAP_##mcdi_cap##_LBN); \
} while (0)
#define SET_CAP(name)  __set_bit(ETHTOOL_LINK_MODE_ ## name ## _BIT, \
				 linkset)
#define CHECK_CAP(mcdi_cap)					\
({								\
	u32 bit = cap & (1 << MC_CMD_PHY_CAP_##mcdi_cap##_LBN);	\
	cap &= ~(1 << MC_CMD_PHY_CAP_##mcdi_cap##_LBN);		\
	bit;							\
})

static void mcdi_fec_to_ethtool_linkset(u32 cap, unsigned long *linkset)
{
	if (cap & (1 << MC_CMD_PHY_CAP_RS_FEC_LBN))
		SET_CAP(FEC_RS);
	if (!(cap & ((1 << MC_CMD_PHY_CAP_RS_FEC_REQUESTED_LBN))))
		SET_CAP(FEC_NONE);
}

void mcdi_to_ethtool_linkset(struct efct_nic *efct, u32 media, u32 cap,
			     unsigned long *linkset)
{
	bitmap_zero(linkset, __ETHTOOL_LINK_MODE_MASK_NBITS);

	switch (media) {
	case MC_CMD_MEDIA_SFP_PLUS:
	case MC_CMD_MEDIA_QSFP_PLUS:
	case MC_CMD_MEDIA_DSFP:
		SET_CAP(FIBRE);
		if (CHECK_CAP(10000FDX)) {
			SET_CAP(10000baseCR_Full);
			SET_CAP(10000baseSR_Full);
		}
		if (CHECK_CAP(25000FDX)) {
			SET_CAP(25000baseCR_Full);
			SET_CAP(25000baseSR_Full);
		}
		break;
	}

	MAP_CAP(PAUSE, Pause);
	MAP_CAP(ASYM, Asym_Pause);
	MAP_CAP(AN, Autoneg);
#ifdef EFCT_NOT_UPSTREAM
	if (cap & MCDI_PORT_SPEED_CAPS) {
		static bool warned;

		if (!warned) {
			netif_notice(efct, probe, efct->net_dev,
				     "This NIC has link speeds that are not supported by your kernel: %#x\n",
				     cap);
			warned = true;
		}
	}
#endif
}

#undef SET_CAP
#undef MAP_CAP
#undef CHECK_CAP

#endif

static u8 mcdi_to_ethtool_media(struct efct_nic *efct)
{
	struct efct_mcdi_phy_data *phy_data = efct->phy_data;

	switch (phy_data->media) {
	case MC_CMD_MEDIA_XAUI:
	case MC_CMD_MEDIA_CX4:
	case MC_CMD_MEDIA_KX4:
		return PORT_OTHER;

	case MC_CMD_MEDIA_XFP:
	case MC_CMD_MEDIA_SFP_PLUS:
	case MC_CMD_MEDIA_QSFP_PLUS:
	case MC_CMD_MEDIA_DSFP:
		return PORT_FIBRE;

	case MC_CMD_MEDIA_BASE_T:
		return PORT_TP;

	default:
		return PORT_OTHER;
	}
}

void efct_link_set_advertising(struct efct_nic *efct,
			       const unsigned long *advertising)
{
	memcpy(efct->link_advertising, advertising,
	       sizeof(__ETHTOOL_DECLARE_LINK_MODE_MASK()));
	if (advertising[0] & ADVERTISED_Autoneg) {
		if (advertising[0] & ADVERTISED_Pause)
			efct->wanted_fc |= (EFCT_FC_TX | EFCT_FC_RX);
		else
			efct->wanted_fc &= ~(EFCT_FC_TX | EFCT_FC_RX);
		if (advertising[0] & ADVERTISED_Asym_Pause)
			efct->wanted_fc ^= EFCT_FC_TX;
	}
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_ETHTOOL_LINKSETTINGS)
void efct_mcdi_phy_get_ksettings(struct efct_nic *efct,
				 struct ethtool_link_ksettings *out)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_LINK_OUT_V2_LEN);
	struct efct_mcdi_phy_data *phy_cfg = efct->phy_data;
	struct ethtool_link_settings *base = &out->base;
	size_t outlen;
	int rc;

	if (netif_carrier_ok(efct->net_dev)) {
		base->speed = efct->link_state.speed;
		base->duplex = efct->link_state.fd ? DUPLEX_FULL : DUPLEX_HALF;
	} else {
		base->speed = 0;
		base->duplex = DUPLEX_UNKNOWN;
	}
	base->port = mcdi_to_ethtool_media(efct);
	base->phy_address = phy_cfg->port;
	base->autoneg = (efct->link_advertising[0] & ADVERTISED_Autoneg) ?
			AUTONEG_ENABLE :
			AUTONEG_DISABLE;
	mcdi_to_ethtool_linkset(efct, phy_cfg->media, phy_cfg->supported_cap,
				out->link_modes.supported);
	memcpy(out->link_modes.advertising, efct->link_advertising,
	       sizeof(__ETHTOOL_DECLARE_LINK_MODE_MASK()));

	rc = efct_mcdi_rpc(efct, MC_CMD_GET_LINK, NULL, 0,
			   outbuf, sizeof(outbuf), &outlen);
	if (rc)
		return;
	if (outlen < MC_CMD_GET_LINK_OUT_V2_LEN)
		return;
	mcdi_to_ethtool_linkset(efct, phy_cfg->media,
				MCDI_DWORD(outbuf, GET_LINK_OUT_V2_LP_CAP),
				out->link_modes.lp_advertising);
	mcdi_fec_to_ethtool_linkset(MCDI_DWORD(outbuf, GET_LINK_OUT_CAP),
				    out->link_modes.advertising);
	mcdi_fec_to_ethtool_linkset(phy_cfg->supported_cap,
				    out->link_modes.supported);
	mcdi_fec_to_ethtool_linkset(MCDI_DWORD(outbuf, GET_LINK_OUT_LP_CAP),
				    out->link_modes.lp_advertising);
}
#endif

static int ethtool_fec_supported(u32 supported_cap, u32 ethtool_cap)

{
	if (ethtool_cap & ETHTOOL_FEC_OFF)
		return 0;

	if (ethtool_cap &&
	    !ethtool_fec_caps_to_mcdi(supported_cap, ethtool_cap))
		return -EINVAL;
	return 0;
}

int efct_mcdi_phy_get_fecparam(struct efct_nic *efct, struct ethtool_fecparam *fec)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_LINK_OUT_V2_LEN);
	u32 caps, active; /* MCDI format */
	size_t outlen;
	int rc;

	BUILD_BUG_ON(MC_CMD_GET_LINK_IN_LEN != 0);
	rc = efct_mcdi_rpc(efct, MC_CMD_GET_LINK, NULL, 0,
			   outbuf, sizeof(outbuf), &outlen);
	if (rc)
		return rc;
	if (outlen < MC_CMD_GET_LINK_OUT_V2_LEN)
		return -EOPNOTSUPP;

	caps = MCDI_DWORD(outbuf, GET_LINK_OUT_V2_CAP);
	fec->fec = mcdi_fec_caps_to_ethtool(caps);

	active = MCDI_DWORD(outbuf, GET_LINK_OUT_V2_FEC_TYPE);
	switch (active) {
	case MC_CMD_FEC_NONE:
		fec->active_fec = ETHTOOL_FEC_OFF;
		break;
	case MC_CMD_FEC_RS:
		fec->active_fec = ETHTOOL_FEC_RS;
		break;
	default:
		netif_warn(efct, hw, efct->net_dev,
			   "Firmware reports unrecognised FEC_TYPE %u\n",
			   active);
		/* We don't know what firmware has picked.  AUTO is as good a
		 * "can't happen" value as any other.
		 */
		fec->active_fec = ETHTOOL_FEC_AUTO;
		break;
	}

	return 0;
}

int efct_mcdi_phy_set_fecparam(struct efct_nic *efct,
			      const struct ethtool_fecparam *fec)
{
	u32 caps = ethtool_linkset_to_mcdi_cap(efct->link_advertising);
	struct efct_mcdi_phy_data *phy_data = efct->phy_data;
	int rc;

	rc = ethtool_fec_supported(phy_data->supported_cap, fec->fec);
	if (rc)
		return rc;

	caps |= ethtool_fec_caps_to_mcdi(phy_data->supported_cap, fec->fec);
	rc = efct_mcdi_set_link(efct, caps, efct_get_mcdi_phy_flags(efct),
			       efct->loopback_mode, SET_LINK_SEQ_IGNORE);
	if (rc)
		return rc;
	efct->fec_config = fec->fec;
	return 0;
}

int efct_mcdi_set_id_led(struct efct_nic *efct, enum efct_led_mode mode)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_SET_ID_LED_IN_LEN);
	int rc;

	BUILD_BUG_ON(EFCT_LED_OFF != MC_CMD_LED_OFF);
	BUILD_BUG_ON(EFCT_LED_ON != MC_CMD_LED_ON);
	BUILD_BUG_ON(EFCT_LED_DEFAULT != MC_CMD_LED_DEFAULT);

	BUILD_BUG_ON(MC_CMD_SET_ID_LED_OUT_LEN != 0);

	MCDI_SET_DWORD(inbuf, SET_ID_LED_IN_STATE, mode);

	rc = efct_mcdi_rpc(efct, MC_CMD_SET_ID_LED, inbuf, sizeof(inbuf),
			   NULL, 0, NULL);

	return rc;
}

static int efct_mcdi_phy_get_module_eeprom_page(struct efct_nic *efct,
						u16 bank,
						u16 page,
						u16 offset,
						u8 *data,
						u16 space);

static int efct_mcdi_phy_get_module_eeprom_byte(struct efct_nic *efct,
						int page,
						u8 byte)
{
	u8 data;
	int rc;

	rc = efct_mcdi_phy_get_module_eeprom_page(efct, 0, page, byte, &data, 1);
	if (rc == 1)
		return data;

	return rc;
}

static int efct_mcdi_phy_sff_8472_level(struct efct_nic *efct)
{
	/* Page zero of the EEPROM includes the DMT level at byte 94. */
	return efct_mcdi_phy_get_module_eeprom_byte(efct, 0, SFF_DMT_LEVEL_OFFSET);
}

static u32 efct_mcdi_phy_module_type(struct efct_nic *efct)
{
	switch (efct_mcdi_phy_get_module_eeprom_byte(efct, 0, 0)) {
	case SFF8024_ID_SFP:
		return MC_CMD_MEDIA_SFP_PLUS;
	case SFF8024_ID_DSFP:
		return MC_CMD_MEDIA_DSFP;
	default:
		return 0;
	}
}

static int efct_sfp_has_diag_info(struct efct_nic *efct, u8 *diag_type_92,
				  u8 *sff_8472_level_94)
{
	u8 d92_94[SFF_READ_BYTES_3];
	int rc;

	/* Read 92, 93, 94 byte of page 0.
	 * byte 92 is Diagnostic Monitoring Type
	 * byte 94 is revision of SFF-8472 the transceiver complies
	 */
	rc = efct_mcdi_phy_get_module_eeprom_page(efct, 0, 0, SFF_DIAG_TYPE_OFFSET, d92_94,
						  SFF_READ_BYTES_3);
	if (rc < 0)
		return -EOPNOTSUPP;
	*diag_type_92 = d92_94[0];
	*sff_8472_level_94 = d92_94[2];

	return 0;
}

static int efct_mcdi_phy_get_module_eeprom_page(struct efct_nic *efct,
						u16 bank,
						u16 page,
						u16 offset,
						u8 *data,
						u16 space)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_PHY_MEDIA_INFO_OUT_LENMAX);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_GET_PHY_MEDIA_INFO_IN_LEN);
	u8 diag_type_92, sff_8472_level_94;
	u32 payload_len;
	size_t outlen;
	u32 to_copy;
	int rc;

	if (offset > ETH_MODULE_EEPROM_PAGE_LEN)
		return -EINVAL;
	to_copy = min_t(u16, space, ETH_MODULE_EEPROM_PAGE_LEN - offset);
	MCDI_POPULATE_DWORD_2(inbuf, GET_PHY_MEDIA_INFO_IN_PAGE,
			      GET_PHY_MEDIA_INFO_IN_DSFP_BANK, bank,
			      GET_PHY_MEDIA_INFO_IN_DSFP_PAGE, page);
	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_GET_PHY_MEDIA_INFO,
				 inbuf, sizeof(inbuf),
				 outbuf, sizeof(outbuf),
			&outlen);
	if (rc)
		return rc;

	if (outlen < (MC_CMD_GET_PHY_MEDIA_INFO_OUT_DATA_OFST +
			ETH_MODULE_EEPROM_PAGE_LEN)) {
		/* There are SFP+ modules that claim to be SFF-8472 compliant
		 * and do not provide diagnostic information, but they don't
		 * return all bits 0 as the spec says they should...
		 */
		if (page >= 2 && page < 4 &&
		    efct_mcdi_phy_module_type(efct) == MC_CMD_MEDIA_SFP_PLUS) {
			rc = efct_sfp_has_diag_info(efct, &diag_type_92, &sff_8472_level_94);
			if (rc)
				return -EIO;
			if (sff_8472_level_94 > 0 && (diag_type_92 & SFF_DIAG_IMPLEMENTED) == 0) {
				memset(data, 0, to_copy);
				return to_copy;
			}
		}

		return -EIO;
	}
	payload_len = MCDI_DWORD(outbuf, GET_PHY_MEDIA_INFO_OUT_DATALEN);
	if (payload_len != ETH_MODULE_EEPROM_PAGE_LEN)
		return -EIO;

	memcpy(data, MCDI_PTR(outbuf, GET_PHY_MEDIA_INFO_OUT_DATA) + offset, to_copy);

	return to_copy;
}

int efct_mcdi_phy_get_module_eeprom_locked(struct efct_nic *efct,
					   struct ethtool_eeprom *ee,
					   u8 *data)
{
	ssize_t space_remaining = ee->len;
	bool ignore_missing;
	int num_pages;
	u32 page_off;
	int  page;
	int bank;
	int rc;

	switch (efct_mcdi_phy_module_type(efct)) {
	case MC_CMD_MEDIA_SFP_PLUS:
		num_pages = efct_mcdi_phy_sff_8472_level(efct) > 0 ?
				SFF_8472_NUM_PAGES : SFF_8079_NUM_PAGES;
		page = 0;
		bank = 0;
		ignore_missing = false;
		break;
	case MC_CMD_MEDIA_DSFP:
		num_pages = SFF_8436_NUM_PAGES;
		/* We obtain the lower page by asking for bank:page -1:-1. */
		page = -1;
		bank = -1;
		ignore_missing = true; /* Ignore missing pages after page 0. */
		break;
	default:
		return -EOPNOTSUPP;
	}

	page_off = ee->offset % ETH_MODULE_EEPROM_PAGE_LEN;
	page += ee->offset / ETH_MODULE_EEPROM_PAGE_LEN;
	while (space_remaining && (page < num_pages)) {
		if (page != -1)
			bank = 0;
		rc = efct_mcdi_phy_get_module_eeprom_page(efct, bank, page,
							  page_off, data, space_remaining);

		if (rc > 0) {
			space_remaining -= rc;
			data += rc;
			page_off = 0;
			page++;
		} else if (rc == 0) {
			space_remaining = 0;
		} else if (ignore_missing && (page > 0)) {
			int intended_size = ETH_MODULE_EEPROM_PAGE_LEN - page_off;

			space_remaining -= intended_size;
			if (space_remaining < 0) {
				space_remaining = 0;
			} else {
				memset(data, 0, intended_size);
				data += intended_size;
				page_off = 0;
				page++;
			}
		} else {
			return rc;
		}
	}

	return 0;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_EEPROM_BY_PAGE)
int efct_mcdi_get_eeprom_page_locked(struct efct_nic *efct,
				     const struct ethtool_module_eeprom *page_data,
				     struct netlink_ext_ack *extack)
{
	u8 diag_type_92, sff_8472_level_94;
	int rc = -EIO;
	int pagemc;
	int bankmc;
	int offset;

	if (page_data->bank) {
		NL_SET_ERR_MSG_MOD(extack, "EEPROM bank read not supported");
		return rc;
	}
	/*Upper page memory starts from 128th byte, FW expects 0*/
	offset = page_data->offset % ETH_MODULE_EEPROM_PAGE_LEN;
	pagemc = page_data->page;
	bankmc = page_data->bank;
	switch (efct_mcdi_phy_module_type(efct)) {
	case MC_CMD_MEDIA_SFP_PLUS:
		/* SFP has 2 I2C address 0x50 and 0x51, Mapping of data to MCDI page is given below
		 * I2C Addr 0x50, offset 0   ==> page 0
		 * I2C Addr 0x50, offset 128 ==> page 1
		 * I2C Addr 0x51, offset 0   ==> page 2
		 * I2C Addr 0x51, offset 128 ==> page 3
		 */
		pagemc = page_data->offset / ETH_MODULE_EEPROM_PAGE_LEN;
		if (page_data->i2c_address == SFF8079_I2C_ADDRESS_HIGH) {
			rc = efct_sfp_has_diag_info(efct, &diag_type_92, &sff_8472_level_94);
			if (rc) {
				NL_SET_ERR_MSG_MOD(extack, "Page read failed");
				return rc;
			}
			if (sff_8472_level_94 == 0 || (diag_type_92 & SFF_DIAG_ADDR_CHANGE))
				NL_SET_ERR_MSG_MOD(extack, "Diagnostic info memory not available");
			else
				pagemc += 2;
		}
		break;
	case MC_CMD_MEDIA_DSFP:
		/* We obtain the lower page by asking for FFFF:FFFF. */
		if (!page_data->page && page_data->offset < ETH_MODULE_EEPROM_PAGE_LEN) {
			pagemc = -1;
			bankmc = -1;
		}
		break;
	default:
		return -EOPNOTSUPP;
	}

	rc = efct_mcdi_phy_get_module_eeprom_page(efct, bankmc,
						  pagemc,
						  offset,
						  page_data->data,
						  page_data->length);
	if (rc < 0)
		NL_SET_ERR_MSG_MOD(extack, "Failed to access module's EEPROM.");
	return rc;
}
#endif

int efct_mcdi_phy_get_module_info_locked(struct efct_nic *efct,
					 struct ethtool_modinfo *modinfo)
{
	u8 diag_type_92, sff_8472_level_94;
	int rc;

	switch (efct_mcdi_phy_module_type(efct)) {
	case MC_CMD_MEDIA_SFP_PLUS:
		rc = efct_sfp_has_diag_info(efct, &diag_type_92, &sff_8472_level_94);
		if (rc < 0)
			return rc;
		if (sff_8472_level_94 == 0 || (diag_type_92 & SFF_DIAG_ADDR_CHANGE)) {
			modinfo->type = ETH_MODULE_SFF_8079;
			modinfo->eeprom_len = ETH_MODULE_SFF_8079_LEN;
		} else {
			modinfo->type = ETH_MODULE_SFF_8472;
			modinfo->eeprom_len = ETH_MODULE_SFF_8472_LEN;
		}
		break;
	case MC_CMD_MEDIA_DSFP:
		modinfo->type = ETH_MODULE_SFF_8436;
		modinfo->eeprom_len = ETH_MODULE_SFF_8436_LEN;
		break;

	default:
		return -EOPNOTSUPP;
	}

	return 0;
}

int efct_mcdi_phy_test_alive(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_PHY_STATE_OUT_LEN);
	size_t outlen;
	int rc;

	BUILD_BUG_ON(MC_CMD_GET_PHY_STATE_IN_LEN != 0);

	rc = efct_mcdi_rpc(efct, MC_CMD_GET_PHY_STATE, NULL, 0,
			   outbuf, sizeof(outbuf), &outlen);
	if (rc)
		return rc;

	if (outlen < MC_CMD_GET_PHY_STATE_OUT_LEN)
		return -EIO;
	if (MCDI_DWORD(outbuf, GET_PHY_STATE_OUT_STATE) != MC_CMD_PHY_STATE_OK)
		return -EINVAL;

	return 0;
}

