// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
* Driver for Xilinx network controllers and boards
* Copyright 2021 Xilinx Inc.
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License version 2 as published
* by the Free Software Foundation, incorporated herein by reference.
*/
/*
 * xlnx_efct.h - efct interface using the auxiliary bus
 *
 * TODO
 * Do we want some versioning like for driverlink?
 */

#ifndef _XLNX_EFCT_H
#define _XLNX_EFCT_H

#include <linux/uio.h>
#include <linux/version.h>

/* This value describes the ABI guarantees of this file, maintained as a
 * semver. See also xlnx_efct_device::version for the runtime equivalent. */
#define XLNX_EFCT_AUX_VERSION  KERNEL_VERSION(8, 0, 0)

/* This is the device name on the auxiliary bus */
#define XLNX_EFCT_DEVNAME	"efct"

/* EFCT client API */
struct xlnx_efct_client;

/**
 * Descriptor of a hugepage allocated by the aux client. These things must
 * have from from hugetlbfs, i.e.
 * assert(PageHuge(page) && file->f_op == &hugetlbfs_file_operations). The
 * net driver doesn't need file, but clients do.
 */
struct xlnx_efct_hugepage {
	struct file* file;
	struct page* page;
};

/**
 * Events the client has to handle
 * @XLNX_EVENT_RESET_DOWN: Generated before device is reset
 *	Generated immediately before a hardware reset. The client must stop all
 *	hardware processing before returning from this method. Callbacks will
 *	be inactive when this method is called.
 *	Context: process, rtnl_lock held
 * @XLNX_EVENT_RESET_UP: Generated after device is reset
 *	Generated after a hardware reset. If @value is true, the client should
 *	state and resume normal operations. If @value is false, the client
 *	should abandon use of the hardware resources. remove() will still be
 *	called.
 *	Context: process, rtnl_lock held
 * @XLNX_EVENT_WAKEUP: Generated after any batch of new packets has been
 *  received, as an opportunity for clients to wake up any sleeping tasks.
 *  @value is (sbseq << 32) | pktix, describing the current 'tail' pointer of
 *  the receive queue: sbseq is the current global superbuf sequence number
 *  (see xlnx_efct_drvops::buffer_start) and pktix is the 0-based index within
 *  that superbuf of the packet. This pointer indicates the place where the
 *  next metadata will arrive, i.e. one more than the last metadata processed.
 *	Context: NAPI.
 * @XLNX_EVENT_RX_FLUSH: Generated when an RXQ release has completed.
 *	Context:
 * @XLNX_EVENT_TIME_SYNC: Generated on a time synchronisation control event.
 *	@value is the 64-bit event extracted from the EFCT event queue, including
 *	a low-resolution timestamp and synchronisation flags.
 *	Context: NAPI
 */
enum xlnx_event_type {
	XLNX_EVENT_RESET_DOWN, /**< Hardware is down for reset */
	XLNX_EVENT_RESET_UP, /**< Hardware is back after reset */
	XLNX_EVENT_WAKEUP, /**< RX data is available */
	XLNX_EVENT_RX_FLUSH, /**< RXQ release has completed */
	XLNX_EVENT_TIME_SYNC, /**< Time synchronisation */
};

struct xlnx_efct_event {
	enum xlnx_event_type	type;
	int			rxq;
	u64			value;
};

/**
 * EFCT driver operations
 *
 * @name: Name of the driver. It is used in log messages.
 * @poll: Called at the top of the NAPI poll handler always, i.e. when any
 *	event is received from the NIC. Note the difference between this and
 *	XLNX_EVENT_WAKEUP: this callback happens before net driver packet
 *	processing (so that it can potentially free superbufs which will shortly
 *	be reused by the net driver), WAKEUP happens after (so it gets the fully
 *	up-to-date 'current position' of the receive queue).
 *	Context: NAPI.
 * @handle_event: Called when an event on a single-function port may
 *	need to be handled by a driver. May be %NULL if the client
 *	driver does not handle events. Returns the NAPI budget consumed.
 *	Context: See per event details
 * @buffer_start: Called when a new superbuf is posted to the given rxq.
 *	Net driver invokes this function from NAPI and rollover_rxq. When
 *	invoked from rollover_rxq, net driver disables the NAPI and serializes
 *	further rollover calls.
 *	'sbseq' provides a monotonic sequence number of superbufs issued to the
 *	NIC, allowing clients to refer to detect gaps in their data stream and to
 *	to use them to describe specific packet locations.
 *	Client returns 1 for accepted superbuf; 0 for rejected superbuf; or
 *	<0 indicating FAILURE.
 *	When accepted by client, refcount is implicitly added to this new superbuf,
 *	which must be dropped by release_superbuf or with buffer_end as appropriate.
 *	The special case where sbid<0 happens when the net driver has run out
 *	of preallocated superbufs belonging to clients and has to fall back to
 *	is own internal list. The aux client cannot access the memory of these
 *	internal superbufs, cannot map them to userspace apps, and must not
 *	call release_superbuf() on them. The callback is essentially only
 *	invoked to keep the client's ring status up-to-date: it should use its
 *	own scheme for marking an opaque bubble in the ring.
 *	Context: NAPI and process.
 * @buffer_end: Called when x3net finished processing given superbuf. That is
 *	it processed all the packets in the superbuf completely or up to a rollover.
 *	Each buffer_start call (with id != -1) is eventually followed with
 *	buffer_end callback in order.
 *	Client returns: 1 if it has done with the superbuf; 0 if decides to keep
 *	it; or <0 indicating FAILURE.
 *	If buffer_end() callback returns 1 a refcount is implicitly removed from
 *	this superbuf and release_superbuf() must not be called.
 *	force flag set indicates that the buffer should be freed by the client
 *	promptly with best effort.
 *	Note: If the buffer had never been accepted by buffer_start(), client is
 *	expected to return 0.
 *	Context: NAPI and process.
 * @alloc_hugepage: Called to request that one more hugepage be allocated.
 *	The net driver must call this only from within a call to bind_rxq from
 *	this client (the implementation may have to do hacks requiring
 *	specific-process context). Hugepages must be freed using the
 *	free_hugepage belonging to the same client which allocated them.
 *	Context: process.
 * @free_hugepage: Free the memory allocated by alloc_hugepage. The hugepage
 *	must not be in use by the NIC or any clients. Unlike alloc_hugepage,
 *	this may be called from any process context or NAPI.
 *	Context: NAPI and process.
 * @hugepage_list_changed: Called when the conceptual array returned by
 *	get_hugepages has changed contents.
 *  Called with NAPI disabled and internal mutex held - trivial operations only.
 *	Context: process, but restrict work to NAPI.
 * @packet_handled: Called to determine if a packet has been handled by client.
 *	The client should return true if the packet with the given metadata
 *	and payload has been handled, and so should not be processed in the
 *	kernel. 'flow_lookup' is the value of the identically-named field in
 *	the receive event; it may be used to avoid accessing any additional cache
 *	lines on a fast path.meta points to the metadata for packet, while
 *	payload points to the start of packet stride with actual payload.
 *	Context: NAPI */
struct xlnx_efct_drvops {
	const char *name;
	int (*poll)(void *driver_data, int rxq, int budget);
	int (*handle_event)(void *driver_data,
	                    const struct xlnx_efct_event *event, int budget);
	int (*buffer_start)(void *driver_data, int rxq, unsigned sbseq,
	                    int sbid, bool sentinel);
	int (*buffer_end)(void *driver_data, int rxq, int sbid,
	                  bool force);
	int (*alloc_hugepage)(void *driver_data,
	                      struct xlnx_efct_hugepage *page);
	void (*free_hugepage)(void *driver_data,
	                      struct xlnx_efct_hugepage *page);
	void (*hugepage_list_changed)(void *driver_data, int rxq);
	bool (*packet_handled)(void *driver_data, int rxq, bool flow_lookup,
	                       const void *meta, const void *payload);
};

/* EFCT server API */

/**
 * Device parameters
 *
 * Parameters are returned or set through an appropriate member of the union
 * @xlnx_efct_param_value.
 *
 * @XLNX_EFCT_CONFIG_MEM: PCI memory bar.
 *	Get only.
 *	Returned through @config_mem.
 * @XLNX_EFCT_NETDEV: The associated netdev for this interface.
 *	Get only.
 *	Returned through @net_dev.
 * @XLNX_EFCT_NETNS: The associated netns for this interface.
 *	Get only.
 *	Returned through @net_ns.
 * @XLNX_EFCT_VARIANT: The HW variant of this interface.
 *	Get only.
 *	Returned through @variant.
 * @XLNX_EFCT_REVISION: The HW revision of this interface.
 *	Get only.
 *	Returned through @value.
 * @XLNX_NIC_RESOURCES: Details of the available event queue range.
 *	The client must provide the pointer to the structure. On
 *	successful return from get_param() the values will be
 *	populated.
 *	Get only.
 *	Returned through @nic_res.
 * @XLNX_IRQ_RESOURCES: Details of available irq resources.
 *	The irq resources will correspond to the NIC resources
 *	retrievable via XLNX_NIC_RESOURCES.
 *	The client must provide the pointer to the structure. On
 *	successful return from get_param() the values will be
 *	populated.
 *	Get only.
 *	Returned through @irq_res.
 * @XLNX_EFCT_POISON_CONFIG: The poison configuration for a queue.
 *	Get or set.
 *	Set through @poison.
 * @XLNX_EFCT_EVQ_WINDOW: The location of control area for event queues.
 *	The base address is for the event queue evq_min provided through
 *	XLNX_EFCT_NIC_RESOURCES. The stride can be used to calculate the
 *	offset of each subsequent event queue from this base.
 *	Get only.
 *	Returned through @evq_window.
 */
enum xlnx_efct_param {
	XLNX_EFCT_CONFIG_MEM,	/**< PCI memory BAR */
	XLNX_EFCT_NETDEV,
	XLNX_EFCT_NETNS,
	XLNX_EFCT_VARIANT,
	XLNX_EFCT_REVISION,
	XLNX_EFCT_NIC_RESOURCES,
	XLNX_EFCT_IRQ_RESOURCES,
	XLNX_EFCT_POISON_CONFIG,
	XLNX_EFCT_EVQ_WINDOW,
	XLNX_EFCT_DESIGN_PARAM,
};

struct xlnx_efct_design_params {
	/* stride between entries in receive window */
	u32 rx_stride;
	/* Length of each receive buffer */
	u32 rx_buffer_len;
	/* Maximum Rx queues available */
	u32 rx_queues;
	/* Maximum Tx apertures available */
	u32 tx_apertures;
	/* Maximum number of receive buffers can be posted */
	u32 rx_buf_fifo_size;
	/* Fixed offset to the frame */
	u32 frame_offset_fixed;
	/* Receive metadata length */
	u32 rx_metadata_len;
	/* Largest window of reordered writes to the CTPIO */
	u32 tx_max_reorder;
	/* CTPIO aperture length */
	u32 tx_aperture_size;
	/* Size of packet FIFO per CTPIO aperture */
	u32 tx_fifo_size;
	/* partial time stamp in sub nano seconds */
	u32 ts_subnano_bit;
	/* Width of sequence number in EVQ_UNSOL_CREDIT_GRANT register */
	u32 unsol_credit_seq_mask;
	/* L4 csm feilds */
	u32 l4_csum_proto;
	/* MAx length of frame data when LEN_ERR indicates runt*/
	u32 max_runt;
	/* Event queue sizes */
	u32 evq_sizes;
	/* Number of filters */
	u32 num_filter;
};

/**
 * Interrupt resource information
 *
 * @flags: currently none
 * @n_ranges: Number of entries in irq_ranges. Must be > 0.
 * @int_prime: Address of the INT_PRIME register.
 * @irq_ranges: Array of interrupts, specified as base vector + range.
 */
struct xlnx_efct_irq_resources {
	u16 flags;
	u16 n_ranges;
	void __iomem *int_prime;
	struct xlnx_efct_irq_range {
		int vector;
		int range;
	} irq_ranges[];
};

/**
 * Event queue resource information
 *
 * @evq_min: index of first available event queue
 * @evq_lim: index of last available event queue + 1
 */
struct xlnx_efct_nic_resources {
	unsigned int evq_min;
	unsigned int evq_lim;
	unsigned int txq_min;
	unsigned int txq_lim;
};

/**
 * RXQ poison configuration
 *
 * @qid: The RX queue the config relates to.
 * @value: The value to use for poisoning. This will be repeated
 *         up to @length.
 * @length: How many bytes of poisoning to use per packet.
 */
struct xlnx_efct_poison_config {
	int qid;
	u64 value;
	size_t length;
};

/**
 * Location of event queue control window.
 *
 * @base: physical address of base of the event queue window
 * @stride: size of each event queue's region within the window
 */
struct xlnx_efct_evq_window {
	resource_size_t base;
	size_t stride;
};

/** Possible values for device parameters */
union xlnx_efct_param_value {
	void __iomem *config_mem;
	struct net_device *net_dev;
	struct net *net_ns;
	struct xlnx_efct_irq_resources *irq_res;
	struct xlnx_efct_nic_resources nic_res;
	struct xlnx_efct_poison_config poison;
	struct xlnx_efct_evq_window evq_window;
	struct xlnx_efct_design_params design_params;
	char variant;
	int value;
	void *ptr;
	bool b;
};

/** Remote Procedure Call to the firmware */
struct xlnx_efct_rpc {
	unsigned int cmd;
	size_t inlen;
	const u32 *inbuf;
	size_t outlen;
	size_t *outlen_actual;
	u32 *outbuf;
};


/**
 * The parameters necessary to specify an event queue.
 *
 * The memory backing the queue must be physically contiguous. It must not be
 * freed until free_evq has returned.
 *
 * @qid: The ID of the queue in the range provided through XLNX_EFCT_NIC_RESOURCES.
 * @irq: The IRQ used with this event queue.
 * @entries: number of entries in queue
 * @q_page: The page containing the start of the queue.
 * @page_offset: The offset within the page of the queue. This must be
	suitably aligned for DMA mapping.
 * @q_size: The size in bytes of the queue.
 * @flags: mcdi flags as input to MC_CMD_INIT_*Q
 * @subscribe_time_sync: Time sync events subscription enable for this queue
 * @unsol_credit: Initial credits to be granted for unsolicited events
 */
struct xlnx_efct_evq_params {
	int qid;
	int irq;
	int entries;
	struct page *q_page;
	size_t page_offset;
	size_t q_size;
	u32 flags;
	bool subscribe_time_sync;
	u16 unsol_credit;
};


/**
 * The parameters necessary to request a TX queue.
 *
 * @evq: The event queue to associate with the allocated TXQ.
 */
struct xlnx_efct_txq_params {
	int evq;
	int label;
	int qid;
};

/**
 * The criteria to satisfy for a requested RXQ.
 *
 * @qid: The requested queue
 * @timestamp_req: True if timestamps must be enabled, false if
 *	they are not required.
 * @n_hugepages: Number of additional hugepages that this client needs to have
 *	as backing memory for superbufs after this bind request returns. If
 *	not already available, these hugepages must be allocated with
 *	alloc_hugepage, then the net driver will pin these pages, DMA-map
 *	them, and add them to the freelist. The net driver is assumed to be
 *	maintaining a per-rxq counter of the total number of hugepages desired
 *	by all clients.
 */
struct xlnx_efct_rxq_params {
	int qid;
	bool timestamp_req;
	size_t n_hugepages;
};

/**
 * Details of a filter install operation.
 *
 * @spec: Details of the requested filter.
 * @mask: CPU affinity of the calling application; used only with
 *	XLNX_EFCT_FILTER_F_ANYQUEUE_*
 * @rxq_out: The RXQ on which the filter was installed, or negative if
 *	filter does not exist on return. See @flags for details.
 * @filter_id_out: Filter id for driver which can be used later to
 *	remove the filter through filter_remove.
 * @filter_handle: Hardware Filter handle received on successful creation of
 *	filter. This filter would be same as filter id received in rx metadata
 *	if the incoming packet matches this filter.
 * @flags: See XLNX_EFCT_FILTER_F_*
 */
struct xlnx_efct_filter_params {
	const struct ethtool_rx_flow_spec *spec;
	const struct cpumask *mask;
	int rxq_out;
	int filter_id_out;
	u32 filter_handle;
	unsigned flags;
};

/**
 * Ignore the value in xlnx_efct_filter_params::spec->ring_cookie and instead
 * pick a suitable default queue based on xlnx_efct_filter_params::mask and
 * any other driver configuration. The resultant queue must be bound to one
 * of the CPUs in 'mask'. If multiple potential queues are possible then
 * the driver should prefer the one with the fewest filters currently assigned
 * to it (except the default rxq, which is the least-preferred).
 */
#define XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT  0x0001

/**
 * Starts off exactly the same as XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT, but if
 * no queue is found on the CPUs in 'mask' then select any queue anywhere.
 */
#define XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE   0x0002

/**
 * The value in xlnx_efct_filter_params::spec->ring_cookie is a preference,
 * not a requirement. If the relevant packets are already being sent to a
 * different queue or an administrative override (using ethtool) has expressed
 * a different requirement then the filter will be added there instead, and
 * rxq_out will reflect this.
 *
 * If neither XLNX_EFCT_FILTER_F_ANYQUEUE_* nor XLNX_EFCT_FILTER_F_PREF_QUEUE
 * is set then the ring_cookie is a hard requirement: filter_insert() will fail
 * if that queue cannot be used for this filter.
 */
#define XLNX_EFCT_FILTER_F_PREF_QUEUE       0x0004

/**
 * Require that an unused rxq is selected, i.e. one with no filters previously
 * on it and not the default rxq. The selected rxq must thenceforth be
 * exclusively reserved to the calling client, i.e. all attempts to add
 * filters to it (including by the net driver) or bind to it by other clients
 * must fail. The exclusive lock is removed once the number of filters
 * targeting that rxq drops back down to 0 again.
 *
 * This flag may be combined with all or none of the other flags. The intended
 * use-case for this feature is applications which want to protect their
 * traffic from snooping by others.
 */
#define XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE  0x0008


struct ethtool_rx_flow_spec;

/**
 * EFCT device operations.
 *
 * @open: Clients need to open a device before using it. This will prevent it
 *	from being removed and provides a handle for further operations.
 *	driver_data is an opaque value passed unaltered and uninspected to the
 *	xlnx_efct_drvops callbacks.
 *	Returns struct xlnx_efct_client pointer on success, or ERR_PTR() on error.
 * @close: Closing a device unlocks it.
 *
 * @get_param: Obtains the xlnx_efct_param.
 * @set_param: Sets the xlnx_efct_param.
 *	See the documentation for each paramter type in @xlnx_efct_param for
 *	details of each available parameter.
 *
 * @fw_rpc: Make a direct request to the NIC firmware.
 *
 * @init_evq: Initialises a specified event queue.
 *	Returns 0 on success, error otherwise.
 * @free_evq: Frees an initalised event queue.
 *	Returns 0 on success, error otherwise.
 * @bind_rxq: Binds to an appropriate RXQ.
 *	The provided qparams specify a set of criteria that must be satisfied.
 *	Once successfully bound to an RXQ this client will begin receiving
 *	notifications about posted buffers to enable traffic reception.
 *	rxq bindings are reference-counted: a matching call to free_rxq is
 *	needed for each call to bind_rxq. Clients will have to call
 *	rollover_rxq after this function succeeds in order to bring the new
 *	binding up-to-date with xlnx_efct_drvops::buffer_start.
 *	Returns queue id on success (same as xlnx_efct_rxq_params::qid, unless
 *	that value was -1), error otherwise.
 * @rollover_rxq: Triggers posting a superbuf to the nic with manual rollover
 *      bit set. This will trigger rollover for all the buffers currently on
 *      NIC. The rollover is asynchronous, to be completed at some time after
 *      this call returns, that means that the return value of this function is
 *      advisory at best.
 *	Returns 0 on success, error otherwise.
 * @free_rxq: Release a bound RXQ. This must trigger a NAPI poll, i.e.
 *	xlnx_efct_drvops::poll must be called in the not-too-distant future.
 *	n_hugepages is the number of hugepages which were added at bind time,
 *	i.e. the same value as xlnx_efct_rxq_params::n_hugepages. The size of
 *	the superbuf pool should be reduced by that amount. The net driver
 *	must discard only hugepages allocated by the calling client, but from
 *	that set it may discard any superbufs from the free pool that it
 *	likes (the reason for the this-client-only limitation is to make
 *	client unload work: if a different client's hugepages were freed then
 *	case (b) below may be unachievable due to references held by other
 *	clients).
 *	If there are not currently enough superbufs in the free pool then
 *	the net driver may choose to perform rollover to free some more and/or
 *	coalesce fragmented superbufs into freeable hugepages.
 *	If there are still not enough then the net driver must mark the
 *	necessary number of hugepages as to-be-lazily-freed when their
 *	contained superbufs next become freed. In that case:
 *	a) The net driver must still maintain its counter of 'desired number
 *	   of hugepages' so that future bind/free requests remain in-sync with
 *	   the client.
 *	b) All pages allocated by a specific client's alloc_hugepage must have
 *	   been passed to that same client's free_hugepage before close()
 *	   returns. This may require manual rollovers.
 * @init_txq: Initialise passed TXQ and associate it with an EVQ
 *	The specified EVQ must have already been successfully inited.
 *	Returns TXQ id on success, error otherwise.
 * @free_txq: Free an allocated TXQ
 * @ctpio_addr: Obtain the bus address of the CTPIO region for a TXQ
 *	On successful return the provided addr will refer to the CTPIO region, and size will provide the size of the region.
 *	The returned address should be IO mapped for access to the region.
 *	Returns 0 on success, error otherwise.
 * @filter_insert: Install a filter to direct traffic to a specific RXQ
 *	Returns 0 on success, error otherwise.
 * @filter_remove: Remove a previously installed filter
 *	Returns 0 on success, error otherwise.
 * @get_hugepages: Populates the 'pages' array with the set of hugepages
 *	currently backing the rxq's superbufs, in ID order, i.e. pages[i]
 *	backs superbufs i*superbufs_per_hugepage to
 *	(i+1)*superbufs_per_hugepage-1 inclusive. n_pages is the size of the
 *	pages array; if it is smaller than the ID of the largest hugepage
 *	allocated then an error is returned.
 *	Once given a position in the array, hugepages must not move around;
 *	they may only be freed. This means that the list may be sparse: any
 *	entries may be filled with NULL. If n_pages is larger than the maximum
 *	number of hugepages supported by the net driver then the values off
 *	the end must likewise be filled with NULL.
 *	Gives additional recounts to each page which the caller must free.
 *	Returns 0 on success, error otherwise.
 * @release_superbuf: Decrements the refcount of a given superbuf. Refs to a
 *	superbuf are created by bind_rxq and by xlnx_efct_drvops::buffer_start.
 *	This function must only be called from NAPI context (i.e. within one of
 *	the xlnx_efct_drvops callbacks) since it doesn't do any locking.
 * @is_filter_supported: Check whether the specified filter is supported.
 *	Fields(type, flow fields, mask) should be initialised with
 *	valid values.
 *	A filter configuration being supported does not guarantee an attempt
 *	to insert a filter of this type will succeed as there are other things
 *	that can cause filter insertion to fail.
 *	Returns true if supported, false otherwise.
 */
struct xlnx_efct_devops {
	struct xlnx_efct_client *(*open)(struct auxiliary_device *adev,
					const struct xlnx_efct_drvops *ops,
					void *driver_data);
	int (*close)(struct xlnx_efct_client *handle);

	int (*get_param)(struct xlnx_efct_client *handle, enum xlnx_efct_param p,
			 union xlnx_efct_param_value *arg);
	int (*set_param)(struct xlnx_efct_client *handle, enum xlnx_efct_param p,
			 union xlnx_efct_param_value *arg);
	int (*fw_rpc)(struct xlnx_efct_client *handle, struct xlnx_efct_rpc *rpc);

	int (*init_evq)(struct xlnx_efct_client *handle,
			struct xlnx_efct_evq_params *params);
	void (*free_evq)(struct xlnx_efct_client *handle, int evq);
	int (*bind_rxq)(struct xlnx_efct_client *handle,
			struct xlnx_efct_rxq_params *params);
	int (*rollover_rxq)(struct xlnx_efct_client *handle, int rxq);
	void (*free_rxq)(struct xlnx_efct_client *handle, int rxq,
	                 size_t n_hugepages);
	int (*init_txq)(struct xlnx_efct_client *handle,
			 struct xlnx_efct_txq_params *params);
	void (*free_txq)(struct xlnx_efct_client *handle, int txq);
	int (*ctpio_addr)(struct xlnx_efct_client *handle, int txq,
			  resource_size_t *addr, size_t *size);

	int (*request_wakeup)(struct xlnx_efct_client *handle, int qid);

	int (*filter_insert)(struct xlnx_efct_client *handle,
	                     struct xlnx_efct_filter_params *params);
	int (*filter_remove)(struct xlnx_efct_client *handle,
			     unsigned int filter_id);

	int (*get_hugepages)(struct xlnx_efct_client *handle, int rxq,
	                     struct xlnx_efct_hugepage *pages, size_t n_pages);
	void (*release_superbuf)(struct xlnx_efct_client *handle, int rxq,
	                         int sbid);
	bool (*is_filter_supported)(struct xlnx_efct_client *handle,
				    const struct ethtool_rx_flow_spec *spec);
};

/**
 * EFCT device interface.
 *
 * @adev: The parent auxiliary bus device.
 * @version: The value XLNX_EFCT_AUX_VERSION, which may be used by clients to
 *	check ABI compatibility.
 * @ops: Device API.
 */
struct xlnx_efct_device {
	struct auxiliary_device adev;
	unsigned version;
	const struct xlnx_efct_devops *ops;
};

static inline
struct xlnx_efct_device *to_xlnx_efct_device(struct auxiliary_device *adev)
{
	return container_of(adev, struct xlnx_efct_device, adev);
}
#endif /* _XLNX_EFCT_H */
