#!/bin/bash

set -e

# -k means keep the kernel kompat kode
# -kk means run kernel_compat.sh at export time
# -o means keep out-of-tree (not-upstream) code
# --suffix=<identifier> adds a suffix to the EFCT_DRIVER_VERSION define
SOURCE_SED_COMMANDS=''
MAKEFILE_SED_COMMANDS=''
UNIFDEF_DEFINES=''
KCOMPAT=
AUTOCOMPAT=
VERSION_SUFFIX=
NOT_UPSTREAM=

if [ -z "$SUBDIR" ]; then
	SUBDIR=drivers/net/ethernet/xilinx/x3
fi

for i in "$@"
do
case $i in
	-k)
		KCOMPAT=1
		;;
	-kk)
		KCOMPAT=1
		AUTOCOMPAT=apply
		;;
	-o)
		AUTOCOMPAT=
		NOT_UPSTREAM=1
		;;
	-s)	shift
		SUBDIR=$1
		;;
	--suffix=*)
		VERSION_SUFFIX="${i#*=}"
		;;
	*)
		# End of arguments
		break
		;;
esac
shift
done

if [ -z "$NOT_UPSTREAM" ]; then
	UNIFDEF_DEFINES="-UEFCT_NOT_UPSTREAM"
	MAKEFILE_SED_COMMANDS='/^ifndef EFCT_UPSTREAM/,/^endif # !EFCT_UPSTREAM/d; '

	if [ -n "$KCOMPAT" ]; then
		if [ "$AUTOCOMPAT" != "apply" ]; then
			AUTOCOMPAT=keep
		fi
		UNIFDEF_DEFINES="$UNIFDEF_DEFINES -DEFCT_USE_KCOMPAT"
	else
		UNIFDEF_DEFINES="$UNIFDEF_DEFINES -UEFCT_USE_KCOMPAT"
		MAKEFILE_SED_COMMANDS="$MAKEFILE_SED_COMMANDS"'/^ifndef EFCT_NO_KCOMPAT/,/^endif # !EFCT_NO_KCOMPAT/d; '
		MAKEFILE_SED_COMMANDS="$MAKEFILE_SED_COMMANDS"'/^#$/d; ' # delete spacing comments
	fi

	UNIFDEF_DEFINES="$UNIFDEF_DEFINES -UCONFIG_XILINX_DEBUGFS -UCONFIG_XILINX_EFCT_TRACING -UCONFIG_XILINX_AUX_EFCT -UCONFIG_EFCT_TEST -UCONFIG_XILINX_MCDI_MON"
fi

UNIFDEF_DEFINES="$UNIFDEF_DEFINES -UEFCT_NOT_EXPORTED -DWITH_MCDI_V2"

if [ -n "$VERSION_SUFFIX" ]; then
	SOURCE_SED_COMMANDS="$SOURCE_SED_COMMANDS s/\(#define EFCT_DRIVER_VERSION.*\)\"/\1-$VERSION_SUFFIX\"/; "
fi

KPATH="$1"
shift

if [ ! -d "$KPATH" ]; then
    echo >&2 "'$KPATH' does not exist"
    exit 2
fi

sources="$*"
#Skip copy below file
if [ -z "$NOT_UPSTREAM" ]; then
	sources=${sources//"efct_auxbus.c"/}
	sources=${sources//"mcdi_mon.c"/}
	sources=${sources//"debugfs.c"/}
	sources=${sources//"ioctl_common.c"/}
fi
if ! UNIFDEF="$(which unifdef 2>/dev/null)"; then
    echo >&2 "unifdef not found; try 'sudo yum install unifdef' or build it from the v5 repository"
    exit 1
fi

mkdir -p $KPATH/$SUBDIR
rm -f $KPATH/$SUBDIR/*.[ch]

if [ -n "$AUTOCOMPAT" ]; then
    # unifdef can't rewrite an #if unless every single symbol is known.
    # Which means we have to include autocompat even if we're going to
    # apply the symbols anyway
    ./kernel_compat.sh -k $KPATH -q > $KPATH/$SUBDIR/autocompat.h
    if [ "$AUTOCOMPAT" = "apply" ]; then
	# Read in from autocompat.h
	while read prefix word value; do
	    # Need to keep any #undefs after #if defined(EFCT_HAVE_XDP_EXT)
	    grep -q "#undef $word" kernel_compat.h && \
		echo ".. keeping $word due to #undef detected in kernel_compat.h" && \
		continue
	    if [  $prefix = "//" ]; then
		UNIFDEF_DEFINES="$UNIFDEF_DEFINES -U$value"
	    elif [ $prefix = "#define" ]; then
		UNIFDEF_DEFINES="$UNIFDEF_DEFINES -D$word"
	    else
		echo >&2 "unable to parse '$prefix $word $value'"
		exit 1
	    fi
	done < $KPATH/$SUBDIR/autocompat.h
    fi
    # Ensure that EFCT_USE_KCOMPAT is defined
    SOURCE_SED_COMMANDS="$SOURCE_SED_COMMANDS /\#include \"autocompat.h\"/i\
#define EFCT_USE_KCOMPAT"
fi

# Handle Makefiles
# Copy Makefile, deleting unwanted sections
sed "$MAKEFILE_SED_COMMANDS" <Makefile >$KPATH/$SUBDIR/Makefile

TOPDIR=$(echo "$SUBDIR" | sed 's#[a-z0-9]*#..#g')
if [ ! -f $KPATH/Makefile ]; then
    sed "$MAKEFILE_SED_COMMANDS" < $TOPDIR/Makefile > $KPATH/Makefile
fi

# Used in make srpm target
if [ -f $TOPDIR/scripts/Makefile.common ]; then
    mkdir -p $KPATH/scripts
    sed "$MAKEFILE_SED_COMMANDS" < $TOPDIR/scripts/Makefile.common > $KPATH/scripts/Makefile.common
fi

# Copy top-level sources, then find required headers and copy them
while [ -n "$sources" ]; do
    missing=""
    for source in $sources; do
	case "$source" in
	    */*.h)
		dest=$KPATH/${SUBDIR:+include}/$source
		;;
	    *)
		dest=$KPATH/$SUBDIR/$source
		;;
	esac
	sourcename=$(basename $source)
	if [ "$sourcename" = kernel_compat.sh -o "$sourcename" = kernel_compat_funcs.sh ]; then
		mkdir -p $(dirname $dest)
		cp $source $dest
		continue
	fi
	if ! [ -f "$source" ]; then
		continue
	fi
	mkdir -p $(dirname $dest)
	if ! [ -f $KPATH/$SUBDIR/$source ]; then
	    if [ -n "$UNIFDEF_DEFINES" ]; then
		# unifdef may return either 0 or 1 for success.
		"$UNIFDEF" $UNIFDEF_DEFINES -k -B <$source \
		    | sed "$SOURCE_SED_COMMANDS" >$dest \
		    && test ${PIPESTATUS[0]} -le 1
	    elif [ -n "$SOURCE_SED_COMMANDS" ]; then
		sed "$SOURCE_SED_COMMANDS" <$source >$dest
	    else
		cp $source $dest
	    fi
	fi
	while read header; do
	    if [ "$header" != config.h -a "$header" != autocompat.h -a \
		 \! -f $KPATH/$SUBDIR/$header ]; then
		missing="$missing $header"
	    fi
	done < <(sed 's/^#include "\([^/]*\)".*$/\1/; t; d' <$dest)
    done
    sources="$missing"
done

if [ -f $KPATH/$SUBDIR/mcdi_pcol.h ];then

	sed -i /MC_CMD_.*_PRIVILEGE_CTG/d $KPATH/$SUBDIR/mcdi_pcol.h

	# Delete multiple Blank lines

	sed -i 'N;/^\n$/D;P;D;' $KPATH/$SUBDIR/mcdi_pcol.h

	# Replace repeated space with single space

	sed -i 's/ \+ / /g' $KPATH/$SUBDIR/mcdi_pcol.h

fi

# Handle Kconfig
if [ -n "$SUBDIR" -a -f $KPATH/$SUBDIR/Kconfig ]; then
    # Copy Kconfig
    cp Kconfig $KPATH/$SUBDIR/
fi

if [ "$SUBDIR" = drivers/net/ethernet/xilinx/x3 ]; then
    # TODO XXXXXXXXXXXX Add a reference in the parent Makefile if it's not there already
    if ! grep -q '^obj-\$(CONFIG_XILINX_EFCT)' $KPATH/drivers/net/ethernet/xilinx/Makefile; then
	sed -i '$a\
obj-$(CONFIG_XILINX_EFCT) += x3/\
' \
	    $KPATH/drivers/net/ethernet/xilinx/Makefile
    fi

    # Add a reference in the parent Kconfig if it's not there already
    if ! grep -q '^source "drivers/net/ethernet/xilinx/x3/Kconfig"' $KPATH/drivers/net/ethernet/xilinx/Kconfig
	then
	sed -i '/^endif/i\
source "drivers/net/ethernet/xilinx/x3/Kconfig"\
' \
	    $KPATH/drivers/net/ethernet/xilinx/Kconfig
    fi
fi

# Update .config with our settings
test -f $KPATH/$SUBDIR/Kconfig || exit 0

if [ -f $KPATH/.config ];
then
    cp $KPATH/.config $KPATH/.config.old
    grep -v -E "CONFIG_XILINX_EFCT|CONFIG_NET_VENDOR_XILINX" $KPATH/.config.old > $KPATH/.config
    echo "CONFIG_NET_VENDOR_XILINX=y" >> $KPATH/.config

    for kernel_config in `grep -E "^config XILINX_EFCT" Kconfig | sed 's/config //'`;
    do
	set -- `grep "export CONFIG_$kernel_config " Makefile | head -1`
	if [ $# -ne 4 ];
	then
		echo "WARNING: Could not determine CONFIG_$kernel_config value from the Makefile:"
		grep "export CONFIG_$kernel_config " Makefile
		continue;
	fi
	echo "CONFIG_$kernel_config=$4" >> $KPATH/.config
    done
else
    test -z "$NOT_UPSTREAM" && echo "WARNING: No .config in $KPATH" || true
fi
