// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
* Driver for Xilinx network controllers and boards
* Copyright 2021 Xilinx Inc.
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License version 2 as published
* by the Free Software Foundation, incorporated herein by reference.
*/
#include <linux/module.h>
#include <linux/miscdevice.h>
#include <linux/device.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/file.h>
#include <linux/delay.h>
#include <linux/auxiliary_bus.h>
#include <linux/net/xilinx/xlnx_efct.h>
#include <linux/ethtool.h>
#include <linux/etherdevice.h>
#include <linux/inet.h>
#include "auxdev_test_ioctl.h"
#include "ktf.h"

MODULE_LICENSE("GPL");

#define AUX_IP4_ADDR_MASK	((__force __be32)~0)
#define AUX_PORT_MASK		((__force __be16)~0)
#define IP_PROTO_FULL_MASK	0xFF
#define NUM_FILTERS 512
#define MAX_RX_QUEUES 16
#define X3_EVENT_TYPE_CONTROL 3
#define EV_TYPE_SHIFT 60
#define EV_TYPE_MASK 0xF
#define TIME_SYNC_COUNT 20
#define TIME_SYNC_SHIFT 53
#define TIME_SYNC_MASK 0x3F
#define X3_CTL_EVENT_SUBTYPE_TIME_SYNC 0x1
#define X3_EV_UNSOL_MAX 64
#define PORT0_REG_HOST_EVQ_UNSOL_CREDIT_GRANT 0x00012000
#define EVQ_STRIDE 0x1000
#define MAC_MULTICAST_DST_MASK   0x0000010000000000
#define MAC_MULTICAST_DST_START  0x0000010000000000
#define ETH_MCAST 1
#define IPV4_MCAST 2
unsigned long client_driver_data;
struct filter_store {
	struct xlnx_efct_filter_params filter_params;
	bool is_free;
};
static struct filter_store filters[NUM_FILTERS] = { 0 };

#define DEVNAME "efctdev"

static struct auxiliary_device *aux_dev;
static struct xlnx_efct_device *xlnx_efct_dev;
static struct xlnx_efct_client *client_handle;
static struct xlnx_efct_client *g_client_handle;

static int buffer_start_release_sb = 0;
static int poll_release_sb = 0;
static int sbid_to_free_list[512];
atomic_t num_of_hugepages_allocated = ATOMIC_INIT(0);
static int buffer_start_count = 0;
static int poll_count = 0;

static int count = 0;

KTF_INIT();

static int auxtest_app_handle_event(void *driver_data, const struct xlnx_efct_event *event, int budget)
{
//       tlog(T_INFO, "%s, event received = %d, rxq = %d, value = %lld, budget = %d\n",
  //          __FUNCTION__, event->type, event->rxq, event->value, budget);
 //      tlog(T_INFO, "Driver data = 0x%lx\n", *(unsigned long *)driver_data);
       return 0;
}

static int auxtest_app_poll(void *driver_data, int rxq, int budget)
{
//       tlog(T_INFO, "%s called, rxq =%d\n", __FUNCTION__, rxq);
       return 0;
}

static int auxtest_app_buffer_start(void *driver_data, int rxq, unsigned sbseq, int sbid, bool sentinel)
{
       tlog(T_INFO, "rxq:%d sbseq:%u sbid:%d sentinel:%d\n", rxq, sbseq, sbid, sentinel);

       if (sbid == -1) {
               tlog(T_INFO, "sbid = -1");
               return -1;
       }

       return 1;
}

static int auxtest_app_buffer_end(void *driver_data, int rxq, int sbid, bool force)
{
       tlog(T_INFO, "rxq:%d sbid:%d sentinel:%d\n", rxq, sbid, force);

       return 1;
}

static int auxtest_app_alloc_hugepage(void *driver_data, struct xlnx_efct_hugepage *page)
{

       page->page = alloc_pages(__GFP_COMP | GFP_KERNEL, 9); //allocating 2MB
       //TODO : need to allocate memory for File as well.
       page->file = NULL;
       tlog(T_INFO, "%s alloc page %px order 9\n", __FUNCTION__, page->page);
	if (!page->page)
		return -1;
       return 0;
}

static void auxtest_app_free_hugepage(void *driver_data, struct xlnx_efct_hugepage *page)
{
       tlog(T_INFO, "%s free page %px\n", __FUNCTION__, page->page);

       put_page(page->page);
	page->page = NULL;
}

static void auxtest_app_hugepage_list_changed(void *driver_data, int rxq)
{
//       tlog(T_INFO, "%s, rxq: %d\n", __FUNCTION__, rxq);
}

static bool auxtest_app_pkt_handled(void *driver_data,  int rxq, bool flow_lookup,
                               const void *meta, const void *payload)
{
//       tlog(T_INFO, "%s\n", __FUNCTION__);
       return false;
}

const struct xlnx_efct_drvops auxtest_app_drv_ops = {
       .name = "auxtest_app",
       .poll = auxtest_app_poll,
       .handle_event = auxtest_app_handle_event,
       .buffer_start = auxtest_app_buffer_start,
       .buffer_end = auxtest_app_buffer_end,
       .alloc_hugepage = auxtest_app_alloc_hugepage,
       .free_hugepage = auxtest_app_free_hugepage,
       .hugepage_list_changed = auxtest_app_hugepage_list_changed,
       .packet_handled = auxtest_app_pkt_handled,
};

static int auxtest_handle_event(void *driver_data, const struct xlnx_efct_event *event, int budget)
{
//	tlog(T_INFO, "%s, event received = %d, rxq = %d, value = %lld, budget = %d\n",
//	     __FUNCTION__, event->type, event->rxq, event->value, budget);
//	tlog(T_INFO, "Driver data = 0x%lx\n", *(unsigned long *)driver_data);
	return 0;
}

//Also in free_rxq, passing hugepages as 1 , to avoid compilation error.
static int auxtest_poll(void *driver_data, int rxq, int budget)
{
	poll_count++;
#if 0
	struct xlnx_efct_client *handle_client = *(struct xlnx_efct_client **)g_client_handle;

	if (poll_release_sb == 1) {
		for (i = 0; i < 512 ; i++) {
			if (sbid_to_free_list[i] != -1) {
				xlnx_efct_dev->ops->release_superbuf(handle_client,
						rxq, sbid_to_free_list[i]);
				sbid_to_free_list[i] = -1;
			}
		}
	}
#endif

	tlog(T_INFO, "%s called, rxq =%d\n", __FUNCTION__, rxq);
	return 0;
}

static int auxtest_buffer_start(void *driver_data, int rxq, unsigned sbseq, int sbid, bool sentinel)
{
	tlog(T_INFO, "rxq:%d sbseq:%u sbid:%d sentinel:%d\n", rxq, sbseq, sbid, sentinel);

	if (sbid == -1) {
		tlog(T_INFO, "sbid = -1");
		return -1;
	} else
		buffer_start_count++;

	if ((buffer_start_release_sb == 1) || (poll_release_sb == 1))
		return -1;

	return 1;
}

static int auxtest_buffer_end(void *driver_data, int rxq, int sbid, bool force)
{
	tlog(T_INFO, "rxq:%d sbid:%d sentinel:%d\n", rxq, sbid, force);

	if ((buffer_start_release_sb == 1) || (poll_release_sb == 1))
		return -1;

	return 1;
}

static int auxtest_alloc_hugepage(void *driver_data, struct xlnx_efct_hugepage *page)
{

	page->page = alloc_pages(__GFP_COMP | GFP_KERNEL, 9); //allocating 2MB
	page->file = NULL;
	tlog(T_INFO, "%s alloc page %px order 9\n", __FUNCTION__, page->page);
	if (!page->page)
		return -1;
	atomic_inc(&num_of_hugepages_allocated);

	return 0;
}

static void auxtest_free_hugepage(void *driver_data, struct xlnx_efct_hugepage *page)
{
	tlog(T_INFO, "%s free page %px\n", __FUNCTION__, page->page);

	put_page(page->page);

	atomic_dec(&num_of_hugepages_allocated);
}

static void auxtest_hugepage_list_changed(void *driver_data, int rxq)
{
//	tlog(T_INFO, "%s, rxq: %d\n", __FUNCTION__, rxq);
}

static bool auxtest_pkt_handled(void *driver_data,  int rxq, bool flow_lookup,
				const void *meta, const void *payload)
{
//	tlog(T_INFO, "%s\n", __FUNCTION__);
	return false;
}

const struct xlnx_efct_drvops auxtest_drv_ops = {
	.name = "auxtest",
	.poll = auxtest_poll,
	.handle_event = auxtest_handle_event,
	.buffer_start = auxtest_buffer_start,
	.buffer_end = auxtest_buffer_end,
	.alloc_hugepage = auxtest_alloc_hugepage,
	.free_hugepage = auxtest_free_hugepage,
	.hugepage_list_changed = auxtest_hugepage_list_changed,
	.packet_handled = auxtest_pkt_handled,
};


static const struct auxiliary_device_id auxtest_id_table[] = {
	{ .name = "xilinx_efct."XLNX_EFCT_DEVNAME, },
	{},
};

MODULE_DEVICE_TABLE(auxiliary, auxtest_id_table);

static int auxtest_probe(struct auxiliary_device *adev,
				const struct auxiliary_device_id *id)
{
	if ( count != 0)
		return 0;

	aux_dev = adev;
	xlnx_efct_dev = to_xlnx_efct_device(adev);

	count++;
	if (NULL == xlnx_efct_dev->ops) {
		terr("Failed to get valid ops\n");
		return -1;
	}
	client_driver_data = 0xAA55;

	client_handle = xlnx_efct_dev->ops->open(adev, &auxtest_app_drv_ops, &client_driver_data);

	if (NULL == client_handle) {
		terr("Failed to open netdevice\n");
		return -1;
	}

	return 0;
}

static void auxtest_remove(struct auxiliary_device *adev)
{
	if (!aux_dev || (adev->id != aux_dev->id))
		return;

	if (client_handle)
		xlnx_efct_dev->ops->close(client_handle);

	count--;

	client_handle = NULL;
	aux_dev  = NULL;
}

static struct auxiliary_driver auxtest_driver = {
	.name = "auxtest",
	.probe = auxtest_probe,
	.remove = auxtest_remove,
	.id_table = auxtest_id_table,
};

static void decref_hugepages(struct xlnx_efct_hugepage *pages, size_t n_pages)
{
	int i;
	for (i = 0; i < n_pages; i++) {
		if (pages[i].page != NULL) {
			put_page(pages[i].page);
			if (pages[i].file != NULL)
				fput(pages[i].file);
		}
	}
}

/*
** This function will be called when we open the Misc device file
*/
static int auxtest_open(struct inode *inode, struct file *file)
{
    pr_info("Aux misc device open\n");
    return 0;
}
/*
** This function will be called when we close the Misc Device file
*/
static int auxtest_close(struct inode *inodep, struct file *filp)
{
    pr_info("Aux misc device close\n");
    return 0;
}

static int verify_hugepages(struct xlnx_efct_hugepage *page, int num_hugepages)
{
	int i = 0, j = 0;
	int nhp = 0;

	for (i = 0; i < num_hugepages; i++) {
		if (page[i].page != NULL) {
			nhp++;
			if (poll_release_sb == 1) {
				sbid_to_free_list[j++] = i;
				sbid_to_free_list[j++] = i+1;
			}
			tlog(T_INFO, "page address is %px for index %d", page[i].page, i);
		}
	}
	tlog(T_INFO, "number of hugepages : %d\n", nhp);

	return nhp;
}
static long auxtest_ioctl(struct file *filp, unsigned int req,
				unsigned long arg)
{
	int rc = 0;

	if (NULL == client_handle) {
		terr("No client handle");
		return -ENODEV;
	}

	switch (req) {
		case AUXTEST_ADD_EVQ:
		{
			struct xlnx_efct_evq_params evq_params = { 0 };

			evq_params.qid = arg;
			evq_params.q_page = alloc_pages(__GFP_COMP | GFP_KERNEL, 0);//allocating one page
			evq_params.page_offset = 0;
			evq_params.q_size = PAGE_SIZE;
			evq_params.entries = (evq_params.q_size / 8);

			rc = xlnx_efct_dev->ops->init_evq(client_handle, &evq_params);
			if (!rc)
				pr_info("Eventq added successfully\n");
			else
				pr_info("Eventq addition failed\n");

			break;
		}
		case AUXTEST_FREE_EVQ:
		{
			int qid;

			qid = arg;
			xlnx_efct_dev->ops->free_evq(client_handle, qid);
			rc = 0;
			break;
		}
		case AUXTEST_BIND_RXQ:
		{
			int qid, nhp;
			struct xlnx_efct_hugepage page[8]; /* Testing for only 1 hugepage of 1MB */
			struct xlnx_efct_rxq_params rxq_params;

			rxq_params.qid = arg;
			rxq_params.n_hugepages = 1;
			qid = xlnx_efct_dev->ops->bind_rxq(client_handle, &rxq_params);
			tlog(T_INFO, "Rx Qid = 0x%x\n", qid);
			if (qid < 0) {
				pr_info("bind_rxq call failed\n");
				rc = qid;
				goto out;
			}

			/* Verify that the alloc_hugepage function in the driver gets called. */
			rc = xlnx_efct_dev->ops->get_hugepages(client_handle, qid, page, 8);
			if (rc) {
				pr_info("get_hugepages call failed\n");
				goto out;
			}
			nhp = verify_hugepages(page, 8);
			decref_hugepages(page, 8);

			break;
		}
		case AUXTEST_FREE_RXQ:
		{
			int qid;

			qid = arg;
			xlnx_efct_dev->ops->free_rxq(client_handle, qid, 1);
			rc = 0;
			break;
		}
		case AUXTEST_ALLOC_TXQ:
		{
			int tx_qid;
			struct xlnx_efct_txq_params txq_params = {0, -1, 2};
			txq_params.evq = arg;
			tx_qid = xlnx_efct_dev->ops->init_txq(client_handle, &txq_params);
			if (tx_qid < 0) {
				rc = tx_qid;
				pr_err("init_txq call failed tx_qid = %d", tx_qid);
				break;
			}

			pr_info("tx_qid = %d", tx_qid);

			break;
		}
		case AUXTEST_FREE_TXQ:
		{
			int qid;
			qid = arg;
			xlnx_efct_dev->ops->free_txq(client_handle, qid);
			rc = 0;
			break;
		}
		case AUXTEST_ROLLOVER_RXQ:
		{
			int qid;
			qid = arg;
			rc = xlnx_efct_dev->ops->rollover_rxq(client_handle, qid);
			if (rc) {
				pr_info("rollover_rxq call failed\n");
				goto out;
			}
			break;
		}
		case AUXTEST_GET_HUGEPAGES:
		{
			int qid,nhp;
			struct xlnx_efct_hugepage page[64];
			qid = arg;
			rc = xlnx_efct_dev->ops->get_hugepages(client_handle, qid, page, 64);
			if (rc) {
				pr_info("get_hugepages call failed\n");
				goto out;
			}
			nhp = verify_hugepages(page, 64);
			decref_hugepages(page, 64);
			break;
		}
		case AUXTEST_FILTER_INSERT:
		{
			struct user_filter_params params;
			struct ethtool_rx_flow_spec *spec;
			cpumask_var_t cpumask;
			int i, index = -1;

			for (i = 0; i < NUM_FILTERS; i++) {
				if (filters[i].is_free == 0) {
					index = i;
					break;
				}
			}
			if (index < 0) {
                                pr_info("NO free filter index is found \n");
				rc = -EINVAL;
				goto out;
			}

			rc = copy_from_user(&params, (char *)arg, sizeof(params));
			spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
			filters[index].filter_params.flags = params.flags;
			spec->h_u.tcp_ip4_spec.ip4dst = params.dst_ip;
			if (params.dst_ip)
				spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
			spec->h_u.tcp_ip4_spec.pdst = params.pdst;
			if (params.pdst)
				spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
			spec->flow_type = params.flow_type;
			spec->ring_cookie = params.qid;
			spec->location = RX_CLS_LOC_ANY;
			filters[index].filter_params.spec = spec;
			if (params.mask) {
				zalloc_cpumask_var(&cpumask, GFP_KERNEL);
				filters[index].filter_params.mask = cpumask;
				for (i = 0; i < MAX_RX_QUEUES; i++) {
					if (params.mask & (1 << i))
						cpumask_set_cpu(i, cpumask);
				}
			}
			if (params.flow_type == ETHER_FLOW) {
				if (params.mcast == ETH_MCAST) {
					u64_to_ether_addr(MAC_MULTICAST_DST_MASK, spec->m_u.ether_spec.h_dest);
				} else if (params.mcast == IPV4_MCAST) {
					memcpy(spec->m_u.ether_spec.h_dest, params.eth_mask, sizeof(params.eth_mask));
				} else {
					memcpy(spec->m_u.ether_spec.h_dest, params.eth_mask, sizeof(params.eth_mask));
				}
				memcpy(spec->h_u.ether_spec.h_dest, params.eth_addr, sizeof(params.eth_addr));
				spec->flow_type = params.flow_type;
				spec->ring_cookie = params.qid;
				spec->location = RX_CLS_LOC_ANY;
			} else if (params.flow_type == IP_USER_FLOW) {
				spec->flow_type = IP_USER_FLOW;
				spec->ring_cookie = params.qid;
				spec->location = RX_CLS_LOC_ANY;
				if (params.l4proto) {
					spec->m_u.usr_ip4_spec.proto = IP_PROTO_FULL_MASK;
					spec->h_u.usr_ip4_spec.proto = params.l4proto;
				} else {
					pr_info("IP proto value is missing\n");
				}
			}
			rc = xlnx_efct_dev->ops->filter_insert(client_handle, &filters[index].filter_params);
                        if (rc && rc != -EEXIST) {
                                pr_info("Filter insert call failed\n");
                                goto out;
                        } else if (rc == -EEXIST) {
				pr_info("Filter already exists\n");
			} else {
				filters[index].is_free = 1;
				pr_info("Added filter index %d", index);
			}
			if (params.mask)
				free_cpumask_var(cpumask);
			kfree(spec);
			break;
		}
		case AUXTEST_FILTER_REMOVE:
		{
			int index = arg;

			if (!filters[index].is_free) {
				pr_info("Invalid filter index \n");
				rc = -EINVAL;
				goto out;
			}

			rc = xlnx_efct_dev->ops->filter_remove(client_handle, filters[index].filter_params.filter_id_out);
                        if (rc) {
                                pr_info("filter remove call failed\n");
                                goto out;
                        }
			filters[index].is_free = 0;
			pr_info("Removed filter index %d", index);
			break;
		}
		default:
			rc = -EINVAL;
	}

out :
	return rc;
}

TEST(auxtest, N001_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = 0;
	filter_params.spec = spec;

	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(NULL, &filter_params));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, NULL));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(NULL, NULL));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_remove(NULL, 0));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_remove(g_client_handle, 0));
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));

	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

TEST(auxtest, N002_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;
	cpumask_var_t cpumask;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	filter_params.spec = spec;
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	filter_params.mask = cpumask;
	cpumask_set_cpu(MAX_RX_QUEUES, cpumask);

	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

TEST(auxtest, N003_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = 0;
	filter_params.spec = spec;

	spec->m_u.tcp_ip4_spec.ip4dst = htonl(0xf0000000);
	spec->location = 0;
	EXPECT_INT_EQ(false, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

TEST(auxtest, N004_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = 0xFFF0000;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = 0;
	filter_params.spec = spec;

	EXPECT_INT_EQ(false, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));

}

TEST(auxtest, N005_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = 0;
	filter_params.spec = spec;

	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_EQ(1, filter_params.rxq_out);
	spec->ring_cookie = 2;
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_EQ(1, filter_params.rxq_out);
	spec->ring_cookie = MAX_RX_QUEUES;
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

TEST(auxtest, N006_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	union xlnx_efct_param_value param = { 0 };
	struct ethtool_rx_flow_spec *spec;
	u16 port = 5001;
	int i;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);
	ASSERT_INT_EQ(0, xlnx_efct_dev->ops->get_param(client_handle, XLNX_EFCT_DESIGN_PARAM, &param));
	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);
	ASSERT_INT_NE(0, param.design_params.num_filter);
	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(port);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = 0;
	filter_params.spec = spec;
	for (i = 0; i < param.design_params.num_filter - 1; i++) {
		EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
		port++;
		spec->h_u.tcp_ip4_spec.pdst = htons(port);

	}

	memset(spec, 0,sizeof(*spec));
	u64_to_ether_addr(MAC_MULTICAST_DST_START, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(MAC_MULTICAST_DST_MASK, spec->m_u.ether_spec.h_dest);
	spec->flow_type = 0x12; //ETHER
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;
	filter_params.flags = 0;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	memset(spec, 0,sizeof(*spec));

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.2");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(port);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

TEST(auxtest, N007_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 0;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	filter_params.spec = spec;

	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_GT(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));

}

TEST(auxtest, N008_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = 2;

	filter_params.flags = 0;
	filter_params.spec = spec;
	//Location not supported
	EXPECT_INT_EQ(false, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

//Duplicate multicast
TEST(auxtest, N009_filter_op)
{

	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("224.1.2.3");
	spec->m_u.tcp_ip4_spec.ip4dst = htonl(0xe0000000);
	spec->flow_type = 0x2; //UDP
	spec->h_u.tcp_ip4_spec.pdst = 0;
	filter_params.flags = 0;
	EXPECT_INT_EQ(false, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	memset(spec, 0,sizeof(*spec));
	u64_to_ether_addr(MAC_MULTICAST_DST_START, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(MAC_MULTICAST_DST_MASK, spec->m_u.ether_spec.h_dest);
	spec->flow_type = 0x12; //ETHER
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;
	filter_params.flags = 0;
	filter_params.spec = spec;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

//Source IP,port
TEST(auxtest, N010_filter_op)
{

	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4src = in_aton("224.1.2.3");
	spec->m_u.tcp_ip4_spec.ip4src = htonl(0xf0000000);
	spec->h_u.tcp_ip4_spec.psrc = htons(5001);
	spec->m_u.tcp_ip4_spec.psrc = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = 0;
	filter_params.spec = spec;
	EXPECT_INT_EQ(false, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

//Source MAC address
TEST(auxtest, N011_filter_op)
{

	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	u64_to_ether_addr(0x0000012345678955, spec->h_u.ether_spec.h_source);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_source);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = 0x12; //ether
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = 0;
	filter_params.spec = spec;
	EXPECT_INT_EQ(false, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

//MAC proto
TEST(auxtest, N012_filter_op)
{

	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);
	spec->h_u.ether_spec.h_proto = 0x0800;
	spec->m_u.ether_spec.h_proto = 0xffff;
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = 0x12; //ether
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = 0;
	filter_params.spec = spec;
	EXPECT_INT_EQ(false, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

//Dest MAC with mask
TEST(auxtest, N013_filter_op)
{

	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffff00, spec->m_u.ether_spec.h_dest);
	spec->flow_type = 0x12; //ether
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = 0;
	filter_params.spec = spec;
	EXPECT_INT_EQ(false, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

//Dest MAC with vlan mask
TEST(auxtest, N014_filter_op)
{

	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x00ff);
	filter_params.flags = 0;
	filter_params.spec = spec;
	EXPECT_INT_EQ(false, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

/* Exclusive, anyqueue strict on default queue(0)
 * filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT
 * cpumask= 0x01
 * output= filter insert Fail "Unable to find the queue ID for given mask"
 */
TEST(auxtest, N015_filter_op)
{

	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = 0x12; //ether
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	filter_params.spec = spec;
	cpumask_set_cpu(0, cpumask);
	filter_params.mask = cpumask;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	free_cpumask_var(cpumask);
cleanup:
	kfree(spec);
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}
/* 1 client1: Programs mac vlan filters in queue 1
 * 2 client2: Program filter with qid X, cpumask=0x03
 *   filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT
 * output: filter insert fail. client2 cannot program filter
 */

TEST(auxtest, N016_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec = NULL;
	struct xlnx_efct_client *handle, *handle1;
	cpumask_var_t cpumask;
	unsigned long client_hndl_test;

	ASSERT_ADDR_NE(aux_dev, NULL);

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_INT_EQ(IS_ERR(handle), 0);
	handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup2);

	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup2);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	filter_params.spec = spec;
	filter_params.flags = 0;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	EXPECT_INT_EQ(1, filter_params.rxq_out);

	spec->h_ext.vlan_tci = htons(0x8);
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	cpumask_set_cpu(1, cpumask);
	filter_params.mask = cpumask;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle1, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));

	free_cpumask_var(cpumask);
cleanup2:
	kfree(spec);
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));

}

/* Program macvlan filter for rxq 3,
 * Program same filter with another client rxq=1,
 * filter_params.flags = 0
 * output:2nd client fail to program filter
 */

TEST(auxtest, N017_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);

        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	spec->ring_cookie = 3;
	filter_params.spec = spec;
	filter_params.flags = 0;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	spec->ring_cookie = 1;
	filter_params.flags = 0;
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	kfree(spec);
cleanup1:
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* Program macvlan filter for rxq 1,
 * Program same filter with rxq=3,
 * filter_params.flags = XLNX_EFCT_FILTER_F_PREF_QUEUE
 * output:Same client cannot re-insert same filter
 */

TEST(auxtest, N018_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.spec = spec;
	filter_params.flags = 0;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_EQ(1, filter_params.rxq_out);
	spec->ring_cookie = 2;
	filter_params.flags = XLNX_EFCT_FILTER_F_PREF_QUEUE;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

//TCP with vlan
TEST(auxtest, N019_filter_op)
{

	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);
	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = TCP_V4_FLOW | FLOW_EXT; //TCP and VLAN
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	filter_params.flags = 0;
	filter_params.spec = spec;
	EXPECT_INT_EQ(false, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

TEST(auxtest, T001_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;
	cpumask_var_t cpumask;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);;
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = 0;
	filter_params.spec = spec;
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	filter_params.mask = cpumask;

	/*Add 4 filters to queue 1*/
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	spec->h_u.tcp_ip4_spec.pdst = htons(5002);
	spec->ring_cookie = 1;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	spec->h_u.tcp_ip4_spec.pdst = htons(5003);
	spec->ring_cookie = 1;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	spec->h_u.tcp_ip4_spec.pdst = htons(5004);
	spec->ring_cookie = 1;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	/*Add 3 filters to queue 2*/
	spec->h_u.tcp_ip4_spec.pdst = htons(5005);
	spec->ring_cookie = 2;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	spec->h_u.tcp_ip4_spec.pdst = htons(5006);
	spec->ring_cookie = 2;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	spec->h_u.tcp_ip4_spec.pdst = htons(5007);
	spec->ring_cookie = 2;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	/*Add 2 filters to queue 3*/
	spec->h_u.tcp_ip4_spec.pdst = htons(5008);
	spec->ring_cookie = 3;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	spec->h_u.tcp_ip4_spec.pdst = htons(5009);
	spec->ring_cookie = 3;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	/*Add 1 filters with strict flag and cpu mask reflecting 1,2,3 queues*/
	spec->h_u.tcp_ip4_spec.pdst = htons(5010);
	cpumask_set_cpu(1, cpumask);
	cpumask_set_cpu(2, cpumask);
	cpumask_set_cpu(3, cpumask);
	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_EQ(3, filter_params.rxq_out);

	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

TEST(auxtest, T002_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = 0;
	filter_params.spec = spec;

	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(g_client_handle, filter_params.filter_id_out));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

TEST(auxtest, T003_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec = NULL;
	struct xlnx_efct_rxq_params rxq_params;
	struct xlnx_efct_client *handle, *handle1;
	unsigned long client_hndl_test;
	int qid = 0;

	ASSERT_ADDR_NE(aux_dev, NULL);

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_INT_EQ(IS_ERR(handle), 0);
	handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup2);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	filter_params.spec = spec;

	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	/*Trying inserting the filter for same client, same queue but exclusive flag set again*/
	spec->h_u.tcp_ip4_spec.pdst = htons(5002);
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	EXPECT_INT_EQ(1, filter_params.rxq_out);
	rxq_params.qid = 1;
	rxq_params.n_hugepages = 0;

	qid = xlnx_efct_dev->ops->bind_rxq(handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);
	/*Trying to bind from different client*/
	qid = xlnx_efct_dev->ops->bind_rxq(handle1, &rxq_params);
	EXPECT_INT_LT(qid, 0);
	/*Trying to insert filter from different client*/
	spec->h_u.tcp_ip4_spec.pdst = htons(5003);
	filter_params.flags = 0;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle1, spec));
	EXPECT_INT_GT(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));

cleanup1:
	kfree(spec);
cleanup2:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));

}

TEST(auxtest, T004_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;
	cpumask_var_t cpumask;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE;
	filter_params.spec = spec;
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	filter_params.mask = cpumask;
	cpumask_set_cpu(MAX_RX_QUEUES, cpumask);

	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_NE(0, filter_params.rxq_out);

	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

TEST(auxtest, T006_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;
	cpumask_var_t cpumask;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE;
	filter_params.spec = spec;
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	filter_params.mask = cpumask;
	cpumask_set_cpu(0, cpumask);

	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_NE(filter_params.rxq_out, 0);

	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

TEST(auxtest, T007_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;
	cpumask_var_t cpumask;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	filter_params.spec = spec;
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	filter_params.mask = cpumask;
	cpumask_set_cpu(0, cpumask);
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	cpumask_set_cpu(1, cpumask);
	spec->h_u.tcp_ip4_spec.pdst = htons(5002);
	/*Verify that default queue is not assigned if another queue is set in the mask*/
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_EQ(1, filter_params.rxq_out);

	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

//Multiple filters on exclusive queue by same client
TEST(auxtest, T008_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec = NULL;
	struct xlnx_efct_rxq_params rxq_params;
	struct xlnx_efct_client *handle, *handle1;
	unsigned long client_hndl_test;
	cpumask_var_t cpumask;
	int qid = 0;

	ASSERT_ADDR_NE(aux_dev, NULL);

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_INT_EQ(IS_ERR(handle), 0);
	handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup2);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	filter_params.spec = spec;

	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	/*Trying inserting the filter for same client, same queue but exclusive flag set again*/
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	//assumed rxq 1 is as cpu 1. How to get cpu for rxq?
	cpumask_set_cpu(1, cpumask);
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE |
				XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	spec->h_u.tcp_ip4_spec.pdst = htons(5002);
	filter_params.mask = cpumask;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	EXPECT_INT_EQ(1, filter_params.rxq_out);
	rxq_params.qid = 1;
	rxq_params.n_hugepages = 0;

	qid = xlnx_efct_dev->ops->bind_rxq(handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);
	/*Trying to bind from different client*/
	qid = xlnx_efct_dev->ops->bind_rxq(handle1, &rxq_params);
	EXPECT_INT_LT(qid, 0);
	/*Trying to insert filter from different client*/
	spec->h_u.tcp_ip4_spec.pdst = htons(5003);
	filter_params.flags = 0;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle1, spec));
	EXPECT_INT_GT(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
cleanup2:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));

}

/* Program mac filter with
 * filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT
 * cpumask= 0xFF
 * output: qid > 0
 */
TEST(auxtest, T009_filter_op)
{

	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec = NULL;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = 0x12; //ether
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	filter_params.spec = spec;
	filter_params.mask = cpu_online_mask;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_GT(filter_params.rxq_out, 0);
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

/* Programs filters in queue 1,2,3,4,5,7
 * filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT
 * cpumask= 0xFF
 * output: qid > 0
 */
TEST(auxtest, T010_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;
	cpumask_var_t cpumask;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);;
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = 0;
	filter_params.spec = spec;
	filter_params.mask = cpumask;

	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	spec->h_u.tcp_ip4_spec.pdst = htons(5002);
	spec->ring_cookie = 2;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	spec->h_u.tcp_ip4_spec.pdst = htons(5003);
	spec->ring_cookie = 3;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	memset(spec, 0,sizeof(*spec));
	u64_to_ether_addr(0x000001234567afbf, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = 0x12; //ether
	spec->ring_cookie = 4;
	spec->location = RX_CLS_LOC_ANY;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	u64_to_ether_addr(0x00000123456bafbf, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->ring_cookie = 5;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	u64_to_ether_addr(0x00000123a56bafbf, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->h_u.tcp_ip4_spec.pdst = htons(5006);
	spec->ring_cookie = 7;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	/*Add 1 filters with strict flag and cpu mask reflecting 1,2,3 queues*/
	memset(spec, 0,sizeof(*spec));
	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);;
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5010);
	spec->location = RX_CLS_LOC_ANY;
	spec->ring_cookie = 2;
	spec->flow_type = 2; //UDP
	spec->location = RX_CLS_LOC_ANY;
	cpumask_set_cpu(0, cpumask);
	cpumask_set_cpu(1, cpumask);
	cpumask_set_cpu(2, cpumask);
	cpumask_set_cpu(3, cpumask);
	cpumask_set_cpu(4, cpumask);
	cpumask_set_cpu(5, cpumask);
	cpumask_set_cpu(6, cpumask);
	cpumask_set_cpu(7, cpumask);
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	filter_params.mask = cpumask;
	spec->h_u.tcp_ip4_spec.pdst = htons(5008);
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_EQ(6, filter_params.rxq_out);

	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}
/* 1. Programs mac vlan filters in queue 1
 * 2 Program filter with qid X, cpumask=0x03
 *   filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT
 * output: qid == 1, same client with multiple filters in exclusive queue
 */

TEST(auxtest, T011_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec = NULL;
	struct xlnx_efct_client *handle;
	cpumask_var_t cpumask;
	unsigned long client_hndl_test;

	ASSERT_ADDR_NE(aux_dev, NULL);

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_INT_EQ(IS_ERR(handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	filter_params.spec = spec;
	filter_params.flags = 0;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	EXPECT_INT_EQ(1, filter_params.rxq_out);

	spec->h_ext.vlan_tci = htons(0x8);
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	cpumask_set_cpu(1, cpumask);
	filter_params.mask = cpumask;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	EXPECT_INT_EQ(1, filter_params.rxq_out);

	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));

}

/* Program 3 IP filter in rxq 1,
 * program 2 mac vlan filter rxq 2,
 * Program 1 mac filter in rxq 3
 * Program filter with qid X, cpumask=0x0F
 * filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT
 * output: filter programmed in qid == 3,(minimum filters)
 */
TEST(auxtest, T012_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;
	cpumask_var_t cpumask;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);;
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->ring_cookie = 1;
	spec->location = RX_CLS_LOC_ANY;

	filter_params.flags = 0;
	filter_params.spec = spec;
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	filter_params.mask = cpumask;

	/*Add 3 filters to queue 1*/
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	spec->h_u.tcp_ip4_spec.pdst = htons(5002);
	spec->ring_cookie = 1;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	spec->h_u.tcp_ip4_spec.pdst = htons(5003);
	spec->ring_cookie = 1;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	/*Add 2 filters to queue 2*/
	memset(spec, 0,sizeof(*spec));
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	spec->ring_cookie = 2;
	filter_params.flags = 0;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	spec->h_ext.vlan_tci = htons(0x10);
	spec->ring_cookie = 2;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	/*Add 1 filters to queue 3*/
	memset(spec, 0,sizeof(*spec));
	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);;
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP
	spec->h_u.tcp_ip4_spec.pdst = htons(5009);
	spec->ring_cookie = 3;
	spec->location = RX_CLS_LOC_ANY;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));

	/*Add 1 filters with strict flag and cpu mask reflecting 1,2,3 queues*/
	memset(spec, 0,sizeof(*spec));
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->location = RX_CLS_LOC_ANY;
	cpumask_set_cpu(0, cpumask);
	cpumask_set_cpu(1, cpumask);
	cpumask_set_cpu(2, cpumask);
	cpumask_set_cpu(3, cpumask);
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_EQ(3, filter_params.rxq_out);

	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}
/* Program mac vlan filter with
 * cpumask=0x03
 * filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE
 * output: rxq == 1
*/
TEST(auxtest, T013_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;
	cpumask_var_t cpumask;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	spec->location = RX_CLS_LOC_ANY;
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE;
	filter_params.spec = spec;
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	cpumask_set_cpu(0, cpumask);
	cpumask_set_cpu(1, cpumask);
	filter_params.mask = cpumask;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_EQ(filter_params.rxq_out, 1);

	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

/* Program mac vlan filter in rxq 1
 * Program different client filter with:
 * cpumask=0x03
 * filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE
 * output: rxq of 2nd client filter != 0 and != 1
 */

TEST(auxtest, T014_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;
	struct xlnx_efct_client *handle, *handle1;
	cpumask_var_t cpumask;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
	ASSERT_ADDR_NE(handle, NULL);
	ASSERT_INT_EQ(IS_ERR(handle), 0);

	handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
	ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup2);

	u64_to_ether_addr(0x00000a234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	spec->ring_cookie = 1;
	filter_params.flags = 0;
	filter_params.spec = spec;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));

	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE;
	filter_params.spec = spec;
	spec->h_ext.vlan_tci = htons(0x13);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	cpumask_set_cpu(0, cpumask);
	cpumask_set_cpu(1, cpumask);
	filter_params.mask = cpumask;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle1, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	EXPECT_INT_NE(filter_params.rxq_out, 0);
	EXPECT_INT_NE(filter_params.rxq_out, 1);

	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
cleanup2:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}
/* Program IP filter with
 * cpumask=0x01
 * filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE
 * output: rxq > 0
 */

TEST(auxtest, T015_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;
	struct xlnx_efct_client *handle;
	cpumask_var_t cpumask;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
	ASSERT_ADDR_NE(handle, NULL);
	ASSERT_INT_EQ(IS_ERR(handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	spec->h_u.tcp_ip4_spec.ip4dst = in_aton("192.168.1.1");
	spec->m_u.tcp_ip4_spec.ip4dst = AUX_IP4_ADDR_MASK;
	spec->h_u.tcp_ip4_spec.pdst = htons(5001);;
	spec->m_u.tcp_ip4_spec.pdst = AUX_PORT_MASK;
	spec->flow_type = 2; //UDP

	spec->location = RX_CLS_LOC_ANY;
	cpumask_set_cpu(0, cpumask);
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE | XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE;
	filter_params.mask = cpumask;
	filter_params.spec = spec;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));

	EXPECT_INT_GT(filter_params.rxq_out, 0);

	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}
/* Program mac vlan exclusive filters in rx1, rx2,rx3
 * Program another filter with cpumask=0x0E,
 * filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT
 * output: rxq either 1 or 2 or 3
*/

TEST(auxtest, T016_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;
	cpumask_var_t cpumask;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	spec->ring_cookie = 1;
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	filter_params.spec = spec;
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	filter_params.mask = cpumask;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_EQ(filter_params.rxq_out, 1);
	spec->h_ext.vlan_tci = htons(0x5);
	spec->ring_cookie = 2;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_EQ(filter_params.rxq_out, 2);
	spec->h_ext.vlan_tci = htons(0x8);
	spec->ring_cookie = 3;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_EQ(filter_params.rxq_out, 3);
	cpumask_set_cpu(1, cpumask);
	cpumask_set_cpu(2, cpumask);
	cpumask_set_cpu(3, cpumask);
	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	spec->h_ext.vlan_tci = htons(0xa);
	filter_params.mask = cpumask;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_GT(filter_params.rxq_out, 0);
	EXPECT_INT_LT(filter_params.rxq_out, 4);
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

/* Program mac vlan filter with cpumask=0x80,
 * filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT
 * output: rxq == 7
 */

TEST(auxtest, T017_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct ethtool_rx_flow_spec *spec;
	cpumask_var_t cpumask;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
	ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);

	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	filter_params.spec = spec;
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	cpumask_set_cpu(7, cpumask);
	filter_params.mask = cpumask;
	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(g_client_handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(g_client_handle, &filter_params));
	EXPECT_INT_EQ(filter_params.rxq_out, 7);
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

/* Program macvlan filter for rxq 3,
 * Program same filter with another client rxq=1,
 * filter_params.flags = XLNX_EFCT_FILTER_F_PREF_QUEUE
 * output: rxq == 3
 */

TEST(auxtest, T018_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);

        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	spec->ring_cookie = 3;
	filter_params.spec = spec;
	filter_params.flags = 0;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	spec->ring_cookie = 1;
	filter_params.flags = XLNX_EFCT_FILTER_F_PREF_QUEUE;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	EXPECT_INT_EQ(filter_params.rxq_out, 3);
	kfree(spec);
cleanup1:
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}


/* Program macvlan filter for rxq 3,
 * Program same filter with another client
 * rxq=1, filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE
 * output: fail, rxq 3 already has same filter
 */

TEST(auxtest, T019_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);

        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	spec->ring_cookie = 3;
	filter_params.spec = spec;
	filter_params.flags = 0;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	spec->ring_cookie = 1;
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	kfree(spec);
cleanup1:
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* Program macvlan filter for rxq 3,
 * Program same filter, same client with
 * Exclusive=1, rxq=1, filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE
 * output: fail, same client cannot program same filter
 * Repeat test with rxq = -1
 */

TEST(auxtest, T020_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle;
	struct ethtool_rx_flow_spec *spec;
	int i, rxq;
	int id1;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	for (i = 0, rxq = 3; i < 2; i++, rxq = -1) {
		u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
		spec->ring_cookie = rxq;
		filter_params.spec = spec;
		filter_params.flags = 0;
		EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		id1 = filter_params.filter_id_out;
		spec->ring_cookie = 1;
		filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
		EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		//Repeat
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
		spec->h_ext.vlan_tci = htons(0x6);
	}
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* Program macvlan filter for rxq 3,
 * Program same filter, same client with
 * rxq=1, filter_params.flags = XLNX_EFCT_FILTER_F_PREF_QUEUE
 * output: fail, Same client cannot program same filter twice
 * Repeat test with rxq = -1
 */

TEST(auxtest, T021_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle;
	struct ethtool_rx_flow_spec *spec;
	int i, rxq;
	int id1;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	for (i = 0, rxq = 3; i < 2; i++, rxq = -1) {
		u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
		spec->ring_cookie = rxq;
		filter_params.spec = spec;
		filter_params.flags = 0;
		EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		id1 = filter_params.filter_id_out;
		spec->ring_cookie = 1;
		filter_params.flags = XLNX_EFCT_FILTER_F_PREF_QUEUE;
		EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		//Repeat
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
		spec->h_ext.vlan_tci = htons(0x6);
	}
	kfree(spec);
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq 3
 * 2) client2: Program same filter with  rxq=3
 * 3) client1: Program another filter with rxq = 3, filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE
 * Note: Repeat test with rxq =  RX_CLS_FLOW_DISC(-1)
 * output: rxq = 3, Filter program : PASS, PASS, FAIL
 *	   rxq= -1, Filter program : PASS, PASS, PASS
 */

TEST(auxtest, T022_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec;
	int i, rxq;
	int id1, id2;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);

        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	for (i = 0, rxq = 3; i < 2; i++, rxq = -1) {
		u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
		spec->ring_cookie = rxq;
		filter_params.flags = 0;
		filter_params.spec = spec;
		EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		id1 = filter_params.filter_id_out;
		EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
		id2 = filter_params.filter_id_out;
		EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		u64_to_ether_addr(0x000001234567abcd, spec->h_u.ether_spec.h_dest);
		filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
		if (rxq == -1)
			EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		else
			EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle1, id2));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
	}
	kfree(spec);
cleanup1:
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq 3
 * 2) client2: client2: Program same filter with  rxq=3,  pref = XLNX_EFCT_FILTER_F_PREF_QUEUE
 * 3) client1: Program another filter with rxq = 3, filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE
 * Note: Repeat test with rxq =  RX_CLS_FLOW_DISC(-1)
 * output: Filter program : PASS, PASS, fail
 */

TEST(auxtest, T023_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec;
	int i, rxq;
	int id1, id2;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);

        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	for (i = 0, rxq = 3; i < 2; i++, rxq = -1) {
		u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
		filter_params.flags = 0;
		spec->ring_cookie = rxq;
		filter_params.spec = spec;
		EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		id1 = filter_params.filter_id_out;
		EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		filter_params.flags = XLNX_EFCT_FILTER_F_PREF_QUEUE;
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
		id2 = filter_params.filter_id_out;
		EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		u64_to_ether_addr(0x000001234567abcd, spec->h_u.ether_spec.h_dest);
		filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
		if (rxq == -1)
			EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		else
			EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle1, id2));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
	}
	kfree(spec);
cleanup1:
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}


/* 1) client1: Program macvlan filter for rxq 3
 * 2) client2: Program same filter with  rxq=1,  pref = XLNX_EFCT_FILTER_F_PREF_QUEUE
 * 3) client1: Program another filter with rxq = 3, filter_params.flags = 0
 * Note: Repeat test with rxq =  RX_CLS_FLOW_DISC(-1)
 * output: Filter program : PASS, PASS, PASS
 *	   rxq= - 1: PASS, FAIL, PASS
 */

TEST(auxtest, T024_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec;
	int i, rxq;
	int id1, id2;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);

        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	for (i = 0, rxq = 3; i < 2; i++, rxq = -1) {
		u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
		filter_params.flags = 0;
		spec->ring_cookie = rxq;
		filter_params.spec = spec;
		EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		id1 = filter_params.filter_id_out;
		EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		filter_params.flags = XLNX_EFCT_FILTER_F_PREF_QUEUE;
		spec->ring_cookie = 1;
		if (rxq == -1) {
			EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
		} else {
			EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
			id2 = filter_params.filter_id_out;
			EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		}
		spec->ring_cookie = rxq;
		u64_to_ether_addr(0x000001234567abcd, spec->h_u.ether_spec.h_dest);
		filter_params.flags = 0;
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, filter_params.filter_id_out));
		if (rxq != -1)
			EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle1, id2));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
	}
	kfree(spec);
cleanup1:
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq 3
 * 2) client2: Program same filter with  XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE,  cpumask=0x03
 * 3) client1: Program another filter with rxq = 3, filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE
 * Note: Repeat test with rxq =  RX_CLS_FLOW_DISC(-1)
 * output: Filter program : rxq = 3 PASS, PASS, fail
 *			    rxq = -1 PASS, fail, PASS
 */

TEST(auxtest, T025_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;
	int i, rxq;
	int id1, id2;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup1);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	for (i = 0, rxq = 3; i < 2; i++, rxq = -1) {
		u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
		filter_params.flags = 0;
		spec->ring_cookie = rxq;
		filter_params.spec = spec;
		EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		id1 = filter_params.filter_id_out;
		EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE;
		cpumask_set_cpu(0, cpumask);
		cpumask_set_cpu(1, cpumask);
		filter_params.mask = cpumask;
		if (rxq == -1) {
			EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
		} else {
			EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
			id2 = filter_params.filter_id_out;
			EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		}
		spec->ring_cookie = rxq;
		filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
		u64_to_ether_addr(0x000001234567abcd, spec->h_u.ether_spec.h_dest);
		filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
		if (rxq == -1) {
			EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		} else {
			EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
			EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle1, id2));
		}
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
	}
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq 3
 * 2) client2: Program same filter with  XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE,  cpumask=0x05
 * 3) client1: Program another filter with rxq = 3, filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE, cpumask=0x05
 * Note: Repeat test with rxq =  RX_CLS_FLOW_DISC(-1)
 * output: Filter program : PASS, PASS, PASS
 *	   rxq = -1: PASS,FAIL,PASS
 */

TEST(auxtest, T026_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;
	int i, rxq;
	int id1, id2;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup1);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	for (i = 0, rxq = 3; i < 2; i++, rxq = -1) {
		u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
		filter_params.flags = 0;
		spec->ring_cookie = rxq;
		filter_params.spec = spec;
		EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		id1 = filter_params.filter_id_out;
		EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE;
		cpumask_set_cpu(0, cpumask);
		cpumask_set_cpu(3, cpumask);
		filter_params.mask = cpumask;
		filter_params.filter_handle = 0xffff;
		if (rxq == -1) {
			EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
		} else {
			EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
			id2 = filter_params.filter_id_out;
			EXPECT_INT_EQ(filter_params.rxq_out, rxq);
			EXPECT_INT_NE(filter_params.filter_handle, 0xffff);
		}
		spec->ring_cookie = rxq;
		u64_to_ether_addr(0x000001234567abcd, spec->h_u.ether_spec.h_dest);
		filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE;
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, filter_params.filter_id_out));
		if (rxq != -1)
			EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle1, id2));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
	}
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq 3
 * 2) client2: Program same filter with  XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT,  cpumask=0x03
 * 3) client1: Program another filter with rxq = 3, filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE
 * Note: Repeat test with rxq =  RX_CLS_FLOW_DISC(-1)
 * output: Filter program : PASS, fail, PASS
 */

TEST(auxtest, T027_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;
	int i, rxq;
	int id1;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup1);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	for (i = 0, rxq = 3; i < 2; i++, rxq = -1) {
		u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
		filter_params.flags = 0;
		spec->ring_cookie = rxq;
		filter_params.spec = spec;
		EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		id1 = filter_params.filter_id_out;
		EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
		cpumask_set_cpu(0, cpumask);
		cpumask_set_cpu(1, cpumask);
		filter_params.mask = cpumask;
		spec->ring_cookie = 0;
		EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
		spec->ring_cookie = rxq;
		u64_to_ether_addr(0x000001234567abcd, spec->h_u.ether_spec.h_dest);
		filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, filter_params.filter_id_out));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
	}
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}
int get_hugepage_count(int expected)
{
	rcu_barrier();
	return atomic_read(&num_of_hugepages_allocated);
}

/* 1) client1: Program macvlan filter for rxq 3
 * 2) client2: Program same filter with  XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT,  cpumask=0x05
 * 3) client1: Program another filter with rxq = 3, filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE
 * Note: Repeat test with rxq =  RX_CLS_FLOW_DISC(-1)
 * output: Filter program : PASS, PASS, fail
 *	   rxq = -1: PASS,fail,PASS
 */

TEST(auxtest, T028_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;
	int i, rxq;
	int id1,id2;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup1);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	for (i = 0, rxq = 3; i < 2; i++, rxq = -1) {
		u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
		filter_params.flags = 0;
		spec->ring_cookie = rxq;
		filter_params.spec = spec;
		EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		id1 = filter_params.filter_id_out;
		EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
		cpumask_set_cpu(0, cpumask);
		cpumask_set_cpu(3, cpumask);
		filter_params.mask = cpumask;
		if (rxq == -1) {
			EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
		} else {
			EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
			id2 = filter_params.filter_id_out;
			EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		}
		spec->ring_cookie = rxq;
		u64_to_ether_addr(0x000001234567abcd, spec->h_u.ether_spec.h_dest);
		filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
		if (rxq == -1)
			EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		else
			EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
		if (rxq != -1)
			EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle1, id2));
	}
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq 3
 * 2) client2: Program same filter with  XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT |  XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE,
		cpumask=0x05
 * 3) client1: Program another filter with rxq = 3, filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE
 * Note: Repeat test with rxq =  RX_CLS_FLOW_DISC(-1)
 * output: Filter program : PASS, fail, PASS
 */

TEST(auxtest, T029_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;
	int i, rxq;
	int id1;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup1);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	for (i = 0, rxq = 3; i < 2; i++, rxq = -1) {
		u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
		filter_params.flags = 0;
		spec->ring_cookie = rxq;
		filter_params.spec = spec;
		EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		id1 = filter_params.filter_id_out;
		EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT | XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
		cpumask_set_cpu(0, cpumask);
		cpumask_set_cpu(3, cpumask);
		filter_params.mask = cpumask;
		EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
		spec->ring_cookie = rxq;
		u64_to_ether_addr(0x000001234567abcd, spec->h_u.ether_spec.h_dest);
		filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, filter_params.filter_id_out));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
	}
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq 1
 * 2) client1: Program mac filter with:
 *	       cpumask=0x03, filter_params.flags =XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE | XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE
 * 3) client2: Program  filter same as in 1) with filter_params.flags =  XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE
 * output: Filter program : PASS, PASS,  fail
 */

TEST(auxtest, T030_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;
	int id1, id2;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup1);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	filter_params.flags = 0;
	spec->ring_cookie = 1;
	filter_params.spec = spec;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	id1 = filter_params.filter_id_out;
	EXPECT_INT_EQ(filter_params.rxq_out, 1);
	spec->h_ext.vlan_tci = 0;
	spec->m_ext.vlan_tci = 0;
	spec->ring_cookie = 0;
	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE | XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	cpumask_set_cpu(0, cpumask);
	cpumask_set_cpu(1, cpumask);
	filter_params.mask = cpumask;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	id2 = filter_params.filter_id_out;
	spec->ring_cookie = 1;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE;
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id2));
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq 1 with flag = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE
 * 2) client2: Program mac filter with rxq = 1, flag = XLNX_EFCT_FILTER_F_PREF_QUEUE
 * output: Filter program : PASS, PASS(rxq != 1)
 */

TEST(auxtest, T031_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	spec->ring_cookie = 1;
	filter_params.spec = spec;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	EXPECT_INT_EQ(filter_params.rxq_out, 1);
	spec->h_ext.vlan_tci = 0;
	spec->m_ext.vlan_tci = 0;
	spec->ring_cookie = 1;
	filter_params.flags = XLNX_EFCT_FILTER_F_PREF_QUEUE;
	filter_params.rxq_out = 0xffff;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	EXPECT_INT_NE(filter_params.rxq_out, 0xffff);
	EXPECT_INT_NE(filter_params.rxq_out, 0x1);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program filter on rxq 1 with flag =  XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE
 * 2) client2: Program same filter with flag = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT, mask = 0x03
 * output: Filter program : PASS, fail(rxq != 1)
 */

TEST(auxtest, T032_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	spec->ring_cookie = 1;
	filter_params.spec = spec;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	EXPECT_INT_EQ(filter_params.rxq_out, 1);
	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	cpumask_set_cpu(1, cpumask);
	filter_params.mask = cpumask;
	filter_params.rxq_out = 0xffff;
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq 3
 * 2) client2: Program same filter with  XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT | XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE,  cpumask=0x03
 * output: Filter program : PASS, fail(rxq != 1)
 */

TEST(auxtest, T033_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	filter_params.flags = 0;
	spec->ring_cookie = 3;
	filter_params.spec = spec;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	EXPECT_INT_EQ(filter_params.rxq_out, 3);
	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT | XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	cpumask_set_cpu(0, cpumask);
	cpumask_set_cpu(1, cpumask);
	filter_params.mask = cpumask;
	filter_params.rxq_out = 0xffff;
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq 3
 * 2) client2: Program same filter with  XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT | XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE,  cpumask=0x08
 * output: Filter program : PASS, fail(rxq != 1)
 */


TEST(auxtest, T034_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	spec->ring_cookie = 3;
	filter_params.spec = spec;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	EXPECT_INT_EQ(filter_params.rxq_out, 3);
	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT | XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	cpumask_set_cpu(0, cpumask);
	cpumask_set_cpu(3, cpumask);
	filter_params.mask = cpumask;
	filter_params.rxq_out = 0xffff;
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}
/* 1) client1: Program filter on rxq 1 with flag =  XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE
 * 2) client2:  Program same filter with flag =  XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE, mask = 0x03
 * output: Filter program : PASS, fail(rxq != 1)
 */

TEST(auxtest, T035_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	spec->ring_cookie = 1;
	filter_params.spec = spec;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	EXPECT_INT_EQ(filter_params.rxq_out, 1);
	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE;
	cpumask_set_cpu(0, cpumask);
	cpumask_set_cpu(1, cpumask);
	filter_params.mask = cpumask;
	filter_params.rxq_out = 0xffff;
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}
/* 1) client1: Program macvlan filter for rxq 1
 * 2) client2: Program same filter with  XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE | XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE,  cpumask=0x03
 * output: Filter program : PASS, fail(rxq != 1)
 */

TEST(auxtest, T036_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
	spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	filter_params.flags = XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	spec->ring_cookie = 1;
	filter_params.spec = spec;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	EXPECT_INT_EQ(filter_params.rxq_out, 1);
	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE | XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
	cpumask_set_cpu(0, cpumask);
	cpumask_set_cpu(1, cpumask);
	filter_params.mask = cpumask;
	filter_params.rxq_out = 0xffff;
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq 3
 * 2) client2: Program same filter with  rxq=3,  pref = XLNX_EFCT_FILTER_F_PREF_QUEUE | XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE
 * Note: Repeat test with rxq =  RX_CLS_FLOW_DISC(-1)
 * output: Filter program : PASS, fail, rxq = -1: PASS, PASS
 */

TEST(auxtest, T037_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec;
	int i, rxq;
	int id1;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);

        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	for (i = 0, rxq = 3; i < 2; i++, rxq = -1) {
		u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
		filter_params.flags = 0;
		spec->ring_cookie = rxq;
		filter_params.spec = spec;
		EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
		id1 = filter_params.filter_id_out;
		EXPECT_INT_EQ(filter_params.rxq_out, rxq);
		filter_params.flags = XLNX_EFCT_FILTER_F_PREF_QUEUE | XLNX_EFCT_FILTER_F_EXCLUSIVE_QUEUE;
		if (rxq == -1)
			EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
		else
			EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
	}
	kfree(spec);
cleanup1:
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq -1
 * 2) client2: Program same filter with  XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE,  cpumask=0x03
 * output: Filter program : PASS, fail,
 */

TEST(auxtest, T038_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;
	int id1;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup1);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	filter_params.flags = 0;
	spec->ring_cookie = -1;
	filter_params.spec = spec;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	id1 = filter_params.filter_id_out;
	EXPECT_INT_EQ(filter_params.rxq_out, -1);
	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_LOOSE;
	cpumask_set_cpu(0, cpumask);
	cpumask_set_cpu(1, cpumask);
	filter_params.mask = cpumask;
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq -1
 * 2) client2: Program same filter with  XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT,  cpumask=0x03
 * output: Filter program : PASS, fail,
 */

TEST(auxtest, T039_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	cpumask_var_t cpumask;
	int id1;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
	zalloc_cpumask_var(&cpumask, GFP_KERNEL);
	ASSERT_ADDR_NE_GOTO(cpumask, NULL, cleanup);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup1);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	filter_params.flags = 0;
	spec->ring_cookie = -1;
	filter_params.spec = spec;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	id1 = filter_params.filter_id_out;
	EXPECT_INT_EQ(filter_params.rxq_out, -1);
	filter_params.flags = XLNX_EFCT_FILTER_F_ANYQUEUE_STRICT;
	cpumask_set_cpu(0, cpumask);
	cpumask_set_cpu(1, cpumask);
	filter_params.mask = cpumask;
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
	free_cpumask_var(cpumask);
cleanup1:
	kfree(spec);
         EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

/* 1) client1: Program macvlan filter for rxq = -1
 * 2) client2: Program same filter with  rxq = -1
 * output: Filter program : PASS, PASS
 */

TEST(auxtest, T040_filter_op)
{
	struct xlnx_efct_filter_params filter_params;
	struct xlnx_efct_client *handle, *handle1;
	struct ethtool_rx_flow_spec *spec = NULL;
	int id1;

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle);
        ASSERT_ADDR_NE(handle, NULL);
        ASSERT_INT_EQ(IS_ERR(handle), 0);
        handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &handle1);
        ASSERT_ADDR_NE(handle1, NULL);
        ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);
        spec = kzalloc(sizeof(*spec), GFP_ATOMIC);
        ASSERT_ADDR_NE_GOTO(spec, NULL, cleanup1);
	u64_to_ether_addr(0x0000ffffffffffff, spec->m_u.ether_spec.h_dest);
	spec->flow_type = ETHER_FLOW | FLOW_EXT; //ether
	spec->location = RX_CLS_LOC_ANY;
	spec->h_ext.vlan_tci = htons(0x3);
	spec->m_ext.vlan_tci = htons(0x0fff);
	u64_to_ether_addr(0x000001234567efef, spec->h_u.ether_spec.h_dest);
	filter_params.flags = 0;
	spec->ring_cookie = -1;
	filter_params.spec = spec;
	EXPECT_INT_EQ(true, xlnx_efct_dev->ops->is_filter_supported(handle, spec));
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle, &filter_params));
	id1 = filter_params.filter_id_out;
	EXPECT_INT_EQ(filter_params.rxq_out, -1);
	filter_params.rxq_out = 0xABAB;
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_insert(handle1, &filter_params));
	EXPECT_INT_EQ(filter_params.rxq_out, -1);
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->filter_remove(handle, id1));
cleanup1:
	kfree(spec);
        EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}


TEST(auxtest, T001A_probe_open)
{
	struct xlnx_efct_rxq_params rxq_params;
	struct xlnx_efct_client *handle_client;
	unsigned long client_hndl_test;
	int qid = 0;

	ASSERT_ADDR_NE(aux_dev, NULL);
	ASSERT_ADDR_NE(client_handle, NULL);

	handle_client = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_INT_EQ(IS_ERR(handle_client), 0);
	ASSERT_ADDR_NE(handle_client, NULL);

	rxq_params.qid = 0;
	rxq_params.n_hugepages = 1;

	atomic_set(&num_of_hugepages_allocated, 0);
	qid = xlnx_efct_dev->ops->bind_rxq(handle_client, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup);

	/* For first ref bind, expect 1 extra huge page */
	EXPECT_INT_EQ(2, get_hugepage_count(2));

	xlnx_efct_dev->ops->free_rxq(handle_client, qid, 1);

	EXPECT_INT_EQ(0, get_hugepage_count(0));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle_client));
}

TEST(auxtest, T001_probe_open)
{
	struct xlnx_efct_client *handle, *handle1;
	unsigned long client_hndl_test;

	ASSERT_ADDR_NE(aux_dev, NULL);

	handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_INT_EQ(IS_ERR(handle), 0);
	handle1 = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_INT_EQ_GOTO(IS_ERR(handle1), 0, cleanup);

	EXPECT_ADDR_NE(handle, handle1);
	EXPECT_ADDR_NE(handle, NULL);
	EXPECT_ADDR_NE(handle, client_handle);

	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle));
}

TEST(auxtest, T001B_probe_open)
{
	struct xlnx_efct_rxq_params rxq_params;
	struct xlnx_efct_client *handle_client;
	unsigned long client_hndl_test;
	int qid = 0;

	ASSERT_ADDR_NE(aux_dev, NULL);

	ASSERT_ADDR_NE(client_handle, NULL);

	handle_client = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_ADDR_NE(handle_client, NULL);
	ASSERT_INT_EQ(IS_ERR(handle_client), 0);

	rxq_params.qid = 0;
	rxq_params.n_hugepages = 1;

	atomic_set(&num_of_hugepages_allocated, 0);
	qid = xlnx_efct_dev->ops->bind_rxq(handle_client, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup);

	EXPECT_INT_EQ(1, get_hugepage_count(1));
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle_client));
	EXPECT_INT_EQ(0, get_hugepage_count(0));
}

TEST(auxtest, N001_probe_open)
{
	unsigned long client_hndl_test;
	EXPECT_ADDR_EQ(-EINVAL, xlnx_efct_dev->ops->open(NULL, &auxtest_drv_ops, &client_hndl_test));

	EXPECT_ADDR_EQ(-EINVAL, xlnx_efct_dev->ops->open(aux_dev, NULL, &client_hndl_test));

	EXPECT_ADDR_EQ(-EINVAL, xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, NULL));

	EXPECT_ADDR_EQ(-EINVAL, xlnx_efct_dev->ops->open(NULL, NULL, NULL));
}

TEST(auxtest, T002_get_param)
{
	union xlnx_efct_param_value param = { 0 };
	int i;

	ASSERT_ADDR_NE(client_handle, NULL);

	for (i = XLNX_EFCT_CONFIG_MEM; i <= XLNX_EFCT_DESIGN_PARAM; ++i) {
		ASSERT_INT_EQ(0, xlnx_efct_dev->ops->get_param(client_handle, i, &param));
		switch (i) {
		case XLNX_EFCT_VARIANT:
			tlog(T_INFO, "variant 0x%x\n", param.variant);
			break;
		case XLNX_EFCT_REVISION:
			tlog(T_INFO, "value 0x%x\n", param.value);
			break;
		case XLNX_EFCT_NIC_RESOURCES:
			tlog(T_INFO, "evq base 0x%x\n", param.nic_res.evq_min);
			tlog(T_INFO, "evq limit 0x%x\n", param.nic_res.evq_lim);
			tlog(T_INFO, "txq base 0x%x\n", param.nic_res.txq_min);
			tlog(T_INFO, "txq limit 0x%x\n", param.nic_res.txq_lim);
			break;
		case XLNX_EFCT_EVQ_WINDOW:
			tlog(T_INFO, "evq stride 0x%lx\n", param.evq_window.stride);
			tlog(T_INFO, "evq base address 0x%llx\n", param.evq_window.base);
			break;
		case XLNX_EFCT_DESIGN_PARAM:
			tlog(T_INFO, "evq_sizes 0x%x\n", param.design_params.evq_sizes);
			tlog(T_INFO, "frame_offset_fixed 0x%x\n", param.design_params.frame_offset_fixed);
			tlog(T_INFO, "l4_csum_proto 0x%x\n", param.design_params.l4_csum_proto);
			tlog(T_INFO, "max_runt 0x%x\n", param.design_params.max_runt);
			tlog(T_INFO, "num_filter 0x%x\n", param.design_params.num_filter);
			tlog(T_INFO, "rx_buffer_len 0x%x\n", param.design_params.rx_buffer_len);
			tlog(T_INFO, "rx_buf_fifo_size 0x%x\n", param.design_params.rx_buf_fifo_size);
			tlog(T_INFO, "rx_metadata_len 0x%x\n", param.design_params.rx_metadata_len);
			tlog(T_INFO, "rx_queues 0x%x\n", param.design_params.rx_queues);
			tlog(T_INFO, "rx stride 0x%x\n", param.design_params.rx_stride);
			tlog(T_INFO, "ts_subnano_bit 0x%x\n", param.design_params.ts_subnano_bit);
			tlog(T_INFO, "tx_apertures 0x%x\n", param.design_params.tx_apertures);
			tlog(T_INFO, "tx_aperture_size 0x%x\n", param.design_params.tx_aperture_size);
			tlog(T_INFO, "tx_fifo_size 0x%x\n", param.design_params.tx_fifo_size);
			tlog(T_INFO, "tx_max_reorder 0x%x\n", param.design_params.tx_max_reorder);
			tlog(T_INFO, "unsol_credit_seq_mask 0x%x\n", param.design_params.unsol_credit_seq_mask);
		default:
			break;
		}
	}
}

TEST(auxtest, N002_get_param)
{
	union xlnx_efct_param_value param = { 0 };

	EXPECT_INT_NE(0, xlnx_efct_dev->ops->get_param(NULL, XLNX_EFCT_CONFIG_MEM, &param));

	EXPECT_INT_NE(0, xlnx_efct_dev->ops->get_param(client_handle, XLNX_EFCT_CONFIG_MEM - 1, &param));

	EXPECT_INT_NE(0, xlnx_efct_dev->ops->get_param(client_handle, XLNX_EFCT_DESIGN_PARAM + 1, &param));

	EXPECT_INT_NE(0, xlnx_efct_dev->ops->get_param(client_handle, XLNX_EFCT_CONFIG_MEM, NULL));
}

TEST(auxtest, T003A_init_evq_n_id)
{
	union xlnx_efct_param_value param = { 0 };
	unsigned int minq, maxq, i;
	struct xlnx_efct_client *handle_client;
	unsigned long client_hndl_test;
	struct xlnx_efct_evq_params evq_params = { 0 };
	struct page *pg;

	handle_client = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_ADDR_NE(handle_client, NULL);
	ASSERT_INT_EQ(IS_ERR(handle_client), 0);

	evq_params.subscribe_time_sync = false;
	evq_params.page_offset = 0;
	evq_params.q_size = PAGE_SIZE;
	evq_params.entries = (evq_params.q_size / 8);
	/* Get queue parameters */
	ASSERT_INT_EQ_GOTO(0, (xlnx_efct_dev->ops->get_param(handle_client, XLNX_EFCT_NIC_RESOURCES, &param)), cleanup1);

	/* Test queue range */
	minq = param.nic_res.evq_min;
	maxq = param.nic_res.evq_lim;
	pg = alloc_pages(__GFP_COMP | GFP_KERNEL, order_base_2(maxq - minq));
	ASSERT_ADDR_NE_GOTO(NULL, pg, cleanup1);
	for (i = minq; i < maxq; i++) {
		evq_params.qid = i;
		evq_params.q_page = pg + (i - minq);
		ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->init_evq(handle_client, &evq_params), cleanup);
	}

cleanup:
	for (minq = param.nic_res.evq_min; minq < i; minq++)
		xlnx_efct_dev->ops->free_evq(handle_client, minq);
	put_page(pg);
	evq_params.q_page = NULL;
cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle_client));
}

TEST(auxtest, T003B_init_evq_n_entries)
{
	int order;
	union xlnx_efct_param_value param = { 0 };
	unsigned int i, entries[] = { 128, 256, 512, 1024, 2048, 4096 };
	unsigned long client_hndl_test;
	struct xlnx_efct_client *handle_client;

	handle_client = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_ADDR_NE(handle_client, NULL);
	ASSERT_INT_EQ(IS_ERR(handle_client), 0);

	/* Get queue parameters */
	ASSERT_INT_EQ_GOTO(0, (xlnx_efct_dev->ops->get_param(handle_client, XLNX_EFCT_NIC_RESOURCES, &param)), cleanup);

	for (i = 0; i < ARRAY_SIZE(entries); ++i) {
		struct xlnx_efct_evq_params evq_params = { 0 };
		evq_params.qid = param.nic_res.evq_min;
		evq_params.page_offset = 0;
		evq_params.q_size = entries[i] * 8;
		evq_params.entries = entries[i];

		if (evq_params.q_size < PAGE_SIZE)
			order = 0;
		else
			order = (evq_params.q_size / PAGE_SIZE);
		evq_params.q_page = alloc_pages(__GFP_COMP | GFP_KERNEL, order);
		ASSERT_ADDR_NE_GOTO(NULL, evq_params.q_page, cleanup);

		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->init_evq(handle_client, &evq_params));
		xlnx_efct_dev->ops->free_evq(handle_client, param.nic_res.evq_min);
		put_page(evq_params.q_page);
		evq_params.q_page = NULL;
	}
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle_client));
}

TEST(auxtest, N003_init_evq)
{
	union xlnx_efct_param_value param = { 0 };
	struct xlnx_efct_evq_params evq_params = { 0 };
	struct xlnx_efct_client *handle_client;
	unsigned long client_hndl_test;

	handle_client = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_ADDR_NE(handle_client, NULL);
	ASSERT_INT_EQ(IS_ERR(handle_client), 0);

	evq_params.q_page = alloc_pages(__GFP_COMP | GFP_KERNEL, 0);//allocating one page
	ASSERT_ADDR_NE_GOTO(NULL, evq_params.q_page, cleanup);

	evq_params.qid = 2;
	evq_params.page_offset = 0;
	evq_params.q_size = PAGE_SIZE;
	evq_params.entries = (evq_params.q_size / 8);

	EXPECT_INT_NE(0, (xlnx_efct_dev->ops->init_evq(NULL, &evq_params)));
	EXPECT_INT_NE(0, (xlnx_efct_dev->ops->init_evq(handle_client, NULL)));
	EXPECT_INT_NE(0, (xlnx_efct_dev->ops->init_evq(NULL, NULL)));

	/* Get queue parameters */
	ASSERT_INT_EQ_GOTO(0, (xlnx_efct_dev->ops->get_param(handle_client, XLNX_EFCT_NIC_RESOURCES, &param)), cleanup1);

	/* Invalid n desc size */
	evq_params.qid = param.nic_res.evq_min;
	evq_params.entries = 777;

	EXPECT_INT_NE(0, (xlnx_efct_dev->ops->init_evq(handle_client, &evq_params)));

	/* Invalid qid */
	evq_params.qid = param.nic_res.evq_lim + 1;
	evq_params.entries = (evq_params.q_size / 8);
	EXPECT_INT_NE(0, (xlnx_efct_dev->ops->init_evq(handle_client, &evq_params)));
cleanup1:
	/* Free invalid event queue */
	xlnx_efct_dev->ops->free_evq(NULL, -1);
	xlnx_efct_dev->ops->free_evq(handle_client, -1);
	put_page(evq_params.q_page);
	evq_params.q_page = NULL;
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle_client));
}

TEST(auxtest, T007_bind_free_rxq)
{
	struct xlnx_efct_rxq_params rxq_params;
	int qid = 0, arr_size = 300;

	struct xlnx_efct_hugepage *page = kzalloc(arr_size * sizeof(struct xlnx_efct_hugepage), GFP_KERNEL);
	ASSERT_ADDR_NE(page, NULL);

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE_GOTO(g_client_handle, NULL, cleanup0);
	ASSERT_INT_EQ_GOTO(IS_ERR(g_client_handle), 0, cleanup0);

	rxq_params.qid = qid;
	rxq_params.n_hugepages = 1;

	atomic_set(&num_of_hugepages_allocated, 0);

	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);

	EXPECT_INT_EQ(2, get_hugepage_count(2));

	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, qid, page, arr_size), cleanup1);
	//tlog(T_INFO, "page address is %p for qid %d", page.page, qid);

	poll_release_sb = 1;

	EXPECT_INT_EQ(2, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);

	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, qid));

	msleep(1000);
	xlnx_efct_dev->ops->free_rxq(g_client_handle, qid, 1);

	poll_release_sb = 0;

	/* Expect all HP freed on last free_rxq() call */
	EXPECT_INT_EQ(0, get_hugepage_count(0));
cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
cleanup0:
	kfree(page);
}

TEST(auxtest, T008A_bind_rollover_free_rxq)
{
	struct xlnx_efct_rxq_params rxq_params;
	int qid = 0, arr_size = 300;

	struct xlnx_efct_hugepage *page = kzalloc(arr_size * sizeof(struct xlnx_efct_hugepage), GFP_KERNEL);
	ASSERT_ADDR_NE(page, NULL);

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE_GOTO(g_client_handle, NULL, cleanup0);
	ASSERT_INT_EQ_GOTO(IS_ERR(g_client_handle), 0, cleanup0);

	rxq_params.qid = qid;
	rxq_params.n_hugepages = 2;

	atomic_set(&num_of_hugepages_allocated, 0);
	buffer_start_count = 0;
	poll_count = 0;

	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);

	EXPECT_INT_EQ(3, get_hugepage_count(3));

	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, qid, page, arr_size));
	//tlog(T_INFO, "page address is %p for qid %d", page.page, qid);

	poll_release_sb = 1;

	EXPECT_INT_EQ(3, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);

	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, qid));

	msleep(1000);

	EXPECT_INT_EQ(4, buffer_start_count);
	xlnx_efct_dev->ops->free_rxq(g_client_handle, qid, 2);

	poll_release_sb = 0;

	/* Expect all HP freed on last free_rxq() call */
	EXPECT_INT_EQ(0, get_hugepage_count(0));

cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));

	buffer_start_count = 0;
	poll_count = 0;
cleanup0:
	kfree(page);
}

TEST(auxtest, T008B_bind_rollover_free_rxq)
{
	struct xlnx_efct_rxq_params rxq_params;
	int qid = 0, arr_size = 300;

	struct xlnx_efct_hugepage *page = kzalloc(arr_size * sizeof(struct xlnx_efct_hugepage), GFP_KERNEL);
	ASSERT_ADDR_NE(page, NULL);

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE_GOTO(g_client_handle, NULL, cleanup0);
	ASSERT_INT_EQ_GOTO(IS_ERR(g_client_handle), 0, cleanup0);

	rxq_params.qid = qid;
	rxq_params.n_hugepages = 1;

	atomic_set(&num_of_hugepages_allocated, 0);
	buffer_start_count = 0;
	poll_count = 0;

	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);

	EXPECT_INT_EQ(2, get_hugepage_count(2));

	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, qid, page, arr_size), cleanup1);
	//tlog(T_INFO, "page address is %p for qid %d", page.page, qid);

	poll_release_sb = 1;

	EXPECT_INT_EQ(2, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);

	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, qid));

	msleep(1000);

	EXPECT_INT_EQ(4, buffer_start_count);
	xlnx_efct_dev->ops->free_rxq(g_client_handle, qid, 1);

	poll_release_sb = 0;

	/* Expect all HP freed on last free_rxq() call */
	EXPECT_INT_EQ(0, get_hugepage_count(0));

cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));

	buffer_start_count = 0;
	poll_count = 0;
cleanup0:
	kfree(page);
}

TEST(auxtest, T008C_bind_rollover_free_rxq)
{
	struct xlnx_efct_rxq_params rxq_params;
	int qid = 0, arr_size = 300;

	struct xlnx_efct_hugepage *page = kzalloc(arr_size * sizeof(struct xlnx_efct_hugepage), GFP_KERNEL);
	ASSERT_ADDR_NE(page, NULL);

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE_GOTO(g_client_handle, NULL, cleanup0);
	ASSERT_INT_EQ_GOTO(IS_ERR(g_client_handle), 0, cleanup0);

	rxq_params.qid = qid;
	rxq_params.n_hugepages = 6;

	atomic_set(&num_of_hugepages_allocated, 0);
	buffer_start_count = 0;
	poll_count = 0;

	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);

	EXPECT_INT_EQ(7, get_hugepage_count(7));
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, qid, page, arr_size), cleanup1);
	//tlog(T_INFO, "page address is %p for qid %d", page.page, qid);
	poll_release_sb = 1;

	EXPECT_INT_EQ(7, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);

	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, qid));

	msleep(1000);

	EXPECT_INT_EQ(4, buffer_start_count);
	xlnx_efct_dev->ops->free_rxq(g_client_handle, qid, 6);

	poll_release_sb = 0;

	/* Expect all HP freed on last free_rxq() call */
	EXPECT_INT_EQ(0, get_hugepage_count(0));

cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));

	buffer_start_count = 0;
	poll_count = 0;
cleanup0:
	kfree(page);
}

TEST(auxtest, T007B_bind_rxq_n_qid)
{
	unsigned int i, qids[] = { 0, 1, 2, 3, 4, 5, 6, 7};
	struct xlnx_efct_rxq_params rxq_params = { 0 };
	int qid, limit, arr_size = 300;

	struct xlnx_efct_hugepage *page = kzalloc(arr_size * sizeof(struct xlnx_efct_hugepage), GFP_KERNEL);
	ASSERT_ADDR_NE(page, NULL);

	limit = num_online_cpus();

	if (limit > ARRAY_SIZE(qids))
		limit = ARRAY_SIZE(qids);

	ASSERT_ADDR_NE_GOTO(client_handle, NULL, cleanup0);
	rxq_params.n_hugepages = 1;
	for (i = 0; i < limit; i++) {
		rxq_params.qid = qids[i];

		qid = xlnx_efct_dev->ops->bind_rxq(client_handle, &rxq_params);
		ASSERT_INT_GT_GOTO(qid, -1, cleanup0);
		tlog(T_INFO, "Rx Qid = 0x%x\n", qid);

		/* Verify that the alloc_hugepage function in the driver gets called. */
		EXPECT_INT_NE(-1, xlnx_efct_dev->ops->get_hugepages(client_handle, qid, page, arr_size));
		EXPECT_INT_EQ(2, verify_hugepages(page, arr_size));
		decref_hugepages(page, arr_size);

		xlnx_efct_dev->ops->free_rxq(client_handle, qid, rxq_params.n_hugepages);
	}
cleanup0:
	kfree(page);
}

TEST(auxtest, T007B_bind_rxq_release_buf)
{
	struct xlnx_efct_rxq_params rxq_params = { 0 };
	int qid;

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE(g_client_handle, NULL);
	ASSERT_INT_EQ(IS_ERR(g_client_handle), 0);

	rxq_params.qid = 0;
	rxq_params.n_hugepages = 1;

	atomic_set(&num_of_hugepages_allocated, 0);
	poll_release_sb = 1;
	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup);
	tlog(T_INFO, "Rx Qid = 0x%x\n", qid);

	/* For first ref bind, expect 1 extra huge page */
	EXPECT_INT_EQ(2, get_hugepage_count(2));

	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, qid));

	msleep(1000);
	xlnx_efct_dev->ops->free_rxq(g_client_handle, qid, 1);

	poll_release_sb = 0;

	EXPECT_INT_EQ(0, get_hugepage_count(0));

cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
}

TEST(auxtest, T007C_bind_rxq)
{
	struct xlnx_efct_rxq_params rxq_params = { 0 };
	int qid, arr_size = 300;

	struct xlnx_efct_hugepage *page = kzalloc(arr_size * sizeof(struct xlnx_efct_hugepage), GFP_KERNEL);
	ASSERT_ADDR_NE(page, NULL);

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE_GOTO(g_client_handle, NULL, cleanup0);
	ASSERT_INT_EQ_GOTO(IS_ERR(g_client_handle), 0, cleanup0);

	rxq_params.qid = 0;
	rxq_params.n_hugepages = 0;

	atomic_set(&num_of_hugepages_allocated, 0);
	/*passing zero huge pages*/
	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);
	tlog(T_INFO, "Rx Qid = 0x%x\n", qid);

	EXPECT_INT_EQ(0, get_hugepage_count(0));

	xlnx_efct_dev->ops->free_rxq(g_client_handle, qid, 0);

	rxq_params.n_hugepages = 2;

	atomic_set(&num_of_hugepages_allocated, 0);
	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);
	tlog(T_INFO, "Rx Qid = 0x%x\n", qid);

	EXPECT_INT_EQ(3, get_hugepage_count(3));
	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->get_hugepages(g_client_handle, qid, page, arr_size));

	buffer_start_release_sb = 1;

	EXPECT_INT_EQ(3, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);

	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, qid));

	msleep(1000);

	xlnx_efct_dev->ops->free_rxq(g_client_handle, qid, 2);

	EXPECT_INT_EQ(0, get_hugepage_count(0));
	buffer_start_release_sb = 0;

	/*calling bind_rxq() multiple times*/
	rxq_params.n_hugepages = 1;

	//atomic_set(&num_of_hugepages_allocated, 0);
	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);
	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);

	EXPECT_INT_EQ(3, get_hugepage_count(3));

	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->get_hugepages(g_client_handle, qid, page, arr_size));

	buffer_start_release_sb = 1;

	EXPECT_INT_EQ(3, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);

	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, qid));

	msleep(1000);

	xlnx_efct_dev->ops->free_rxq(g_client_handle, qid, 1);
	xlnx_efct_dev->ops->free_rxq(g_client_handle, qid, 1);
	buffer_start_release_sb = 0;

	EXPECT_INT_EQ(0, get_hugepage_count(0));

	msleep(1000);
cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));

	EXPECT_INT_EQ(0, get_hugepage_count(0));
cleanup0:
	kfree(page);
}

TEST(auxtest, T007D_bind_free_rxq_huge_pages)
{
	struct xlnx_efct_rxq_params rxq_params;
	int qid = 0, arr_size = 300;

	struct xlnx_efct_hugepage *page = kzalloc(arr_size * sizeof(struct xlnx_efct_hugepage), GFP_KERNEL);
	ASSERT_ADDR_NE(page, NULL);

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE_GOTO(g_client_handle, NULL, cleanup0);
	ASSERT_INT_EQ_GOTO(IS_ERR(g_client_handle), 0, cleanup0);

	atomic_set(&num_of_hugepages_allocated, 0);
	rxq_params.qid = 0;
	rxq_params.n_hugepages = 257;
	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_EQ_GOTO(rxq_params.qid, qid, cleanup1);
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(258, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);
cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
	EXPECT_INT_EQ(0, get_hugepage_count(0));
cleanup0:
	kfree(page);

}

TEST(auxtest, T007E_bind_rollover_close)
{
	struct xlnx_efct_rxq_params rxq_params;
	int qid = 0, arr_size = 300;

	struct xlnx_efct_hugepage *page = kzalloc(arr_size * sizeof(struct xlnx_efct_hugepage), GFP_KERNEL);
	ASSERT_ADDR_NE(page, NULL);

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE_GOTO(g_client_handle, NULL, cleanup0);
	ASSERT_INT_EQ_GOTO(IS_ERR(g_client_handle), 0, cleanup0);

	rxq_params.qid = qid;
	rxq_params.n_hugepages = 1;

	atomic_set(&num_of_hugepages_allocated, 0);

	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);

	EXPECT_INT_EQ(2, get_hugepage_count(2));

	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	//tlog(T_INFO, "page address is %p for qid %d", page.page, qid);

	poll_release_sb = 1;

	EXPECT_INT_EQ(2, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);

	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, 0));

	msleep(1000);

	poll_release_sb = 0;

	/* As after rollover, free_alloc_hugepages will not be called*/
	EXPECT_INT_EQ(2, get_hugepage_count(2));
cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
	EXPECT_INT_EQ(0, get_hugepage_count(0));
cleanup0:
	kfree(page);
}

TEST(auxtest, T007F_bind_free_rxq)
{
	struct xlnx_efct_rxq_params rxq_params;
	int qid = 0, arr_size = 300;

	struct xlnx_efct_hugepage *page = kzalloc(arr_size * sizeof(struct xlnx_efct_hugepage), GFP_KERNEL);
	ASSERT_ADDR_NE(page, NULL);

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE_GOTO(g_client_handle, NULL, cleanup0);
	ASSERT_INT_EQ_GOTO(IS_ERR(g_client_handle), 0, cleanup0);

	atomic_set(&num_of_hugepages_allocated, 0);
	rxq_params.qid = 0;
	rxq_params.n_hugepages = 4;
	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(5, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);

	rxq_params.qid = 0;
	rxq_params.n_hugepages = 4;
	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);
	poll_release_sb = 1;
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(9, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);

	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, 0));

	msleep(1000);
	poll_release_sb = 0;
	xlnx_efct_dev->ops->free_rxq(g_client_handle, 0, 2);
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(7, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);

	xlnx_efct_dev->ops->free_rxq(g_client_handle, 0, 6);
	msleep(1000);
cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));

	EXPECT_INT_EQ(0, get_hugepage_count(0));
cleanup0:
	kfree(page);

}

TEST(auxtest, T007G_multiple_rollover)
{
	struct xlnx_efct_rxq_params rxq_params;
	int qid = 0, arr_size = 300;

	struct xlnx_efct_hugepage *page = kzalloc(arr_size * sizeof(struct xlnx_efct_hugepage), GFP_KERNEL);
	ASSERT_ADDR_NE(page, NULL);

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE_GOTO(g_client_handle, NULL, cleanup0);
	ASSERT_INT_EQ_GOTO(IS_ERR(g_client_handle), 0, cleanup0);

	atomic_set(&num_of_hugepages_allocated, 0);
	rxq_params.qid = 0;
	rxq_params.n_hugepages = 2;
	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);
	poll_release_sb = 1;
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(3, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);
	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, 0));
	msleep(1000);

	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(5, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);
	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, 0));
	msleep(1000);

	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(7, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);
	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, 0));
	msleep(1000);
	xlnx_efct_dev->ops->free_rxq(g_client_handle, 0, 2);
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(5, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);

	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(7, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);
	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, 0));
	msleep(1000);

	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(9, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);
	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(g_client_handle, 0));
	msleep(1000);

	xlnx_efct_dev->ops->free_rxq(g_client_handle, 0, 2);
	msleep(1000);
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(7, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);
	xlnx_efct_dev->ops->free_rxq(g_client_handle, 0, 2);
	msleep(1000);
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(5, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);
	xlnx_efct_dev->ops->free_rxq(g_client_handle, 0, 2);
	msleep(1000);
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(3, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);
	xlnx_efct_dev->ops->free_rxq(g_client_handle, 0, 2);
	msleep(1000);
	poll_release_sb = 0;
cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));

	EXPECT_INT_EQ(0, get_hugepage_count(0));
cleanup0:
	kfree(page);

}

TEST(auxtest, T007H_bind_free_free)
{
	struct xlnx_efct_rxq_params rxq_params;
	int qid = 0, arr_size = 300;

	struct xlnx_efct_hugepage *page = kzalloc(arr_size * sizeof(struct xlnx_efct_hugepage), GFP_KERNEL);
	ASSERT_ADDR_NE(page, NULL);

	g_client_handle = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &g_client_handle);
	ASSERT_ADDR_NE_GOTO(g_client_handle, NULL, cleanup0);
	ASSERT_INT_EQ_GOTO(IS_ERR(g_client_handle), 0, cleanup0);

	atomic_set(&num_of_hugepages_allocated, 0);
	rxq_params.qid = 0;
	rxq_params.n_hugepages = 6;
	qid = xlnx_efct_dev->ops->bind_rxq(g_client_handle, &rxq_params);
	ASSERT_INT_GE_GOTO(qid, 0, cleanup1);
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(g_client_handle, 0, page, arr_size), cleanup1);
	EXPECT_INT_EQ(7, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);

	xlnx_efct_dev->ops->free_rxq(g_client_handle, 0, 4);
	/*Observe manual error here as free_rxq() does not return anything*/
	xlnx_efct_dev->ops->free_rxq(g_client_handle, 0, 4);
	EXPECT_INT_EQ(0, get_hugepage_count(0));
cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(g_client_handle));
cleanup0:
	kfree(page);
}

TEST(auxtest, N007B_bind_rxq)
{
	struct xlnx_efct_rxq_params rxq_params = { 0 };
	struct xlnx_efct_hugepage *page;
	int qid, arr_size = 300;

	ASSERT_ADDR_NE(client_handle, NULL);

	page = kzalloc(arr_size * sizeof(struct xlnx_efct_hugepage), GFP_KERNEL);
	ASSERT_ADDR_NE(page, NULL);

	rxq_params.qid = 0;
	rxq_params.n_hugepages = 1;

	qid = xlnx_efct_dev->ops->bind_rxq(client_handle, &rxq_params);
	ASSERT_INT_GT_GOTO(qid, -1, cleanup0);
	tlog(T_INFO, "Rx Qid = 0x%x\n", qid);

	/* Verify that the alloc_hugepage function in the driver gets called. */

	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->get_hugepages(client_handle, qid, page, arr_size));
	/* For first ref bind, expect 1 extra huge page */
	EXPECT_INT_EQ(2, verify_hugepages(page, arr_size));
	//tlog(T_INFO, "page address is %p for qid %d", page.page, qid);
	decref_hugepages(page, arr_size);

	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(NULL, qid));
	EXPECT_INT_NE(-1, xlnx_efct_dev->ops->rollover_rxq(client_handle, -1));
	/*passing invalid params to release_superbuf*/
	xlnx_efct_dev->ops->release_superbuf(NULL, qid, 0);
	xlnx_efct_dev->ops->release_superbuf(client_handle, 3, 0);
	xlnx_efct_dev->ops->release_superbuf(client_handle, qid, -1);

	xlnx_efct_dev->ops->free_rxq(client_handle, qid, 1);
cleanup0:
	kfree(page);
}

TEST(auxtest, N007_get_huge_page)
{
	int qid = 0, arr_size = 300;
	struct xlnx_efct_hugepage *page;
	struct xlnx_efct_rxq_params rxq_params = { 0 };

	ASSERT_ADDR_NE(client_handle, NULL);

	page = kzalloc(arr_size * sizeof(struct xlnx_efct_hugepage), GFP_KERNEL);
	ASSERT_ADDR_NE(page, NULL);

	rxq_params.qid = qid;
	rxq_params.n_hugepages = 4;
	qid = xlnx_efct_dev->ops->bind_rxq(client_handle, &rxq_params);
	pr_info("qid = %d\n", qid);
	ASSERT_INT_GT_GOTO(qid, -1, cleanup0);

	/* Passing hugpages less than hugepges requested */
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->get_hugepages(client_handle, qid, page, 1));
	/* Passing hugpages more than hugepges requested */
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(client_handle, qid, page, arr_size), cleanup1);
	/* For first ref bind, expect 1 extra huge page */
	EXPECT_INT_EQ(5, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);
	/*Free the RXQ and bind again*/
	xlnx_efct_dev->ops->free_rxq(client_handle, qid, 4);
	rxq_params.n_hugepages = 1;
	qid = xlnx_efct_dev->ops->bind_rxq(client_handle, &rxq_params);
	pr_info("qid = %d\n", qid);
	ASSERT_INT_GT_GOTO(qid, -1, cleanup0);
	/* Passing client_handle NULL */
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->get_hugepages(NULL, qid, page, 1));
	/* Passing invalid QID */
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->get_hugepages(client_handle, 2, page, 1));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->get_hugepages(client_handle, 1, page, 1));
	/*Testing for valid queue ID*/
	pr_info("qid = %d\n", qid);
	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->get_hugepages(client_handle, qid, page, arr_size), cleanup1);
	/* For first ref bind, expect 1 extra huge page */
	EXPECT_INT_EQ(2, verify_hugepages(page, arr_size));
	decref_hugepages(page, arr_size);

cleanup1:
	xlnx_efct_dev->ops->free_rxq(client_handle, qid, 1);
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->get_hugepages(client_handle, qid, page, arr_size));
cleanup0:
	kfree(page);
}

TEST(auxtest, N007_bind_rxq)
{
	int qid = 0;
	struct xlnx_efct_rxq_params rxq_params = { 0 };

	ASSERT_ADDR_NE(client_handle, NULL);

	//TODO FIXME : Passing rxq_params gives larger frame error -Wframe-larger-than=
	//EXPECT_INT_NE(0, xlnx_efct_dev->ops->bind_rxq(NULL, &rxq_params));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->bind_rxq(client_handle, NULL));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->bind_rxq(NULL, NULL));

	/*passing invalid qid*/
	rxq_params.qid = 1024;
	rxq_params.n_hugepages = 1;
	EXPECT_INT_LT(xlnx_efct_dev->ops->bind_rxq(client_handle, &rxq_params), 0);

	/* Testing free_rxq() with invalid args*/
	rxq_params.qid = 0;
	rxq_params.n_hugepages = 1;
	qid = xlnx_efct_dev->ops->bind_rxq(client_handle, &rxq_params);
	ASSERT_INT_GT(qid, -1);

	/* Invalid client pointer */
	xlnx_efct_dev->ops->free_rxq(NULL, -1, 0);
	/*Invalid queue ID*/
	xlnx_efct_dev->ops->free_rxq(client_handle, -1, 0);
	xlnx_efct_dev->ops->free_rxq(client_handle, 0, 2);
	/*passing removed queue ID*/
	xlnx_efct_dev->ops->free_rxq(client_handle, 0, 1);
}

TEST(auxtest, T005_init_txq)
{
	int qid;
	union xlnx_efct_param_value param = { 0 };
	struct xlnx_efct_evq_params evq_params = { 0 };
	struct xlnx_efct_txq_params txq_params = { 0 };
	struct xlnx_efct_client *handle_client;
	unsigned long client_hndl_test;

	handle_client = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_ADDR_NE(handle_client, NULL);
	ASSERT_INT_EQ(IS_ERR(handle_client), 0);

	evq_params.q_page = alloc_pages(__GFP_COMP | GFP_KERNEL, 0);//allocating one page
	ASSERT_ADDR_NE_GOTO(evq_params.q_page, NULL, cleanup);

	/* Get queue parameters */
	ASSERT_INT_EQ_GOTO(0, (xlnx_efct_dev->ops->get_param(handle_client, XLNX_EFCT_NIC_RESOURCES, &param)), cleanup1);

	evq_params.qid = param.nic_res.evq_min;
	evq_params.page_offset = 0;
	evq_params.q_size = PAGE_SIZE;
	evq_params.entries = (evq_params.q_size / 8);

	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->init_evq(handle_client, &evq_params), cleanup1);

	txq_params.evq = param.nic_res.evq_min;
	txq_params.label = -1;
	txq_params.qid = param.nic_res.txq_min;

	qid = xlnx_efct_dev->ops->init_txq(handle_client, &txq_params);
	tlog(T_INFO, "Tx Qid = 0x%x\n", qid);
	if (qid != txq_params.qid)
		terr("init_txq call failed\n");

	EXPECT_INT_EQ(qid, txq_params.qid);

	xlnx_efct_dev->ops->free_txq(handle_client, qid);
	xlnx_efct_dev->ops->free_evq(handle_client, evq_params.qid);
cleanup1:
	put_page(evq_params.q_page);
	evq_params.q_page = NULL;
cleanup:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle_client));
}

#define MAX_TX_QUEUES 16
TEST(auxtest, T008_alloc_multiple_txq_with_single_eventq)
{
	int qid, i, j, num_tx_queues = 0;
	int tx_q[MAX_TX_QUEUES];
	union xlnx_efct_param_value param = { 0 };
	struct xlnx_efct_evq_params evq_params = { 0 };
	struct xlnx_efct_txq_params txq_params = { 0 };
	struct xlnx_efct_client *handle_client;
	unsigned long client_hndl_test;

	handle_client = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_ADDR_NE(handle_client, NULL);
	ASSERT_INT_EQ(IS_ERR(handle_client), 0);

	evq_params.q_page = alloc_pages(__GFP_COMP | GFP_KERNEL, 0);//allocating one page
	ASSERT_ADDR_NE_GOTO(evq_params.q_page, NULL, cleanup1);

	/* Get queue parameters */
	ASSERT_INT_EQ_GOTO(0, (xlnx_efct_dev->ops->get_param(handle_client, XLNX_EFCT_NIC_RESOURCES, &param)), cleanup);

	pr_info("evq_lim: %d", param.nic_res.evq_lim);
	pr_info("evq_min: %d", param.nic_res.evq_min);
	//pr_info("txq_lim", )

	evq_params.qid = param.nic_res.evq_min;
	evq_params.page_offset = 0;
	evq_params.q_size = PAGE_SIZE;
	evq_params.entries = (evq_params.q_size / 8);

	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->init_evq(handle_client, &evq_params), cleanup);

	txq_params.evq = param.nic_res.evq_min;
	txq_params.label = -1;
	txq_params.qid = param.nic_res.txq_min;

	for (i = param.nic_res.evq_min,j = param.nic_res.txq_min; i < param.nic_res.evq_lim && j < param.nic_res.txq_lim; i++, j++) {
		txq_params.qid = j;
		qid = xlnx_efct_dev->ops->init_txq(handle_client, &txq_params);
		ASSERT_INT_EQ_GOTO(qid, txq_params.qid, cleanup2);

		tx_q[j] = qid; // Maintaining this list for free_txq
		tlog(T_INFO, "Tx Qid [%d] = 0x%x\n", j, qid);
	}
cleanup2:
	num_tx_queues = j;
	for  (j = param.nic_res.txq_min; j < num_tx_queues; j++) {
		xlnx_efct_dev->ops->free_txq(handle_client, tx_q[j]);
	}

	xlnx_efct_dev->ops->free_evq(handle_client, evq_params.qid);
cleanup:
	put_page(evq_params.q_page);
	evq_params.q_page = NULL;
cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle_client));
}

TEST(auxtest, T009_alloc_multiple_txq_with_multiple_eventq)
{
	int qid, i, j, num_tx_queues;
	int tx_q[MAX_TX_QUEUES];
	union xlnx_efct_param_value param = { 0 };
	struct xlnx_efct_evq_params evq_params = { 0 };
	struct xlnx_efct_txq_params txq_params = { 0 };
	struct xlnx_efct_client *handle_client;
	unsigned long client_hndl_test;
	struct page *pg;
	bool failed_at_alloc_txq = false;

	handle_client = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_ADDR_NE(handle_client, NULL);
	ASSERT_INT_EQ(IS_ERR(handle_client), 0);

	/* Get queue parameters */
	ASSERT_INT_EQ_GOTO(0, (xlnx_efct_dev->ops->get_param(handle_client, XLNX_EFCT_NIC_RESOURCES, &param)), cleanup1);

	pr_info("evq_lim: %d", param.nic_res.evq_lim);
	pr_info("evq_min: %d", param.nic_res.evq_min);

	evq_params.qid = param.nic_res.evq_min;
	evq_params.page_offset = 0;
	evq_params.q_size = PAGE_SIZE;
	evq_params.entries = (evq_params.q_size / 8);
	pg = alloc_pages(__GFP_COMP | GFP_KERNEL, order_base_2(param.nic_res.evq_lim - param.nic_res.evq_min));
	ASSERT_ADDR_NE_GOTO(pg, NULL, cleanup1);

	for (i = param.nic_res.evq_min, j = param.nic_res.txq_min; i < param.nic_res.evq_lim && j < param.nic_res.txq_lim; i++, j++) {
		evq_params.qid = i;
		evq_params.q_page = pg + (i - param.nic_res.evq_min);
		ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->init_evq(handle_client, &evq_params), cleanup);

		txq_params.evq = evq_params.qid;
		txq_params.label = -1;
		txq_params.qid = j;
		qid = xlnx_efct_dev->ops->init_txq(handle_client, &txq_params);
		ASSERT_INT_EQ_GOTO(qid, txq_params.qid, cleanup2);
		tx_q[j] = qid; // Maintaining this list for free_txq
		tlog(T_INFO, "Tx Qid [%d] = 0x%x\n", j, qid);
	}

	goto cleanup;
cleanup2:
	failed_at_alloc_txq = true;
cleanup:
	num_tx_queues = j;
	for  (i = param.nic_res.txq_min; i < param.nic_res.txq_lim; i++)
		xlnx_efct_dev->ops->free_txq(handle_client, tx_q[i]);
	for  (i = param.nic_res.evq_min; i < param.nic_res.evq_lim; i++)
		xlnx_efct_dev->ops->free_evq(handle_client, i);
	if(failed_at_alloc_txq)
		xlnx_efct_dev->ops->free_evq(handle_client, i + 1);

	put_page(pg);
	evq_params.q_page = NULL;
cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle_client));
}

TEST(auxtest, T010_multiple_txq_call_close_before_free_test)
{
	int qid, i, j, num_tx_queues;
	int tx_q[MAX_TX_QUEUES];
	union xlnx_efct_param_value param = { 0 };
	struct xlnx_efct_evq_params evq_params = { 0 };
	struct xlnx_efct_txq_params txq_params = { 0 };
	struct xlnx_efct_client *handle_client;
	unsigned long client_hndl_test;
	struct page *pg;
	bool failed_at_alloc_txq = false;

	handle_client = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_ADDR_NE(handle_client, NULL);
	ASSERT_INT_EQ(IS_ERR(handle_client), 0);

	/* Get queue parameters */
	ASSERT_INT_EQ_GOTO(0, (xlnx_efct_dev->ops->get_param(handle_client, XLNX_EFCT_NIC_RESOURCES, &param)), cleanup2);

	pr_info("evq_lim: %d", param.nic_res.evq_lim);
	pr_info("evq_min: %d", param.nic_res.evq_min);

	evq_params.qid = param.nic_res.evq_min;
	evq_params.page_offset = 0;
	evq_params.q_size = PAGE_SIZE;
	evq_params.entries = (evq_params.q_size / 8);
	pg = alloc_pages(__GFP_COMP | GFP_KERNEL, order_base_2(param.nic_res.evq_lim - param.nic_res.evq_min));
	ASSERT_ADDR_NE_GOTO(pg, NULL, cleanup2);
	/* Allocate all the Tx Queues */
	for (i = param.nic_res.evq_min, j = param.nic_res.txq_min; i < param.nic_res.evq_lim && j < param.nic_res.txq_lim; i++, j++) {
		evq_params.qid = i;
		evq_params.q_page = pg + (i - param.nic_res.evq_min);
		ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->init_evq(handle_client, &evq_params), cleanup);

		txq_params.evq = i;
		txq_params.label = -1;
		txq_params.qid = j;

		qid = xlnx_efct_dev->ops->init_txq(handle_client, &txq_params);
		ASSERT_INT_EQ_GOTO(qid, txq_params.qid, cleanup3);
		tx_q[j] = qid; // Maintaining this list for free_txq
	}

	/* Call close with freeing the queues */
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle_client));

	/* Again open and allocate all Tx Queues */
	handle_client = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_ADDR_NE_GOTO(handle_client, NULL, cleanup1);

	/* Get queue parameters */
	ASSERT_INT_EQ_GOTO(0, (xlnx_efct_dev->ops->get_param(handle_client, XLNX_EFCT_NIC_RESOURCES, &param)), cleanup1);

	pr_info("evq_lim: %d", param.nic_res.evq_lim);
	pr_info("evq_min: %d", param.nic_res.evq_min);

	evq_params.qid = param.nic_res.evq_min;
	evq_params.page_offset = 0;
	evq_params.q_size = PAGE_SIZE;
	evq_params.entries = (evq_params.q_size / 8);
	/* Allocate all the Tx Queues */
	for (i = param.nic_res.evq_min, j = param.nic_res.txq_min; i < param.nic_res.evq_lim && j < param.nic_res.txq_lim; i++, j++) {
		evq_params.qid = i;
		evq_params.q_page = pg + (i - param.nic_res.evq_min);
		ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->init_evq(handle_client, &evq_params), cleanup);

		txq_params.evq = i;
		txq_params.label = -1;
		txq_params.qid = j;
		qid = xlnx_efct_dev->ops->init_txq(handle_client, &txq_params);
		ASSERT_INT_EQ_GOTO(qid, txq_params.qid, cleanup3);
		tx_q[j] = qid; // Maintaining this list for free_txq
		tlog(T_INFO, "Tx Qid [%d] = 0x%x\n", j, qid);
	}
	goto cleanup;
cleanup3:
	failed_at_alloc_txq = true;
cleanup:
	num_tx_queues = j;
	for  (i = param.nic_res.txq_min; i < param.nic_res.txq_lim; i++) {
		xlnx_efct_dev->ops->free_txq(handle_client, tx_q[i]);
	}
	for  (i = param.nic_res.evq_min; i < param.nic_res.evq_lim; i++) {
		xlnx_efct_dev->ops->free_evq(handle_client, i);
	}
	if(failed_at_alloc_txq)
		xlnx_efct_dev->ops->free_evq(handle_client, i + 1);
cleanup1:
	put_page(pg);
	evq_params.q_page = NULL;
cleanup2:
	if(handle_client)
		EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle_client));
}

TEST(auxtest, N005_init_txq)
{
	int qid;
	union xlnx_efct_param_value param = { 0 };
	struct xlnx_efct_evq_params evq_params = { 0 };
	struct xlnx_efct_txq_params txq_params = { 0 };

	evq_params.q_page = alloc_pages(__GFP_COMP | GFP_KERNEL, 0);//allocating one page
	ASSERT_ADDR_NE(NULL, evq_params.q_page);

	evq_params.qid = 2;
	evq_params.page_offset = 0;
	evq_params.q_size = PAGE_SIZE;
	evq_params.entries = (evq_params.q_size / 8);

	ASSERT_ADDR_NE_GOTO(client_handle, NULL, cleanup);

	EXPECT_INT_NE(0, xlnx_efct_dev->ops->init_evq(NULL, &evq_params));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->init_evq(client_handle, NULL));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->init_evq(NULL, NULL));

	/* Get queue parameters */
	ASSERT_INT_EQ_GOTO(0, (xlnx_efct_dev->ops->get_param(client_handle, XLNX_EFCT_NIC_RESOURCES, &param)), cleanup);

	/* Check invalid event qid */
	evq_params.qid =  param.nic_res.evq_lim;

	EXPECT_INT_LT(xlnx_efct_dev->ops->init_evq(client_handle, &evq_params), 0);

	/* Invalid event qid */
	txq_params.evq = param.nic_res.evq_lim;
	txq_params.label = -1;
	txq_params.qid = param.nic_res.txq_lim;
	qid = xlnx_efct_dev->ops->init_txq(client_handle, &txq_params);
	ASSERT_INT_LT(qid, 0);

	/* Free invalid txq */
	xlnx_efct_dev->ops->free_txq(NULL, -1);
	xlnx_efct_dev->ops->free_txq(client_handle, -1);

	/* Allocate valid txq */
	txq_params.evq = param.nic_res.evq_min;
	txq_params.label = -1;
	txq_params.qid = param.nic_res.txq_min;

	qid = xlnx_efct_dev->ops->init_txq(client_handle, &txq_params);
	tlog(T_INFO, "Tx Qid = 0x%x\n", qid);
	ASSERT_INT_LT_GOTO(qid, 0, cleanup);

	/* Try freeing evq first. Although it should free the evq, try freeing it after txq */
	xlnx_efct_dev->ops->free_evq(client_handle, evq_params.qid);
	xlnx_efct_dev->ops->free_txq(client_handle, qid);
	xlnx_efct_dev->ops->free_evq(client_handle, evq_params.qid);
	/*Free the txq again*/
	xlnx_efct_dev->ops->free_txq(client_handle, qid);
	/*Passing invalid queue*/
	xlnx_efct_dev->ops->free_txq(NULL, qid);
	xlnx_efct_dev->ops->free_txq(client_handle, param.nic_res.evq_lim);
cleanup:
	put_page(evq_params.q_page);
	evq_params.q_page = NULL;
}

TEST(auxtest, T006_ctpio_addr)
{
	int qid;
	resource_size_t addr;
	size_t size;
	union xlnx_efct_param_value param = { 0 };
	struct xlnx_efct_evq_params evq_params = { 0 };
	struct xlnx_efct_txq_params txq_params = { 0 };
	struct xlnx_efct_client *handle_client;
	unsigned long client_hndl_test;

	handle_client = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_ADDR_NE(handle_client, NULL);

	evq_params.q_page = alloc_pages(__GFP_COMP | GFP_KERNEL, 0);//allocating one page
	ASSERT_ADDR_NE_GOTO(evq_params.q_page, NULL, cleanup2);

	/* Get queue parameters */
	ASSERT_INT_EQ_GOTO(0, (xlnx_efct_dev->ops->get_param(handle_client, XLNX_EFCT_NIC_RESOURCES, &param)), cleanup1);

	evq_params.qid = param.nic_res.evq_min;
	evq_params.page_offset = 0;
	evq_params.q_size = PAGE_SIZE;
	evq_params.entries = (evq_params.q_size / 8);

	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->init_evq(handle_client, &evq_params), cleanup1);

	txq_params.evq = param.nic_res.evq_min;
	txq_params.label = -1;
	txq_params.qid = param.nic_res.txq_min;

	qid = xlnx_efct_dev->ops->init_txq(handle_client, &txq_params);
	tlog(T_INFO, "Tx Qid = 0x%x\n", qid);
	ASSERT_INT_EQ_GOTO(qid, txq_params.qid, cleanup);

	/* Call the ctpio API */
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->ctpio_addr(handle_client, qid, &addr, &size));
	//TODO Address validation pending. Currently no way to do. config_mem return virtual address. This call returns physical address */
	tlog(T_INFO, "CTPIO addr Received = 0x%llx\n", addr);
	EXPECT_INT_EQ(4096, size);

	xlnx_efct_dev->ops->free_txq(handle_client, qid);
cleanup:
	xlnx_efct_dev->ops->free_evq(handle_client, evq_params.qid);
cleanup1:
	put_page(evq_params.q_page);
	evq_params.q_page = NULL;
cleanup2:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle_client));
}

TEST(auxtest, N006_ctpio_addr)
{
	int qid;
	resource_size_t addr;
	size_t size;
	union xlnx_efct_param_value param = { 0 };
	struct xlnx_efct_evq_params evq_params = { 0 };
	struct xlnx_efct_txq_params txq_params = { 0 };

	evq_params.q_page = alloc_pages(__GFP_COMP | GFP_KERNEL, 0);//allocating one page
	ASSERT_ADDR_NE(evq_params.q_page, NULL);

	ASSERT_ADDR_NE_GOTO(client_handle, NULL, cleanup1);

	/* Get queue parameters */
	ASSERT_INT_EQ_GOTO(0, (xlnx_efct_dev->ops->get_param(client_handle, XLNX_EFCT_NIC_RESOURCES, &param)), cleanup1);

	evq_params.qid = param.nic_res.evq_min;
	evq_params.page_offset = 0;
	evq_params.q_size = PAGE_SIZE;
	evq_params.entries = (evq_params.q_size / 8);

	ASSERT_INT_EQ_GOTO(0, xlnx_efct_dev->ops->init_evq(client_handle, &evq_params), cleanup1);

	txq_params.evq = param.nic_res.evq_min;
	txq_params.label = -1;
	txq_params.qid = param.nic_res.txq_min;

	qid = xlnx_efct_dev->ops->init_txq(client_handle, &txq_params);

	tlog(T_INFO, "Tx Qid = 0x%x\n", qid);
	ASSERT_INT_EQ_GOTO(qid, txq_params.qid, cleanup);

	/* Call the ctpio API .-- negative test cases*/
	//EXPECT_INT_NE(0, xlnx_efct_dev->ops->ctpio_addr(client_handle, qid + 1, &addr, &size)); //passing invalid queue -- application fault. Driver may not handle this.
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->ctpio_addr(NULL, qid, &addr, &size));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->ctpio_addr(client_handle, qid, NULL, &size));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->ctpio_addr(client_handle, qid, &addr, NULL));
	EXPECT_INT_NE(0, xlnx_efct_dev->ops->ctpio_addr(NULL, qid + 1, NULL, NULL));

	xlnx_efct_dev->ops->free_txq(client_handle, qid);
cleanup:
	xlnx_efct_dev->ops->free_evq(client_handle, evq_params.qid);
cleanup1:
	put_page(evq_params.q_page);
	evq_params.q_page = NULL;
}

TEST(auxtest, T011_time_sync_events)
{
	union xlnx_efct_param_value param = { 0 }, config_param = { 0 };
	struct xlnx_efct_evq_params evq_params = { 0 };
	void *buf_addr;
	bool time_sync_found = false;
	unsigned long int value, i;
	struct xlnx_efct_client *handle_client;
	unsigned long client_hndl_test;

	handle_client = xlnx_efct_dev->ops->open(aux_dev, &auxtest_drv_ops, &client_hndl_test);
	ASSERT_ADDR_NE(handle_client, NULL);
	/* Get queue parameters */
	ASSERT_INT_EQ_GOTO(0, (xlnx_efct_dev->ops->get_param(handle_client, XLNX_EFCT_NIC_RESOURCES, &param)), cleanup1);
	EXPECT_INT_EQ(0, (xlnx_efct_dev->ops->get_param(handle_client, XLNX_EFCT_CONFIG_MEM, &config_param)));

	/*Initialize event queue*/
	evq_params.q_page = alloc_pages(__GFP_COMP | GFP_KERNEL, 0);//allocating one page
	ASSERT_ADDR_NE_GOTO(evq_params.q_page, NULL, cleanup1);
	evq_params.qid = param.nic_res.evq_min;
	evq_params.page_offset = 0;
	evq_params.q_size = PAGE_SIZE;
	evq_params.entries = (evq_params.q_size / 8);
	evq_params.subscribe_time_sync = true;
	evq_params.unsol_credit = X3_EV_UNSOL_MAX;
	ASSERT_INT_EQ_GOTO(0, (xlnx_efct_dev->ops->init_evq(handle_client, &evq_params)), cleanup);

	/*poll for time sync events*/
	buf_addr = page_address(evq_params.q_page) + evq_params.page_offset;
	for (i = 0; i < TIME_SYNC_COUNT; i++) {
		value = *(unsigned long int *)buf_addr;
		if (((value >> EV_TYPE_SHIFT) & EV_TYPE_MASK) == X3_EVENT_TYPE_CONTROL) {
			if (((value >> TIME_SYNC_SHIFT) & TIME_SYNC_MASK) == X3_CTL_EVENT_SUBTYPE_TIME_SYNC) {
				time_sync_found = true;
				break;
			}
		}
		mdelay(100);
	}

	/*clean up*/
	xlnx_efct_dev->ops->free_evq(handle_client, evq_params.qid);
cleanup:
	put_page(evq_params.q_page);
	evq_params.q_page = NULL;
	EXPECT_INT_NE(0,time_sync_found);
cleanup1:
	EXPECT_INT_EQ(0, xlnx_efct_dev->ops->close(handle_client));
}

static void add_tests(void)
{
	/*Filter test cases*/
	ADD_TEST(T001_filter_op);
	ADD_TEST(T002_filter_op);
	ADD_TEST(T003_filter_op);
	ADD_TEST(T004_filter_op);
	ADD_TEST(T006_filter_op);
	ADD_TEST(T007_filter_op);
	ADD_TEST(T008_filter_op);
	ADD_TEST(T009_filter_op);
	ADD_TEST(T010_filter_op);
	ADD_TEST(T011_filter_op);
	ADD_TEST(T012_filter_op);
	ADD_TEST(T013_filter_op);
	ADD_TEST(T014_filter_op);
	ADD_TEST(T015_filter_op);
	ADD_TEST(T016_filter_op);
	ADD_TEST(T017_filter_op);
	ADD_TEST(T018_filter_op);
	ADD_TEST(T019_filter_op);
	ADD_TEST(T020_filter_op);
	ADD_TEST(T021_filter_op);
	ADD_TEST(T022_filter_op);
	ADD_TEST(T023_filter_op);
	ADD_TEST(T024_filter_op);
	ADD_TEST(T025_filter_op);
	ADD_TEST(T026_filter_op);
	ADD_TEST(T027_filter_op);
	ADD_TEST(T028_filter_op);
	ADD_TEST(T029_filter_op);
	ADD_TEST(T030_filter_op);
	ADD_TEST(T031_filter_op);
	ADD_TEST(T032_filter_op);
	ADD_TEST(T033_filter_op);
	ADD_TEST(T034_filter_op);
	ADD_TEST(T035_filter_op);
	ADD_TEST(T036_filter_op);
	ADD_TEST(T037_filter_op);
	ADD_TEST(T038_filter_op);
	ADD_TEST(T039_filter_op);
	ADD_TEST(T040_filter_op);
	ADD_TEST(N001_filter_op);
	ADD_TEST(N002_filter_op);
	ADD_TEST(N003_filter_op);
	ADD_TEST(N004_filter_op);
	ADD_TEST(N005_filter_op);
	ADD_TEST(N006_filter_op);
	ADD_TEST(N007_filter_op);
	ADD_TEST(N008_filter_op);
	ADD_TEST(N009_filter_op);
	ADD_TEST(N010_filter_op);
	ADD_TEST(N011_filter_op);
	ADD_TEST(N012_filter_op);
	ADD_TEST(N013_filter_op);
	ADD_TEST(N014_filter_op);
	ADD_TEST(N015_filter_op);
	ADD_TEST(N016_filter_op);
	ADD_TEST(N017_filter_op);
	ADD_TEST(N018_filter_op);
	ADD_TEST(N019_filter_op);
	ADD_TEST(T001_probe_open);
	ADD_TEST(N001_probe_open);
	ADD_TEST(T002_get_param);
	ADD_TEST(N002_get_param);
	ADD_TEST(T003A_init_evq_n_id);
	ADD_TEST(T003B_init_evq_n_entries);
	ADD_TEST(N003_init_evq);
	ADD_TEST(T005_init_txq);
	ADD_TEST(N005_init_txq);
	ADD_TEST(T006_ctpio_addr);
	ADD_TEST(N006_ctpio_addr);
	ADD_TEST(T008_alloc_multiple_txq_with_single_eventq);
	ADD_TEST(T009_alloc_multiple_txq_with_multiple_eventq);
	ADD_TEST(T010_multiple_txq_call_close_before_free_test);
      //RX test cases
	ADD_TEST(T001A_probe_open);
	ADD_TEST(N007B_bind_rxq);
	ADD_TEST(N007_get_huge_page);
	ADD_TEST(N007_bind_rxq);
	ADD_TEST(T007_bind_free_rxq);
	ADD_TEST(T007B_bind_rxq_release_buf);
	ADD_TEST(T007C_bind_rxq);
	ADD_TEST(T007G_multiple_rollover);
	ADD_TEST(T007F_bind_free_rxq);
	ADD_TEST(T008A_bind_rollover_free_rxq);
	ADD_TEST(T008B_bind_rollover_free_rxq);
	ADD_TEST(T008C_bind_rollover_free_rxq);
	ADD_TEST(T007H_bind_free_free);
	ADD_TEST(T007E_bind_rollover_close);
	ADD_TEST(T007D_bind_free_rxq_huge_pages);
	ADD_TEST(T007B_bind_rxq_n_qid);
	ADD_TEST(T011_time_sync_events);
}

static struct file_operations control_fops = {
	.owner = THIS_MODULE,
	.open  = auxtest_open,
	.release = auxtest_close,
	.unlocked_ioctl = auxtest_ioctl,
};

//Misc device structure
struct miscdevice aux_misc_dev = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "efctdev",
	.fops = &control_fops,
};

static int __init auxtest_init(void)
{
	int ret;
	int i;

	/* Add KTF Test cases */
	add_tests();

	for (i = 0; i < 512; i++)
		sbid_to_free_list[i] = -1;

	/* Create char device for application to communicate */
	ret = misc_register(&aux_misc_dev);
	if (ret < 0) {
		terr("Failed to register char device");
		return ret;
	}

	tlog(T_INFO, "Char device registered\n");
	/* Register AUX Driver */
	ret = auxiliary_driver_register(&auxtest_driver);
	if (ret) {
		terr("Failed to register Auxiliary driver");
		misc_deregister(&aux_misc_dev);
		return ret;
	}

	tlog(T_INFO, "auxtest: loaded");
	return 0;
}

static void __exit auxtest_exit(void)
{
	KTF_CLEANUP();
	auxiliary_driver_unregister(&auxtest_driver);
	misc_deregister(&aux_misc_dev);
	tlog(T_INFO, "auxtest: unloaded");
}

module_init(auxtest_init);
module_exit(auxtest_exit);
