// SPDX-License-Identifier: BSD-2-Clause
/********************************************************************
* Copyright(c) 2021 Xilinx, Inc. All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
*
*   * Redistributions of source code must retain the above copyright
*     notice, this list of conditions and the following disclaimer.
*   * Redistributions in binary form must reproduce the above copyright
*     notice, this list of conditions and the following disclaimer in
*     the documentation and/or other materials provided with the
*     distribution.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
* A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
* OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
* SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
* THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "auxdev_test_ioctl.h"

static struct user_filter_params filter_params;

enum command {
	ADD_EVQ,
	FREE_EVQ,
	BIND_RXQ,
	FREE_RXQ,
	ADD_TXQ,
	FREE_TXQ,
	ROLLOVER_RXQ,
	GET_HUGE_PAGE,
	FILTER_INSERT,
	FILTER_REMOVE,
};

static int dev_open(void)
{
	int fd = open("/dev/efctdev", O_RDWR);

	if (fd < 0)
		perror("open");

	return fd;
}

static void get_ether_address(char *eth_src, unsigned char *eth_dst)
{
	unsigned int buf[ETH_ALEN];
	int i;

	sscanf(eth_src, "%2x:%2x:%2x:%2x:%2x:%2x",
		&buf[0], &buf[1], &buf[2], &buf[3], &buf[4], &buf[5]);

	for (i = 0; i < ETH_ALEN; i++)
		eth_dst[i] = (uint8_t)buf[i];
}

static int handle_queue_cmd(enum command cmd, int argc, char *argv[])
{
	int fd, rc, i = 1, index;
	unsigned long long qid;
	struct in_addr addr;
	char * ipaddr = "127.0.0.1";
	char * eth_dst;
	uint16_t pdst;

	while (i < argc) {
		if (!strcmp(argv[i], "qid")) {
			++i;
			qid = strtol(argv[i], NULL, 0);
			filter_params.qid = qid;
		} else if (!strcmp(argv[i], "dstip")) {
			++i;
			ipaddr = argv[i];
			inet_aton(ipaddr, &addr);
			filter_params.dst_ip = addr.s_addr;
		} else if (!strcmp(argv[i], "dstp")) {
			++i;
			pdst = strtol(argv[i], NULL, 0);
			filter_params.pdst = htons(pdst);
		} else if (!strcmp(argv[i], "flags")) {
			++i;
			filter_params.flags = strtol(argv[i], NULL, 0);
		} else if (!strcmp(argv[i], "mask")) {
			++i;
			filter_params.mask = strtol(argv[i], NULL, 0);
		} else if (!strcmp(argv[i], "flow_type")) {
			++i;
			filter_params.flow_type = strtol(argv[i], NULL, 0);
		} else if (!strcmp(argv[i], "mcast")) {
			++i;
			filter_params.mcast = strtol(argv[i], NULL, 0);
		} else if (!strcmp(argv[i], "id")) {
			++i;
			index = strtol(argv[i], NULL, 0);
		} else if (!strcmp(argv[i], "ether_dst")) {
			++i;
			eth_dst = argv[i];
			get_ether_address(eth_dst, filter_params.eth_addr);
		} else if (!strcmp(argv[i], "ether_mask")) {
			i++;
			eth_dst = argv[i];
			get_ether_address(eth_dst, filter_params.eth_mask);
		} else if (!strcmp(argv[i], "l4proto")) {
			++i;
			filter_params.l4proto = strtol(argv[i], NULL, 0);
		} else {
			printf("Unknown command [%s]\n", argv[i]);
			return 1;
		}
		++i;
	}

	fd = dev_open();
	if (fd < 0)
		return fd;

	switch (cmd) {
		case ADD_EVQ:
			rc = ioctl(fd, AUXTEST_ADD_EVQ, qid);
			break;
		case FREE_EVQ:
			rc = ioctl(fd, AUXTEST_FREE_EVQ, qid);
			break;
		case BIND_RXQ:
			rc = ioctl(fd, AUXTEST_BIND_RXQ, qid);
			break;
		case FREE_RXQ:
			rc = ioctl(fd, AUXTEST_FREE_RXQ, qid);
			break;
		case ADD_TXQ:
			rc = ioctl(fd, AUXTEST_ALLOC_TXQ, qid);
			break;
		case FREE_TXQ:
			rc = ioctl(fd, AUXTEST_FREE_TXQ, qid);
			break;
		case ROLLOVER_RXQ:
			rc = ioctl(fd, AUXTEST_ROLLOVER_RXQ, qid);
			break;
		case GET_HUGE_PAGE:
			rc = ioctl(fd, AUXTEST_GET_HUGEPAGES, qid);
			break;
		case FILTER_INSERT:
			rc = ioctl(fd, AUXTEST_FILTER_INSERT, &filter_params);
			break;
		case FILTER_REMOVE:
			rc = ioctl(fd, AUXTEST_FILTER_REMOVE, index);
			break;
		default:
			rc = -1;
	}

	close(fd);
	return rc;
}

static void print_help(void)
{
	printf("Usage: auxdev_test_app [add_evq|free_evq|bind_rxq|free_rxq|add_txq|free_txq|\
			rollover_rxq|get_huge_page] qid <N>\n");
	printf("       auxdev_test_app [filter_insert] qid <N> flowtype <N> dstip <ip> dstp <port> mask <cpumask> flags <mask>\n");
	printf("       auxdev_test_app [filter_remove] id <N>\n");
}

int main(int argc, char *argv[])
{
	int arg;

	printf("running application.\n");
	for (arg = 1; arg < argc; arg++) {
		if (!strcmp(argv[arg], "add_evq")) {
			return handle_queue_cmd(ADD_EVQ, argc - 1, argv + 1);
		} else if (!strcmp(argv[arg], "free_evq")) {
			return handle_queue_cmd(FREE_EVQ, argc - 1, argv + 1);
		} else if (!strcmp(argv[arg], "bind_rxq")) {
			return handle_queue_cmd(BIND_RXQ, argc - 1, argv + 1);
		} else if (!strcmp(argv[arg], "free_rxq")) {
			return handle_queue_cmd(FREE_RXQ, argc - 1, argv + 1);
		} else if (!strcmp(argv[arg], "add_txq")) {
			return handle_queue_cmd(ADD_TXQ, argc - 1, argv + 1);
		} else if (!strcmp(argv[arg], "free_txq")) {
			return handle_queue_cmd(FREE_TXQ, argc - 1, argv + 1);
		} else if (!strcmp(argv[arg], "rollover_rxq")) {
			return handle_queue_cmd(ROLLOVER_RXQ, argc - 1, argv + 1);
		} else if (!strcmp(argv[arg], "get_huge_page")) {
			return handle_queue_cmd(GET_HUGE_PAGE, argc - 1, argv + 1);
		} else if (!strcmp(argv[arg], "filter_insert")) {
			return handle_queue_cmd(FILTER_INSERT, argc - 1, argv + 1);
		} else if (!strcmp(argv[arg], "filter_remove")) {
			return handle_queue_cmd(FILTER_REMOVE, argc - 1, argv + 1);
		} else {
			printf("Invalid command [%s]\n", argv[arg]);
			print_help();
			return -1;
		}
	}

	return 0;
}
