/*
** (c) Copyright 2012-2019 Xilinx, Inc. All rights reserved.
**
** This file contains confidential and proprietary information
** of Xilinx, Inc. and is protected under U.S. and
** international copyright and other intellectual property
** laws.
**
** DISCLAIMER
** This disclaimer is not a license and does not grant any
** rights to the materials distributed herewith. Except as
** otherwise provided in a valid license issued to you by
** Xilinx, and to the maximum extent permitted by applicable
** law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
** WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
** AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
** BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
** INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
** (2) Xilinx shall not be liable (whether in contract or tort,
** including negligence, or under any other theory of
** liability) for any loss or damage of any kind or nature
** related to, arising under or in connection with these
** materials, including for any direct, or any indirect,
** special, incidental, or consequential loss or damage
** (including loss of data, profits, goodwill, or any type of
** loss or damage suffered as a result of any action brought
** by a third party) even if such damage or loss was
** reasonably foreseeable or Xilinx had been advised of the
** possibility of the same.
**
** CRITICAL APPLICATIONS
** Xilinx products are not designed or intended to be fail-
** safe, or for use in any application requiring fail-safe
** performance, such as life-support or safety devices or
** systems, Class III medical devices, nuclear facilities,
** applications related to the deployment of airbags, or any
** other applications that could lead to death, personal
** injury, or severe property or environmental damage
** (individually and collectively, "Critical
** Applications"). Customer assumes the sole risk and
** liability of any use of Xilinx products in Critical
** Applications, subject only to applicable laws and
** regulations governing limitations on product liability.
**
** THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
** PART OF THIS FILE AT ALL TIMES
*/

/**
 * \file
 * \brief This header is used to generate C type definitions and corresponding
 * runtime type information for data structures that are shared by
 * SolarCapture with other processes.
 *
 * In order to create runtime type information a template header file must be
 * created. For example a node called my_node could have a template file
 * my_node_tmpl.h as follows
 *
 *
 *     ST_STRUCT(my_node_stats)
 *       ST_FIELD(double,   some_stat,         config)
 *       ST_FIELD(int,      some_other_stat,   pkt_count)
 *       ST_FIELD(double,   another_stat,      magnitude)
 *       ST_FIELD(double,   yet_one_more_stat, ev_count)
 *       ...
 *     ST_STRUCT_END
 *
 *
 * In the node source file the node must
 *   1. Define SC_TYPE_TEMPLATE to be a header file containing the node's type
 *   template definition.
 *   2. Define SC_DECLARE_TYPES to be the name of the declaration function to
 *   create.
 *   3. Include declare_types.h
 *   4. Call the function defined by (2)
 *   5. Call ::sc_node_export_state to allocate a struct of the type defined in
 *   the node's type template definition.
 *
 * Stats can them be updated by changing the values of the fields in the newly
 * created struct from step (5). If stats need to be updated during runtime a means
 * of accessing this struct should be kept in the nodes nd_private field.
 *
 * For example, a node which would like to create a declaration function with name
 * my_node_declare using the template my_node_tmpl.h would do the following
 *
 *
 *     #define SC_TYPE_TEMPLATE  <my_node_tmpl.h>
 *     #define SC_DECLARE_TYPES my_node_declare
 *     #include <solar_capture/declare_types.h>
 *
 *     ...
 *
 *     static int my_node_init(struct sc_node* node, const struct sc_attr* attr,
 *                     const struct sc_node_factory* factory)
 *     {
 *       ...
 *       my_node_declare(sc_thread_get_session(sc_node_get_thread(node)));
 *       ...
 *       struct my_node_stats* stats;
 *       sc_node_export_state(node, "my_node_stats",
 *                      sizeof(struct my_node_stats), &stats);
 *     }
 *
 *
 *
 */

#ifndef SC_TYPE_TEMPLATE
# error SC_TYPE_TEMPLATE must be defined before including declare_types.h
#endif


/**********************************************************************
 * Generate C type definitions.
 */

/**
 * \brief A constant value in the template definition.
 *
 * After the node has initialised its shared data structures @p name will be
 * used as the field in the stats struct to update this data.
 *
 * \param name      The field name.
 * \param val       The constant.
 */
#define ST_CONSTANT(name, val)         enum { name = val };a
/**
 * \brief Start of the template definition.
 *
 * After the node has initialised its shared data structures the resulting struct
 * type for updating the stats will use @p name for its type.
 *
 * \param name      The name of the template.
 */
#define ST_STRUCT(name)                struct name {
/**
 * \brief A string field in the template definition
 *
 * After the node has initialised its shared data structures @p name will be
 * used as the field in the stats struct to update this data.
 *
 * \param name      The field name.
 * \param len       The length of the string.
 * \param kind      A string to describe the kind of data. Examples used by
 *                  SolarCapture nodes are pkt_count, ev_count, config, const,
 *                  magnitude.
 */
#define ST_FIELD_STR(name, len, kind)  char name[len];
/**
 *  \brief A C basic type field in the template definition.
 *
 *  After the node has initialised its shared data structures @p name will be
 *  used as the field in the stats struct to update this data.
 *
 *  \param type     The basic data type.
 *  \param name     The field name.
 *  \param kind     A string to describe the kind of data. Examples used by
 *                  SolarCapture nodes are pkt_count, ev_count, config, const,
 *                  magnitude.
 */
#define ST_FIELD(type, name, kind)     type name;
/**
 * \brief End of the template definition
 */
#define ST_STRUCT_END                  };

#include SC_TYPE_TEMPLATE

#undef ST_CONSTANT
#undef ST_STRUCT
#undef ST_FIELD_STR
#undef ST_FIELD
#undef ST_STRUCT_END


/**********************************************************************
 * Generate function to declare the types.
 */

#ifdef SC_DECLARE_TYPES
/** \brief Function to declare the runtime type information to be shared with
 * other processes.
 *
 * \param scs       The current solar capture session.
 *
 */
static inline void SC_DECLARE_TYPES(struct sc_session* scs)
{
#define __scs scs
#define ST_CONSTANT(name, val)                  \
  sc_montype_constant(__scs, #name, val);
#define ST_STRUCT(name)                         \
  sc_montype_struct(__scs, #name);
#define ST_FIELD_STR(name, len, kind)                   \
  sc_montype_field(__scs, #name, "str", #kind, #len);
#define ST_FIELD(type, name, kind)                      \
  sc_montype_field(__scs, #name, #type, #kind, NULL);
#define ST_STRUCT_END                           \
  sc_montype_struct_end(__scs);

#include SC_TYPE_TEMPLATE
  sc_montype_flush(scs);

#undef ST_CONSTANT
#undef ST_STRUCT
#undef ST_FIELD_STR
#undef ST_FIELD
#undef ST_STRUCT_END
}

#undef SC_DECLARE_TYPES
#endif


#undef SC_TYPE_TEMPLATE
/** @}*/
