/*
** (c) Copyright 2012-2019 Xilinx, Inc. All rights reserved.
**
** This file contains confidential and proprietary information
** of Xilinx, Inc. and is protected under U.S. and
** international copyright and other intellectual property
** laws.
**
** DISCLAIMER
** This disclaimer is not a license and does not grant any
** rights to the materials distributed herewith. Except as
** otherwise provided in a valid license issued to you by
** Xilinx, and to the maximum extent permitted by applicable
** law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
** WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
** AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
** BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
** INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
** (2) Xilinx shall not be liable (whether in contract or tort,
** including negligence, or under any other theory of
** liability) for any loss or damage of any kind or nature
** related to, arising under or in connection with these
** materials, including for any direct, or any indirect,
** special, incidental, or consequential loss or damage
** (including loss of data, profits, goodwill, or any type of
** loss or damage suffered as a result of any action brought
** by a third party) even if such damage or loss was
** reasonably foreseeable or Xilinx had been advised of the
** possibility of the same.
**
** CRITICAL APPLICATIONS
** Xilinx products are not designed or intended to be fail-
** safe, or for use in any application requiring fail-safe
** performance, such as life-support or safety devices or
** systems, Class III medical devices, nuclear facilities,
** applications related to the deployment of airbags, or any
** other applications that could lead to death, personal
** injury, or severe property or environmental damage
** (individually and collectively, "Critical
** Applications"). Customer assumes the sole risk and
** liability of any use of Xilinx products in Critical
** Applications, subject only to applicable laws and
** regulations governing limitations on product liability.
**
** THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
** PART OF THIS FILE AT ALL TIMES
*/

/**
 * \file
 * \brief sc_vi: Supports receiving packets from the network.
 */

#ifndef __SOLAR_CAPTURE_VI_H__
#define __SOLAR_CAPTURE_VI_H__


struct sc_attr;
struct sc_thread;
struct sc_session;
struct sc_vi_group;
struct sc_node;
/**
 * \struct sc_vi
 * \brief A VI object.
 *
 * Fields in this structure are not exposed, and must not be directly 
 * accessed. Instead use the functions in vi.h.
 */
struct sc_vi;
struct sc_stream;

/**
 * \brief Allocate a VI instance.
 *
 * \param vi_out          The allocated VI is returned here
 * \param attr            Attributes
 * \param thread          The thread the VI will be in
 * \param interface       The network interface to receive packets from
 *
 * \return 0 on success, or a negative error code.
 *
 * A VI is a "virtual network interface" and supports receiving packets
 * from the network.  Packets received by a VI are passed to nodes
 * (sc_node) for processing.
 */
extern int sc_vi_alloc(struct sc_vi** vi_out, const struct sc_attr* attr,
                       struct sc_thread* thread, const char* interface);

/**
 * \brief Set the node a VI should deliver its received packets to.
 *
 * \param vi              The VI receiving packets
 * \param node            The node to deliver packets to
 * \param name_opt        Optional ingress port name (may be NULL)
 *
 * \return 0 on success, or a negative error code.
 *
 * Since SolarCapture 1.1, if @p node is in a different thread from @p vi, then
 * this function automatically creates a link between the threads using
 * mailboxes.
 */
extern int sc_vi_set_recv_node(struct sc_vi* vi, struct sc_node* node,
                               const char* name_opt);

/**
 * \brief Direct a packet stream to a VI.
 *
 * \param vi              The VI receiving packets
 * \param stream          The packet stream
 *
 * \return 0 on success, or a negative error code.
 *
 * Arrange for the packet stream identified by @p stream to be copied or
 * steered to @p vi.
 */
extern int sc_vi_add_stream(struct sc_vi* vi, struct sc_stream* stream);

/**
 * \brief Return the thread associated with a VI.
 *
 * \param vi              The VI
 *
 * \return The thread associated with the VI.
 */
extern struct sc_thread* sc_vi_get_thread(const struct sc_vi* vi);


#if SC_API_VER >= 3
/**
 * \brief Return the name of the network interface associated with a VI.
 *
 * \param vi              The VI
 *
 * \return The name of the network interface associated with the
 * ::sc_vi object
 *
 * This call returns the name of the network interface associated with the
 * ::sc_vi object.  This can be different from the interface name used to
 * create the ::sc_vi when application clustering is used.
 *
 * The network interface name is most often needed so that the application
 * can create an injector on the same interface as a VI.
 */
extern const char* sc_vi_get_interface_name(const struct sc_vi* vi);
#endif


/**
 * \brief Allocate a VI group.
 *
 * \param vi_out          The allocated VI is returned here
 * \param attr            Attributes
 * \param session         The SolarCapture session
 * \param interface       The network interface to receive packets from
 * \param num_vis         The number of VIs in the group
 *
 * \return 0 on success, or a negative error code.
 *
 * A VI group provides a way to distribute packet capture over multiple
 * threads.  A VI group consists of a set of VIs, each of which receives a
 * distinct subset of the streams directed at the group.
 *
 * Streams are directed to a group by calling ::sc_vi_group_add_stream().
 *
 * While a VI allocated from a group receives packets from streams directed
 * to the group (::sc_vi_group_add_stream()), it is also possible to use
 * ::sc_vi_add_stream() to direct a specific stream to a specific member of
 * the group.
 */
extern int sc_vi_group_alloc(struct sc_vi_group** vi_out,
                             const struct sc_attr* attr,
                             struct sc_session* session,
                             const char* interface, int num_vis);

/**
 * \brief Return the session associated with a VI group.
 *
 * \param vi_group        The VI group
 *
 * \return The session associated with the VI group.
 */
extern struct sc_session*
  sc_vi_group_get_session(const struct sc_vi_group* vi_group);

/**
 * \brief Allocate a VI instance from a VI group.
 *
 * \param vi_out          The allocated VI is returned here
 * \param attr            Attributes
 * \param thread          The thread the VI will be in
 * \param vi_group        The VI group
 *
 * \return 0 on success, or a negative error code.
 *
 * See also ::sc_vi_group_alloc() and ::sc_vi_alloc().
 */
extern int sc_vi_alloc_from_group(struct sc_vi** vi_out,
                                  const struct sc_attr* attr,
                                  struct sc_thread* thread,
                                  struct sc_vi_group* vi_group);

/**
 * \brief Direct a packet stream to a group of VIs.
 *
 * \param vi_group        The VI group receiving packets
 * \param stream          The packet stream
 *
 * \return The session associated with the VI group.
 *
 * Arrange for the packet stream identified by @p stream to be copied or
 * steered to the VIs that comprise @p vi_group.
 *
 * Note that packets are spread over the VIs in a group by computing a hash
 * on the addresses in the packet headers.  Normally the hash is computed
 * over the IP addresses, and for TCP packets also the port numbers.  The
 * hash selects a VI within the group, so that packets with the same
 * addresses are consistently delivered to the same VI.
 *
 * If @p stream identifies a set of packets that all have the same source and
 * destination IP addresses (and ports in the case of TCP) then they will
 * all be received by a single VI.
 */
extern int sc_vi_group_add_stream(struct sc_vi_group* vi_group, struct sc_stream* stream);


#endif  /* __SOLAR_CAPTURE_VI_H__ */
/** @} */
