/*
** (c) Copyright 2012-2019 Xilinx, Inc. All rights reserved.
**
** This file contains confidential and proprietary information
** of Xilinx, Inc. and is protected under U.S. and
** international copyright and other intellectual property
** laws.
**
** DISCLAIMER
** This disclaimer is not a license and does not grant any
** rights to the materials distributed herewith. Except as
** otherwise provided in a valid license issued to you by
** Xilinx, and to the maximum extent permitted by applicable
** law: (1) THESE MATERIALS ARE MADE AVAILABLE "AS IS" AND
** WITH ALL FAULTS, AND XILINX HEREBY DISCLAIMS ALL WARRANTIES
** AND CONDITIONS, EXPRESS, IMPLIED, OR STATUTORY, INCLUDING
** BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY, NON-
** INFRINGEMENT, OR FITNESS FOR ANY PARTICULAR PURPOSE; and
** (2) Xilinx shall not be liable (whether in contract or tort,
** including negligence, or under any other theory of
** liability) for any loss or damage of any kind or nature
** related to, arising under or in connection with these
** materials, including for any direct, or any indirect,
** special, incidental, or consequential loss or damage
** (including loss of data, profits, goodwill, or any type of
** loss or damage suffered as a result of any action brought
** by a third party) even if such damage or loss was
** reasonably foreseeable or Xilinx had been advised of the
** possibility of the same.
**
** CRITICAL APPLICATIONS
** Xilinx products are not designed or intended to be fail-
** safe, or for use in any application requiring fail-safe
** performance, such as life-support or safety devices or
** systems, Class III medical devices, nuclear facilities,
** applications related to the deployment of airbags, or any
** other applications that could lead to death, personal
** injury, or severe property or environmental damage
** (individually and collectively, "Critical
** Applications"). Customer assumes the sole risk and
** liability of any use of Xilinx products in Critical
** Applications, subject only to applicable laws and
** regulations governing limitations on product liability.
**
** THIS COPYRIGHT NOTICE AND DISCLAIMER MUST BE RETAINED AS
** PART OF THIS FILE AT ALL TIMES
*/

/**
 * \file
 * \brief Functions for managing time.
 */

#ifndef __SOLAR_CAPTURE_TIME_H__
#define __SOLAR_CAPTURE_TIME_H__


struct sc_attr;
struct sc_thread;


/**
 * \brief Request a callback at a given time
 *
 * \param cb              A callback object
 * \param time            Time at which callback is requested
 *
 * The callback @p cb will be invoked at or after the specified time.  If the
 * time is in the past, then the handler function will be invoked as soon
 * as possible.
 *
 * The time is relative to the system realtime clock (CLOCK_REALTIME),
 * which is the same clock returned by ::sc_thread_get_time().
 */
extern void sc_timer_expire_at(struct sc_callback* cb,
                               const struct timespec* time);

/**
 * \brief Request a callback in the future
 *
 * \param cb              A callback object
 * \param delta_ns        How far in the future in nanoseconds.
 *
 * The callback will be invoked at or after the specified time delta in
 * nanoseconds.  If @p delta_ns is zero or negative then the handler function
 * will be invoked as soon as possible.
 */
extern void sc_timer_expire_after_ns(struct sc_callback* cb, int64_t delta_ns);

/**
 * \brief Push the expiry time further into the future
 *
 * \param cb              A callback object
 * \param delta_ns        How far in the future in nanoseconds.
 *
 * This function pushes the expiry time of a timer callback further into
 * the future.
 *
 * The callback @p cb must either be a currently active timer registered with
 * ::sc_timer_expire_at() or ::sc_timer_expire_after_ns(), or it must be an
 * inactive timer.  ie. The most recent use of @p cb must have been as a
 * timer callback.
 *
 * If @p cb is active, then it is rescheduled at its current expiry time plus
 * @p delta_ns.  If it is not active then it is scheduled at its previous
 * expiry time plus @p delta_ns.
 */
extern void sc_timer_push_back_ns(struct sc_callback* cb, int64_t delta_ns);

/**
 * \brief Return the expiry time of a timer callback.
 *
 * \param cb              A callback object
 * \param ts_out          The expiry time is returned here
 * \return                Zero if @p cb is a timer else -1
 */
extern int sc_timer_get_expiry_time(const struct sc_callback* cb,
                                    struct timespec* ts_out);

/**
 * \brief Convert a timespec struct to nanoseconds
 * \param ts              The timespec struct to convert
 * \return                Time in nanoseconds
 */
static inline uint64_t sc_ns_from_ts(const struct timespec* ts)
{
  return (uint64_t) ts->tv_sec * 1000000000 + ts->tv_nsec;
}

/**
 * \brief Convert a timeval struct to nanoseconds
 * \param tv              The timeval struct to convert
 * \return                Time in nanoseconds
 */
static inline uint64_t sc_ns_from_tv(const struct timeval* tv)
{
  return (uint64_t) tv->tv_sec * 1000000000 + (uint64_t) tv->tv_usec * 1000;
}


/**
 * \brief Convert milliseconds to nanoseconds
 * \param ms              The time in milliseconds to convert
 * \return                Time in nanoseconds
 */
static inline uint64_t sc_ns_from_ms(uint64_t ms)
{
  return ms * 1000000;
}


/**
 * \brief Convert microseconds to nanoseconds
 * \param us              The time in microseconds to convert
 * \return                Time in nanoseconds
 */
static inline uint64_t sc_ns_from_us(uint64_t us)
{
  return us * 1000;
}


#endif  /* __SOLAR_CAPTURE_TIME_H__ */
/** @} */
